package com.sonicsw.mf.common.metrics.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import com.sonicsw.mf.common.metrics.IAggregateMetric;
import com.sonicsw.mf.common.metrics.IAggregateMetricsData;
import com.sonicsw.mf.common.metrics.IAlert;
import com.sonicsw.mf.common.metrics.IHistoricalMetric;
import com.sonicsw.mf.common.metrics.IHistoricalMetricsData;
import com.sonicsw.mf.common.metrics.IMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricsData;
import com.sonicsw.mf.common.metrics.MetricsFactory;

/**
 * Class is not for reference and construction, rather you should use the factory and interface.
 *
 * @see com.sonicsw.mf.common.metrics.MetricsFactory#createMetricsData(short)
 * @see com.sonicsw.mf.common.metrics.IMetric
 */
public final class MetricsData
implements IMetricsData, IHistoricalMetricsData, IAggregateMetricsData, Serializable
{
    private static final long serialVersionUID = 3480759768086297444L;
    private static final short m_serialVersion = 0;

    private static final short NORMAL_TYPES = 0;
    private static final short HISTORICAL_TYPES = 1;
    private static final short AGGREGATE_TYPES = 2;

    private short m_metricTypes = NORMAL_TYPES;
    private long m_timestamp;
    private Object[] m_metrics;

    /**
     * For serialization use only.
     */
    public MetricsData() { }

    public MetricsData(IMetric[] metrics, long currencyTimestamp)
    {
        if (metrics == null)
        {
            throw new IllegalArgumentException("Metrics array cannot be null.");
        }

        if (metrics instanceof IHistoricalMetric[])
        {
            m_metricTypes = HISTORICAL_TYPES;
        }
        else if (metrics instanceof IAggregateMetric[])
        {
            m_metricTypes = AGGREGATE_TYPES;
        }
        m_timestamp = currencyTimestamp;
        m_metrics = metrics;
    }

    public MetricsData(IAggregateMetric[] metrics, long currencyTimestamp)
    {
        if (metrics == null)
        {
            throw new IllegalArgumentException("Metrics array cannot be null.");
        }

        m_metricTypes = AGGREGATE_TYPES;
        m_timestamp = currencyTimestamp;
        m_metrics = metrics;
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricsData#getCurrencyTimestamp()
     */
    @Override
    public long getCurrencyTimestamp()
    {
        if (m_metricTypes == HISTORICAL_TYPES)
        {
            throw new IllegalStateException("Not supported for historic metrics.");
        }

        return m_timestamp;
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricsData#getMetrics()
     */
    @Override
    public IMetric[] getMetrics()
    {
        if (m_metricTypes != NORMAL_TYPES)
        {
            throw new IllegalStateException("Not supported for other than normal metrics.");
        }

        return (IMetric[])m_metrics;
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IHistoricalMetricsData#getHistoricalMetrics()
     */
    @Override
    public IHistoricalMetric[] getHistoricalMetrics()
    {
        if (m_metricTypes != HISTORICAL_TYPES)
        {
            throw new IllegalStateException("Not supported for non-historic metrics.");
        }

        IHistoricalMetric[] historicalMetrics = new IHistoricalMetric[m_metrics.length];
        System.arraycopy(m_metrics, 0, historicalMetrics, 0, m_metrics.length);

        return historicalMetrics;
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IAggregateMetricsData#getAggregateMetrics()
     */
    @Override
    public IAggregateMetric[] getAggregateMetrics()
    {
        if (m_metricTypes != AGGREGATE_TYPES)
        {
            throw new IllegalStateException("Not supported for non-aggregate metrics.");
        }

        IAggregateMetric[] aggregateMetrics = new IAggregateMetric[m_metrics.length];
        System.arraycopy(m_metrics, 0, aggregateMetrics, 0, m_metrics.length);

        return aggregateMetrics;
    }


    //
    // Serialization
    //

    // You can add to these, but never remove
    private static final short TIMESTAMP_FIELD = 0;
    private static final short TYPES_FIELD = 1;
    private static final short METRIC_COUNT = 2;
    private static final short ID_NAME_FIELD = 3;
    private static final short ID_HASH_FIELD = 4;
    private static final short VALUE_FIELD = 5;
    private static final short CURRENCY_FIELD = 6;
    private static final short SOURCE_FIELD = 7;
    private static final short SOURCES_FIELD = 8;
    private static final short TRIGGERED_ALERTS_FIELD = 9;
    private static final short VALUES_FIELD = 10;

    // This is done more efficiently than in other places where we employ version
    // handling serialization code .. as there may be many metrics to describe

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        short fieldCount = 3;

        // we know how many fields we will write and serial version will always be written
        stream.writeShort(fieldCount);
        stream.writeShort(m_serialVersion);

        // Metric specific fields

        stream.writeShort(TYPES_FIELD);
        stream.writeShort(m_metricTypes);

        stream.writeShort(TIMESTAMP_FIELD);
        stream.writeLong(m_timestamp);

        stream.writeShort(METRIC_COUNT);
        stream.writeInt(m_metrics.length);

        for (int i = 0; i < m_metrics.length; i++)
        {
            IMetricIdentity id = m_metricTypes == AGGREGATE_TYPES ? ((IAggregateMetric)m_metrics[i]).getMetricIdentity() : ((IMetric)m_metrics[i]).getMetricIdentity();
            long hash = ((MetricIdentity)id).getHash();

            short perMetricFieldCount = (short)(hash == 0 ? 3 : 4);
            if (m_metricTypes > NORMAL_TYPES)
            {
                perMetricFieldCount++;
            }
            IAlert[] trigAlerts = null;
            if (m_metricTypes != AGGREGATE_TYPES)
            {
                trigAlerts = ((IMetric)m_metrics[i]).getTriggeredAlerts();
                if (trigAlerts != null)
                {
                    perMetricFieldCount++;
                }
            }
            stream.writeShort(perMetricFieldCount);

            stream.writeShort(ID_NAME_FIELD);
            stream.writeObject(id.getNameComponents());

            if (hash != 0)
            {
                stream.writeShort(ID_HASH_FIELD);
                stream.writeLong(hash);
            }

            if (m_metricTypes == AGGREGATE_TYPES)
            {
                stream.writeShort(VALUES_FIELD);
                long[] values = ((IAggregateMetric)m_metrics[i]).getValues();
                stream.writeShort(values.length);
                for (int j = 0; j < values.length; j++)
                {
                    stream.writeLong(values[j]);
                }
            }
            else
            {
                stream.writeShort(VALUE_FIELD);
                stream.writeLong(((IMetric)m_metrics[i]).getValue());
            }

            stream.writeShort(CURRENCY_FIELD);
            stream.writeLong(m_metricTypes == AGGREGATE_TYPES ? ((IAggregateMetric)m_metrics[i]).getCurrencyTimestamp() : ((IMetric)m_metrics[i]).getCurrencyTimestamp());

            if (trigAlerts != null)
            {
                stream.writeShort(TRIGGERED_ALERTS_FIELD);
                stream.writeInt(trigAlerts.length);
                for (int a=0; a < trigAlerts.length; a++)
                {
                    stream.writeLong(trigAlerts[a].getThresholdValue());
                    stream.writeBoolean(trigAlerts[a].isHighThreshold());
                }
            }

            if (m_metricTypes == HISTORICAL_TYPES)
            {
                stream.writeShort(SOURCE_FIELD);
                stream.writeUTF(((IHistoricalMetric)m_metrics[i]).getSource());
            }

            if (m_metricTypes == AGGREGATE_TYPES)
            {
                stream.writeShort(SOURCES_FIELD);
                stream.writeObject(((IAggregateMetric)m_metrics[i]).getSources());
            }
        }
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        short numFields = stream.readShort();
        short serialVer = stream.readShort();
        for (int i = 0; i < numFields; i++)
        {
            short field = stream.readShort();
            switch (serialVer)
            {
                // case olderVersion<n> ...
                default:
                {
                    if (field == TIMESTAMP_FIELD)
                    {
                        m_timestamp = stream.readLong();
                    }
                    if (field == TYPES_FIELD)
                    {
                        m_metricTypes = stream.readShort();
                    }
                    else if (field == METRIC_COUNT)
                    {
                        int count = stream.readInt();

                        if (m_metricTypes == AGGREGATE_TYPES)
                        {
                            ;
                        }

                        if (m_metricTypes == AGGREGATE_TYPES)
                        {
                            m_metrics = new AggregateMetric[count];
                        }
                        else
                        {
                            m_metrics = new Metric[count];
                        }
                        for (int j = 0; j < count; j++)
                        {
                            String[] nameComponents = null;
                            long hash = 0;
                            long value = 0;
                            long[] values = null;
                            long currencyTimestamp = 0;
                            long[] trigAlertsThresholds = null;
                            boolean[] trigAlertsHiLo = null;
                            String source = null;
                            String[] sources = null;

                            short numSubFields = stream.readShort();
                            for (int k = 0; k < numSubFields; k++)
                            {
                                field = stream.readShort();
                                if (field == ID_NAME_FIELD)
                                {
                                    nameComponents = (String[])stream.readObject();
                                }
                                else if (field == ID_HASH_FIELD)
                                {
                                    hash = stream.readLong();
                                }
                                else if (field == VALUE_FIELD)
                                {
                                    value = stream.readLong();
                                }
                                else if (field == VALUES_FIELD)
                                {
                                    values = new long[stream.readShort()];
                                    for (int l = 0; l < values.length; l++)
                                    {
                                        values[l] = stream.readLong();
                                    }
                                }
                                else if (field == CURRENCY_FIELD)
                                {
                                    currencyTimestamp = stream.readLong();
                                }
                                else if (field == TRIGGERED_ALERTS_FIELD)
                                {
                                    int numAlerts = stream.readInt();
                                    trigAlertsThresholds = new long[numAlerts];
                                    trigAlertsHiLo = new boolean[numAlerts];
                                    for (int a = 0; a < numAlerts; a++)
                                    {
                                        trigAlertsThresholds[a] = stream.readLong();
                                        trigAlertsHiLo[a] = stream.readBoolean();
                                    }
                                }

                                else if (field == SOURCE_FIELD)
                                {
                                    source = stream.readUTF();
                                }
                                else if (field == SOURCES_FIELD)
                                {
                                    sources = (String[])stream.readObject();
                                }
                            }

                            IMetricIdentity id = MetricsFactory.createMetricIdentity(nameComponents);
                            ((MetricIdentity)id).setHash(hash);

                            if (m_metricTypes == AGGREGATE_TYPES)
                            {
                                m_metrics[j] = MetricsFactory.createMetric(sources, id, values, currencyTimestamp);
                            }
                            else
                            {
                                m_metrics[j] = MetricsFactory.createMetric(id, value, currencyTimestamp);
                                if (trigAlertsThresholds != null)
                                {
                                    IAlert[] trigAlerts = new IAlert[trigAlertsThresholds.length];
                                    for (int a = 0; a < trigAlertsThresholds.length; a++)
                                    {
                                        trigAlerts[a] = MetricsFactory.createAlert(id, trigAlertsHiLo[a], trigAlertsThresholds[a]);
                                    }

                                    ((Metric)m_metrics[j]).setTriggeredAlerts(trigAlerts);
                                }

                                if (m_metricTypes == HISTORICAL_TYPES)
                                {
                                    m_metrics[j] = MetricsFactory.createMetric(source, (IMetric)m_metrics[j]);
                                }
                            }
                        }
                    }
                    break;
                }
            }
        }
    }
}
