package com.sonicsw.mf.common.metrics.manager;

/**
 * A historical statistic provides a set of raw statistical values. This metrics manager
 * tells a historical statistic how many values it should maintain and when it should
 * cycle those values.
 *
 * A statistic is interpreted by a metric analyzer to derive a particular metric value.
 *
 * @see IStatistic
 * @see IHistoricalSampledStatistic
 * @see IMetricAnalyzer
 * @see MetricsManager
 */
public interface IHistoricalStatistic
extends IStatistic
{
    /**
     * Returns the buffer of values for which historical information is available. The number
     * of values returned may be less than the number of values that can be recorded; this could
     * occur early in the life cycle of the statistic (i.e. before the buffer overflows) or after
     * a reset of the number of slots for historical data the statistic will hold.
     *
     * Returns null if historical information has not yet been initialized (for example, before
     * first refresh cycle)
     *
     * @see #setNumValues(int)
     */
    public long[] getLastValues();

    /**
     * Returns the buffer of update counts for each refresh interval for which historical information is available.
     *
     * The number of counts returned may be less than the number of counts that can be recorded;
     * this could occur early in the life cycle of the statistic (i.e. before the buffer overflows)
     * or after a reset of the number of slots for historical data the statistic will hold.
     * Historical statistics that do not record the history of the number of updates that occurred
     * in each refresh cycle will return null.
     *
     * @see IStatistic#getLastUpdateCount   note that, in contrast, this method returns the running update count
     *                                      since the last reset()
     */
    public long[] getLastUpdateCounts();

    /**
     * This will be called by the MetricsManager to allocate the number of slots for historical
     * data (as calculated from the refresh and collection intervals of the metrics manager),
     * effectively setting the buffer size. If this is reset at runtime, the implementation
     * should invalidate all existing data.
     */
    public void setNumValues(int numValues);

    /**
     * This will be called by the MetricsManager at the end of each refresh interval. This moves
     * the current value into the buffer, expires the oldest historical value if the buffer is full
     * and reinitializes the current value (if indicated).
     */
    @Override
    public void refresh();

    /**
     * Clears the current and historical statistical data and resets the statistic back to its
     * initial value.
     */
    @Override
    public void reset();
}
