/**
 * Copyright (c) 1998-2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Progress.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mf.common.metrics.manager;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.metrics.IAlert;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IMetricsData;

/**
 * A metrics manager isolates the following aspects:
 * <p>
 *  - the cataloging of what metrics are exposed by a component (via a list of
 *    metrics meta-data) and which metrics are enabled
 *  - the collection of raw statistical data
 *  - the interpretation and presentation of the statistical data as a set of metrics
 * <p>
 * Error handling convention: invalid items are ignored if passed as part of a list of metric identities
 * or alerts, rather than causing an exception.
 */
public interface IMetricsManager
{
    /**
     * Sets the component's context
     */
    public void init(IComponentContext context);

    /**
     * Cleanup the manager instance including stopping any refresh thread
     */
    public void cleanup();

    /**
     * Gets the metric meta-data for all the metrics the component supports.
     */
    public IMetricInfo[] getMetricsInfo();

    /**
     * Gets the current set of metric data for the requested set of active metrics,
     * or for all active metrics.
     * <p>
     * Optionally includes a list of triggered alerts for these metrics.
     *
     * @params ids  a list of active metrics, either individual metrics or instances, or null
     *              to indicate all active metrics
     * @params  returnTriggeredAlerts    whether to include a list of triggered (exceeded) alerts for these metrics
     *
     * @see IMetric#getTriggeredAlerts()
     * @see IMetricsManager#getActiveMetrics()
     */
    public IMetricsData getMetricsData(IMetricIdentity[] ids, Boolean returnTriggeredAlerts);

    /**
     * Gets the set of metrics that are currently active for a list of specified metrics, or all metrics.
     * <p>
     * Active metrics are those metrics, either individual metrics or instance metrics, for which
     * statistics data is currently being collected.
     * <p>
     * This method can be useful to get the active instances of one or more instance metric parents.
     *
     * @param ids   A list of metrics identified by a set of branches on the metrics tree, that is,
     *              metric parents, individual metrics or instance, or null to request all active metrics.
     *              Note that instance patterns are ignored by this method. To obtain enabled instance patterns,
     *              use getEnabledMetrics(IMetricIdentity[]) instead.
     *
     * @return      The expanded list of active metrics, including individual metrics and metric instances.
     *
     * @see IMetricsManager#getEnabledMetrics(IMetricIdentity[])
     */
    public IMetricIdentity[] getActiveMetrics(IMetricIdentity[] ids);

    /**
     * Gets the set of enabled metrics and metric patterns for a list of metrics, or for all metrics.
     * <p>
     * This method can be useful to get the enabled instance patterns of one or more instance metric parents.
     *
     * @param ids   A list of metrics identified by a set of branches on the metrics tree,
     *              including metric parents or individual metrics, or instance patterns, or all enabled metrics if null.
     *
     * @return      The expanded set of enabled metrics, either individual metrics or instance patterns.
     *
     * @see IMetricsManager#getActiveMetrics(IMetricIdentity[])
    */
    public IMetricIdentity[] getEnabledMetrics(IMetricIdentity[] ids);

    /**
     * Enables a list of metrics.
     *
     * @params ids  a list of metrics identified as a combination of branches of the metric
     *              tree (parent nodes), individual metrics or instance patterns.
     *
     * @return an expanded list of individual metrics.
     */
    public IMetricIdentity[] enableMetrics(IMetricIdentity[] ids);

    /**
     * Disables a list of metrics.
     * <p>
     * To disable all metrics call getEnabledMetrics(null) and pass the result to this method.
     *
     * @param ids   a list of metrics to disable
     * @return      the list of metrics actually disabled.
     */
    public IMetricIdentity[] disableMetrics(IMetricIdentity[] ids);

    /**
     * Reinitializes the statistics on which all enabled metrics are based.
     * For non-historical statistics it clears the current statistical data and
     * resets the statistic back to its initial value. For historical statistics
     * it additionally clears the history.
     */
    public void resetMetrics();

    /**
     * Gets the set of alerts that are currently active for a list of specified metrics, or all metrics.
     * <p>
     * Active metrics are those metrics, either individual metrics or instance metrics, for which
     * statistics data is currently being collected.
     * <p>
     *
     * @return      A list of active alerts.
     *
     * @see IMetricsManager#getEnabledMetrics(IMetricIdentity[])
     */
    public IAlert[] getActiveAlerts();

    /**
     * Enables a list of alerts on one or more metrics.
     * <p>
     * An alert to be enabled must specify:
     *  - a metric identity expressed as an individual metric or instance (not a pattern or parent node)
     *  - a threshold value
     *  - a type (high or low)
     *
     * @param alerts    a list of alerts to enable
     * @return          the list of alerts actually enabled
     */
    public IAlert[] enableAlerts(IAlert[] alerts);

    /**
     * Disables a list of alerts on one or more metrics.
     * <p>
     * An alert to be disabled must specify:
     *  - a metric identity expressed as an individual metric or an instance (not a pattern or parent node)
     *  - a threshold value
     *  - a type (high or low)
     * <p>
     * To disable all alerts on a specific metric (individual or instance),
     * call getEnabledAlerts(IMetricIdentity[]) and pass the result to this method.
     *
     * @param alerts    a list of alerts to disable
     * @return          the list of alerts actually disabled
     *
     * @see IMetricsManager#getEnabledAlerts(IMetricIdentity[])
     */
    public IAlert[] disableAlerts(IAlert[] alerts);

    /**
     * Gets the list of enabled alerts for the specified metrics, or for all metrics.
     *
     * @param ids   a list of metrics expressed as parent nodes, individual metrics or instances, or
     *              null to indicate all metrics.
     */
    public IAlert[] getEnabledAlerts(IMetricIdentity[] ids);

    /**
     * Gets the current value of the metrics collection interval, in minutes.
     *
     * @return   the current value of the metrics collection interval (in minutes).
     */
    public long getCollectionInterval();

    /**
     * Sets the current value of the metrics collection interval, in minutes.
     *
     * @param collectionInterval   value of the metrics collection interval (in minutes).
     */
    public void setCollectionInterval(long collectionInterval) throws MFException;

    /**
     * Gets the current value of the metrics refresh interval, in seconds.
     *
     * @return   the current value of the metrics refresh interval (in seconds).
     */
    public long getRefreshInterval();

    /**
     * Sets the current value of the metrics refresh interval, in seconds.
     *
     * @param refreshInterval   value of the metrics refresh interval (in seconds).
     */
    public void setRefreshInterval(long refreshInterval) throws MFException;

    public void setRefreshAndCollectionInterval(long refreshInterval, long collectionInterval)throws MFException;

    /**
     * Gets the current value of the repeat metric alerts flag.
     *
     * @return   the current value of the repeat metric alerts flag.
     */
    public boolean getRepeatMetricAlerts();

    /**
     * Sets the current value of the repeat metric alerts flag.
     *
     * @param repeatMetricAlerts   value of the repeat metric alerts flag.
     */
    public void setRepeatMetricAlerts(boolean repeatMetricAlerts) throws MFException;
}

