package com.sonicsw.mf.common.metrics.manager;

import java.util.Map;

import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.metrics.IMetricIdentity;

/**
 * A metrics registrar isolates the metrics management calls by a component that exposes metrics
 * using the management framework's metrics model.
 * <p>
 * Components obtain a reference to a metrics registrar via their component context.
 *
 * @see com.sonicsw.mf.common.IComponentContext#initMetricsManagement(IMetricInfo[])
 */
public interface IMetricsRegistrar
{
    /**
     * Adds the metric to the list of actively managed metrics. An actively managed metric
     * will appear in the list of enabled metrics and its value accessible when requests for
     * metrics data are made. If the statistic is a sampled statistic, the MetricsManager
     * will add the statistic to the list of statistics to be sampled at the refresh interval.
     *
     * Since a metric analyzer is not supplied, the MetricsManager will looks up the metric's
     * meta-data to find its value type and uses the default analyzer for that metric type.
     *
     * If the metric is already enabled with the same statistic instance then no action will
     * be taken. If the metric is enabled with a different statistic instance then the metric
     * is automatically unregistered, then re-registered with the new statistic.
     * 
     * @param id        The identity of the metric to be registered.
     * @param statistic The statistic to be used to evaluate the metric.
     *
     * @see #registerMetric(IMetricIdentity, IMetricAnalyzer, IStatistic)
     */
    public void registerMetric(IMetricIdentity id, IStatistic statistic);

    /**
     * Allows provision of a user specified analyzer.
     *
     * @param id        The identity of the metric to be registered.
     * @param analyzer  The metrics analyzer to be used to evaluate the metric value.
     * @param statistic The statistic to be used to evaluate the metric.
     *
     * @see #registerMetric(IMetricIdentity, IStatistic)
     */
    public void registerMetric(IMetricIdentity id, IMetricAnalyzer analyzer, IStatistic statistic);

    /**
     * Called by a component to unregister a metric.
     * <p>
     * The component signals that it has disabled a metric, in particular, if an instance has been destroyed.
     * <p>
     * Note: The associated statistic may still be in use by another metric.
     * <p>
     * Unregistering a metric does NOT affect any instance patterns, including a pattern equal to the instance name
     * Patterns are removed by the client calling disableMetric()
     *
     * @param id The identity of the metric to be unregistered.
     */
    public void unregisterMetric(IMetricIdentity id);
    
    /**
     * Called by a component to unregister a metric.  If 'statistic' is null this method is
     * equivalent to unregisterMetric(IMetricIdentity id).  However, if 'statistic' is non-null
     * it is checked against the statistic currently registered with the metric.  If they match
     * (i.e. both reference the same underlying statistic object) the unregister completes,
     * otherwise the method is a no-op.
     * <p>
     * The component signals that it has disabled a metric, in particular, if an instance has been destroyed.
     * <p>
     * Note: The associated statistic may still be in use by another metric.
     * <p>
     * Unregistering a metric does NOT affect any instance patterns, including a pattern equal to the instance name
     * Patterns are removed by the client calling disableMetric()
     *
     * @param id The identity of the metric to be unregistered.
     * @param statistic The statistic believed to be associated with the metric, or null.
     */
    public void unregisterMetric(IMetricIdentity id, IStatistic statistic);    

    /**
     * Called by a component to determine if the instance metric identified by the given metric identity
     * should be enabled.
     *
     * @param id The identity of the metric to be unregistered.
     */
    public boolean isInstanceEnabled(IMetricIdentity id);

    /**
     * Set by a component to specify a the category for alert notifications and a fixed
     * set of notification attributes (name/value pairs) to be added to all alert
     * notifications generated on behalf of the component.
     *
     * @param category   The alert notification category
     * @param attributes The predefined set of alert notification attributes
     */
    public void registerAlertNotificationProperties(short category, Map attributes);

        /**
     * Get the interval at which the manager will:
     *
     *  - request the statistic providers of a sampled statistic to update
     *    the statistic value
     *  - request each historical statistic to cycle its data
     *
     * @return The refresh interval in milliseconds.
     */
    public long getRefreshInterval();

    /**
     * Set the interval at which the manager will:
     *
     *  - request the statistic providers of a sampled statistic to update
     *    the statistic value
     *  - request each historical statistic to cycle its data
     *
     * When reset the number of historical values stored by historical statistics
     * will be recalculated and reset in each historical statistic.
     *
     * @param interval The refresh interval in milliseconds.
     *
     * @see IHistoricalStatistic
     * @see IHistoricalSampledStatistic
     */
    public void setRefreshInterval(long interval)
    throws MFException;

    /**
     * Get the window over which historical statistics will be maintained
     * for historical statistsics.
     *
     * @return The collection interval in milliseconds.
     */
    public long getCollectionInterval();

    /**
     * Set the window over which historical statistics will be maintained
     * for historical statistsics.
     *
     * When reset the number of historical values stored by historical statistics
     * will be recalculated and reset in each historical statistic.
     *
     * @param interval The collection interval in milliseconds.
     *
     * @see IHistoricalStatistic
     * @see IHistoricalSampledStatistic
     */
    public void setCollectionInterval(long interval)
    throws MFException;

    /**
     * Gets the current value of the repeat metric alerts flag.
     *
     * @return   the current value of the repeat metric alerts flag.
     */
    public boolean getRepeatMetricAlerts();

    /**
     * Sets the current value of the repeat metric alerts flag.
     *
     * @param repeatMetricAlerts   value of the repeat metric alerts flag.
     */
    public void setRepeatMetricAlerts(boolean repeatMetricAlerts) throws MFException;
}