package com.sonicsw.mf.common.metrics.manager;

/**
 * A statistic provides a single raw statistical value. This metrics manager has no special
 * treatment for classes which implement this interface (i.e. unlike its subclass interfaces),
 * and the value of the statistic is maintained outside of control of the metrics manager.
 *
 * A statistic is interpreted by a metric analyzer to derive a particular metric value.
 *
 * @see ISampledStatistic
 * @see IHistoricalStatistic
 * @see IHistoricalSampledStatistic
 * @see IMetricAnalyzer
 * @see MetricsManager
 */
public interface IStatistic
{
    /**
     * A statistic with this mode is a simple value that is a straight forward representation of
     * the updated value.
     */
    public static final short VALUE_MODE = 0;
    /**
     * A statistic with this mode is a value that represents the difference between the last two
     * recorded values (or the initial value if two updates have not been recorded).
     *
     * @see #setInitialValue(long)
     */
    public static final short DIFFERENCE_MODE = 1;
    /**
     * A statistic with this mode interprets updates as increments (positive values) or decrements
     * (negative values) to the current value.
     */
    public static final short COUNTER_MODE = 2;
    /**
     * A statistic with this mode is a value that represents the maximum value amongst the update values
     * applied to this statistic.
     */
    public static final short MAXIMUM_MODE = 3;
    /**
     * A statistic with this mode is a value that represents the minimum value amongst the update values
     * applied to this statistic.
     */
    public static final short MINIMUM_MODE = 4;
    /**
     * A statistic with this mode interprets updates as increments (positive values) or decrements
     * (negative values) to the current value and evaluates the maximum within a refresh cycle.
     */
    public static final short MAXIMUM_COUNTER_MODE = 5;
    /**
     * A statistic with this mode interprets updates as increments (positive values) or decrements
     * (negative values) to the current value and evaluates the minimum within a refresh cycle.
     */
    public static final short MINIMUM_COUNTER_MODE = 6;

    /**
     * Update the statistic value.
     */
    public void updateValue(long value);

    /**
     * Gets the most current statistical value of this statistic. For a non-sampled statistic
     * the current value is a real time value.
     * If this statistic is sampled, the current value is equal to the last refreshed value.
     *
     * @return The current value of the statistic.
     */
    public long getCurrentValue();

    /**
     * Gets the value of this statistic as captured at the expiration of the most recent refresh interval.
     * If interval mode is specified, returns the difference between the value at the end of the last refresh
     * interval and the value at the previous refresh interval.
     * @return The value of the statistic as recorded at the last refresh.
     */
    public long getLastValue();

    /**
     * Gets the real-time running count of the number of updates since the last reset().
     */
    public long getLastUpdateCount();

    /**
     * For a non-sampled, non-historical statistic, returns the current time.
     * Else, gets the time the statistic was last refreshed, or -1 if a complete refresh cycle has
     * not yet been made.
     */
    public long getCurrencyTimestamp();

    /**
     * Gets actual duration of the last refresh cycle applied to this statistic.
     */
    public long getLastRefreshCycleDuration();

    /**
     * At the end of each refresh interval, the MetricsManager updates all sampled statistics,
     * then calls this method for all statistics.
     */
    public void refresh();

    /**
     * Set the initial value to be assigned to the statistic after its creation or when reset() is called.
     * Any initial value must be set prior to the first update after creation or reset().
     *
     * @see #updateValue(long)
     */
    public void setInitialValue(long value);

    /**
     * Clears the current statistical data and resets the statistic back to its initial value.
     */
    public void reset();
}
