package com.sonicsw.mf.common.metrics.manager.impl;

import com.sonicsw.mf.common.metrics.manager.IHistoricalStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.metrics.manager.StatisticsFactory;

/**
 * The default implementation of IHistoricalStatistic.
 *
 * @see MetricsManager#createStatistic(short, IStatisticProvider[], boolean)
 */
public class HistoricalStatistic
extends Statistic
implements IHistoricalStatistic
{
    // the array of values recorded during the collection interval
    protected long[] m_historyValues;
    protected int m_beginIndex;
    protected int m_endIndex;
    protected boolean m_newHistory = true;  // new history has been initialized - wait 2 refreshes to start recording

    protected IHistoricalStatistic m_updateCountsStatistic;

    private boolean DEBUG = false;

    /**
     * Creates a historical statistic with the given mode (see <>_MODE values).
     *
     * @param updateMode       The update mode of the statistic value
     * @param intervalMode     If true, the value as of the last refresh interval is returned as the
     *                         difference between the actual value at the last refresh interval and the value
     *                         as of the previous refresh interval.
     * @param saveUpdateCounts In addition to recording the values at the end of each refesh interval, record
     *                         the number of updates that occured in that interval.
     */
    public HistoricalStatistic(short updateMode, boolean intervalMode, boolean saveUpdateCounts)
    {
        super(updateMode, intervalMode);
        m_beginIndex = -1;
        if (saveUpdateCounts)
        {
            // always keep count with intervalMode true (= interval diff)
            m_updateCountsStatistic = (IHistoricalStatistic)StatisticsFactory.createStatistic(IStatistic.VALUE_MODE, true, null, (short)1);
        }
     }

    /**
     * Override to always return the time of the most recent refresh interval
     * @see IStatistic#getCurrencyTimestamp()
     */
    @Override
    public synchronized long getCurrencyTimestamp() {  return m_currencyTimestamp;}

    /**
     * @see IHistoricalStatistic#getLastValues()
     */
    @Override
    public synchronized long[] getLastValues()
    {
        if (m_historyValues == null)
        {
            throw new IllegalStateException("Number of values not yet set.");
        }

        if (m_beginIndex < 0)
        {
            return null;
        }

        long[] values = null;

        if (m_beginIndex > m_endIndex) // wrapped (normal case)
        {
            int tailCount = m_historyValues.length - m_beginIndex;
            int headCount = m_endIndex + 1;
            values = new long[tailCount + headCount];
            System.arraycopy(m_historyValues, m_beginIndex, values, 0, tailCount);
            System.arraycopy(m_historyValues, 0, values, tailCount, headCount);
        }
        else if (m_beginIndex < m_endIndex) // not wrapped
        {
            int count = m_endIndex - m_beginIndex + 1;
            values = new long[count];
            System.arraycopy(m_historyValues, m_beginIndex, values, 0, count);
        }
        else
        {
            values = new long[] { m_historyValues[m_beginIndex] };
        }

        return values;
    }

    /**
     * @see IHistoricalStatistic#getLastUpdateCounts()
     */
    @Override
    public long[] getLastUpdateCounts()
    {
        if (m_updateCountsStatistic == null)
        {
            return null;
        }

        return m_updateCountsStatistic.getLastValues();
    }

    /**
     * @see IHistoricalStatistic#setNumValues(int)
     */
    @Override
    public void setNumValues(int numValues)
    {
        if (numValues < 2)
        {
            throw new IllegalArgumentException("Number of values must be > 1.");
        }

        if (DEBUG)
        {
            System.out.println(System.currentTimeMillis() + " (" + m_name + ") HistoricalStatistic: setNumValues = " + numValues);
        }
        synchronized(this)
        {
            m_beginIndex = m_endIndex = -1;
            m_historyValues = new long[numValues];
            if (m_updateCountsStatistic != null)
            {
                m_updateCountsStatistic.setNumValues(numValues);
            }

            // invalidate current and last interval values
            m_currentIntervalValue = m_lastIntervalValue = m_initialValue;
            m_lastCurrencyTimestamp = m_currencyTimestamp = INVALID_TIME;

            // mark history as new, so that first refresh interval can be skipped
            m_newHistory = true;
        }
    }

    /**
     * @see IHistoricalStatistic#refresh()
     */
    @Override
    public void refresh()
    {
        if (m_historyValues == null)
        {
            throw new IllegalStateException("Number of values not yet set.");
        }

        synchronized(this)
        {
            super.refresh();

            if (m_updateCountsStatistic != null)
            {
                m_updateCountsStatistic.updateValue(getLastUpdateCount());
                m_updateCountsStatistic.refresh();
            }

            // Start fresh calculation of maximum or minimum for this interval
            if (m_updateMode == MAXIMUM_MODE || m_updateMode == MINIMUM_MODE)
            {
                m_currentUpdateValue = m_lastUpdateValue;
            }

            // wait until 2d refresh to record history (for differences)
            if (m_newHistory)
            {
                m_newHistory = false;
                return;
            }

            if (m_beginIndex > m_endIndex) // (normal case)
            {
                m_beginIndex--;
                m_endIndex--;

                if (m_endIndex < 0)
                {
                    m_endIndex = m_historyValues.length - 1;
                }
            }
            else if (m_beginIndex == -1)
            {
                m_beginIndex = m_endIndex = m_historyValues.length - 1;
            }
            else
            {
                m_beginIndex--;
                if (m_beginIndex < 0) // wrap the head
                {
                    m_beginIndex = m_historyValues.length - 1;
                    m_endIndex--;
                    if (DEBUG)
                    {
                        System.out.println(" HistoricalStatistic.refresh *** Collection interval complete!! " + m_name);
                        printValues();
                    }
                }
            }

            m_historyValues[m_beginIndex] = getLastValue();
        }
    }

    /**
     * @see IHistoricalStatistic#reset()
     */
    @Override
    public synchronized void reset()
    {
        // re-initialize update values and timestamps
        super.reset();

        // re-initialize refresh interval values, counts
        m_currentIntervalValue = m_lastIntervalValue = m_currentUpdateValue;
        if (m_historyValues != null)
        {
            setNumValues(m_historyValues.length);
        }
    }

    // debug
    synchronized void printValues()
    {
        System.out.println(System.currentTimeMillis() + " (" + m_name + ") History values: begin = " + m_beginIndex + " end = " + m_endIndex);
        String values = " ";
        for (int i=0; i < m_historyValues.length; i++)
        {
            values += " " +  m_historyValues[i];
            if (values.length() > 72)
            {
                System.out.println(values);
                values = " ";
            }
        }
        System.out.println(values);
    }

    @Override
    public void setName(String name)
    {
        super.setName(name);
        if (m_updateCountsStatistic != null)
        {
            ((Statistic)m_updateCountsStatistic).m_name = "Count for " + m_name;
        }
    }

    @Override
    public String toString() { return "HistoricalStatistic@" + Integer.toHexString(this.hashCode()) + ": name=" + m_name + ", updateMode=" + m_updateMode + ", intervalMode=" + m_intervalMode ;}
}
