package com.sonicsw.mf.common.metrics.manager.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.metrics.IAlert;
import com.sonicsw.mf.common.metrics.IMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IMetricsData;
import com.sonicsw.mf.common.metrics.IValueType;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.impl.Alert;
import com.sonicsw.mf.common.metrics.impl.Metric;
import com.sonicsw.mf.common.metrics.impl.MetricIdentity;
import com.sonicsw.mf.common.metrics.impl.MetricInfo;
import com.sonicsw.mf.common.metrics.manager.IHistoricalStatistic;
import com.sonicsw.mf.common.metrics.manager.IMetricAnalyzer;
import com.sonicsw.mf.common.metrics.manager.IMetricsManager;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.metrics.manager.ISampledStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatisticProvider;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.sdf.AbstractDiagnosticsProvider;
import com.sonicsw.sdf.DiagnosticsManagerAccess;
import com.sonicsw.sdf.IDiagnosticsManager;
import com.sonicsw.sdf.IStateWriter;
import com.sonicsw.sdf.ITracer;

/**
 * The MetricsManager provides a default implementation of the IMetricsManager interface.
 *
 * Most components should use this default metrics manager implementation unless they have
 * specialized requirements that cannot be satisfied by this implementation. Components should
 * obtain an instance of this class using the MetricsFactory class.
 */
public class MetricsManager
implements IMetricsManager, IMetricsRegistrar
{
    // SDF diagnostics
    private static final int SDF_TRACE_NOTHING = 0;
    private static final int SDF_TRACE_VERBOSE = 1;
    private static final int SDF_TRACE_REGISTRATION_COUNTS = 2;

    private static final String SDF_DOIID_STATISTICS = "Statistics";
    private static final String SDF_DOIID_METRICS = "Metrics";

    public static int m_sdfStatisticsTraceMask = SDF_TRACE_NOTHING;
    public static int m_sdfMetricsTraceMask = SDF_TRACE_NOTHING;

    private long m_statisticRegistrations = -1; // don't count m_elapsedCollectionTimeStatistic
    private long m_statisticUnregistrations = 0;
    private long m_metricRegistrations = 0;
    private long m_metricUnregistrations = 0;
    
    private static HashSet m_metricsManagers = new HashSet();
    
    private static IDiagnosticsManager m_diagnosticsManager;
    private static MetricsManagerDiagnostics m_metricsManagerDiagnostics;
    // end SDF diagnostics
    
    private static boolean DEBUG_CONFIG = false;
    private static boolean DEBUG_METRICS = false; 
    private static boolean DEBUG_STATS = false;
    private static boolean DEBUG_ALERTS = false;
    private static boolean DEBUG_AGENT = false;

    private static String METRICID_NONNULL = "Metric identity cannot be null";
    private static String STATISTIC_NONNULL = "Statistic cannot be null";
    private IComponentContext m_mfContext;
    private String m_componentName = "";
    private boolean m_isAgent;

    private boolean m_isClosing = false;

    protected HashMap m_metricInfos = new HashMap();    // key = id, value = info
    protected HashMap m_activeMetrics = new HashMap(); // key = id, value = registered metric: statistic + analyzer + hash
    protected HashSet m_statistics = new HashSet();     // obj = statistic
    protected HashSet m_enabledMetricsPatterns = new HashSet();   // obj = instance patterns
    protected HashMap m_enabledAlerts = new HashMap();   // key = id, value = Set(Alerts)
    protected HashSet m_activeAlerts = new HashSet();   // obj = Alert
    protected HashSet m_idleAlerts = new HashSet();

    protected long m_currentMetricId = 0;               // generated hash for registered metrics
    private Map m_alertNotificationAttributes = null;  // set by component
    private short m_alertNotificationCategory = INotification.SYSTEM_CATEGORY;   // set by component

    // the time the last refresh cycle occurred representing the currency
    // of the last values held in a statistic
    protected long m_currencyTimestamp;
    // the elasped collection time statistic and its analyzer
    protected HistoricalStatistic m_elapsedCollectionTimeStatistic;
    protected IMetricAnalyzer m_elapsedCollectionTimeAnalyzer;
    protected Timer m_intervalTimer;
    // controls on the interval settings are the min/max below +
    // rule that: collection / refresh <= 250 (i.e. max slots is 250)
    protected static final long MIN_REFRESH_INTERVAL = 5000; // min 5 secs
    protected static final long MAX_REFRESH_INTERVAL = 900000; // max 15 minutes
    protected static final long MIN_COLLECTION_INTERVAL = 60000; // min 1 minute
    protected static final long MAX_COLLECTION_INTERVAL = 86400000; // max 24 hours
    protected static final int MAX_HISTORY_VALUES = 250;

    protected long m_refreshInterval = (long) (IContainerConstants.REFRESH_INTERVAL_DEFAULT * 1000);  //convert to milliseconds
    protected long m_collectionInterval = (long) (IContainerConstants.COLLECTION_INTERVAL_DEFAULT * 60000); //convert to seconds
    protected boolean m_repeatAlerts = IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_DEFAULT;
    protected int m_numHistoricalValues = (int)(m_collectionInterval / m_refreshInterval);

    protected static final IMetricInfo[] EMPTY_INFO_ARRAY = new IMetricInfo[0];
    protected static final IMetric[] EMPTY_METRIC_ARRAY = new IMetric[0];
    protected static final IAlert[] EMPTY_ALERT_ARRAY = new IAlert[0];

    // index by metric value type into this table
    protected IMetricAnalyzer[] m_defaultAnalyzers = new IMetricAnalyzer[IValueType.NUM_VALUE_TYPES];

    static
    {
        m_diagnosticsManager = DiagnosticsManagerAccess.createManager();
        m_metricsManagerDiagnostics = new MetricsManagerDiagnostics();
        m_metricsManagerDiagnostics.register();
    }

    /**
     * Creates the default metrics manager implementation.
     *
     * @param metricInfos An array of meta-data for all the metrics the component is capable of
     *                    supporting.
     */
    public MetricsManager(IMetricInfo[] metricInfos)
    {
        for (int i = 0; i < metricInfos.length; i++)
        {
            if (DEBUG_CONFIG)
            {
                System.out.println("MetricsManager " + metricInfos[i]);
            }
            m_metricInfos.put(metricInfos[i].getMetricIdentity(), metricInfos[i]);

            // static metrics are always enabled
            if (!metricInfos[i].isDynamic())
            {
                m_enabledMetricsPatterns.add(metricInfos[i].getMetricIdentity());
            }
        }
        
        MetricsManager.m_metricsManagers.add(this);
    }
    
    /**
     * Sets the component's context
     */
    @Override
    public synchronized void init(IComponentContext mfContext)
    {
        if (m_mfContext != null)
        {
            throw new IllegalStateException("Context already set");
        }

        m_mfContext = mfContext;
        m_componentName = m_mfContext.getComponentName().getComponentName();
        m_isAgent = m_componentName.equals("AGENT");

        // create a statistic and anaylzer to calculate the elapsed collection time
        m_elapsedCollectionTimeAnalyzer = getMetricAnalyzer(IValueType.TOTAL);
        m_elapsedCollectionTimeStatistic = new HistoricalSampledStatistic(Statistic.VALUE_MODE, true, false,
                new IStatisticProvider[] { new IStatisticProvider()
                                            {
                                                @Override
                                                public void updateStatistic(ISampledStatistic statistic)
                                                {
                                                    statistic.updateValue(System.currentTimeMillis());
                                                }
                                                @Override
                                                public void resetStatistic(ISampledStatistic statistic)
                                                {
                                                    statistic.updateValue(System.currentTimeMillis());
                                                }
                                            }});
       m_elapsedCollectionTimeStatistic.m_name = "TimeKeeper";  // for debug
       registerStatistic(m_elapsedCollectionTimeStatistic);
    }

    /**
     * Cleanup the manager instance including stopping any refresh thread
     */
    @Override
    public synchronized void cleanup()
    {
        MetricsManager.m_metricsManagers.remove(this);

        m_isClosing = true;
        if (m_intervalTimer != null)
        {
            m_intervalTimer.cancel();
        }

        m_activeMetrics.clear();
        m_statistics.clear();
        m_enabledMetricsPatterns.clear();
        m_enabledAlerts.clear();
        m_activeAlerts.clear();
        if (m_alertNotificationAttributes != null)
        {
            m_alertNotificationAttributes.clear();
        }
    }

    /**
     * Set by the component: optional alert notification category and attributes
     */
    @Override
    public void registerAlertNotificationProperties(short category, Map attributes)
    {
        m_alertNotificationCategory = category;
        m_alertNotificationAttributes = attributes;
    }
    
    public int getStatisticCount()
    {
        return m_statistics.size();
    }

    //
    // IMetricsManager implementations
    //

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricsManager#getMetricsInfo()
     */
    @Override
    public IMetricInfo[] getMetricsInfo() { return internalGetMetricsInfo(false); }

    /**
     * Returns metrics data for the specified list of metrics, or all metrics if the list is null.
     * @see com.sonicsw.mf.common.metrics.IMetricsManager#getMetricsData(IMetricIdentity[])
     */
    @Override
    public synchronized IMetricsData getMetricsData(IMetricIdentity[] ids, Boolean getTriggeredAlerts)
    {
         return internalGetMetricsData(ids, getTriggeredAlerts, false);
    }


    /**
     * Gets the list of all matching active metrics.
     *
     * @see com.sonicsw.mf.common.metrics.IMetricsManager#getActiveMetrics()
     */
    @Override
    public synchronized IMetricIdentity[] getActiveMetrics(IMetricIdentity[] ids)
    {
        return internalGetActiveMetrics(ids, false);
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricsManager#resetMetrics()
     */
    @Override
    public synchronized void resetMetrics()
    {

        if (DEBUG_STATS)
        {
            if (DEBUG_AGENT || !m_isAgent)
            {
                System.out.println(System.currentTimeMillis() + " [" + m_componentName + "] *** MetricsManager.resetMetrics");
            }
        }

        if (m_isClosing)
        {
            return;
        }

        Iterator iterator = m_statistics.iterator();
        while (iterator.hasNext())
        {
            IStatistic statistic = (IStatistic)iterator.next();
            statistic.reset();

            // if its a sampled statistic then call the providers to reset
            if (statistic instanceof ISampledStatistic)
            {
                IStatisticProvider[] providers = ((ISampledStatistic)statistic).getStatisticProviders();
                for (int i = 0; i < providers.length; i++)
                {
                    providers[i].resetStatistic((ISampledStatistic)statistic);
                }
            }
        }

        // clear history slots and restart timer
        startRefreshing();
    }

    /**
     * Returns the expanded list of enabled metrics and metric patterns for the given list of metrics,
     * or all enabled metrics patterns if the parameter is null.
     *
     * @see com.sonicsw.mf.common.metrics.IMetricsManager#getEnabledMetrics(IMetricIdentity[])
     */
    @Override
    public synchronized IMetricIdentity[] getEnabledMetrics(IMetricIdentity[] ids)
    {
        return internalGetEnabledMetrics(ids, false);
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricsRegistrar#isInstanceEnabled(IMetricIdentity)
     */
    @Override
    public synchronized boolean isInstanceEnabled(IMetricIdentity id)
    {
        if (id == null)
        {
            throw new IllegalArgumentException(METRICID_NONNULL);
        }

        IMetricIdentity[] idPatterns = (IMetricIdentity[])m_enabledMetricsPatterns.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);

        for (int i = 0; i < idPatterns.length; i++)
        {
            if ((idPatterns[i] != null) && (id.isInstanceOf(idPatterns[i])))
            {
                return true;
            }
        }

        return false;
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricsManager#getEnabledAlerts(IMetricIdentity[])
     */
    @Override
    public synchronized IAlert[] getEnabledAlerts(IMetricIdentity[] ids)
    {
        ArrayList retAlerts = new ArrayList();
        Iterator it = m_enabledAlerts.keySet().iterator();
        while (it.hasNext())
        {
            IMetricIdentity id = (IMetricIdentity) it.next();
            // process this id?
            boolean check = false;
            if (ids == null)
            {
                check = true;
            }
            else
            {
                for (int i=0; i < ids.length; i++)
                {
                    if (id.isInstanceOf(ids[i]))
                    {
                        check = true;
                    }
                }
            }
            if (!check)
            {
                continue;
            }

            Set alerts =(Set) m_enabledAlerts.get(id);
             if (alerts != null)
            {
                Iterator it1 = alerts.iterator();
                while (it1.hasNext())
                {
                    retAlerts.add(it1.next());
                }
            }
        }
        if (DEBUG_ALERTS)
        {
            System.out.println("[" + m_componentName + "] Active Alerts:");
            synchronized (m_activeAlerts)
            {
                Iterator it1 = m_activeAlerts.iterator();
                while (it1.hasNext())
                {
                    System.out.println((IAlert) it1.next());
                }
            }
        }

        return (IAlert[])retAlerts.toArray(EMPTY_ALERT_ARRAY);
    }

    /**
     * Returns all active alerts
    **/
    @Override
    public IAlert[] getActiveAlerts()
    {
        ArrayList retAlerts = new ArrayList();
        synchronized (m_activeAlerts)
        {
            Iterator it = m_activeAlerts.iterator();
            while (it.hasNext())
            {
                retAlerts.add(it.next());
            }
        }

        return (IAlert[])retAlerts.toArray(EMPTY_ALERT_ARRAY);
    }

   /**
    * @see com.sonicsw.mf.common.metrics.IMetricsManager#enableAlerts(IAlert[])
    */
    @Override
    public synchronized IAlert[] enableAlerts(IAlert[] enableAlerts)
    {
        if (enableAlerts == null)
        {
            throw new IllegalArgumentException("List of alerts cannot be null");
        }

        if (m_isClosing)
        {
            return EMPTY_ALERT_ARRAY;
        }

        ArrayList retEnabled = new ArrayList(); // return the set actually enabled

        for (int i=0; i< enableAlerts.length; i++)
        {
            IMetricIdentity id = enableAlerts[i].getMetricIdentity();
            if (getMetricNodeId(id) == null)
            {
                // not defined, or a metric parent
                continue;
            }
            if (isInstancePattern(id))
            {
                // cannot enable alerts with patterns (yet)
                continue;
            }

            // does this metric support this type of alert?
            IMetricInfo info = getMetricInfo(id);
            if ((enableAlerts[i].isHighThreshold() && !info.supportsHighThresholdAlerts())
                || (!enableAlerts[i].isHighThreshold() && !info.supportsLowThresholdAlerts()))
            {
                continue;
            }

            // get the existing alerts for this metric
            Set alerts = (Set)m_enabledAlerts.get(id);
            if (alerts == null)
            {
                alerts = new HashSet();
                m_enabledAlerts.put(id, alerts);
                if (DEBUG_ALERTS)
                {
                    System.out.println("[" + m_componentName + "] Enabling alert " + id);
                }
            }
            // add and activate alert if metric doesn't have matching one
            if (matchAlert(alerts, enableAlerts[i]) == null)
            {
                alerts.add(enableAlerts[i]);
                // activate
                if (!info.isInstanceMetric()) 
                {
                    activateAlert(enableAlerts[i]); // individual metric
                    if (DEBUG_ALERTS)
                    {
                        System.out.println("[" + m_componentName + "] Activating alert " + enableAlerts[i]);
                    }
                }
                else
                {
                    // instance parent or instance
                    IMetricIdentity[] instances = getActiveInstanceMetrics(new IMetricIdentity[] { id });
                    for (int c = 0; c < instances.length; c++)
                    {
                        IAlert childAlert = MetricsFactory.createAlert(instances[c], enableAlerts[i].isHighThreshold(), enableAlerts[i].getThresholdValue());
                        activateAlert(childAlert);
                        if (DEBUG_ALERTS)
                        {
                            System.out.println("[" + m_componentName + "] Activating alert " + childAlert);
                        }
                    }
                }
            }
            // add to enabled list
            retEnabled.add(enableAlerts[i]);
        }
        return (IAlert[]) retEnabled.toArray(EMPTY_ALERT_ARRAY);
    }

    // activate an alert on an individual metric or an instance
    private void activateAlert(IAlert alert)
    {
        synchronized(m_activeAlerts)
        {
            if (matchAlert(m_activeAlerts, alert) == null)  // make sure it is not already there.
            {
                if (DEBUG_ALERTS)
                {
                    System.out.println("[" + m_componentName + "] Activating " + alert);
                }

                if (!m_repeatAlerts) {
                    //Before activating it, check if there is another with
                    //the same identity but different thresholds and make it idle
                    IAlert otherAlert = matchMetricIdentity(m_activeAlerts, alert);
                    if( otherAlert != null)
                    {
                        m_activeAlerts.remove(otherAlert);
                        m_idleAlerts.add(otherAlert);
                    }
                }
                m_activeAlerts.add(alert);
            }
        }
    }

    // deactivate an alert and its instance children, if any
    private void deactivateAlert(IAlert alert)
    {
        synchronized (m_activeAlerts)
        {
            Iterator it = m_activeAlerts.iterator();
            while (it.hasNext())
            {
                IAlert match = (IAlert) it.next();
                if (((Alert)match).isInstanceOf(alert))
                {
                    if (DEBUG_ALERTS)
                    {
                        System.out.println("[" + m_componentName + "] deactivating " + match);
                    }
                    it.remove();
                }
            }

            if (!m_repeatAlerts) {
                //Check if there is an idle alert with the same metric
                //identity and re enable it
                IAlert otherAlert = matchMetricIdentity(m_idleAlerts, alert);
                if(otherAlert != null)
                {
                    m_idleAlerts.remove(otherAlert);
                    m_activeAlerts.add(otherAlert);
                }
            }
        }
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricsManager#disableAlerts(IAlert[])
     */
    @Override
    public synchronized IAlert[] disableAlerts(IAlert[] disableAlerts)
    {
        if (disableAlerts == null)
        {
            throw new IllegalArgumentException("List of alerts cannot be null");
        }

        ArrayList retDisabled = new ArrayList();
        for (int i=0; i < disableAlerts.length; i++)
        {
            IMetricIdentity id = disableAlerts[i].getMetricIdentity();
            if (getMetricNodeId(id) == null)
            {
                continue;
            }
            if (isInstancePattern(id))
            {
                continue;
            }

            //TODO: disabling parent should disable all enabled alert instances

            Set alerts = (Set)m_enabledAlerts.get(id); // enabled alerts for this metric
            if (alerts == null)
            {
                continue;
            }
            IAlert match = matchAlert(alerts, disableAlerts[i]);
            if (match != null)
            {
                if (DEBUG_ALERTS)
                {
                    System.out.println("[" + m_componentName + "] Disabling alert " + match);
                }
                alerts.remove(match);
                deactivateAlert(match);
            }
            // can we remove the alerts for this metric
            if (alerts.isEmpty())
            {
                m_enabledAlerts.remove(id);
            }
        }
        return (IAlert[]) retDisabled.toArray(EMPTY_ALERT_ARRAY);
    }

    //
    // Other public methods of this implementation
    //

    /*
     * Replace the currently enabled metrics for the given metridIds with a new set of metrics.
     * This method is not exposed through the IMetricIdentity interface. 
     * Is it intended for use by config changes only.
     */
    public synchronized IMetricIdentity[][] replaceEnabledMetrics(IMetricIdentity[] metricIds)
    {
        IMetricIdentity[][] result = new IMetricIdentity[2][];
        
        // Find currently enabled metrics
        IMetricIdentity[] enabledMetricIds = getEnabledMetrics(null);

        // Disable any that are not in replace list.
        // We do this rather than disable all metrics and re-enable those in the replace list in order to retain gathered data.
        if (enabledMetricIds.length > 0)
        {
            ArrayList<IMetricIdentity> disableMetricIdList = new ArrayList<IMetricIdentity>();
            for (int i = 0; i < enabledMetricIds.length; i++)
            {
                boolean disable = true;
                for (int j = 0; metricIds != null && j < metricIds.length; j++)
                {
                    if (enabledMetricIds[i].equals(metricIds[j]))
                    {
                        disable = false;
                        break;
                    }
                }
                if (disable)
                {
                    disableMetricIdList.add(enabledMetricIds[i]);
                }
            }
            result[0] = disableMetrics((IMetricIdentity[])disableMetricIdList.toArray(new IMetricIdentity[disableMetricIdList.size()]));
        }
        else
        {
            result[0] = IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY;
        }

        // Enable the given set of metricIds (if a metricId is already enabled it will be ignored)
        result[1] = metricIds == null ? IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY : enableMetrics(metricIds);
        return result;
    }

    /*
     * Replace the currently enabled alerts for the given metridIds with a new set of alerts.
     * This method is not exposed through the IMetricIdentity interface. 
     * Is it intended for use by config changes only.
     */
    public synchronized IAlert[] replaceEnabledAlerts(IAlert[] alerts)
    {
        // Find currently enabled alerts
        IAlert[] enabledAlerts = getEnabledAlerts(null);
        
        // Disable any that are not in replace list.
        // We do this rather than disable all alerts and re-enable those in the replace list in order to retain gathered data.
        if (enabledAlerts.length > 0)
        {
            ArrayList<IAlert> disableAlertList = new ArrayList<IAlert>();
            for (int i = 0; i < enabledAlerts.length; i++)
            {
                boolean disable = true;
                for (int j = 0; alerts != null && j < alerts.length; j++)
                {
                    if (enabledAlerts[i].equals(alerts[j]))
                    {
                        disable = false;
                        break;
                    }
                }
                if (disable)
                {
                    disableAlertList.add(enabledAlerts[i]);
                }
            }
            disableAlerts((IAlert[])disableAlertList.toArray(new IAlert[disableAlertList.size()]));
        }

        // Enable the given set of metricIds (if a metricId is already enabled it will be ignored)
        return alerts == null ? MetricsManager.EMPTY_ALERT_ARRAY : enableAlerts(alerts);
    }
        
    /**
     * Enable the matching given metric(s) including hidden metrics. Hidden metrics are not exposed through the external API.
     *
     * This method is equivalent of the non-hidden version, however it allows enabling of both hidden and non-hidden metrics.
     *
     * @see #enaableMetric(IMetricIdentity[])
     */
    public synchronized IMetricIdentity[] enableAllMetrics(IMetricIdentity[] ids)
    {
         return internalEnableMetrics(ids, true);
    }

    /**
     * Disable the given metric(s) including hidden metrics. Hidden metrics are not exposed through the external API.
     *
     * This method is equivalent of the non-hidden version, however it allows disabling of both hidden and non-hidden metrics.
     *
     * @see #disableMetric(IMetricIdentity[])
     */
    public synchronized IMetricIdentity[] disableAllMetrics(IMetricIdentity[] ids)
    {
         return internalDisableMetrics(ids, true);
    }

    /**
     * Gets the metrics meta-data for all metrics including hidden metric(s). Hidden metrics are not exposed through the external API.
     *
     * @see #getMetricsInfo()
     */
    public IMetricInfo[] getAllMetricsInfo() { return internalGetMetricsInfo(true); }

    /**
     * Get the enabled metrics patterns for the specified list of metrics including hidden metrics.
     * Hidden metrics are not exposed through the external API.
     *
     * @see #getEnabledMetrics(IMetricIdentity[])
    **/
    public synchronized IMetricIdentity[] getAllEnabledMetrics(IMetricIdentity[] ids)
    {
        return internalGetEnabledMetrics(ids, true);
    }

    /**
     * Gets the list of all matching active metrics including hidden metric(s). Hidden metrics are not exposed through the external API.
     *
     * @see #getAllActiveMetrics()
     */
    public synchronized IMetricIdentity[] getAllActiveMetrics(IMetricIdentity[] ids)
    {
         return internalGetActiveMetrics(ids, true);
    }

    /**
     * Gets the metrics data for all specified metrics including hidden metric(s). Hidden metrics are not exposed through the external API.
     *
     * @see #getAllMetricsData(IMetricIdentity[])
     */
    public synchronized IMetricsData getAllMetricsData(IMetricIdentity[] ids, Boolean returnTriggeredAlerts)
    {
         return internalGetMetricsData(ids, returnTriggeredAlerts, true);
    }

    /**
     * Gets the metrics data for all active metrics including hidden metric(s). Hidden metrics are not exposed through the external API.
     *
     * This method is optimized to have the least overhead for internal benchmark use.
     *
     * @see #getAllMetricsData()
     */
    public synchronized IMetricsData getAllMetricsData()
    {
        return internalGetMetricsData(m_activeMetrics.values());
    }

    /**
     * Returns the list of active instance metrics of a metric id parent, or the metric id itself
     * if it is an instance.
     * Caller must synchronize on MetricsManager.this
      */
    private IMetricIdentity[] getActiveInstanceMetrics(IMetricIdentity[] ids)
    {
        ArrayList ret = new ArrayList();
        for (int i=0; i < ids.length; i++)
        {
            IMetricInfo info = getMetricInfo(ids[i]);
            if (!info.isInstanceMetric())
            {
                continue;
            }

            Iterator it = m_activeMetrics.keySet().iterator();
            while (it.hasNext())
            {
                IMetricIdentity match = (IMetricIdentity)it.next();
                if (match.isInstanceOf(ids[i]))
                {
                    ret.add(match);
                }
            }
        }
        return (IMetricIdentity[])ret.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
    }

   /**
     * Adds the metric to the list of actively managed metrics. An actively managed metric
     * will appear in the list of enabled metrics and its value accessible when requests for
     * metrics data are made. If the statistic is a sampled statistic, the MetricsManager
     * will add the statistic to the list of statistics to be sampled at the refresh interval.
     *
     * Since a metric analyzer is not supplied, the MetricsManager will looks up the metric's
     * meta-data to find its value type and uses the default analyzer for that metric type.
     *
     * If the metric is already enabled with the same statistic instance then no action will
     * be taken. If the metric is enabled with a different statistic instance then the metric
     * is automatically unregistered, then re-registered with the new statistic.
     *
     * An IllegalArgumentException will be thrown if either a null IMetricIdentity or null IStatistic 
     * reference is passed in as an input parameter
     * 
     * @param id        The identity of the metric to be enabled.
     * @param statistic The statistic to be used to evaluate the metric.
     *
     * @exception IllegalArgumentException
     */
    @Override
    public void registerMetric(IMetricIdentity id, IStatistic statistic)
    {
        // metric id must be specified
        if (id == null)
        {
            throw new IllegalArgumentException(METRICID_NONNULL);
        }

        // statistic must likewise be specified
        if (statistic == null)
        {
            throw new IllegalArgumentException(STATISTIC_NONNULL);
        }
        
        // get the metric info so we can get the value type and hence determine the appropriate
        // analyzer
        IMetricInfo info = getMetricInfo(id);
        if (info.isInstanceMetric() && info.getMetricIdentity().equals(id))
        {
            throw new IllegalArgumentException("Cannot register an instance metric parent");
        }

        // get default analyzer
        IMetricAnalyzer analyzer = getMetricAnalyzer(info.getValueType());
        registerMetric(id, analyzer, statistic);
    }

    /**
     * Allows provision of a user specified analyzer.
     *
     * An IllegalArgumentException will be thrown if either a null IMetricIdentity 
     * or IStatistic reference is passed in as an input parameter
     * 
     * @param id        The identity of the metric to be enabled.
     * @param analyzer  The metrics analyzer to be used to evaluate the metric value.
     * @param statistic The statistic to be used to evaluate the metric.
     *
     * @see #enableMetric(IMetricIdentity, IStatistic)
     */
    @Override
    public synchronized void registerMetric(IMetricIdentity id, IMetricAnalyzer analyzer, IStatistic statistic)
    {
        if (DEBUG_CONFIG || DEBUG_METRICS)
        {
            System.out.println("[" + m_componentName + "] Registering metric " + id  + " for statistic = " + statistic + " with analyzer = " + analyzer);
        }
        
        // metric id must be specified
        if (id == null)
        {
            throw new IllegalArgumentException(METRICID_NONNULL);
        }

        // statistic must likewise be specified
        if (statistic == null)
        {
            throw new IllegalArgumentException(STATISTIC_NONNULL);
        }
        
        IMetricInfo info = getMetricInfo(id);
        if (info.isInstanceMetric() && info.getMetricIdentity().equals(id))
        {
            throw new IllegalArgumentException("Cannot register an instance metric parent");
        }

        if (!info.isInstanceMetric() && !info.getMetricIdentity().equals(id))
        {
            throw new IllegalArgumentException("Cannot register a metric parent");
        }

        if (m_isClosing)
        {
            return;
        }

        // TODO: could re-use custom analyzers by keeping them in a HashSet(). Would have to distinguish
        // them from default analyzers, which are automatically re-used.

        // deal with metric already being registered
        RegisteredMetric existingMetric = (RegisteredMetric) m_activeMetrics.get(id);        
        if (existingMetric != null)
        {
            IStatistic existingStatistic = existingMetric.getStatistic();
            if (statistic == existingStatistic)
            {
                // metric already registered for this same statistic - don't register it again - just return
                if (DEBUG_CONFIG || DEBUG_METRICS)
                {
                    System.out.println("[" + m_componentName + "] Metric " + id  + " already registered with statistic = " + statistic + " - returning");
                }
                return;
            }
            else
            {
                // metric registered but with a different statistic - unregister existing metric before we continue
                if (DEBUG_CONFIG || DEBUG_METRICS)
                {
                    System.out.println("[" + m_componentName + "] Metric " + id  + " already registered, but with statistic = " + existingStatistic + " - unregistering existing metric");
                }
                unregisterMetric(id, existingStatistic);
            }
            
        }

        registerStatistic(statistic);   // register statistic if needed
        m_activeMetrics.put(id, new RegisteredMetric(id, statistic, analyzer, ++m_currentMetricId));
        m_metricRegistrations++;

        // activate any alerts specified for this metric
        Set alerts = (Set) m_enabledAlerts.get(id);   // does this metric have enabled alerts
        if (alerts == null)
        {
            // see if instance parent has alerts
            alerts = (Set) m_enabledAlerts.get(getMetricNodeId(id));
        }

        if (alerts != null)
        {
            Iterator it = alerts.iterator();
            while (it.hasNext())
            {
                IAlert parent = (IAlert)it.next();
                // Construct a child alert, if necessary
                IAlert alert = MetricsFactory.createAlert(id, parent.isHighThreshold(), parent.getThresholdValue());
                activateAlert(alert);
            }
        }

        if (m_intervalTimer == null && !m_isClosing)
        {
            startRefreshing();
        }
    }

    /**
     * Register a statistic
     * 
     * An IllegalArgumentException will be thrown if a null IStatistic reference 
     * is passed in as an input parameter
    */
    public synchronized void registerStatistic(IStatistic stat)
    {
        if (DEBUG_CONFIG)
        {
            System.out.println("[" + m_componentName + "] Registering statistic " + stat);
        }

        // statistic must be specified
        if (stat == null)
        {
            throw new IllegalArgumentException(STATISTIC_NONNULL);
        }
     
        if (m_isClosing)
        {
            return;
        }

        if (!m_statistics.contains(stat))
        {
            if (DEBUG_CONFIG)
            {
                System.out.println("[" + m_componentName + "] Adding statistic " + stat);
            }

            m_statistics.add(stat);
            m_statisticRegistrations++;

            // if its a historical statistic then we need to set the number of values if were already
            // refreshing (otherwise it is handled by startRefreshing()
            if (stat instanceof IHistoricalStatistic  && m_intervalTimer != null)
            {
                ((IHistoricalStatistic)stat).setNumValues(m_numHistoricalValues);
            }
        }
        else
            if (DEBUG_CONFIG)
            {
                System.out.println("[" + m_componentName + "] Statistic " + stat + " already registered");
            }
    }

    /**
     * Unregister a statistic if no more metrics are using it
     * 
    */
    public synchronized void unregisterStatistic(IStatistic stat)
    {
        if (DEBUG_CONFIG)
        {
            System.out.println("[" + m_componentName + "] Unregistering statistic " + stat);
        }

        // statistic must be specified
        if (stat == null)
        {
            throw new IllegalArgumentException(STATISTIC_NONNULL);
        }
        
        if (m_statistics.contains(stat))
        {
            boolean found = false;
            Iterator it = m_activeMetrics.values().iterator();
            while (it.hasNext())
            {
                if (((RegisteredMetric)it.next()).getStatistic() == stat)
                {
                    found = true;
                    break;
                }
            }
            if (!found)
            {
                m_statistics.remove(stat);
                m_statisticUnregistrations++;
                MetricsManager.m_metricsManagerDiagnostics.traceStatisticRegistrations(this);
                if (DEBUG_CONFIG)
                {
                    System.out.println("[" + m_componentName + "] Removing statistic " + stat);
                }
            }
            else
                if (DEBUG_CONFIG)
                {
                    System.out.println("[" + m_componentName + "] Statistic " + stat + " still in use");
                }
        }
        else
            if (DEBUG_CONFIG)
            {
                System.out.println("[" + m_componentName + "] Statistic " + stat + " not registered");
            }

    }

    /**
     * Helper methods
    */
    /**
     * Get node id of a metric:
     * - the metric itself if not an instance
     * - the metric parent if an instance
     * - null if a metric parent
    */
    private IMetricIdentity getMetricNodeId(IMetricIdentity id)
    {
        IMetricInfo info = getMetricInfo(id);
        if (info == null)
        {
            return null;
        }
        else
        {
            return info.getMetricIdentity();
        }
    }

    // get instance name or pattern, or null if not an instance
    private String getInstanceName(IMetricIdentity id)
    {
        IMetricInfo info = getMetricInfo(id);
        if (info == null || !info.isInstanceMetric() || id.equals(info.getMetricIdentity()))
        {
            return null;
        }
        else
        {
            String[] components = id.getNameComponents();
            return components[components.length-1];
        }
    }

    // is it an instance pattern
    private boolean isInstancePattern(IMetricIdentity id)
    {
        String instanceName = getInstanceName(id);
        if (instanceName == null)
        {
            return false;
        }
        else
        {
            return containsWildcard(instanceName);
        }
    }

    // Does this component name contain a true wildcard character, 
    // or is it one of the exceptions? (Currently the only exception
    // is the ".* string from temp queue names.)
    private boolean containsWildcard(String name) 
    {
    	String wildcard = "*";
    	// Let's get rid of the no wildcard char case quickly
    	int index = name.indexOf(wildcard);
    	if (index == -1)
        {
            return false;
        }
    	
    	// Strings which contain '*' where it is not used as a wildcard
    	// (Right now the array only contains ".*" used in temp queue names.)
    	String[] exceptionStrings = new String[] { ".*" };
    	int[] wildcardPos = new int[exceptionStrings.length];
    	for (int i = 0; i < exceptionStrings.length; i++) {
    		wildcardPos[i] = exceptionStrings[i].indexOf(wildcard);
    		// This is really a programming error - exceptionStrings entry is invalid
    		if (wildcardPos[i] == -1) {
    			return true;
    		}
    	}
    	
   		StringBuffer buf = new StringBuffer(name);
   		boolean isExceptionString = false;
		while (index != -1) {
			isExceptionString = false;
			for (int j = 0; j < wildcardPos.length; j++) {
				int fromIndex = index - wildcardPos[j];
				if (fromIndex >= 0) {
					int exceptionIndex = buf.indexOf(exceptionStrings[j], fromIndex);
					if (exceptionIndex + wildcardPos[j] == index) {
						isExceptionString = true;
						break;
					}
				}
			}
			// Found a wildcard that was not contained in an exception string
			if (!isExceptionString)
            {
                return true;
            }
			
			// The last wildcard char was an exception; look for the next one
			if (index != -1 && index < buf.length())
            {
                index = buf.indexOf(wildcard, index+1);
            }
		}
    		return index != -1 && !isExceptionString;
    }


    /**
     * Enables the given set of metrics
     * Each metric in the given list can be either:
     * - an individual metric (metric node, including an instance parent)
     * - a parent node
     * - an instance pattern
     *
     * @return An expanded list of individual ids to pass to the component.
     */
    @Override
    public synchronized IMetricIdentity[] enableMetrics(IMetricIdentity[] ids)
    {
        return internalEnableMetrics(ids, false);
    }
    /**
     * Disable the given set of metric identities which may include one or more identities
     * that are branches of the metrics tree, in which case all the leaves (individual metrics)
     * of that branch will be disabled. Metrics already disabled or those that are not dynamic
     * will not be disabled.
     * If any of the indentities are instance metrics, then disable the patterns they represent.
     * Does not disable hidden metrics.
     *
     * @return An expanded list of ids to pass to the component.
     *
     * @see #disableAllMetrics(IMetricIdentity[])
     */
    @Override
    public synchronized IMetricIdentity[] disableMetrics(IMetricIdentity[] ids)
    {
         return internalDisableMetrics(ids, false);
    }

    /**
     * Called by the component to unregister a metric.
     * The component signals that it has disabled a metric, in particular, if an instance has been destroyed.
     * Note that the associated statistic may still be in use by another metric.
     *
     * Unregistering a metric does NOT affect any instance patterns, including a pattern equal to the instance name
     * Patterns are removed by the client calling disableMetric()
     */
    @Override
    public void unregisterMetric(IMetricIdentity id)
    {
        unregisterMetric(id, null);
    }
    
    /**
     * Called by a component to unregister a metric.  If 'statistic' is null this method is
     * equivalent to unregisterMetric(IMetricIdentity id).  However, if 'statistic' is non-null
     * it is checked against the statistic currently registered with the metric.  If they match
     * (i.e. both reference the same underlying statistic object) the unregister completes,
     * otherwise the method is a no-op.
     * 
     * The component signals that it has disabled a metric, in particular, if an instance has been destroyed.
     * Note: The associated statistic may still be in use by another metric.
     * Unregistering a metric does NOT affect any instance patterns, including a pattern equal to the instance name
     */
    @Override
    public synchronized void unregisterMetric(IMetricIdentity id, IStatistic statistic)
    {
        if (DEBUG_CONFIG || DEBUG_METRICS)
        {
            if (statistic == null)
            {
                System.out.println("[" + m_componentName + "] Unregistering metric " + id);
            }
            else
            {
                System.out.println("[" + m_componentName + "] Unregistering metric " + id + ", statistic " + statistic);
            }                
        }

        IMetricInfo info = getMetricInfo(id);
        if (info.isInstanceMetric() && info.getMetricIdentity().equals(id))
        {
            throw new IllegalArgumentException("Cannot unregister an instance metric parent");
        }

        // ignore static metrics
        if (!info.isDynamic())
        {
            return;
        }

        // check metric is registered
        RegisteredMetric metricDetails = (RegisteredMetric) m_activeMetrics.get(id);
        if (metricDetails == null)
         {
            return;     // metric not registered
        }
        
        // check statistic parameter (if supplied) matches currently registered statistic
        IStatistic registeredStatistic = metricDetails.getStatistic();              
        if (statistic != null && statistic != registeredStatistic)
        {            
            if (DEBUG_CONFIG || DEBUG_METRICS)
            {
                System.out.println("[" + m_componentName + "] Unregister skipped for metric " + id + ", statistic parameter does not match currently registered statistic");
            }
            
            // we don't unregister the statistic here since that should have happened
            // when the metric was previously re-registered with the new statistic
            return;
        }
        
        // remove from active metrics
        m_activeMetrics.remove(id);
        m_metricUnregistrations++;
        MetricsManager.m_metricsManagerDiagnostics.traceMetricRegistrations(this);

        // unregister statistic unless another metric still uses it
        unregisterStatistic(registeredStatistic);  // will check whether still in use by another metric
        
        // de-activate all alerts for this metric
        deactivateAlerts(id);        
    }    

    /**
     * deactivate all alerts for a metric
     * Caller must synchronize on MetricsManager
    **/
    private void deactivateAlerts(IMetricIdentity metricId)
    {
        synchronized (m_activeAlerts)
        {
            Iterator it = m_activeAlerts.iterator();
            while (it.hasNext())
            {
                IAlert alert = (IAlert) it.next();
                if (metricId.equals(alert.getMetricIdentity()))
                {
                    it.remove();
                }
            }
        }
    }

	
    /**     
     * validate the RefreshInterval 
    **/
    private void checkValidRefreshInterval(long refreshInterval)
    throws MFException
    {
        if (refreshInterval < MIN_REFRESH_INTERVAL || refreshInterval > MAX_REFRESH_INTERVAL )
        {
            throw new MFException("Refresh interval=["+ refreshInterval +"] cannot be < " + MIN_REFRESH_INTERVAL +" or >"+ MAX_REFRESH_INTERVAL +" milliseconds.");
        }
    }    
	
    /**     
     * validate the CollectionInterval 
    **/
    private void checkValidCollectionInterval(long collectionInterval)
    throws MFException
    {
        if (collectionInterval < MIN_COLLECTION_INTERVAL || collectionInterval > MAX_COLLECTION_INTERVAL)
        {
            throw new MFException("Collection interval=["+ collectionInterval +"] cannot be < " + MIN_COLLECTION_INTERVAL +" or >"+ MAX_COLLECTION_INTERVAL + " milliseconds.");
        }
    }    

    private void checkValidIntervalRatio(long refreshInterval, long collectionInterval)
    throws MFException
    { 
        if ((int)(collectionInterval / refreshInterval) > MAX_HISTORY_VALUES)
        {
            // reset to default values (otherwise, exceptions might constantly occur when reset attempts are made)
            m_refreshInterval = (long) IContainerConstants.REFRESH_INTERVAL_DEFAULT * 1000;  //convert from seconds to milliseconds
            m_collectionInterval = (long) IContainerConstants.COLLECTION_INTERVAL_DEFAULT * 60000; //convert from minutes to milliseconds
            throw new MFException("Collection interval divided by Refresh interval cannot be > " + MAX_HISTORY_VALUES + '.');
        }
    }
    

    private void validateRefreshAndCollectionInterval(long refreshInterval, long collectionInterval)
    throws MFException
    {
        checkValidRefreshInterval(refreshInterval);
        checkValidCollectionInterval(collectionInterval);
        checkValidIntervalRatio( m_collectionInterval, refreshInterval);
    }
    
    
    /**
     * Get the interval at which the manager will:
     *
     *  - request the statistic providers of a sampled statistic to update
     *    the statistic value
     *  - request each historical statistic to cycle its data
     *
     * @return The refresh interval in milliseconds.
     */
    @Override
    public long getRefreshInterval() { return m_refreshInterval; }

    /**
     * Set the interval at which the manager will:
     *
     *  - request the statistic providers of a sampled statistic to update
     *    the statistic value
     *  - request each historical statistic to cycle its data
     *
     * When reset the number of historical values stored by historical statistics
     * will be recalculated and reset in each historical statistic.
     *
     * @param interval The refresh interval in milliseconds.
     *
     * @see IHistoricalStatistic
     * @see IHistoricalSampledStatistic
     */
     
    @Override
    public synchronized void setRefreshInterval(long refreshInterval)
    throws MFException
    {
        validateRefreshAndCollectionInterval(refreshInterval, m_collectionInterval);

        if (refreshInterval != m_refreshInterval)
        {
            m_refreshInterval = refreshInterval;
            m_numHistoricalValues = (int)(m_collectionInterval / m_refreshInterval);
            if (DEBUG_STATS)
            {
                if (DEBUG_AGENT || !m_isAgent)
                {
                    System.out.println(System.currentTimeMillis() + " [" + m_componentName + "] *** MetricsManager.setRefreshInterval = " + refreshInterval);
                }
            }

            if (m_isClosing)
            {
                return;
            }

            startRefreshing();
        }
    }

    /**
     * Get the window over which historical statistics will be maintained
     * for historical statistsics.
     *
     * @return The collection interval in milliseconds.
     */
    @Override
    public long getCollectionInterval() { return m_collectionInterval; }

    /**
     * Set the window over which historical statistics will be maintained
     * for historical statistsics.
     *
     * When reset the number of historical values stored by historical statistics
     * will be recalculated and reset in each historical statistic.
     *
     * @param interval The collection interval in milliseconds.
     *
     * @see IHistoricalStatistic
     * @see IHistoricalSampledStatistic
     */
    @Override
    public synchronized void setCollectionInterval(long collectionInterval)
    throws MFException
    {
        validateRefreshAndCollectionInterval(m_refreshInterval, collectionInterval);

        if ( collectionInterval != m_collectionInterval)
        {
            m_collectionInterval = collectionInterval;
            m_numHistoricalValues = (int)(m_collectionInterval / m_refreshInterval);
            if (DEBUG_STATS)
            {
                if (DEBUG_AGENT || !m_isAgent)
                {
                    System.out.println(System.currentTimeMillis() + " [" + m_componentName + "] *** MetricsManager.setCollectionInterval = " + collectionInterval);
                }
            }

            if (m_isClosing)
            {
                return;
            }

            startRefreshing();
        }
    }

    @Override
    public synchronized void setRefreshAndCollectionInterval(long refreshInterval, long collectionInterval)
    throws MFException
    {
        validateRefreshAndCollectionInterval(refreshInterval, collectionInterval);

        if ( ( refreshInterval != m_refreshInterval ) || ( collectionInterval != m_collectionInterval ) )
        {
            m_refreshInterval    = refreshInterval;
            m_collectionInterval = collectionInterval;
			
            m_numHistoricalValues = (int)(m_collectionInterval / m_refreshInterval);
            if (DEBUG_STATS)
            {
                if (DEBUG_AGENT || !m_isAgent)
                {
                    System.out.println(System.currentTimeMillis() + " [" + m_componentName + "] *** MetricsManager.setRefreshAndCollectionInterval(" + refreshInterval + ","+collectionInterval+")");
                }
            }

            if (m_isClosing)
            {
                return;
            }

            startRefreshing();
        }
    }


    @Override
    public synchronized boolean getRepeatMetricAlerts() { return m_repeatAlerts; }

    @Override
    public synchronized void setRepeatMetricAlerts(boolean repeatMetricAlerts)
    throws MFException
    {
        m_repeatAlerts = repeatMetricAlerts;
    }

    /**
     * Get an accurate figure for the number of milliseconds from the start of the
     * first refresh cycle to the end of the last completed refresh cycle.
     */
    synchronized long getElapsedCollectionTime()
    {
        long[] values = m_elapsedCollectionTimeAnalyzer.evaluateValue(m_elapsedCollectionTimeStatistic);
        if (DEBUG_STATS)
        {
            if (DEBUG_AGENT || !m_isAgent)
            {
                System.out.println(System.currentTimeMillis() + " [" + m_componentName + "] ElapsedCollectionTime = " + values[0] + " timestamp = " + values[1]);
            }
        }

        return values[0];
    }

    /**
     * Get the elapsed time for the last n refresh cycles. This method is used to compute rates for
     * metrics that have not yet completed a full collection cycle.
    **/
    long getElapsedCollectionTime(int maxCycles)
    {
        long[] values;
        synchronized (this)
        {
            values = m_elapsedCollectionTimeStatistic.getLastValues();
        }
        if (values == null)
        {
            return 0;
        }
        int len = values.length;
        if (maxCycles < len)
        {
            len = maxCycles;
        }

        long ret = 0;
        for (int i=0; i < len; i++)
        {
            ret += values[i];
        }
        return ret;
    }

    /**
     * Gets the time of the last completed refresh cycle.
     */
    public long getCurrencyTimestamp() { return m_currencyTimestamp; }

    /**
     * Gets a default implementation for the given value type (see IValueType).
     *
     * @exception IllegalArgumentException Thrown when a no default implementation exists
     *                                     for the given type.
     */
    public IMetricAnalyzer getMetricAnalyzer(short valueType)
    {
        if (valueType >= m_defaultAnalyzers.length)
        {
            throw new IllegalArgumentException("No default metric analyzer is available for the value type: [type=" + valueType + ']');
        }

        IMetricAnalyzer analyzer = m_defaultAnalyzers[valueType];

        if (analyzer == null)
        {
            if (DEBUG_CONFIG)
            {
                System.out.println("[" + m_componentName + "] Creating analyzer for value type: " + valueType);
            }
            switch (valueType)
            {
                case IValueType.VALUE:
                    analyzer = new Value();
                    break;
                case IValueType.COUNT:
                    analyzer = new Count();
                    break;
                case IValueType.TOTAL:
                    analyzer = new Total();
                    break;
                case IValueType.MINIMUM:
                    analyzer = new Minimum();
                    break;
                case IValueType.MAXIMUM:
                    analyzer = new Maximum();
                    break;
                case IValueType.AVERAGE:
                    analyzer = new Average();
                    break;
               case IValueType.UTILIZATION:
                    analyzer = new Rate(this, 100);
                    break;
                case IValueType.PEAK_UTILIZATION:
                    analyzer = new PeakRate(this, 100);
                    break;
                case IValueType.PER_SECOND_RATE:
                    analyzer = new Rate(this, 1000);
                    break;
                case IValueType.PEAK_PER_SECOND_RATE:
                    analyzer = new PeakRate(this, 1000);
                    break;
                case IValueType.PER_MINUTE_RATE:
                    analyzer = new Rate(this, 60000);
                    break;
                case IValueType.PEAK_PER_MINUTE_RATE:
                    analyzer = new PeakRate(this, 60000);
                    break;
                case IValueType.PER_HOUR_RATE:
                    analyzer = new Rate(this, 3600000);
                    break;
                case IValueType.PEAK_PER_HOUR_RATE:
                    analyzer = new PeakRate(this, 3600000);
                    break;
                default:
                    throw new IllegalArgumentException("No default metric analyzer is available for the value type: [type=" + valueType + ']');
            }
            m_defaultAnalyzers[valueType] = analyzer; // save for re-use
         }
         return analyzer;
    }

    /**
    * Generate an alert notification based on an MF metric alert
    * Assumes the notification info has also been statically defined.
    */
    private void sendAlertNotification(IAlert alert, long value)
    {
        IMetricIdentity metricId = alert.getMetricIdentity();
        String metricName = getMetricNodeId(metricId).getName();    // instance parent or self, null if metric parent
        String units = getMetricInfo(metricId).getUnits();
        String instance = getInstanceName(metricId);    // null if not an instance
        long threshold = alert.getThresholdValue();
        boolean isHighThreshold = alert.isHighThreshold();
        GregorianCalendar initialAlertTime = ((Alert)alert).getInitialAlertTime();
        if (DEBUG_ALERTS)
        {
            System.out.println(System.currentTimeMillis()  + " [" + m_componentName + "]  **** sendAlertNotification for metric = " + metricName + " instance = " + instance
                    + " Units = " + units + (isHighThreshold? " High": " Low") + " Threshold = " + threshold + " value = " + value);
        }

        if (m_mfContext == null)
        {
            throw new IllegalStateException("Context not set");
        }
        INotification notification = m_mfContext.createNotification(m_alertNotificationCategory, INotification.SUBCATEGORY_TEXT[INotification.ALERT_SUBCATEGORY], metricName, Level.INFO);
        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("Threshold", new Long(threshold));
        notification.setAttribute("Value", new Long(value));
        notification.setAttribute("InitialAlert", initialAlertTime);
        notification.setAttribute("Repeat", ((Alert)alert).isRepeat());
        notification.setAttribute("ThresholdType", isHighThreshold ? "high" : "low");
        notification.setAttribute("Units", units);
        if (instance != null)
        {
            notification.setAttribute("Instance", instance);
        }

        // add common alert notification attributes
        if (m_alertNotificationAttributes != null && !m_alertNotificationAttributes.isEmpty())
        {
            Iterator it = m_alertNotificationAttributes.keySet().iterator();
            while (it.hasNext())
            {
                String attr = (String) it.next();
                notification.setAttribute(attr, m_alertNotificationAttributes.get(attr));
            }
        }
        m_mfContext.sendNotification(notification);
    }

    //
    // Protected and other implementation private methods
    //

    /**
     * Starts (or restarts) the statistic refreshing schedule.
     *
     * Called:
     *
     *  - when the first metric is enabled
     *  - when the refresh interval or collection interval is changed (to a different value)
     *  - when the current schedule gets so far behind that the current refresh cycle is
     *    being executed late by at least the collection interval
     */
    protected synchronized void startRefreshing()
    {
        if (m_isClosing)
        {
            return;
        }

        if (DEBUG_STATS)
        {
            if (DEBUG_AGENT || !m_isAgent)
            {
                System.out.println(System.currentTimeMillis() + " [" + m_componentName + "] Start refreshing");
            }
        }

        // reset he number of history slots in historical statistics
        // including the elapsed collection time statistic
        Iterator iterator = m_statistics.iterator();
        while (iterator.hasNext())
        {
            IStatistic statistic = (IStatistic) iterator.next();
            
            // only set the number of slots on historical statistsic
            if (statistic instanceof IHistoricalStatistic)
            {
                ((IHistoricalStatistic)statistic).setNumValues(m_numHistoricalValues);
            }
        }

        if (m_intervalTimer != null)
        {
            m_intervalTimer.cancel();
        }
        m_intervalTimer = new Timer(true);  // daemon

        // (re-) initialize time keeper before first refresh for calculating the first diff
        m_elapsedCollectionTimeStatistic.updateValue(System.currentTimeMillis());

        // schedule refresh at fixed intervals
        m_intervalTimer.scheduleAtFixedRate(new MetricRefresher(), 0, m_refreshInterval);
    }

    /**
     * Determines if the given metric is a "hidden" metric that is not exposed via the external API
     */
    private boolean isHiddenMetric(IMetricIdentity id)
    {
        IMetricInfo info = getMetricInfo(id);
        if (info == null)
        {
            return false;
        }

        return ((MetricInfo)info).isHidden();
    }

    /**
     * Get meta-data for all metrics.
     * Filter on whether to include hidden metrics.
     **/
    private IMetricInfo[] internalGetMetricsInfo(boolean includeAll)
    {
        if (includeAll)
        {
            return (IMetricInfo[])m_metricInfos.values().toArray(EMPTY_INFO_ARRAY);
        }

        // else be selective - exclude hidden metrics
        ArrayList metricInfos = new ArrayList();
        Iterator iterator = m_metricInfos.values().iterator();
        while (iterator.hasNext())
        {
            IMetricInfo info = (IMetricInfo)iterator.next();
            if (!((MetricInfo)info).isHidden())
            {
                metricInfos.add(info);
            }
        }
        return (IMetricInfo[])metricInfos.toArray(EMPTY_INFO_ARRAY);
    }

    /**
     * Get the set of enabled metrics and metric patterns for a given list of metrics, or all metrics if null.
     * Filters depending on whether to show hidden metrics.
     * Caller is responsible for synchronization.
    */
    private IMetricIdentity[] internalGetEnabledMetrics(IMetricIdentity[] ids, boolean showHidden)
    {
        if (ids == null && showHidden)
        {
            // return all elements in enabled table
            return (IMetricIdentity[])m_enabledMetricsPatterns.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
        }
        else
        {
            ArrayList retPatterns = new ArrayList();
            Iterator it = m_enabledMetricsPatterns.iterator();
            // filter by given list and whether to show hidden
            while (it.hasNext())
            {
                IMetricIdentity pat = (IMetricIdentity) it.next();
                if (!isHiddenMetric(pat) || showHidden)
                {
                    if (ids == null)
                    {
                        retPatterns.add(pat);
                    }
                    else
                    {
                        for (int i=0; i < ids.length; i++)
                        {
                            if (pat.isInstanceOf(ids[i]))
                            {
                                retPatterns.add(pat);
                            }
                        }
                    }
                }
            }
            return (IMetricIdentity[])retPatterns.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
        }
    }

    /**
     * Returns the expanded list of matching active metrics, including instances, that match the given set of metrics, or
     * all metrics if the parameter is null.
     * Filters on whether to include hidden metrics.
     * Metric identities of active metrics are tagged with an internal hash code, if implemented.
     * Caller is responsible for synchronization.
     **/
    private IMetricIdentity[] internalGetActiveMetrics(IMetricIdentity[] ids, boolean includeAll)
    {
        ArrayList retActive = new ArrayList();
        Iterator iterator = m_activeMetrics.keySet().iterator();
        // filter on list and whether hidden
        while (iterator.hasNext())
        {
            IMetricIdentity id = (IMetricIdentity)iterator.next();
            IMetricInfo info = getMetricInfo(id);
            if (includeAll || !((MetricInfo)info).isHidden())
            {
                RegisteredMetric details = (RegisteredMetric) m_activeMetrics.get(id);
                if (details != null)
                {
                    ((MetricIdentity)id).setHash(details.getHash());
                }
                if (ids == null)
                {
                    retActive.add(id);
                }
                else
                {
                    // filter on given list
                    for (int i=0; i < ids.length; i++)
                    {
                        if (id.isInstanceOf(ids[i]))
                        {
                            retActive.add(id);
                        }
                    }
                }
            }
        }
        return (IMetricIdentity[])retActive.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
    }

    private IMetricIdentity[] new_internalEnableMetrics(IMetricIdentity[] ids, boolean includeHidden)
    {
        if (ids == null)
        {
            throw new IllegalArgumentException("List of metric identities cannot be null");
        }

        if (DEBUG_CONFIG || DEBUG_METRICS)
        {
            System.out.println("[" + m_componentName + "] Enabling metrics:");
            for (int i = 0; i < ids.length; i++)
            {
                System.out.println(" - " + ids[i]);
            }
        }

        if (m_isClosing)
        {
            return IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY;
        }

        HashSet retEnabled = new HashSet(); // catch duplicates

        // for regular expression support, get the regular expression [from each input metric identity] 
        // and then compare each registered metric info [of the MetricsManager] with the expression
        IMetricIdentity mid = null;
        String metricNamePattern = null;
        for (int i = 0; i < ids.length; i++)
        {
            Pattern pattern = Pattern.compile(ids[i].getAbsoluteName());
            Matcher matcher = pattern.matcher("");
            Pattern instancePattern = null; //used to test for instance metric name match
            Matcher instanceMatcher = null;
            String[] nameComponents = ids[i].getNameComponents();
            IMetricIdentity tempID = null;

            if (nameComponents.length > 2)
            {
                String[] parentName = null;
                if (ids[i].getAbsoluteName().endsWith("..*"))
                {
                    parentName = new String[nameComponents.length - 2];
                    System.arraycopy(nameComponents, 0, parentName, 0, nameComponents.length - 2);
                }
                else
                {
                    parentName = new String[nameComponents.length - 1];
                    System.arraycopy(nameComponents, 0, parentName, 0, nameComponents.length - 1);
                }
                tempID = new MetricIdentity(parentName);
            }

            Iterator iterator = m_metricInfos.values().iterator();
            while (iterator.hasNext())
            {
                IMetricInfo info = (IMetricInfo) iterator.next();
                if ( ( (MetricInfo) info).isHidden() && !includeHidden)
                {
                    continue;
                }
                IMetricIdentity idNode = info.getMetricIdentity();

                matcher.reset(idNode.getAbsoluteName());
                if (matcher.matches()) // check for exact match
                {
                    retEnabled.add(ids[i]);
                    // since IMetricInfo objects do not exist for instance metrics (they exist only for normal metrics
                    // and instance metric parents), and since there was an exact match, this IMetricInfo object
                    // must be for a normal metric.  So, add the exact metric ID to the list of enabled metrics.
                    m_enabledMetricsPatterns.add(idNode);
                }
                else if (matcher.lookingAt()) // beginning matches...
                {
                    // may be for an instance metric...
                    if (ids[i].getAbsoluteName().endsWith("..*"))
                    {
                        // get the root of the input metricID (everything before "..*")
                        IMetricIdentity pat = null;
                        IMetricIdentity parentID = getMetricParent(ids[i]);
                        if (parentID.isInstanceOf(idNode) && info.isInstanceMetric())
                        {
                            retEnabled.add(idNode);
                            m_enabledMetricsPatterns.add(ids[i]);
                        }
                        else if (idNode.isInstanceOf(parentID))
                        {
                            retEnabled.add(idNode);
                            if (info.isInstanceMetric())
                            {
                                pat = MetricsFactory.createMetricIdentity(idNode.getAbsoluteName() + "..*");
                            }
                            else
                            {
                                pat = idNode;
                            }
                            m_enabledMetricsPatterns.add(pat);
                        }
                    }
                }
                else
                {
                    // see if input metric is an instance metric...
                    if (tempID != null)
                    {
                        instancePattern = Pattern.compile(idNode.getAbsoluteName());
                        instanceMatcher = instancePattern.matcher(tempID.getAbsoluteName());
                        if (instanceMatcher.lookingAt()) // For the moment, just try matching from the beginning of the metric id
                        {
                            //determine if instance metric and whether to enable parent...
                            // add to enabled list(s)...
                            m_enabledMetricsPatterns.add(ids[i]); // enable instance pattern
                            retEnabled.add(tempID); // but return instance parent node
                        }
                    }
                }
            }
        }

        IMetricIdentity[] retIds = (IMetricIdentity[])retEnabled.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);

        return retIds;
    }

    private IMetricIdentity getMetricParent(IMetricIdentity id)
    {
        String[] nameComponents = id.getNameComponents();
        int len = nameComponents.length;
        IMetricIdentity parentID = null;
        int index = 0;
        String[] parent = null;
        if ((len > 2)  && (nameComponents[len-1].equals("*")) && (nameComponents[len-2].equals("")))
        {
            index = len-2;
            parent = new String[len-2];
        }
        else
        {
            index = len - 1;
            parent = new String[len-1];
        }
        System.arraycopy(nameComponents, 0, parent, 0, index);
        parentID = new MetricIdentity(parent);
        return parentID;
    }

       /* Enable all matching metrics, including instance patterns.
        * Filter depending on whether hidden metrics are included
        * Caller is responsible for synchronization
        */
       private IMetricIdentity[] internalEnableMetrics(IMetricIdentity[] inputMetricIds, boolean includeHidden)
       {
           if (inputMetricIds == null)
        {
            throw new IllegalArgumentException("List of metric identities cannot be null");
        }

           if (DEBUG_CONFIG || DEBUG_METRICS)
           {
               System.out.println("[" + m_componentName + "] Enabling metrics:" );
               for (int i=0; i < inputMetricIds.length; i++)
            {
                System.out.println(" - " + inputMetricIds[i]);
            }
           }

           if (m_isClosing)
        {
            return IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY;
        }

           HashSet retNewlyEnabled = new HashSet();     // catch duplicates

           // iterate through all of the metrics defined for the component, and enable each defined metric that matches an input metric ID (or input pattern)
           Iterator iterator = m_metricInfos.values().iterator();  // iterator of all metrics defined for the component
           while (iterator.hasNext())  // iterate through list of metrics defined for the component
           {
               IMetricInfo definedInfo = (IMetricInfo)iterator.next();  // get MetricInfo record for defined metric
               if (((MetricInfo)definedInfo).isHidden() && !includeHidden)
            {
                continue;
            }

               IMetricIdentity definedIdNode = definedInfo.getMetricIdentity();
               for (int i = 0; i < inputMetricIds.length; i++)
               {
                   // is defined metric node a parent or individual metric (incl. instance parent)
                   if (definedIdNode.isInstanceOf(inputMetricIds[i]))  // Tests if the input metric is the parent the defined metric.
                   {
                       // SNC00076391 We no longer support enabling all instances by specifying the parent without a ".*" wild card
                       if (definedInfo.isInstanceMetric())
                    {
                        continue;
                    }

                       if (!m_enabledMetricsPatterns.contains(definedIdNode))  // make sure metric (or metric pattern) has not already been enabled (skip it if it already has been enabled)
                       {
                           m_enabledMetricsPatterns.add(definedIdNode);

                           if (!retNewlyEnabled.contains(definedIdNode))
                        {
                            retNewlyEnabled.add(definedIdNode);
                        }
                       }

                   }
                   else  // is defined metric node an instance pattern
                   {
                       // is it an instance pattern
                       if (inputMetricIds[i].isInstanceOf(definedIdNode) && definedInfo.isInstanceMetric()) // Tests if defined metric is the parent of the input metric && if the defined metric info record is marked as an instance parent (the "isInstanceMetric" method name is a bit of a misnomer - it should probably be "isInstanceMetricParent")
                       {
                           if (definedInfo.getMetricIdentity().getNameComponents().length < inputMetricIds[i].getNameComponents().length - 1)
                           {
                               // then they probably specified an invalid instance metric by not escaping "." with "%."
                               throw new IllegalArgumentException("Invalid metric identity: " + inputMetricIds[i] + " (instance names containing '.' should be escaped with '%')");
                           }
                           
                           if (!m_enabledMetricsPatterns.contains(inputMetricIds[i]))
                           {
                               m_enabledMetricsPatterns.add(inputMetricIds[i]);   // add input instance metric pattern to list of enabled metrics...
                               if (!retNewlyEnabled.contains(definedIdNode))
                             {
                                retNewlyEnabled.add(definedIdNode);            // ...but add instance *parent* metric node to list of newly-enabled metrics that is to be returned (if and only if it isn't already on the list)
                            }
                           }
                       }
                   }
               }
           }

           IMetricIdentity[] retNewlyEnabledIds = (IMetricIdentity[])retNewlyEnabled.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
           if (DEBUG_CONFIG || DEBUG_METRICS)
           {
               System.out.println("[" + m_componentName + "] Enable list for component " + retNewlyEnabledIds.length);
               for (int i = 0; i < retNewlyEnabledIds.length; i++)
            {
                System.out.println(" - " + retNewlyEnabledIds[i]);
            }
           }
           return retNewlyEnabledIds;
       }

    /**
     * Disable the set of matching metrics patterns.
     * When disabling instance parent nodes, patterns <node> or <node>.* are considered equivalent.
     *
     * Disable is called by the client and is only used for patterns.
     * The actual metric is disabled when the component calls back the MetricsManager with unregisterMetric()
     * Caller must synchronize on MetricsManager.this
     */
    private IMetricIdentity[] new_internalDisableMetrics(IMetricIdentity[] ids, boolean includeAll)
    {
        if (ids == null)
        {
            throw new IllegalArgumentException("List of metric identities cannot be null");
        }

        if (DEBUG_CONFIG || DEBUG_METRICS)
        {
            for (int i = 0; i < ids.length; i++)
            {
                System.out.println(" - " + ids[i]);
            }
        }

        HashSet disabled = new HashSet();

        IMetricIdentity mid = null;
        String metricNamePattern = null;

        for (int i = 0; i < ids.length; i++)
        {
            IMetricInfo info = getMetricInfo(ids[i]);

            // if its a parent (instance or otherwise) remove all the children
            if (info == null || (info.isInstanceMetric() && info.getMetricIdentity().equals(ids[i])))
            {
                mid = (IMetricIdentity) ids[i];
                metricNamePattern = mid.getAbsoluteName();
                Pattern pattern = Pattern.compile(metricNamePattern);
                Matcher matcher = pattern.matcher("");
                Iterator iterator = m_enabledMetricsPatterns.iterator();
                while (iterator.hasNext())
                {
                    IMetricIdentity enabledID = (IMetricIdentity)iterator.next();
                    if (enabledID.isInstanceOf(ids[i]))
                    {
                        IMetricInfo enaInfo = getMetricInfo(enabledID);
                        if (enaInfo != null && enaInfo.isDynamic())
                        {
                            iterator.remove();
                            disabled.add(enaInfo.getMetricIdentity()); // ensures we put instance parents in return list
                        }
                    }
                }
            }
            else if (info.isDynamic())  // "dynamic" indicates metric can be enabled/disabled at runtime!
            {
                m_enabledMetricsPatterns.remove(ids[i]);
                disabled.add(info.getMetricIdentity());
            }
            else  // else don't disable statics
            {
            	// no-op 
            }
        }

        IMetricIdentity[] retIds = (IMetricIdentity[])disabled.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
        if (DEBUG_CONFIG || DEBUG_METRICS)
        {
           System.out.println("[" + m_componentName + "] Disable list for component " + retIds.length);
           for (int i=0; i < retIds.length; i++)
        {
            System.out.println(" - " + retIds[i]);
        }
        }

        return retIds;
    }

    /**
     * Disable the set of matching metrics patterns.
     *
     * Disable is called by the client and is only used for patterns.
     * The actual metric is disabled when the component calls back the MetricsManager with unregisterMetric()
     * Caller must synchronize on MetricsManager.this
     */
    private IMetricIdentity[] internalDisableMetrics(IMetricIdentity[] ids, boolean includeAll)
    {
        if (ids == null)
        {
            throw new IllegalArgumentException("List of metric identities cannot be null");
        }

        if (DEBUG_CONFIG || DEBUG_METRICS)
        {
            for (int i=0; i < ids.length; i++)
            {
                System.out.println(" - " + ids[i]);
            }
        }

        HashSet disabled = new HashSet();

        for (int i = 0; i < ids.length; i++)
        {
            IMetricInfo info = getMetricInfo(ids[i]);

            // if its an instance parent then just disable any alerts defined for the parent
            if (info != null && (info.isInstanceMetric() && info.getMetricIdentity().equals(ids[i])))
            {
            	disableAlerts(getEnabledAlerts(new IMetricIdentity[] { ids[i] }));
            	continue;
            }

            // if its a parent (instance or otherwise) remove all the children
            if (info == null)
            {
                Iterator iterator = m_enabledMetricsPatterns.iterator();
                while (iterator.hasNext())
                {
                    IMetricIdentity enabledID = (IMetricIdentity)iterator.next();
                    if (enabledID.isInstanceOf(ids[i]))
                    {
                        IMetricInfo enabledInfo = getMetricInfo(enabledID);
                        if (enabledInfo.isDynamic())
                        {
                            iterator.remove();
                            disabled.add(enabledInfo.getMetricIdentity()); // ensures we put instance parents in return list
                        }
                    }
                }
            }
            else
            if (info.isDynamic())
            {
                m_enabledMetricsPatterns.remove(ids[i]);
                disabled.add(info.getMetricIdentity());
            }
            // else don't disable statics
        }

        IMetricIdentity[] retIds = (IMetricIdentity[])disabled.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
        if (DEBUG_CONFIG || DEBUG_METRICS)
        {
           System.out.println("[" + m_componentName + "] Disable list for component " + retIds.length);
           for (int i=0; i < retIds.length; i++)
        {
            System.out.println(" - " + retIds[i]);
        }
        }
        
        return retIds;
    }

    /**
     * Return expanded list of metric nodes that are not yet enabled.
     * Filter on whether hidden are allowed.
     * Caller must synchronize on MetricsManager.this
     */
    private IMetricIdentity[] internalEvaluateEnableMetrics(IMetricIdentity[] ids, boolean showAll)
    {
        HashSet retEval = new HashSet();    // catch duplicates
        Iterator it = m_metricInfos.values().iterator();
        while (it.hasNext())
        {
            IMetricInfo info = (IMetricInfo) it.next();
            if (((MetricInfo) info).isHidden() && !showAll)
            {
                continue;
            }

            IMetricIdentity id = info.getMetricIdentity();
            if (!m_enabledMetricsPatterns.contains(id))
            {
                retEval.add(id);        // return all nodes not specifically enabled
            }
            else
            {
                if (ids != null)
                {
                    for (int i = 0; i < ids.length; i++)
                    {
                        if ((id.isInstanceOf(ids[i]) // it's a parent node or individual (incl. instance parent) node
                            || (ids[i].isInstanceOf(id) && isInstancePattern(ids[i]))) // it's an instance pattern
                            && !m_enabledMetricsPatterns.contains(id))
                        {
                            retEval.add(ids[i]);
                        }
                    }
                }
            }
        }
        return (IMetricIdentity[])retEval.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
    }

    /**
     * Get metrics data for specified set of active metrics.
     * Add triggered alerts if requested
     * Filter on whether hidden or not
     * Caller is responsible for synchronization
    **/
    private IMetricsData internalGetMetricsData(IMetricIdentity[] ids, Boolean includeTriggeredAlerts, boolean includeAll)
    {
        Collection registeredMetrics;
        if (ids != null)    // for efficiency, only loop over given set of metrics
        {
            registeredMetrics = new HashSet();

            for (int i = 0; i < ids.length; i++)
            {
                // if input metric ID is in list of active metrics, add it to the hashset of registered metrics for which metric data will be obtained
                Object rm = m_activeMetrics.get(ids[i]);
                if (rm != null)  // found exact match for input metric in list of active metrics for this component
                {
                    registeredMetrics.add(rm);
                }
                else  // check for wildcard metric name
                {
                    // wildcard-check-and-handling
                    String[] inputNameComponents = ids[i].getNameComponents();
                    int inputLength = inputNameComponents.length;
                    if (inputNameComponents[inputLength-1].endsWith("*"))
                    {
                        // iterate through active metrics, searching for those that match the wildcard patter
                        Iterator it = m_activeMetrics.keySet().iterator();
                        while (it.hasNext())
                        {
                            IMetricIdentity activeMetricID = (IMetricIdentity) it.next();
                            String[] activeNameComponents = activeMetricID.getNameComponents();
                            if (activeNameComponents.length > inputLength)  // for now, will skip over active metric if its array of name components is not the same length as that of the input metric....
                            {
                                //TODO: will eventually want to allow this case, to allow for a greater range of wildcard pattern support
                                continue;
                            }
                            else if (inputLength > activeNameComponents.length)  // for now, will skip over active metric if its array of name components is not the same length as that of the input metric....
                            {
                                continue;
                            }

                            boolean match = true;  // assume that the active metric name matches the input wildcard metric id name...
                            for (int j = 0; j < activeNameComponents.length-1; j++)  // compare name components, except for last name component  (i.e. compare "broker.connections.Count" to "broker.connections.*", skipping the comparison between "Count" and "*")
                            {
                                if (!activeNameComponents[j].equals(inputNameComponents[j]))
                                {
                                    match = false;
                                    break;  // out of "for (int j = 0; ..."
                                }
                            }
                            if (match)
                            {
                                rm = m_activeMetrics.get(activeMetricID);
                                registeredMetrics.add(rm);
                            }
                        }
                    }
                }
            }
        }
        else
        {
            registeredMetrics = m_activeMetrics.values();
        }

        // get data for selected metrics
        IMetricsData data =  internalGetMetricsData(registeredMetrics);
        if (!includeTriggeredAlerts.booleanValue())
        {
            return data;
        }
        else
        {
            return updateTriggeredAlerts(data);
        }
    }

    // given a set of metrics data, add triggered alerts for these metrics
    private IMetricsData updateTriggeredAlerts(IMetricsData data)
    {
        IMetric[] metrics = data.getMetrics();
        if (metrics == null)
        {
            return data;
        }

        // get exceeded alerts
        List exceededAlerts = new ArrayList();
        synchronized (m_activeAlerts)
        {
            Iterator it = m_activeAlerts.iterator();
            while (it.hasNext())
            {
                Alert al = (Alert) it.next();
                if (al.isExceeded())
                {
                    exceededAlerts.add(al);
                }
            }
        }
        if (exceededAlerts.isEmpty())
         {
            return data;    // no exceeded alerts
        }

        // add triggered alerts to matching metrics
        for (int i=0; i < metrics.length; i++)
        {
            ArrayList triggeredAls = null;   // may not need it
            Iterator it = exceededAlerts.iterator();
            while (it.hasNext())
            {
                IAlert al = (IAlert) it.next();
                if (al.getMetricIdentity().equals(metrics[i].getMetricIdentity()))
                {
                    if (triggeredAls == null)
                    {
                        triggeredAls = new ArrayList();
                    }
                    triggeredAls.add(al);
                }
            }
            if (triggeredAls == null)
            {
                continue;
            }
            ((Metric)metrics[i]).setTriggeredAlerts((IAlert[])triggeredAls.toArray(EMPTY_ALERT_ARRAY));
        }
        return MetricsFactory.createMetricsData(metrics, data.getCurrencyTimestamp());
    }

    // Get metrics data for a given collection of metrics.
    // Other public methods are filters that end up calling this method.
    // caller must synchronize on MetricsManager.this
    private IMetricsData internalGetMetricsData(Collection registeredMetrics)
    {
        ArrayList metrics = new ArrayList();    // data
        long currencyTimestamp = 0;
        long minCurrencyTimestamp = Long.MAX_VALUE;
        long maxCurrencyTimestamp = Long.MIN_VALUE;

        // NOTE: There is no easy way to ensure we get correctly timestamped data in an
        //       efficient way - we could put a timestamp on each statistic as to when
        //       it was refreshed, but that would be a lot of overhead.
        //       The compromise is that we will at least *try* to capture the metrics
        //       outside of the bounds of a refresh cycle.
        short retryCount = 0;
        retryLoop:
        while (retryCount < 2)
        {
            Iterator iterator = registeredMetrics.iterator();
            currencyTimestamp = m_currencyTimestamp;
            long nextRefreshTime = m_currencyTimestamp + m_refreshInterval;
            while (iterator.hasNext())
            {
                RegisteredMetric enabledMetricDetails = (RegisteredMetric)iterator.next();

                IMetricIdentity id = enabledMetricDetails.getMetricIdentity();
                ((MetricIdentity)id).setHash(enabledMetricDetails.getHash());

                long[] value = null;
                value = enabledMetricDetails.evaluate();
                // early sanity check - make assumption that the currencyTimestamp
                // is somewhere near the refresh scheduled time (and if we don't
                // get out early here there is always a check when we get out of this
                // loop
                if (value[1] > nextRefreshTime)
                {
                    retryCount++;
                    continue retryLoop;
                }
                if (value[1] < minCurrencyTimestamp)
                {
                    minCurrencyTimestamp = value[1];
                }
                if (value[1] > maxCurrencyTimestamp)
                {
                    maxCurrencyTimestamp = value[1];
                }
                 metrics.add(MetricsFactory.createMetric(id, value[0], value[1]));
            }

            // if no refresh cycle occured while we were evaluating the metrics, return the
            // data we got otherwise try once more
            if (m_currencyTimestamp != currencyTimestamp)
            {
                retryCount++;
                continue retryLoop;
            }

            // else we got a set of values between refresh cycles
            break;
        }

        // to have the data objects currency timestamp be a little more representative, we'll take the
        // max and min currency timestamps of the individual metrics and average them
        currencyTimestamp = (minCurrencyTimestamp + maxCurrencyTimestamp) / 2;

        return MetricsFactory.createMetricsData((IMetric[])metrics.toArray(EMPTY_METRIC_ARRAY), currencyTimestamp);
    }

    /**
     * get a matching alert in a set.
     * caller must ensure synchronization
     */
    private IAlert matchAlert(Set set, IAlert alert)
    {
        IAlert ret = null;
        Iterator it = set.iterator();
        while (it.hasNext())
        {
            IAlert match = (IAlert) it.next();
            if (alert.equals(match))
            {
                ret = match;
                break;
            }
        }
        return ret;
    }

    /**
     * get a matching alert in a set within
     * the metric identity only.
     * caller must ensure synchronization
     */
    private IAlert matchMetricIdentity(Set set, IAlert alert)
    {
        IAlert ret = null;
        Iterator it = set.iterator();
        while (it.hasNext())
        {
            IAlert match = (IAlert) it.next();
            if (alert.sameMetricIdentity(match.getMetricIdentity()))
            {
                ret = match;
                break;
            }
        }
        return ret;
    }

    /**
         * Need to use this method to get the associated info object as in the case of an
         * instance metric member we need to get the info for the parent.
         */
        public IMetricInfo getMetricInfo(IMetricIdentity id)
        {
            if (id == null)
            {
                return null;
            }

            IMetricInfo info = (IMetricInfo)m_metricInfos.get(id);

            if (info == null)
            {
                // try getting the parent and verifying it is an instance metric
                String[] nameComponents = id.getNameComponents();
                if (nameComponents.length == 1)
                {
                    return null;
                }

                String[] parentNameComponents = new String[nameComponents.length - 1];
                System.arraycopy(nameComponents, 0, parentNameComponents, 0, parentNameComponents.length);

                // create the parent id and look its info object
                id = MetricsFactory.createMetricIdentity(parentNameComponents);
                info = (IMetricInfo)m_metricInfos.get(id);

                if (info == null || !info.isInstanceMetric())
                {
                    return null;
                }
            }
            return info;
        }

    /**
     * Need to use this method to get the associated info object as in the case of an
     * instance metric member we need to get the info for the parent.
     */
    public IMetricInfo new_getMetricInfo(IMetricIdentity id)
    {
        if (id == null)
        {
            return null;
        }

        IMetricInfo info = (IMetricInfo)m_metricInfos.get(id);

        if (info == null)
        {
            // try getting the parent and verifying it is an instance metric
            String[] nameComponents = id.getNameComponents();
            if (nameComponents.length == 1)
            {
                return null;
            }

            // create the parent id and look at its info object
            id = getParentMetricID(id);
            info = (IMetricInfo)m_metricInfos.get(id);

            if (info == null || !info.isInstanceMetric())
            {
                return null;
            }
        }
        return info;
    }

    /**
     *  get parent metric identity
     */
    private synchronized IMetricIdentity getParentMetricID( IMetricIdentity instanceID)
    {
        IMetricIdentity parentID = null;
        String absoluteName = instanceID.getAbsoluteName();
        int pos = absoluteName.lastIndexOf(".");
        String sub = absoluteName.substring(0,pos);
        while (sub.lastIndexOf(".") == (sub.length()-1))
        {
            sub = sub.substring(0,sub.lastIndexOf("."));
        }
        parentID = MetricsFactory.createMetricIdentity(sub);
        return parentID;
    }

    /**
     * Refresh all statistics
    */
    private synchronized void refresh()
    {
        if (DEBUG_STATS)
        {
            if (DEBUG_AGENT || !m_isAgent)
            {
                System.out.println(System.currentTimeMillis() + " [" + m_componentName + "] Refreshing ...");
            }
        }

        // update refresh time
        MetricsManager.this.m_currencyTimestamp = System.currentTimeMillis();


        // perform this refresh cycle on registered statistics
        // elapsedCollectionTimeStatistic will be refreshed as well, as it is a registered statistic
        Object[] statistics = m_statistics.toArray();
        for (int i = 0; i < statistics.length; i++)
        {
            IStatistic statistic = (IStatistic)statistics[i];
            // if its a sampled statistic then call the providers to update
            if (statistic instanceof ISampledStatistic)
            {
                IStatisticProvider[] providers = ((ISampledStatistic)statistic).getStatisticProviders();
                for (int j = 0; j < providers.length; j++)
                {
                    providers[j].updateStatistic((ISampledStatistic)statistic);
                }
            }
            statistic.refresh(); // synchronized
        }

        if (DEBUG_STATS)
        {
            if (DEBUG_AGENT || !m_isAgent)
            {
                m_elapsedCollectionTimeStatistic.printValues();
            }
        }

        // Are there any alerts?
        // Must evaluate all associated metrics
        synchronized (m_activeAlerts)
        {
            if (!m_activeAlerts.isEmpty())
            {
                Iterator activeMetrics = m_activeMetrics.entrySet().iterator();
                while (activeMetrics.hasNext())
                {
                    Map.Entry activeMetric = (Map.Entry)activeMetrics.next();
                    IMetricIdentity id = (IMetricIdentity)activeMetric.getKey();
                    
                    Alert highAlert = null;
                    Alert lowAlert = null;
                    
                    Iterator activeAlerts = m_activeAlerts.iterator();
                    long[] value = null;

                    // look through all of the alerts to see if any match the identity of the active metric
                    while (activeAlerts.hasNext())
                    {
                        Alert alert = (Alert)activeAlerts.next();
                        if (alert.getMetricIdentity().equals(id))
                        {
                            if (value == null)
                            {
                                value = ((RegisteredMetric)activeMetric.getValue()).evaluate();
                            }
                            
                            // should a metric alert notification be sent
                            if (alert.check(value[0], m_repeatAlerts))
                            {
                                // .. include test to see if its higher/lower than an already found high/low threshold,
                                // since in the case of multiple defined thresholds, we should only send an alert for
                                // the most extreme
                                if (alert.isHighThreshold()) // its a high threshold
                                {
                                    if (highAlert == null || highAlert.getThresholdValue() < alert.getThresholdValue())
                                    {
                                        highAlert = alert;
                                    }
                                }
                                else // its a low threshold
                                {
                                    if (lowAlert == null || lowAlert.getThresholdValue() > alert.getThresholdValue())
                                    {
                                        lowAlert = alert;
                                    }
                                }
                            }
                        }
                    }
                    
                    // should a high threshold alert be sent
                    if (highAlert != null)
                    {
                        sendAlertNotification((IAlert)highAlert, value[0]);
                    }
                    // should a low threshold alert be sent
                    if (lowAlert != null)
                    {
                        sendAlertNotification((IAlert)lowAlert, value[0]);
                    }
                }
            }
        }
    }

    private class MetricRefresher
    extends TimerTask
    {
        @Override
        public void run()
        {
            synchronized(MetricsManager.this)
            {
                if (MetricsManager.this.m_isClosing)
                {
                    return;
                }

                long currentTime = System.currentTimeMillis();
                long scheduledTime = super.scheduledExecutionTime();

                if (DEBUG_STATS)
                {
                    if (DEBUG_AGENT || !m_isAgent)
                    {
                        System.out.println(currentTime + " [" + m_componentName + "] MetricsManager$MetricRefresher: scheduled=" + scheduledTime + " late= " + (currentTime - scheduledTime));
                    }
                }

                // if we are hopelessly behind then restart the whole schedule and statistic history
                if (currentTime - scheduledTime > MetricsManager.this.m_collectionInterval)
                {
                    if (DEBUG_STATS)
                    {
                        if (DEBUG_AGENT || !m_isAgent)
                        {
                            System.out.println(currentTime + " [" + m_componentName + "] MetricsManager$MetricRefresher: Start Over! ");
                        }
                    }
                    MetricsManager.this.startRefreshing(); // will cause existing historical values to be cleared
                    return;
                }

                // refresh
                MetricsManager.this.refresh();
            }
        }
    }
    
    // holds the association of a metric with its analyzer and statistic
    private class RegisteredMetric
    {
        private IMetricIdentity m_metricId;
        private IStatistic m_statistic;
        private IMetricAnalyzer m_analyzer;
        private long m_hash;

        RegisteredMetric(IMetricIdentity id, IStatistic stat, IMetricAnalyzer ana, long hash)
        {
            m_metricId = id;
            m_statistic = stat;
            m_analyzer = ana;
            m_hash = hash;
        }

        IMetricIdentity getMetricIdentity() { return m_metricId;}
        IStatistic getStatistic() { return m_statistic;}
        IMetricAnalyzer getAnalyzer() { return m_analyzer;}
        long getHash() { return m_hash;}

        // evaluate metric
        long[] evaluate()
        {
            if (DEBUG_STATS)
            {
                if (DEBUG_AGENT || !m_isAgent)
                {
                    if (m_statistic instanceof HistoricalStatistic)
                    {
                        ((HistoricalStatistic)m_statistic).printValues();
                        m_elapsedCollectionTimeStatistic.printValues();
                    }
                }
            }
            long[] value = m_analyzer.evaluateValue(m_statistic);
            if (DEBUG_STATS)
            {
                if (DEBUG_AGENT || !m_isAgent)
                {
                    System.out.println("[" + m_componentName + "] Id = " + m_metricId + " Value = " + value[0] + " Time = " + value[1]);
                }
            }
            return value;
        }
    }

    private static class MetricsManagerDiagnostics
    extends AbstractDiagnosticsProvider
    {
        private static String[] OPERATIONS;
        private static HashMap SHOW_TRACE_LEVEL_PARAM_DESCIPTOR = new HashMap();
        private static HashMap UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR = new HashMap();
        private static HashMap PARAM_DESCRIPTOR = new HashMap();
        private static HashMap DUMP_STATE_PARAM_DESCIPTOR = new HashMap();
        private static HashMap DESCRIBE_PARAM_DESCIPTOR = new HashMap();
        private static HashMap LIST_DIAGNOSTICS_INSTANCES_PARAM_DESCIPTOR = new HashMap();

        String description = "Use this subsystem to help diagnose the Sonic MF metrics management subsystem."
                           + NEWLINE
                           + NEWLINE
                           + "The diagnosis includes tracing of statistics and their metric usage."
                           + NEWLINE
                           + "Two diagnosed object instances are used to distinguish between statistics (\"Statistics\") and metrics (\"Metrics\")."
                           + NEWLINE
                           + "Tracing can be configured so to provide various levels of verbosity. The various trace options are:"
                           + NEWLINE
                           + " 1 - verbose : provide more detailed or frequent tracing."
                           + NEWLINE
                           + " 2 - registration counts : for every 500 (100 with verbose bit set) unregistrations dumps registration counts"
                           + NEWLINE
                           + NEWLINE
                           + "Examples:"
                           + NEWLINE
                           + "  sonic.mf.metrics updateTraceLevel doiID=Statistics integerTraceLevel=2"
                           + NEWLINE
                           + "  sonic.mf.metrics dumpState"
                           + NEWLINE;

        static
        {
            UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR.put(DIAGNOSTICS_OBJECT_ID_PARAM, "values: \"" + SDF_DOIID_STATISTICS + "\", \"" + SDF_DOIID_METRICS + "\"");
            UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR.put(INTEGER_TRACE_LEVEL_PARAM, "bits: 1 - verbose, 2 - registration counts");

            PARAM_DESCRIPTOR.put(DUMP_STATE_OP, DUMP_STATE_PARAM_DESCIPTOR);
            PARAM_DESCRIPTOR.put(DESCRIBE_OP, DESCRIBE_PARAM_DESCIPTOR);
            PARAM_DESCRIPTOR.put(SHOW_TRACE_LEVEL_OP, SHOW_TRACE_LEVEL_PARAM_DESCIPTOR);
            PARAM_DESCRIPTOR.put(UPDATE_TRACE_LEVEL_OP, UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR);
            PARAM_DESCRIPTOR.put(LIST_DIAGNOSTICS_INSTANCES_OP, LIST_DIAGNOSTICS_INSTANCES_PARAM_DESCIPTOR);

            OPERATIONS = toOpnameArray(PARAM_DESCRIPTOR);
        }

        MetricsManagerDiagnostics()
        {
            super("sonic.mf.metrics");
        }

        @Override
        public String describe()
        {
            return this.description;
        }

        @Override
        public String[] getOperations()
        {
            return OPERATIONS;
        }

        @Override
        public HashMap describeParameters(String operationName)
        {
            return (HashMap)PARAM_DESCRIPTOR.get(operationName);
        }

        @Override
        public String[] getDOInstances()
        {
            return new String[] { SDF_DOIID_STATISTICS, SDF_DOIID_METRICS };
        }

        @Override
        public void updateTraceLevel(String doiID, HashMap parameters, StringBuffer buffer)
        {
            try
            {
                if (doiID.equals(SDF_DOIID_STATISTICS))
                {
                    MetricsManager.m_sdfStatisticsTraceMask = parseTraceLevel(doiID, parameters, buffer, MetricsManager.m_sdfStatisticsTraceMask, MetricsManager.m_sdfMetricsTraceMask);
                }
                if (doiID.equals(SDF_DOIID_METRICS))
                {
                    MetricsManager.m_sdfMetricsTraceMask = parseTraceLevel(doiID, parameters, buffer, MetricsManager.m_sdfMetricsTraceMask, MetricsManager.m_sdfStatisticsTraceMask);
                }
            }
            catch (Exception e)
            {
                e.printStackTrace();
                buffer.append(e.toString());
            }
        }

        @Override
        public void showTraceLevel(String doiID, HashMap parameters, StringBuffer buffer)
        {
            if (doiID.equals(SDF_DOIID_STATISTICS))
            {
                buffer.append("Trace level for \"").append(super.m_subsystemName).append("\" [").append(doiID).append("] is ").append(MetricsManager.m_sdfStatisticsTraceMask);
            }
            if (doiID.equals(SDF_DOIID_METRICS))
            {
                buffer.append("Trace level for \"").append(super.m_subsystemName).append("\" [").append(doiID).append("] is ").append(MetricsManager.m_sdfMetricsTraceMask);
            }
        }

        @Override
        public void appendStateDump(String doiID, HashMap Parameters, StringBuffer buffer)
        {
            IStateWriter writer = null;
            try
            {
                writer = m_diagnosticsContext.getStateWriter();

                Iterator metricsManagers = ((HashSet)MetricsManager.m_metricsManagers.clone()).iterator();
                while (metricsManagers.hasNext())
                {
                    dumpMetricsManager(writer, (MetricsManager)metricsManagers.next());
                }
            }
            catch (Exception e)
            {
                buffer.append("Failed to use state file: " + e);
                return;
            }
            finally
            {
                if (writer != null)
                {
                    writer.close();
                }
            }

            if (writer != null) {
            	buffer.append("Dump of \"").append(super.m_subsystemName).append("\" written to ").append(writer.getFilePath());
            }
        }

        private String createIndent(int offset)
        {
            String indent = "";
            for (int i = 0; i < offset; i++)
            {
                indent += "  ";
            }

            return indent;
        }


        private void dumpMetricsManager(IStateWriter writer, MetricsManager metricsManager)
        throws Exception
        {
            writer.write(new Date().toString());
            writer.write(NEWLINE);
            writer.write("+ <metrics manager> " + metricsManager.m_componentName);
            writer.write(NEWLINE);
            
            HashSet usedStatistics = new HashSet();
            
            synchronized(metricsManager)
            {
                writer.write("  + <active metrics>  (count=" + metricsManager.m_activeMetrics.size() + ")");
    
                if (metricsManager.m_activeMetrics.size() == 0)
                {
                    writer.write(NEWLINE);
                    writer.write("    - <none>");
                }
                else
                {
                    Iterator activeMetrics = metricsManager.m_activeMetrics.values().iterator();
                    while (activeMetrics.hasNext())
                    {
                        RegisteredMetric registeredMetric = (RegisteredMetric)activeMetrics.next();
                        writer.write(NEWLINE);
                        writer.write("    - " + registeredMetric.m_metricId.getName() + " (statistic=" + registeredMetric.m_statistic.hashCode() + ")");
                        
                        usedStatistics.add(registeredMetric.m_statistic);
                    }
                }
                
                writer.write(NEWLINE);
                // we don't report the hidden m_elapsedCollectionTimeStatistic statistic
                writer.write("  + <statistics>  (count=" + (metricsManager.m_statistics.size() - 1) + ")");
    
                if (metricsManager.m_statistics.size() == 1)
                {
                    writer.write(NEWLINE);
                    writer.write("    - <none>");
                }
                else
                {
                    Iterator statistics = metricsManager.m_statistics.iterator();
                    while (statistics.hasNext())
                    {
                        IStatistic statistic = (IStatistic)statistics.next();
                        
                        if (statistic == metricsManager.m_elapsedCollectionTimeStatistic)
                        {
                            continue;
                        }
                        
                        writer.write(NEWLINE);
                        String type = statistic.getClass().getName();
                        type = type.substring(type.lastIndexOf('.') + 1);
                        writer.write("    - " + statistic.hashCode());
                        if (!usedStatistics.contains(statistic))
                        {
                            writer.write("*");
                        }
                        writer.write(" (type=" + type + ")");
                    }
                }
            }
            writer.write(NEWLINE);
        }

        private int parseTraceLevel(String doiID, HashMap parameters, StringBuffer buffer, int currentTraceMask, int otherTraceMask)
        throws Exception
        {
            String traceLevel = (String)parameters.get(INTEGER_TRACE_LEVEL_PARAM);
            if (traceLevel == null)
            {
                traceLevel = new String("0");
            }

            int traceMask = Integer.parseInt(traceLevel);

            if (traceMask == SDF_TRACE_NOTHING)
            {
                if (currentTraceMask > 0 && otherTraceMask == 0)
                {
                    if (super.m_tracer != null)
                    {
                        super.m_tracer.close();
                        super.m_tracer = null;
                    }
                }
            }
            if (traceMask > SDF_TRACE_NOTHING)
            {
                if (super.m_tracer == null)
                {
                    super.m_tracer = super.m_diagnosticsContext.getTracer();
                }
                buffer.append("Trace file for \"").append(super.m_subsystemName).append("\" [").append(doiID).append("] is \"").append(super.m_tracer.getFilePath()).append("\"");
                super.m_tracer.trace("Start tracing " + super.m_subsystemName + '.' + doiID + " with tracing level " + traceLevel, false);
            }

            return traceMask;
        }

        private void traceStatisticRegistrations(MetricsManager metricsManager)
        {
            if ((MetricsManager.m_sdfStatisticsTraceMask & MetricsManager.SDF_TRACE_REGISTRATION_COUNTS) == 0)
            {
                return;
            }
            
            ITracer tracer = m_tracer;
            if (tracer != null)
            {
                synchronized(metricsManager)
                {
                    int factor = (MetricsManager.m_sdfStatisticsTraceMask & MetricsManager.SDF_TRACE_VERBOSE) == 0 ? 500 : 100;
                    if (metricsManager.m_statisticUnregistrations % factor != 0)
                    {
                        return;
                    }
                
                    StringBuffer sb = new StringBuffer();
                    
                    sb.append("<statistic registrations> " + metricsManager.m_componentName);
                    sb.append(" registrations=").append(metricsManager.m_statisticRegistrations);
                    sb.append(", unregistrations=").append(metricsManager.m_statisticUnregistrations);
                    
                    try
                    {
                        tracer.trace(sb.toString(), false);
                    }
                    catch (IOException e)
                    {
                        e.printStackTrace();
                    }
                }
            }
        }

        private void traceMetricRegistrations(MetricsManager metricsManager)
        {
            if ((MetricsManager.m_sdfMetricsTraceMask & MetricsManager.SDF_TRACE_REGISTRATION_COUNTS) == 0)
            {
                return;
            }
            
            ITracer tracer = m_tracer;
            if (tracer != null)
            {
                synchronized(metricsManager)
                {
                    int factor = (MetricsManager.m_sdfMetricsTraceMask & MetricsManager.SDF_TRACE_VERBOSE) == 0 ? 500 : 100;
                    if (metricsManager.m_metricUnregistrations % factor != 0)
                    {
                        return;
                    }
                
                    StringBuffer sb = new StringBuffer();
                    
                    sb.append("<metric registrations> " + metricsManager.m_componentName);
                    sb.append(" registrations=").append(metricsManager.m_metricRegistrations);
                    sb.append(", unregistrations=").append(metricsManager.m_metricUnregistrations);
                    
                    try
                    {
                        tracer.trace(sb.toString(), false);
                    }
                    catch (IOException e)
                    {
                        e.printStackTrace();
                    }
                }
            }
        }
    }
}
