package com.sonicsw.mf.common.metrics.manager.impl;

import com.sonicsw.mf.common.metrics.manager.IHistoricalStatistic;
import com.sonicsw.mf.common.metrics.manager.IMetricAnalyzer;
import com.sonicsw.mf.common.metrics.manager.IStatistic;

/**
 * A metric analyzer that returns the peak rate based on a single value or set of historical values.
 *
 * The following rates equate to a given metric type:
 *
 *  100     - peak utilization
 *  1000    - peak per second rate
 *  60000   - peak per minute rate
 *  3600000 - peak per hour rate
 */
public class PeakRate
extends AbstractTimeBasedAnalyzer
implements IMetricAnalyzer
{
    protected long m_factor;

    public PeakRate(MetricsManager metricsManager, long factor)
    {
        super(metricsManager);

        if (factor < 1)
        {
            throw new IllegalArgumentException("Factor cannot be less than 1");
        }

        m_factor = factor;
    }

    /**
     * @see IMetricsAnalyzer#evaluateValue(IStatistic)
     */
    @Override
    public long[] evaluateValue(IStatistic statistic)
    {
        long[] value = new long[2];
        value[0] = 0;
        value[1] = statistic.getCurrencyTimestamp();

        if (statistic instanceof IHistoricalStatistic)
        {
            long[] values;
            long[] timeValues;
            synchronized(super.m_metricsManager)
            {
                synchronized(statistic)
                {
                    values = ((IHistoricalStatistic)statistic).getLastValues();
                }
                timeValues = super.m_metricsManager.m_elapsedCollectionTimeStatistic.getLastValues();
            }

            // get the max rate over all intervals
            if (!(values == null))
            {
                int len = values.length;
                if (timeValues.length < len)
                 {
                    len = timeValues.length;    // shouldn't happen
                }
                for (int i = 0; i < len; i++)
                {
                    long intervalRate = 0;
                    long intervalTime = timeValues[i];
                    if (intervalTime > 0)
                    {
                        // avoid div/0
                        intervalRate = Math.round((double)values[i] * m_factor / (double)intervalTime);
                    }

                    if (intervalRate > value[0])
                    {
                        value[0] = intervalRate;
                    }
                }
            }
        }
        else
        {
            // non-historical statistic - not really meaningful, same as rate
            long elapsedTime;
            synchronized(super.m_metricsManager)
            {
                synchronized(statistic)
                {
                    value[0] = statistic.getCurrentValue();
                    elapsedTime = System.currentTimeMillis() - ((Statistic)statistic).m_resetTimestamp;
                }
            }
            if (elapsedTime > 0)
            {
                // ensure full refresh cycle has elapsed
                value[0] = Math.round((double)value[0] * m_factor / (double)elapsedTime);
            }
        }
        return value;
    }
}
