package com.sonicsw.mf.common.metrics.manager.impl;

import com.sonicsw.mf.common.metrics.manager.IHistoricalStatistic;
import com.sonicsw.mf.common.metrics.manager.IMetricAnalyzer;
import com.sonicsw.mf.common.metrics.manager.IStatistic;

/**
 * A metric analyzer that returns the per rate of a single or set of historical values.
 *
 * The following rates equate to a given metric type:
 *
 *  100     - utilization
 *  1000    - per second rate
 *  60000   - per minute rate
 *  3600000 - per hour rate
 */
public class Rate
extends AbstractTimeBasedAnalyzer
implements IMetricAnalyzer
{
    protected long m_factor;
    boolean DEBUG = false;

    public Rate(MetricsManager metricsManager, long factor)
    {
        super(metricsManager);

        if (factor < 1)
        {
            throw new IllegalArgumentException("Factor cannot be less than 1");
        }

        m_factor = factor;
    }

    /**
     * @see IMetricsAnalyzer#evaluateValue(IStatistic)
     */
    @Override
    public long[] evaluateValue(IStatistic statistic)
    {
        long elapsedTime = 0;
        long[] value = new long[2];
        long lastValue = 0;

        if (statistic instanceof IHistoricalStatistic)
        {
            long[] values = null;
            synchronized(super.m_metricsManager)
            {
                synchronized(statistic)
                {
                    values = ((IHistoricalStatistic)statistic).getLastValues();
                    value[1] = statistic.getCurrencyTimestamp();
                }
                if (values != null)
                {
                    elapsedTime = super.m_metricsManager.getElapsedCollectionTime(values.length);
                }
                if (DEBUG)
                {
                    System.out.println("Computing " + this + " for historical statistic " + statistic);
                    ((HistoricalStatistic)statistic).printValues();
                    System.out.println("Elapsed Time:");
                    super.m_metricsManager.m_elapsedCollectionTimeStatistic.printValues();
                }
            }
            if (!(values == null))
            {
                for (int i = 0; i < values.length; i++)
                {
                    lastValue += values[i];
                }
            }
        }
        else
        {
            // Non-historical statistic - calculate rate since last reset() based on
            // current time and value.
            synchronized(super.m_metricsManager)
            {
                synchronized(statistic)
                {
                    lastValue = statistic.getCurrentValue();
                    value[1] = statistic.getCurrencyTimestamp();
                    elapsedTime = value[1] - ((Statistic)statistic).m_resetTimestamp;
                }
            }
        }

        if (elapsedTime > 0)
        {
            // ensure full refresh cycle has elapsed
            value[0] = Math.round((double)lastValue * m_factor / (double)elapsedTime);
        }

        if (DEBUG)
        {
            System.out.println(this + " for " + statistic +  " last value = " + lastValue + " elapsed = " + elapsedTime + " value = " + value[0]);
        }

        return value;
    }

    @Override
    public String toString()
    {
        return "Rate *" + m_factor;
    }
}
