package com.sonicsw.mf.common.metrics.manager.impl;

import com.sonicsw.mf.common.metrics.manager.ISampledStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatistic;

/**
 * The default implementation of IStatistic.
 *
 * @see StatisticsFactory#createStatistic(short, boolean, IStatisticProvider[], short)
 */
public class Statistic
implements IStatistic
{
    public short m_updateMode;
    public boolean m_intervalMode;
    private boolean DEBUG = false;

    // the working value
    protected long m_currentUpdateValue;
    // the working min or max value for counter style updates
    protected long m_currentMinMaxValue;
    // the actual value of the last update for maximum or minimum or difference mode
    protected long m_lastUpdateValue;
    // the running update count
    protected long m_numUpdates;

    // the value at the last refresh interval
    protected long m_currentIntervalValue;
    // the value as of the previous to last refresh
    protected long m_lastIntervalValue;

    // the timestamp of when the value was last refreshed
    protected long m_currencyTimestamp = INVALID_TIME;
    // the timestamp of when the value was previously refreshed
    protected long m_lastCurrencyTimestamp = INVALID_TIME;
    // the timestamp of last reset
    public long m_resetTimestamp = INVALID_TIME;

    // initial value on reset
    protected long m_initialValue = 0;

    protected boolean m_isVirgin = true;

    protected static final long INVALID_TIME = -1;

    // for debug
    String m_name;

    /**
     * Creates a statistic with the given mode (see <>_MODE values).
     *
     * @param updateMode      The update mode of the statistic value.
     * @param intervalMode    If true, a historical statistic returns the difference between the actual value
     *                        at the last refresh interval and the value at the previous refresh interval.
     */
    public Statistic(short updateMode, boolean intervalMode)
    {
        m_updateMode = updateMode;
        m_intervalMode = intervalMode;
        initializeStatistic();
    }

    /**
     * @see IStatistic#getCurrentValue()
     */
    @Override
    public synchronized long getCurrentValue()
    {
        // for MIN/MAX counter mode we need to have assessed over a full refresh interval (can't take a partial interval
        // and hence can't have the instantaneous value)
        if (m_updateMode == MAXIMUM_COUNTER_MODE || m_updateMode == MINIMUM_COUNTER_MODE)
        {
            return m_currentIntervalValue;
        }
        return m_currentUpdateValue;
    }

    /**
     * @see IStatistic#getLastValue()
     */
    @Override
    public synchronized long getLastValue(){ return (m_intervalMode ? m_currentIntervalValue - m_lastIntervalValue : m_currentIntervalValue);
    }

    /**
     * @see IStatistic#getLastUpdateCount()
     */
    @Override
    public long getLastUpdateCount() { return m_numUpdates; }

    /**
     * @see IStatistic#getCurrencyTimestamp()
     */
    @Override
    public synchronized long getCurrencyTimestamp()
    {
        if (this instanceof ISampledStatistic)
        {
            return m_currencyTimestamp;
        }
        else
        {
            // actual time for non-sampled statistics
            return System.currentTimeMillis();
        }
    }

    /**
     * @see IStatistic#getLastRefreshCycleDuration()
     */
    @Override
    public synchronized long getLastRefreshCycleDuration()
    {
        if (m_lastCurrencyTimestamp == INVALID_TIME || m_currencyTimestamp == INVALID_TIME)
        {
            return INVALID_TIME;
        }

        return m_currencyTimestamp - m_lastCurrencyTimestamp;
    }

    /**
     * Update the statistic value. The given value will be interpreted in different ways dependent upon
     * the mode setting of this statistic (see <>_MODE values).
     */
    @Override
    public synchronized void updateValue(long value)
    {
        m_numUpdates++;
        
        long previousValue = m_lastUpdateValue;  // remember previous update value
        m_lastUpdateValue = value;  // actual value
        switch (m_updateMode)
        {
            case VALUE_MODE:
                m_currentUpdateValue = value;
                break;
            case DIFFERENCE_MODE:
                m_currentUpdateValue = value - previousValue;
                break;
            case COUNTER_MODE:
                m_currentUpdateValue += value;
                break;
            case MAXIMUM_COUNTER_MODE:
                m_currentUpdateValue += value;
                if (m_currentUpdateValue > m_currentMinMaxValue)
                {
                    m_currentMinMaxValue = m_currentUpdateValue;
                }
                break;
            case MINIMUM_COUNTER_MODE:
                m_currentUpdateValue += value;
                if (m_currentUpdateValue < m_currentMinMaxValue)
                {
                    m_currentMinMaxValue = m_currentUpdateValue;
                }
                break;
            case MAXIMUM_MODE:
                if (value > m_currentUpdateValue)
                {
                    m_currentUpdateValue = value;
                }
                break;
            case MINIMUM_MODE:
                if (value < m_currentUpdateValue)
                {
                    m_currentUpdateValue = value;
                }
                break;
        }

        m_isVirgin = false;

        if (DEBUG)
        {
            System.out.println(System.currentTimeMillis() + " (" + this + ") Updated value = " + value + " currentUpdateValue = " + m_currentUpdateValue + " count = " + m_numUpdates);
        }
    }

    /**
     * @see IStatistic#refresh()
     */
    @Override
    public synchronized void refresh()
    {
        if (DEBUG)
        {
            System.out.println(System.currentTimeMillis() + " (" + this + ") Refreshing Statistic - current value = " + m_currentUpdateValue);
        }

        // record value and count at current and previous refresh cycle
        m_lastIntervalValue = m_currentIntervalValue;
        switch (m_updateMode)
        {
            case VALUE_MODE:
            case DIFFERENCE_MODE:
            case COUNTER_MODE:
            case MAXIMUM_MODE:
            case MINIMUM_MODE:
                m_currentIntervalValue = m_currentUpdateValue;
                break;
            case MAXIMUM_COUNTER_MODE:
            case MINIMUM_COUNTER_MODE:
                m_currentIntervalValue = m_currentMinMaxValue;
                m_currentMinMaxValue = m_currentUpdateValue;
                break;
        }

        m_lastCurrencyTimestamp = m_currencyTimestamp;
        m_currencyTimestamp = System.currentTimeMillis();
        m_isVirgin = false;
    }

    private void initializeStatistic() {
        m_currentUpdateValue = m_lastUpdateValue = m_initialValue;
        m_numUpdates = 0;
        m_currentIntervalValue = m_lastIntervalValue = m_currentUpdateValue;
        m_currencyTimestamp = INVALID_TIME;
        m_lastCurrencyTimestamp = INVALID_TIME;
        m_resetTimestamp = System.currentTimeMillis();
        m_isVirgin = true;
    }
    /**
     * @see IStatistic#reset()
     */
    @Override
    public synchronized void reset()
    {
        initializeStatistic();
    }

    /**
     * @see IStatistic#setInitialValue(long)
     */
    @Override
    public synchronized void setInitialValue(long value)
    {
        if (!m_isVirgin)
        {
            throw new IllegalStateException("Cannot reset initial value after usage.");
        }
        m_initialValue = value;
        reset();
    }

    public void setName(String name) { m_name = name;}

    @Override
    public String toString() { return "Statistic@" + Integer.toHexString(this.hashCode()) + ": name=" + m_name + ", updateMode=" + m_updateMode + ", intervalMode=" + m_intervalMode ;}
}
