package com.sonicsw.mf.common.runtime;

import java.io.Serializable;

import com.sonicsw.mf.common.metrics.IHistoricalMetric;


/**
 * The IMonitoredMetrics interface represents a collection of stored metrics
 * that are retrieved from a CollectionsMonitor instance, along with the timestamp of the
 * last retrieved metric and a flag to indicate if there are any additional stored
 * metrics to be retrieved from the CollectionsMonitor.
 * <p>
 * A concrete instance of IMonitoredMetrics is obtained through the invocation of a CollectionsMonitor
 * instance's "getStoredMetrics" method.  The returned IMonitoredMetrics instance
 * will contain a set of metrics that were retrieved from the CollectionsMonitor instance's store [if
 * there are any stored metrics], the timestamp of the last metric retrieved from the store, and a
 * boolean flag that indicates if there are any more metrics to retrieve.
 * <p>
 * It is important to note that the "getStoredMetrics" method will return only approximately 1 MB worth of
 * stored metrics.  If the total size of the metrics stored by a given CollectionsMonitor instance,
 * for the specified time interval, exceeds 1 MB, then a caller will have to invoke the "getStoredMetrics"
 * method multiple times.
 * <p>
 * In order for a caller to determine if there are more stored metrics to retrieve for the specified time interval,
 * the caller must check the appropriate flag by invoking the IMonitoredMetrics instance's "hasMoreDataToRetrieve" method.
 * <p>
 * If multiple invocations of "getStoredMetricsData" must be made in order to retrieve all of the stored metrics,
 * then each invocation [after the initial invocation] should use an adjusted "lookback" value, based on the timestamp
 * value returned by invoking "getLatestTimestamp".  So, the pattern of execution would be:
 *   (a) initial invocation of "getStoredMetrics, passing in desired timestamp and lookback arguments
 *   (b) invoke "getMetrics" method on the returned IMonitoredMetrics instance in order to obtain retrieved metrics
 *   (c) invoke "hasMoreDataToRetrieve" method on the returned IMonitoredMetrics instance, in order
 *       to determine if a successive invocation must be made
 *   (d) if "hasMoreDataToRetrieve returns "true", invoke the "getLatestTimestamp" method
 *       of the returned IMonitoredMetrics instance and use the returned timestamp value
 *       to calculate a new "lookback" value, and use the new lookback value as an input
 *       parameter to a follow-on invocation of "getStoredMetricsData",
 *       along with the timestamp value used [as the "lastReceivedTimestamp" parameter] for the initial invocation
 *   (e) repeat steps (b) through (d), until the returned value of "hasMoreDataToRetrieve"
 *       is "false".
 * <p>
 * Invoking the "getMetrics()" method of an IMonitoredMetrics instance will return an array of IHistoricalMetric object instances.
 * If no stored metrics were found for the specified time interval, a zero-length array will be returned from the invocation of
 * "getMetrics".
 * <p>
 * Invoking "getLatestTimestamp()" will return the timestamp of the last metric returned from a CollectionsMonitor instance's store.
 * If there were no stored metrics for the specified time period, the returned timestamp will correspond to the time
 * of the invocation of "getStoredMetrics" method of the CollectionsMonitor instance.  If multiple invocations of "getStoredMetrics"
 * are required in order to obtain all of the stored metrics for the specified time period, the returned timestamp will
 * correspond to the timestamp of the last returned metric PLUS one millisecond.
 * <p>
 * Invoking "hasMoreDataToRetrieve()" will return a boolean value that indicates whether there are more
 * stored metrics to retrieve for the time interval specified in the invocation of "getStoredMetrics".
 * If the returned value is "true", then successive invocations of "getStoredMetrics" will return the
 * remaining stored metrics, and the value returned by "getLatestTimestamp()" should be used to adjust
 * the maximum lookback value passed as an input argument for the successive invocation of "getStoredMetrics".
 * <p>
 * @see com.sonicsw.mf.common.runtime.impl.MonitoredMetrics
 */
public interface IMonitoredMetrics
extends Serializable
{
    public IHistoricalMetric[] getMetrics();
    public long getLatestTimestamp();
    public boolean hasMoreDataToRetrieve();
}