package com.sonicsw.mf.common.runtime;

import java.io.Serializable;

/**
 * The IMonitoredNotifications interface represents a collection of stored notifications
 * that are retrieved from a CollectionsMonitor instance, along with the timestamp of the
 * last retrieved notification and a flag to indicate if there are any additional stored
 * notifications to be retrieved from the CollectionsMonitor.
 * <p>
 * A concrete instance of IMonitoredNotifications is obtained through the invocation of a CollectionsMonitor
 * instance's "getStoredNotifications" method.  The returned IMonitoredNotifications instance
 * will contain a set of notifications that were retrieved from the CollectionsMonitor instance's store [if
 * there are any stored notifications], the timestamp of the last notification retrieved from the store, and a
 * boolean flag that indicates if there are any more notifications to retrieve.
 * <p>
 * It is important to note that the "getStoredNotifications" method will return only approximately 1 MB worth of
 * stored notifications.  If the total size of the notifications stored by a given CollectionsMonitor instance,
 * for the specified time interval, exceeds 1 MB, then a caller will have to invoke the "getStoredNotifications"
 * method multiple times.
 * <p>
 * In order for a caller to determine if there are more stored notifications to retrieve for the specified time interval,
 * the caller must check the appropriate flag by invoking the IMonitoredNotifications instance's "hasMoreDataToRetrieve" method.
 * <p>
 * If multiple invocations of "getStoredNotificationsData" must be made in order to retrieve all of the stored notifications,
 * then each invocation [after the initial invocation] should use an adjusted "lookback" value, based on the timestamp
 * value returned by invoking "getLatestTimestamp".  So, the pattern of execution would be:
 *   (a) initial invocation of "getStoredNotifications, passing in desired timestamp and lookback arguments
 *   (b) invoke "getNotifications" method on the returned IMonitoredNotifications instance in order to obtain retrieved notifications
 *   (c) invoke "hasMoreDataToRetrieve" method on the returned IMonitoredNotifications instance, in order
 *       to determine if a successive invocation must be made
 *   (d) if "hasMoreDataToRetrieve returns "true", invoke the "getLatestTimestamp" method
 *       of the returned IMonitoredNotifications instance and use the returned timestamp value
 *       to calculate a new "lookback" value, and use the new lookback value as an input
 *       parameter to a follow-on invocation of "getStoredNotificationsData",
 *       along with the timestamp value used [as the "lastReceivedTimestamp" parameter] for the initial invocation
 *   (e) repeat steps (b) through (d), until the returned value of "hasMoreDataToRetrieve"
 *       is "false".
 * <p>
 * Invoking the "getNotifications()" method of an IMonitoredNotifications instance will return an array of INotification object instances.
 * If no stored notifications were found for the specified time interval, a zero-length array will be returned from the invocation of
 * "getNotifications".
 * <p>
 * Invoking "getLatestTimestamp()" will return the timestamp of the last notification returned from a CollectionsMonitor instance's store.
 * If there were no stored notifications for the specified time period, the returned timestamp will correspond to the time
 * of the invocation of "getStoredNotifications" method of the CollectionsMonitor instance.  If multiple invocations of "getStoredNotifications"
 * are required in order to obtain all of the stored notifications for the specified time period, the returned timestamp will
 * correspond to the timestamp of the last returned notification PLUS one millisecond.
 * <p>
 * Invoking "hasMoreDataToRetrieve()" will return a boolean value that indicates whether there are more
 * stored notifications to retrieve for the time interval specified in the invocation of "getStoredNotifications".
 * If the returned value is "true", then successive invocations of "getStoredNotifications" will return the
 * remaining stored notifications, and the value returned by "getLatestTimestamp()" should be used to adjust
 * the maximum lookback value passed as an input argument for the successive invocation of "getStoredNotifications".
 * <p>
 * @see com.sonicsw.mf.common.runtime.impl.MonitoredNotifications
 */
public interface IMonitoredNotifications
extends Serializable
{
    public INotification[] getNotifications();
    public long getLatestTimestamp();
    public boolean hasMoreDataToRetrieve();
}