// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.mf.common.runtime;

import java.io.Serializable;
import java.util.HashMap;

/**
 * The notification interface represents a notification emitted from a Sonic
 * Management Framework compliant component.
 * <p>
 * When a notifiable event occurs, components create a notification. Event
 * information is included in the notification by adding attributes
 * (name/value pairs).
 * <p>
 * Management applications subscribe to notifications through the JMS connector
 * and may cast JMX notifications to INotification in order to access the
 * additional functionality of this interface.
 *
 * @see com.sonicsw.mf.jmx.client.JMSConnectorClient
 * @see com.sonicsw.mf.jmx.client.JMSConnectorClient#addNotificationListener(ObjectName, NotificationListener, NotificationFilter, Object)
 */
public interface INotification
extends Serializable
{
    public static final String CLASSNAME = "com.sonicsw.mf.jmx.client.MFNotification";

    //
    // Permissible values for notification categories
    //

    /** Management framework system notifications (reserved for internal use only) */
    public final short SYSTEM_CATEGORY = 0;
    /** @deprecated */
    public final short SECURITY_CATEGORY = 1;
    public final short APPLICATION_CATEGORY = 2;
    public final short CONFIGURATION_CATEGORY = 3;


    // and their textual equivalents
    /**
     * A textual representation of the notification category may be obtained using the
     * return value from getCategory() as an index into the array.
     *
     * @see #getCategory()
     */
    public final String[] CATEGORY_TEXT = new String[] { "system", "security", "application", "configuration" };

    //
    // Values for well used notification subcategories
    //

    public final short STATE_SUBCATEGORY = 0;
    public final short LOG_SUBCATEGORY = 1;
    public final short MONITOR_SUBCATEGORY = 2;
    public final short STORAGE_SUBCATEGORY = 3;
    public final short ALERT_SUBCATEGORY = 4;
    public final short SECURITY_SUBCATEGORY = 5;

    // and their textual equivalents
    /**
     * A textual representation of the notification subcategory may be obtained using the
     * return value from getSubCategory() as an index into the array.
     *
     * @see #getSubCategory()
     */
    public final String[] SUBCATEGORY_TEXT = new String[] { "state", "log", "monitor", "storage", "alert", "security" };

    //
    // Permissible values for log types.
    // These map to Microsoft Event Log information types.
    //

    public final short INFORMATION_TYPE = 0;
    public final short WARNING_TYPE = 1;
    public final short ERROR_TYPE = 2;
    public final short SUCCESS_AUDIT_TYPE = 3;
    public final short FAILURE_AUDIT_TYPE = 4;

    // and their textual equivalents
    /**
     * A textual representation of the notification log type may be obtained using the
     * return value from getLogType() as an index into the array.
     *
     * @see #getLogType()
     */
    public final String[] LOGTYPE_TEXT = new String[] { "information", "warning", "error", "audit success", "audit failure" };

    /**
     * Each notification belongs to a major category.
     * <p>
     * Permissible values are 0 thru 3 (see <>_CATEGORY constants).
     */
    public short getCategory();

    /**
     * Each notification has a minor category.
     * <p>
     * A subset of well known values are defined as constants (see <>_SUBCATEGORY constants).
     */
    public String getSubCategory();

    /**
     * Each notification type is unique by category/subcategory/event name. Event names
     * are descriptive of what event generated the notification.
     */
    public String getEventName();

    /**
     * Each notification type is unique by category/subcategory/event name. This method returns
     * the full notification type name as a string.
     */
    public String getType();

    /**
     * Each notification has a severity associated with it. Severities can be used by
     * client applications to prioritize multiple notifications. Additionally the severity
     * of certain notifications will be used to derive an aggregated state for a container.
     */
    public int getSeverity();

    /**
     * Each notification can be assigned a log type. If not set explicitly it will derived
     * from the severity.
     * <p>
     * Log types are intended for indexing of information types by log\audit services (e.g.
     * Microsoft Event Viewer). They provide a narrower/alternate view of severity.
     *
     * @param logType Permissible values are 0 thru 4 (see <>_TYPE constants).
     */
    public void setLogType(short logType);

    /**
     * @see #setLogType(short)
     */
    public short getLogType();

    /**
     * Gets the source of the notification. The source of the notification is set by the
     * container prior to sending the notification to interested parties.
     *
     * @return The MF runtime identity of the component that generated the notification.
     */
    public IComponentIdentity getSourceIdentity();

    /**
     * Sets the source of the notification. The source of the notification is set by the
     * container prior to sending the notification to interested parties.
     *
     * Once set the sourceIdenity cannot be reset.
     */
    public void setSourceIdentity(IComponentIdentity sourceIdentity);

    /**
     * Gets the source of the notification. The source of the notification is set by the
     * container prior to sending the notification to interested parties.
     *
     * @return The host IP of the machine hosting the container/component source that
     *         generated the notification.
     */
    public String getSourceHost();

    /**
     * Gets the sequence number for the notification. Sequence numbers are unique within a
     * component and imply an order based on the time the sequence number is assigned to
     * the notification. Sequence numbers start at 0 and increment by 1, but some sequence 
     * numbers may appear to be skipped due to the fact that notifications are also used 
     * internally.
     *
     * @return The sequence number of the notification.
     */
    public long getSequenceNumber();

    /**
     * Gets the timestamp of the notification. This will be a timestamp based on the
     * system clock of the machine on which the notification was generated and this should be
     * taken into account by recipients of notifications that reside on different machines with
     * a possibly conflicting system clock setting.
     *
     * @return The notification timestamp.
     */
    public long getTimeStamp();

    /**
     * Sets a notification attribute (a named piece of event information). Each attribute should
     * be uniquely named (to avoid overwriting already specified attributes).
     *
     * @param name  The attribute name.
     * @param value The attribute value (should be base types or arrays of base types).
     */
    public void setAttribute(String name, Object value);

    /**
     * Gets the event information specified as attributes.
     *
     * @return The attribute table or null if there are no attributes.
     */
    public HashMap getAttributes();
}
