package com.sonicsw.mf.common.runtime;

/**
 * A concrete implementation of the IStateManager interface acts as a central point
 * for managing the state of an entity, such as the fault tolerant state of the DS or AM.
 * <p>
 * A concrete state manager implementation is configured with a set.
 * <p>
 * State controllers (implementers of IStateController) may be registered to effect
 * the behavioral adjustments associated with a particular state transition.
 * <p>
 * State listeners (implementors of IStateListener) may be added to be informed when
 * a state transitions starts and whether the transition succeeds or fails.
 * <p>
 * A developer should not attempt to request a state change from either a state
 * controller or a state listener.
 */
public interface IStateManager
{
    /**
     * Get the state of the specified object.
     *
     * For a concrete implementation that is designed to managed only a single object's state,
     * the input argument could be null.
     * For a concrete implementation that is designed to manage the states of multiple
     * objects, the input argument would be the reference to the object instance whose
     * state is to be returned.
     *
     * @param  Object   The object instance whose current state is to be returned.
     * @return short    The current state of the object instance.
     */
    public short getState(Object obj);

    /**
     * The thread that detects the condition that requires a state change will call this
     * method to effect the state change.
     * <p>
     * The calling thread is expected to have previously obtained the current state and
     * to have made the decision as to what is the next state that should be applied. To
     * ensure synchronization between calls, the caller is required to provide the expected
     * current state so that it can possibly retry if another thread changed the state
     * between the callers calls.
     *
     * If the current state, from the state manager's perspective, does not match the current
     * state expected by the caller, then a value of "false" should be returned.  Likewise, if
     * the transition to the requested desired state from the current state does not represent
     * an allowable transition, then a value of "false" should be returned.
     *
     * @param expectedCurrentState The current state obtained immediately before calling
     *                             this method
     * @param desiredState         The state the caller wishes to transition to
     * @param obj                  The state-managed object whose state is to be changed
     * @return                     Return true if the state transition was successful, false if not.
     */
    public boolean requestStateChange(short expectedCurrentState, short desiredState, Object obj)
    throws NonRecoverableStateChangeException, RecoverableStateChangeException;

    /**
     * Register a state controller that will cause the behvioral changes associated
     * with the transition from one specified state to another specified state.
     *
     * @param stateController  The concrete implementation of IStateController that will perform the steps necessary for the state transition
     * @param fromState        The state at which the state-managed entity must be in order for the controller will be invoked
     * @param toState          The state to which the state-managed entity must transition in order for the controller to be invoked
     * @param obj              The state-managed object with which the controller is associated
     */
    public void registerStateController(IStateController stateController, short fromState, short toState, Object obj);

    /**
     * Register a listener for the changes in state of the state-managed object.
     *
     * @param stateListener  The concrete implementation of IStateListener
     * @param obj            The state-managed object to which the listener will be associated
     */
    public void registerStateListener(IStateListener stateListener, Object obj);
}