package com.sonicsw.mf.common.runtime.impl;

import java.util.StringTokenizer;

import com.sonicsw.mf.common.runtime.ICanonicalName;
import com.sonicsw.mf.common.runtime.IComponentIdentity;

/**
 * Represents the canonical name of a component
 */
public class CanonicalName
implements ICanonicalName
{
    private String m_canonical;
    private String m_domain;
    private String m_container;
    private String m_component;
    private String m_nodeName;
    private String m_localContainerName;

    /**
     * The assumption is the name has the correct form:
     *  - a container form being "<domain>.<container>"
     *  - a component form being "<domain>.<container>:ID=<componentID>"
     */
    public CanonicalName (String canonicalName)
    {
         init(canonicalName);
    }

    /**
     * For a container the component should be an empty string ("").
     */
    public CanonicalName (String domain, String container, String component)
    {
         init(domain + "." + container + (component.length() == 0 ? "" : ':' + IComponentIdentity.ID_PREFIX + component));
    }

    public final void init(String name)
    {
         m_canonical = name;

         StringTokenizer st = new StringTokenizer(name, ".:");
         m_domain = st.nextToken();
         m_container = st.nextToken();
         m_component = st.hasMoreTokens() ? st.nextToken() : "";
         if (m_component.length() > 0)
         {
            m_component = m_component.substring(IComponentIdentity.ID_PREFIX_LENGTH); // strip off the "ID="
        }
         CompoundContainerName compoundName = new CompoundContainerName(m_container);
         m_nodeName = compoundName.m_nodeName;
         m_localContainerName = compoundName.m_localContainerName;
    }

    public String getNodeName()
    {
        return m_nodeName;
    }

    public String getLocalContainerName()
    {
        return m_localContainerName;
    }

    @Override
    public String getCanonicalName()
    {
        return m_canonical;
    }

    @Override
    public String getDomainName()
    {
        return m_domain;
    }

    @Override
    public String getContainerName()
    {
        return m_container;
    }

    @Override
    public String getComponentName()
    {
        return m_component;
    }

    @Override
    public boolean equals(Object object)
    {
        if (this == object)
        {
            return true;
        }
        if (!(object instanceof ICanonicalName))
        {
            return false;
        }
        return m_canonical.equals(((ICanonicalName)object).getCanonicalName());
    }

    @Override
    public int hashCode()
    {
        return m_canonical.hashCode();
    }
    
    @Override
    public int compareTo(Object object)
    throws ClassCastException
    {
        return m_canonical.compareTo(((CanonicalName)object).getCanonicalName());
    }

    static class CompoundContainerName
    {
         private static final char NODE_AT_CHAR = '@';
         String m_localContainerName;
         String m_nodeName;

         CompoundContainerName(String containerName)
         {
             int nodeDelimiter = containerName.indexOf(NODE_AT_CHAR);
             if (nodeDelimiter != -1)
             {
                 m_localContainerName =  containerName.substring(0, nodeDelimiter);
                 m_nodeName = containerName.substring(nodeDelimiter+1);
             }
             else
             {
                 m_localContainerName = containerName;
                 m_nodeName = "";
             }
         }
    }
}
