package com.sonicsw.mf.common.runtime.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Objects;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.runtime.IChildContainerState;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.IContainerIdentity;

public class ChildContainerState
implements IChildContainerState, Serializable
{
    private static final long serialVersionUID = -7773691032575402133L;
    private static final short m_serialVersion = 0;

    private IContainerIdentity m_containerID;
    private long m_timestamp;
    private short m_state = IChildContainerState.STATE_UNKNOWN;
    private Integer m_exitCode;
    private ArrayList m_logTail = new ArrayList();

    // the max # of lines we will save in the log tail - not cobfigurable at present
    private static final int MAX_LOG_TAIL_SIZE = 200;

    /**
     * Creates a child container state with an initial state of IChildContainerState.STATE_ONLINE and a
     * hostname of the current machines hostname.
     */
    public ChildContainerState(CanonicalName canonicalName, IElementIdentity configID)
    {
        m_containerID = new ContainerIdentity(canonicalName, configID);
        m_timestamp = System.currentTimeMillis();
    }

    @Override
    public com.sonicsw.mf.common.runtime.IIdentity getRuntimeIdentity() { return m_containerID; }

    @Override
    public long getTimeStamp() { return m_timestamp == 0 ? System.currentTimeMillis() : m_timestamp; }

    public void setTimestamp(long timestamp) { m_timestamp = timestamp; }

    @Override
    public short getState() { return m_state; }

    public void setState(short state)
    {
        if (!(state == IChildContainerState.STATE_UNKNOWN || state == IChildContainerState.STATE_ONLINE || state == IChildContainerState.STATE_INACTIVE || state == IChildContainerState.STATE_OFFLINE))
        {
            throw new IllegalArgumentException("Invalid container state: " + state);
        }

        m_state = state;
    }

    @Override
    public String getStateString() { return IChildContainerState.STATE_TEXT[getState()]; }

    public synchronized void setExitCode(short exitCode)
    {
        if (exitCode < 0 || exitCode > IContainerExitCodes.EXIT_CODE_TEXTS.length - 1)
        {
            throw new IllegalArgumentException("Invalid container exit code: " + exitCode);
        }

        m_exitCode = new Integer(exitCode);
    }

    public void clearExitCode() { m_exitCode = null; }

    @Override
    public Integer getExitCode() { return m_exitCode; }

    public synchronized void addLogLine(String logLine)
    {
        while (m_logTail.size() > MAX_LOG_TAIL_SIZE)
        {
            m_logTail.remove(0);
        }
        m_logTail.add(logLine);
    }

    @Override
    public String[] getLogTail() { return (String[])m_logTail.toArray(IEmptyArray.EMPTY_STRING_ARRAY); }

    /**
     * Returns a string representation of the state. While the returned string
     * "textually represents" the state, it does not completely encapsulate it
     * - only the state string is exposed.
     *
     * @return   a string representation of the state.
     */
    @Override
    public String toString()
    {
        return getStateString();
    }

    @Override
    public boolean equals(Object object)
    {
        if (this == object)
        {
            return true;
        }
        if (!(object instanceof IChildContainerState))
        {
            return false;
        }
        return getRuntimeIdentity().equals(((IChildContainerState)object).getRuntimeIdentity()) && getState() == ((IChildContainerState)object).getState();
    }

    @Override
    public int hashCode() {
        return Objects.hash(getRuntimeIdentity(), getState());
    }
    
    @Override
    public int compareTo(Object object)
    throws ClassCastException
    {
        return toString().compareTo(((IChildContainerState)object).toString());
    }

    //
    // Serialization
    //

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(6);

        // MFNotification specific fields

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("containerID");
        stream.writeObject(m_containerID);

        stream.writeUTF("timestamp");
        stream.writeObject(new Long(System.currentTimeMillis()));

        stream.writeUTF("state");
        stream.writeObject(new Short(m_state));

        stream.writeUTF("exitCode");
        stream.writeObject(m_exitCode);

        stream.writeUTF("logTail");
        stream.writeObject(m_logTail);
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore

        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                try { m_containerID = (IContainerIdentity)map.get("containerID"); } catch(Exception e) {}
                try { m_timestamp = ((Long)map.get("timestamp")).longValue(); } catch(Exception e) {}
                try { m_state = ((Short)map.get("state")).shortValue(); } catch(Exception e) {}
                try { m_exitCode = (Integer)map.get("exitCode"); } catch(Exception e) {}
                try { m_logTail = (ArrayList)map.get("logTail"); } catch(Exception e) {}
                break;
            }
        }
    }
}

