package com.sonicsw.mf.common.runtime.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.common.IXMLFragment;
import com.sonicsw.mf.common.runtime.ICollectiveOpStatus;

public class CollectiveOpStatus
implements ICollectiveOpStatus, IXMLFragment, Serializable
{
    private static final long serialVersionUID = 163719925118650399L;
    private static final short m_serialVersion = 1;
    
    private ArrayList m_results;
    
    public CollectiveOpStatus(int resultCount)
    {
        m_results = new ArrayList(resultCount);
    }
    
    public void addResult(String componentName, Object returnValue, Throwable throwable)
    {
        m_results.add(new Object[] { componentName, returnValue, throwable });
    }
    
    /** @see com.sonicsw.mf.common.runtime.ICollectiveOpStatus#getCount() */
    @Override
    public int getCount() { return m_results.size(); }

    /** @see com.sonicsw.mf.common.runtime.ICollectiveOpStatus#getComponentName(String) */
    @Override
    public String getComponentName(int compNum)
    {
        if (compNum > m_results.size())
        {
            throw new IllegalArgumentException("Entry [" + compNum + "] does not exist");
        }

        Object[] entry = (Object[])m_results.get(compNum);
        return (String)entry[0];
    }

    /** @see com.sonicsw.mf.common.runtime.ICollectiveOpStatus#operationIsSuccessful(int) */
    @Override
    public boolean operationIsSuccessful(int compNum)
    {
        if (compNum > m_results.size())
        {
            throw new IllegalArgumentException("Entry [" + compNum + "] does not exist");
        }

        Object[] entry = (Object[])m_results.get(compNum);
        return entry[2] == null;
    }

    /** @see com.sonicsw.mf.common.runtime.ICollectiveOpStatus#getReturnValue(int) */
    @Override
    public Object getReturnValue(int compNum)
    {
        if (compNum > m_results.size())
        {
            throw new IllegalArgumentException("Entry [" + compNum + "] does not exist");
        }

        Object[] entry = (Object[])m_results.get(compNum);
        return entry[1];
    }

    /** @see com.sonicsw.mf.common.runtime.ICollectiveOpStatus#getThrowable(int) */
    @Override
    public Throwable getThrowable(int compNum)
    {
        if (compNum > m_results.size())
        {
            throw new IllegalArgumentException("Entry [" + compNum + "] does not exist");
        }

        Object[] entry = (Object[])m_results.get(compNum);
        return (Throwable)entry[2];
    }

    //
    // Serialization
    //
    
    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(2);
        
        // MFNotification specific fields
        
        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));
        
        stream.writeUTF("results");
        stream.writeObject(m_results);
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }
            
        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore
        
        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            case 0:
            {
                try
                {
                    Object[][] results = (Object[][])map.get("results");
                    for (int i = 0; i < results.length; i++)
                    {
                        m_results.add(results[i]);
                    }
                }
                catch(Exception e) {}
                break;
            }
            default: // the current version or newer versions
            {
                try { m_results = (ArrayList)map.get("results"); } catch(Exception e) {}
                break;
            }
        }
    }
    
    @Override
    public String toString()
    {
        StringBuffer sb = new StringBuffer();
        sb.append('{');
        
        for (int i = 0; i < m_results.size(); i++)
        {
            if (i > 0)
            {
                sb.append(' ');
            }
            Object[] result = (Object[])m_results.get(i);
            sb.append('[').append(result[0]).append(']');
            sb.append(result[(result[2] == null ? 1 : 2)]);
        }
        
        sb.append('}');
        
        return sb.toString();
    }
    
    @Override
    public String toXMLFragment()
    {
        StringBuffer sb = new StringBuffer();
        sb.append("<collectiveOpStatus>");
        
        for (int i = 0; i < m_results.size(); i++)
        {
            Object[] result = (Object[])m_results.get(i);

            sb.append("<result>");
            sb.append("<target>").append(result[0]).append("</target>");

            if (result[2] == null)
            {
                if (result[1] == null)
                {
                    sb.append("<return>null</return>");
                }
                else
                {
                    sb.append("<return>").append(result[1] instanceof IXMLFragment ? ((IXMLFragment)result[1]).toXMLFragment() : result[1].toString()).append("</return>");
                }
            }
            else
            {
                sb.append("<exception>").append(result[2] instanceof IXMLFragment ? ((IXMLFragment)result[2]).toXMLFragment() : result[2].toString()).append("</exception>");
            }

            sb.append("</result>");
        }
        
        sb.append("</collectiveOpStatus>");
        
        return sb.toString();
    }
}
