package com.sonicsw.mf.common.runtime.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.ISubComponentState;
import com.sonicsw.mf.common.runtime.Level;

public final class ComponentState
implements IComponentState, Serializable
{
    private static final long serialVersionUID = 3469829302166258423L;
    private static final short m_serialVersion = 1;

    private IComponentIdentity m_componentID;
    private short m_state;
    private int m_errorLevel;
    private String m_errorDescription;
    private Map<String, ISubComponentState[]> m_subComponentStates;

    public ComponentState(CanonicalName canonicalName, IElementIdentity configID, short state, String errorDescription, int errorLevel, Map<String, ISubComponentState[]> subComponentStates)
    {
        m_componentID = new ComponentIdentity(canonicalName, configID);
        m_state = state;
        m_errorLevel = errorLevel;
        m_errorDescription = errorDescription;
        m_subComponentStates = subComponentStates == null ? new HashMap<String, ISubComponentState[]>() : subComponentStates;
    }

    @Override
    public com.sonicsw.mf.common.runtime.IIdentity getRuntimeIdentity() { return m_componentID; }

    @Override
    public short getState() { return m_state; }

    public void setState(short state)
    {
        if (state < IComponentState.STATE_UNKNOWN || state > IComponentState.STATE_STOPPING)
        {
            throw new IllegalArgumentException("Invalid component state: " + state);
        }

        m_state = state;
    }

    @Override
    public String getStateString() { return IComponentState.STATE_TEXT[getState()]; }

    @Override
    public int getLastErrorLevel() { return m_errorLevel; }

    public void setLastErrorLevel(int errorLevel)
    {
        if (errorLevel < Level.UNKNOWN || errorLevel > Level.INFO)
        {
            throw new IllegalArgumentException("Invalid component error severity: " + errorLevel);
        }

        m_errorLevel = errorLevel;
    }

    @Override
    public String getLastErrorDescription() { return m_errorDescription; }
    public void setLastErrorDescription(String errorDescription) { m_errorDescription = errorDescription; }

    @Override
    public Map<String, ISubComponentState[]> getSubComponentStates()
    {
        return m_subComponentStates;
    }

    /**
     * Returns a string representation of the state. While the returned string
     * "textually represents" the state, it does not completely encapsulate it
     * - only the state string is exposed.
     *
     * @return   a string representation of the state.
     */
    @Override
    public String toString()
    {
        return getStateString();
    }

    @Override
    public boolean equals(Object object)
    {
        if (this == object)
        {
            return true;
        }
        if (!(object instanceof IComponentState))
        {
            return false;
        }
        return getRuntimeIdentity().equals(((IComponentState)object).getRuntimeIdentity()) && getState() == ((IComponentState)object).getState();
    }

    @Override
    public int hashCode() {
        return Objects.hash(getRuntimeIdentity(), getState());
    }
    
    @Override
    public int compareTo(Object object)
    throws ClassCastException
    {
        return toString().compareTo(((IComponentState)object).toString());
    }

    //
    // Serialization
    //

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(6);

        // MFNotification specific fields

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("componentID");
        stream.writeObject(m_componentID);

        stream.writeUTF("state");
        stream.writeObject(new Short(m_state));

        stream.writeUTF("errorDescription");
        stream.writeObject(m_errorDescription);

        stream.writeUTF("errorLevel");
        stream.writeObject(new Integer(m_errorLevel));
        
        stream.writeUTF("subComponentStates");
        if ( m_subComponentStates instanceof Serializable)
        {
            stream.writeObject(m_subComponentStates);
        }
        else
        {
            stream.writeObject(new HashMap<String, ISubComponentState[]>(m_subComponentStates));
        }
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore

        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            case 0: // the current version or newer versions
            {
                try { m_componentID = (IComponentIdentity)map.get("componentID"); } catch(Exception e) {}
                try { m_state = ((Short)map.get("state")).shortValue(); } catch(Exception e) {}
                try { m_errorDescription = (String)map.get("errorDescription"); } catch(Exception e) {}
                try { m_errorLevel = ((Integer)map.get("errorLevel")).intValue(); } catch(Exception e) {}
                break;
            }
            default: // the current version or newer versions
            {
                try { m_componentID = (IComponentIdentity)map.get("componentID"); } catch(Exception e) {}
                try { m_state = ((Short)map.get("state")).shortValue(); } catch(Exception e) {}
                try { m_errorDescription = (String)map.get("errorDescription"); } catch(Exception e) {}
                try { m_errorLevel = ((Integer)map.get("errorLevel")).intValue(); } catch(Exception e) {}
                try { m_subComponentStates = (HashMap<String, ISubComponentState[]>)map.get("subComponentStates"); } catch(Exception e) {}
                break;
            }
        }
    }
}
