package com.sonicsw.mf.common.runtime.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Objects;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

public class ContainerIdentity
implements IContainerIdentity, Serializable
{
    private static final long serialVersionUID = 1958150406706801004L;
    private static final short m_serialVersion = 0;

    private static final String DOMAIN_NAME_ATTR = "DOMAIN_NAME";
    private static final String CONTAINER_NAME_ATTR = "CONTAINER_NAME";

    private String m_domainName;
    private String m_containerName;
    private String m_nodeName;
    private String m_localContainerName;
    private IElementIdentity m_configID;

    public ContainerIdentity(IElement containerConfig)
    {
        IAttributeSet containerAttrs = containerConfig.getAttributes();
        m_domainName = (String)containerAttrs.getAttribute(DOMAIN_NAME_ATTR);
        if (m_domainName == null)
        {
            m_domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
        }
        m_containerName = (String)containerAttrs.getAttribute(CONTAINER_NAME_ATTR);
        m_configID = containerConfig.getIdentity();
        CanonicalName.CompoundContainerName compoundName = new CanonicalName.CompoundContainerName(m_containerName);
        m_localContainerName = compoundName.m_localContainerName;
        m_nodeName = compoundName.m_nodeName;
    }

    public ContainerIdentity(CanonicalName canonicalName, IElementIdentity configID)
    {
        m_domainName = canonicalName.getDomainName();
        m_containerName = canonicalName.getContainerName();
        m_localContainerName = canonicalName.getLocalContainerName();
        m_nodeName = canonicalName.getNodeName();
        m_configID = configID;
    }

    @Override
    public String getDomainName() { return m_domainName; }

    @Override
    public String getContainerName() { return m_containerName; }

    public String getLocalContainerName() { return m_localContainerName; }

    @Override
    public String getNodeName() { return m_nodeName; }

    @Override
    public String getCanonicalName() { return m_domainName + '.' + m_containerName; }

    // The prefix of components contained by this container
    public String getCanonicalNamePrefix() { return m_domainName + '.' + m_containerName + ":"; }

    @Override
    public IElementIdentity getConfigIdentity() { return m_configID; }

    @Override
    public String toString()
    {
        StringBuffer sb = new StringBuffer();

        sb.append(m_domainName);
        sb.append('.');
        sb.append(m_containerName);
        if (m_nodeName.length() > 0)
        {
            sb.append('@').append(m_nodeName);
        }

        return sb.toString();
    }

    @Override
    public boolean equals(Object object)
    {
        if (this == object)
        {
            return true;
        }
        if (!(object instanceof IContainerIdentity))
        {
            return false;
        }
        return toString().equals(object.toString()) && getConfigIdentity().equals(((IContainerIdentity)object).getConfigIdentity());
    }

    @Override
    public int hashCode() {
        return Objects.hash(toString(), getConfigIdentity());
    }
    
    @Override
    public int compareTo(Object object)
    throws ClassCastException
    {
        return toString().compareTo(((IContainerIdentity)object).toString());
    }

    //
    // Serialization
    //

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(4);

        // MFNotification specific fields

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("domainName");
        stream.writeObject(m_domainName);

        stream.writeUTF("containerName");
        stream.writeObject(m_containerName);

        stream.writeUTF("configID");
        stream.writeObject(m_configID);
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore

        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                try { m_domainName = (String)map.get("domainName"); } catch(Exception e) {}
                try { m_containerName = (String)map.get("containerName"); } catch(Exception e) {}
                try { m_configID = (IElementIdentity)map.get("configID"); } catch(Exception e) {}
                CanonicalName canonicalName = new CanonicalName(m_domainName, m_containerName, "");
                m_localContainerName = canonicalName.getLocalContainerName();
                m_nodeName = canonicalName.getNodeName();
                break;
            }
        }
    }
}

