package com.sonicsw.mf.common.runtime.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Objects;

import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.common.runtime.IContainerState;

public class ContainerState
implements IContainerState, Serializable
{
    private static final long serialVersionUID = -7773691032575402133L;
    private static final short m_serialVersion = 0;

    private IContainerIdentity m_containerID;
    private long m_timestamp;
    private short m_state = IContainerState.STATE_ONLINE;
    String m_host;
    private ArrayList m_componentStates = new ArrayList();

    /**
     * Creates a container state with an initial state of IContainerState.STATE_ONLINE and a
     * hostname of the current machines hostname.
     */
    public ContainerState(CanonicalName canonicalName, IElementIdentity configID)
    {
        m_containerID = new ContainerIdentity(canonicalName, configID);
        try
        {
            m_host = InetAddress.getByName(InetAddress.getLocalHost().getHostAddress()).getHostName();
        }
        catch(UnknownHostException e) { m_host = "UNKNOWN"; }
        m_timestamp = System.currentTimeMillis();
    }

    public void addComponentState(IComponentState componentState)
    {
        m_componentStates.add(componentState);
    }

    public void removeComponentState(String componentName)
    {
        for (int i = 0; i < m_componentStates.size(); i++)
        {
            IComponentState componentState = (IComponentState)m_componentStates.get(i);
            IComponentIdentity componentIdentity = (IComponentIdentity)componentState.getRuntimeIdentity();
            if (componentIdentity.getComponentName().equals(componentName))
            {
                m_componentStates.remove(i);
                return;
            }
        }
    }

    @Override
    public com.sonicsw.mf.common.runtime.IIdentity getRuntimeIdentity() { return m_containerID; }

    @Override
    public long getTimeStamp() { return m_timestamp; }

    public void setTimestamp(long timestamp) { m_timestamp = timestamp; }

    @Override
    public short getState() { return m_state; }

    public void setState(short state)
    {
        if (!(state == IContainerState.STATE_UNKNOWN || state == IContainerState.STATE_ONLINE || state == IContainerState.STATE_OFFLINE))
        {
            throw new IllegalArgumentException("Invalid container state: " + state);
        }

        m_state = state;
    }

    @Override
    public String getStateString() { return IComponentState.STATE_TEXT[getState()]; }

    @Override
    public IComponentState[] getComponentStates()
    {
        IComponentState[] states = new IComponentState[m_componentStates.size()];
        return (IComponentState[])m_componentStates.toArray(states);
    }

    @Override
    public String getContainerHost() { return m_host; }

    public void setContainerHost(String host)
    {
        if (host == null || host.length() == 0)
        {
            throw new IllegalArgumentException("Container host cannot be null or empty.");
        }

        m_host = host;
    }

    /**
     * Returns a string representation of the state. While the returned string
     * "textually represents" the state, it does not completely encapsulate it
     * - only the state string is exposed.
     *
     * @return   a string representation of the state.
     */
    @Override
    public String toString()
    {
        return getStateString();
    }

    @Override
    public boolean equals(Object object)
    {
        if (this == object)
        {
            return true;
        }
        if (!(object instanceof IContainerState))
        {
            return false;
        }
        return getRuntimeIdentity().equals(((IContainerState)object).getRuntimeIdentity()) && getState() == ((IContainerState)object).getState();
    }

    @Override
    public int hashCode() {
        return Objects.hash(getRuntimeIdentity(), getState());
    }
    
    @Override
    public int compareTo(Object object)
    throws ClassCastException
    {
        return toString().compareTo(((IContainerState)object).toString());
    }

    //
    // Serialization
    //

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(6);

        // MFNotification specific fields

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("containerID");
        stream.writeObject(m_containerID);

        stream.writeUTF("timestamp");
        stream.writeObject(new Long(m_timestamp));

        stream.writeUTF("state");
        stream.writeObject(new Short(m_state));

        stream.writeUTF("host");
        stream.writeObject(m_host);

        stream.writeUTF("componentStates");
        stream.writeObject(m_componentStates);
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore

        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                try { m_containerID = (IContainerIdentity)map.get("containerID"); } catch(Exception e) {}
                try { m_timestamp = ((Long)map.get("timestamp")).longValue(); } catch(Exception e) {}
                try { m_state = ((Short)map.get("state")).shortValue(); } catch(Exception e) {}
                try { m_host = (String)map.get("host"); } catch(Exception e) {}
                try { m_componentStates = (ArrayList)map.get("componentStates"); } catch(Exception e) {}
                break;
            }
        }
    }
}
