package com.sonicsw.mf.common.runtime.impl;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.MFProxyException;
import com.sonicsw.mf.common.MFProxyRuntimeException;
import com.sonicsw.mf.common.runtime.IRemoteExecResult;

public class ExecUtility
{
	public static final String NEWLINE = System.getProperty("line.separator");

    public static RemoteExecResult exec(String[] cmdarray, String[] envp, String workDirPath, byte[] inputBytes)
    {
        ExecProcess p = null;
        try
        {
            p = new ExecProcess(cmdarray, envp, workDirPath, inputBytes);
            p.exec();
            int exitCode = p.waitFor();
            return new RemoteExecResult(true, p.getIOExceptionsTrace(), exitCode, p.getOutData(), p.getErrData());
        }
        catch (Exception e)
        {
            return new RemoteExecResult(false, printStackTrace(e, "com.sonicsw.mf.framework.agent.ExecUtility failed:"), 1, null, null);
        }
    }

    public static String execResultOutputToString(IRemoteExecResult result)
    {
        byte[] outBytes = result.getProcessStdOutput();
        String outputString = (outBytes != null && outBytes.length > 0) ? new String(outBytes) : null;

        byte[] errBytes = result.getProcessStdError();
        String errString = (errBytes != null && errBytes.length > 0) ? new String(errBytes) : null;

        String rsltString = outputString;
        if (errString != null)
        {
            rsltString = (rsltString == null) ? errString : rsltString + System.getProperty("line.separator") + errString;
        }

        return (rsltString != null) ? rsltString : "";
    }


    private static class ExecProcess
    {
        private String[] m_execCommand;
        private File m_workDir;
        private String[] m_envp;
        private Process m_systemProcess;
        private byte[] m_inputBytes;
        private OutputReader m_stdOutputReader;
        private OutputReader m_stdErrorReader;
        private InputWriter m_stdInputWriter;


        ExecProcess(String[] execCommand, String[] envp, String workingDir, byte[] inputBytes) throws Exception
        {
            m_execCommand = execCommand;
            m_workDir = new File(workingDir != null ? workingDir : ".");
            m_envp = envp;
            m_inputBytes = inputBytes;
            if (!m_workDir.exists())
            {
                throw new Exception("Working directory \"" + m_workDir.getAbsolutePath() + "\" does not exist.");
            }
        }

        byte[] getErrData()
        {
            return m_stdErrorReader.getData();
        }

        byte[] getOutData()
        {
            return m_stdOutputReader.getData();
        }

        String getIOExceptionsTrace()
        {
            StringBuffer sb = new StringBuffer();

            IOException e = m_stdOutputReader.getException();
            if (e != null)
            {
                sb.append(printStackTrace(e, "Reading the std output from the process failed:"));
            }

            e = m_stdErrorReader.getException();
            if (e != null)
            {
                sb.append(printStackTrace(e, "Reading the std error from the process failed:"));
            }

            e = (m_stdInputWriter != null) ? m_stdInputWriter.getException() : null;
            if (e != null)
            {
                sb.append(printStackTrace(e, "Writing the input to the process failed:"));
            }

            String result = sb.toString();
            return (result.length() > 0) ? result : null;
        }

        public void exec() throws Exception
        {
            m_systemProcess = Runtime.getRuntime().exec(m_execCommand, m_envp, m_workDir);
            m_stdOutputReader = new OutputReader("com.sonicsw.mf.framework.agent.ExecUtility Standard Output", m_systemProcess.getInputStream());
            m_stdErrorReader = new OutputReader("com.sonicsw.mf.framework.agent.ExecUtility Standard Error", m_systemProcess.getErrorStream());

            m_stdOutputReader.start();
            m_stdErrorReader.start();

            if (m_inputBytes != null)
            {
                m_stdInputWriter = new InputWriter("com.sonicsw.mf.framework.agent.ExecUtility Standard Input", m_systemProcess.getOutputStream(), m_inputBytes);
                m_stdInputWriter.start();
            }
            else
            {
                m_systemProcess.getOutputStream().close();
            }

        }

        int waitFor()
        {
            while (true)
            {
                try
                {
                    int exiCode = m_systemProcess.waitFor();
                    //tell the read loops to stop
                    m_stdOutputReader.processExited();
                    m_stdErrorReader.processExited();
                    //Make sure the threads that read standard output and error are done
                    m_stdOutputReader.waitForOutput();
                    m_stdErrorReader.waitForOutput();
                    return exiCode;
                }
                catch (InterruptedException e)
                {
                    m_systemProcess.destroy();
                    continue;
                }
            }



        }

        private final class OutputReader extends Thread
        {
            private InputStream m_stream;
            private ArrayList<byte[]> m_buffers;
            private byte[] m_buffer;
            private byte[] m_data;
            private boolean m_outputReady;
            private IOException m_exception;
            private boolean m_stop;
            private volatile boolean m_stopRequest; //After m_stopRequest is set, we try to read one more time to make sure there is no data left in the buffer

            private OutputReader(String threadName, InputStream stream)
            {
                super(threadName);
                setDaemon(true);
                m_stream = stream;
                m_buffer = new byte[8196];
                m_buffers = new ArrayList<byte[]>();
                m_outputReady = false;
                m_stopRequest = false;
                m_stop = false;
            }

            byte[] getData()
            {
                return m_data;
            }

            IOException getException()
            {
                return m_exception;
            }

            void prepareData()
            {
                int totalLength = 0;
                for (int i = 0; i < m_buffers.size(); i++)
                {
                    totalLength += m_buffers.get(i).length;
                }
                m_data = new byte[totalLength];

                int crntOffset = 0;
                for (int i = 0; i < m_buffers.size(); i++)
                {
                    byte[] crntBuffer = m_buffers.get(i);
                    System.arraycopy(crntBuffer, 0, m_data, crntOffset, crntBuffer.length);
                    crntOffset += crntBuffer.length;
                }
            }

            void processExited()
            {
                m_stopRequest = true;
            }

            private synchronized void waitForOutput()
            {
                while (!m_outputReady)
                {
                    try
                    {
                        wait();
                    }
                    catch (InterruptedException e)
                    {
                    }
                }
            }

            private synchronized void outputReady()
            {
                m_outputReady = true;
                notifyAll();
            }

            @Override
            public void run()
            {
                try
                {
                    while (true)
                    {
                        int available = m_stream.available();
                        if (available > 0) {
                            if (available > m_buffer.length)
                            {
                                available = m_buffer.length;
                            }

                            int count = m_stream.read(m_buffer, 0, available);
                            if (count < 0)
                            {
                                break;
                            }
                            else if (count == 0)
                            {
                                continue;
                            }
                            else
                            {
                                byte[] crntBuffer = new byte[count];
                                System.arraycopy(m_buffer, 0, crntBuffer, 0, count);
                                m_buffers.add(crntBuffer);
                            }
                        }
                        else
                        {
                            try
                            {
                                Thread.sleep(100);
                            } catch (InterruptedException e)
                            {
                                // ignore
                            }
                        }
                        if (m_stop)
                        {
                            break;
                        }
                        if (m_stopRequest)
                        {
                            m_stop = true;
                        }
                    }
                    prepareData();
                }
                catch(IOException e)
                {
                    m_exception = e;
                }
                finally
                {
                    outputReady();
                }
            }

        }

        private final class InputWriter extends Thread
        {
            private OutputStream m_stream;
            private byte[] m_outputBytes;
            private IOException m_exception;

            private InputWriter(String threadName, OutputStream stream, byte[] outputBytes)
            {
                super(threadName);
                setDaemon(true);
                m_stream = stream;
                m_outputBytes = outputBytes;
            }

            IOException getException()
            {
                return m_exception;
            }

            @Override
            public void run()
            {
                    try
                    {
                        if (m_outputBytes.length > 0)
                        {
                            m_stream.write(m_outputBytes);
                        }
                        m_stream.close();
                    }
                    catch(IOException e)
                    {
                         m_exception = e;
                    }
            }

        }
    }
    
    public static String printStackTrace(Throwable exception, String message)
    {
        StringBuffer buffer = new StringBuffer(message);
        buffer.append(NEWLINE);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        PrintWriter writer = new PrintWriter(out, true);
        printStackTrace(exception, writer);
        byte[] data = out.toByteArray();

        try
        {
            out.close();
        }
        catch (IOException io) { }

        buffer.append(new String(data));
        return buffer.toString();
    }

    public static void printStackTrace(Throwable exception, PrintWriter writer)
    {
        if (exception == null) {
            writer.println("Exception = null");
            return;
        }

        exception.printStackTrace(writer);
        Throwable cause = exception.getCause();

        Class exceptionClass = exception.getClass();

        try
        {
            Method getTargetExceptionMethod = exceptionClass.getMethod("getTargetException", IEmptyArray.EMPTY_CLASS_ARRAY);
            Throwable targetException = (Throwable)getTargetExceptionMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
            if (targetException != null)
            {
                if (!targetException.equals(cause))
                {
                    writer.print("Caused by: ");
                    printStackTrace(targetException, writer);
                }
                return;
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        try
        {
            Method getTargetErrorMethod = exceptionClass.getMethod("getTargetError", IEmptyArray.EMPTY_CLASS_ARRAY);
            Throwable targetError = (Throwable)getTargetErrorMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
            if (targetError != null)
            {
                if (!targetError.equals(cause))
                {
                    writer.print("Caused by: ");
                    printStackTrace(targetError, writer);
                }
                return;
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        try
        {
            Method getLinkedExceptionMethod = exceptionClass.getMethod("getLinkedException", IEmptyArray.EMPTY_CLASS_ARRAY);
            Throwable linkedException = (Throwable)getLinkedExceptionMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
            if (linkedException != null)
            {
                if (!linkedException.equals(cause))
                {
                    writer.print("Caused by: ");
                    printStackTrace(linkedException, writer);
                }
                return;
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        // the classes below print out the actual exception so only do for other exceptions
        if (!(exceptionClass.getName().equals(MFProxyRuntimeException.class.getName()) || exceptionClass.getName().equals(MFProxyException.class.getName())))
        {
            try
            {
                Method getActualExceptionMethod = exceptionClass.getMethod("getActualException", IEmptyArray.EMPTY_CLASS_ARRAY);
                Throwable actualException = (Throwable)getActualExceptionMethod.invoke(exception, IEmptyArray.EMPTY_OBJECT_ARRAY);
                if (actualException != null)
                {
                    if (!actualException.equals(cause))
                    {
                        writer.print("Caused by: ");
                        printStackTrace(cause, writer);
                    }
                    return;
                }
            }
            catch (NoSuchMethodException nsme)
            {} // ignore
            catch (IllegalAccessException iae)
            {} // ignore
            catch (InvocationTargetException ite)
            {} // ignore
        }
    }

}