package com.sonicsw.mf.common.runtime.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Objects;

import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IIdentity;
import com.sonicsw.mf.common.runtime.IState;
import com.sonicsw.mf.common.runtime.ISubComponentState;

public final class SubComponentState
implements ISubComponentState, Serializable
{
    private static final long serialVersionUID = 1L;
    private static final short m_serialVersion = 0;

    private IComponentIdentity m_componentID;
    private String m_subComponentName;
    private String m_subComponentType;
    private short m_state;

    public SubComponentState(CanonicalName componentName, IElementIdentity componentConfigID, String subComponentName, String subComponentType, short state)
    {
        m_componentID = new ComponentIdentity(componentName, componentConfigID);
        m_subComponentName = subComponentName;
        m_subComponentType = subComponentType;
        m_state = state;
    }

    @Override
    public IIdentity getRuntimeIdentity() { return m_componentID; }

    @Override
    public String getSubComponentName() { return m_subComponentName; }

    @Override
    public String getSubComponentType() { return m_subComponentType; }

    @Override
    public short getState() { return m_state; }

    public void setState(short state)
    {
        if (state < IComponentState.STATE_UNKNOWN || state > IComponentState.STATE_STOPPING)
        {
            throw new IllegalArgumentException("Invalid component state: " + state);
        }

        m_state = state;
    }

    @Override
    public String getStateString() { return ISubComponentState.STATE_TEXT[getState()]; }

    /**
     * Returns a string representation of the state. While the returned string
     * "textually represents" the state, it does not completely encapsulate it
     * - only the state string is exposed.
     *
     * @return   a string representation of the state.
     */
    @Override
    public String toString()
    {
        return getStateString();
    }

    @Override
    public boolean equals(Object object)
    {
        if (this == object)
        {
            return true;
        }
        if (!(object instanceof ISubComponentState))
        {
            return false;
        }
        
        if (!getRuntimeIdentity().equals(((IState)object).getRuntimeIdentity()))
        {
            return false;
        }
        if (!getSubComponentName().equals(((ISubComponentState)object).getSubComponentName()))
        {
            return false;
        }
        
        return getState() == ((ISubComponentState)object).getState();
    }

    @Override
    public int hashCode() {
        return Objects.hash(getRuntimeIdentity(), getSubComponentName(), getState());
    }
    
    @Override
    public int compareTo(Object object)
    throws ClassCastException
    {
        return toString().compareTo(((ISubComponentState)object).toString());
    }

    //
    // Serialization
    //

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(5);

        // MFNotification specific fields

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("componentID");
        stream.writeObject(m_componentID);

        stream.writeUTF("subComponentName");
        stream.writeObject(m_subComponentName);

        stream.writeUTF("subComponentType");
        stream.writeObject(m_subComponentType);

        stream.writeUTF("state");
        stream.writeObject(new Short(m_state));
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore

        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                try { m_componentID = (IComponentIdentity)map.get("componentID"); } catch(Exception e) {}
                try { m_subComponentName = (String)map.get("subComponentName"); } catch(Exception e) {}
                try { m_subComponentType = (String)map.get("subComponentType"); } catch(Exception e) {}
                try { m_state = ((Short)map.get("state")).shortValue(); } catch(Exception e) {}
                break;
            }
        }
    }
}
