package com.sonicsw.mf.common.security;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashMap;

/**
 * Unchecked management exception.
 * <p>
 * Indicates that a caller attempted to directly or indirectly perform some action
 * on configuration data stored in the Directory Service but did not have permission
 * to perform that action.
 */
public final class ConfigurePermissionDeniedException
extends ManagementPermissionDeniedException
{
    private static final long serialVersionUID = 0L;
    private static final short m_serialVersion = 0;

    private String m_logicalPath;
    private int m_requiredPermission;

    /**
     * Constructs a new exception with the specified detail message.  The
     * cause is not initialized, and may subsequently be initialized by
     * a call to {@link #setLinkedException}.
     *
     * @param message The textural details of why the permission was denied.
     * @param logicalPath The logical path to which the permission was denied (or null if this was a storage path with no logical equivalent).
     * @param requiredPermission The permission that was denied.
     * 
     * @see IConfigurePermissionBits
     */
    public ConfigurePermissionDeniedException(String message, String logicalPath, int requiredPermission)
    {
        super(message);
        m_logicalPath = logicalPath;
        m_requiredPermission = requiredPermission;
    }

    /**
     * Gets the logical path to which the permission was denied (or null if this was a storage path with no logical equivalent).
     */
    public String getLogicalPath()
    {
        return m_logicalPath;
    }

    /**
     * Gets the permission that was denied.
     * 
     * @see IManagePermissionBits
     */
    public int getRequiredPermission()
    {
        return m_requiredPermission;
    }
    
    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(3);

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("logicalPath");
        stream.writeObject(m_logicalPath);

        stream.writeUTF("requiredPermission");
        stream.writeObject(new Integer(m_requiredPermission));
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore

        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                // MFRuntimeException fields
                try { m_logicalPath = (String)map.get("logicalPath"); } catch(Exception e) {}
                try { m_requiredPermission = ((Integer)map.get("requiredPermission")).intValue(); } catch(Exception e) {}
                break;
            }
        }
    }
}
