package com.sonicsw.mf.common.security;

import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;

/**
 * This interface is used by administration clients to enable and configure the management security features:
 * <ul>
 * <li>Configuration and runtime checking of management permissions
 * <li>Auditing of configuration and/or runtime events.
 * </ul>
 * Creation and use of an instance of this interface is best demonstrated by the sample located at: 
 * &lt;installdir&gt;/MQ&lt;version&gt;/samples/Management/mgmtSecurityAPI/java
 * <p>
 * 
 */
public interface IManagementSecurityConfigurator
{
    /** *********************************************************************** */
    /** ****************** Management Permissions Methods ******************** */
    /** *********************************************************************** */

    /**
     * Gets the element that contains the domain-wide settings.
     * <p>
     * Although the configuration element containing domain-wide management security
     * settings is available through this API, it is recommened that users configure
     * the domain-wide settings through the Sonic Management Console or strictly follow
     * the sample located at: &lt;installdir&gt;/MQ&lt;version&gt;/samples/Management/mgmtSecurityAPI/java
     * 
     * @return The domain wide element.
     * 
     * @exception DirectoryServiceException Thrown if there is a problem accessing the underlying storage.
     */
    public IDirElement getDomainElement(boolean forUpdate)
    throws DirectoryServiceException;

    /**
     * Applies a modification to the domain-wide settings element.
     * <p>
     * Although the configuration element containing domain-wide management security
     * settings is available through this API, it is recommened that users configure
     * the domain-wide settings through the Sonic Management Console or strictly follow
     * the sample located at: &lt;installdir&gt;/MQ&lt;version&gt;/samples/Management/mgmtSecurityAPI/java
     * 
     * @param element The delta element to be applied to the DS.
     * 
     * @exception DirectoryServiceException Thrown if there is a problem accessing the underlying storage
     *                                      or the folder does not exist.
     */
    public void setDomainElement(IDeltaDirElement element)
    throws DirectoryServiceException;

    /**
     * Set management permissions.
     * <p>
     * For "manage" permissions, the path of components in containers is
     * specified as /&lt;folder1&gt;/.../&lt;foldern&gt;/&lt;ContainerName&gt;:ID=&lt;ComponentName&gt;
     * <p>
     * Will throw a runtime ConfigurePermissionDeniedException if the caller does not have
     * IConfigurePermissionBits.ALLOW_SET_PERMISSIONS permission on any of the paths.
     * 
     * @param paths
     *        The logical names of folders or configurations (for "configure" permissions) or folders, containers or
     *        components for "manage" permissions.
     * @param type
     *        "manage" or "configure", for manage or configure permissions.
     * @param permissions
     *        An array of permissions for each one of the paths.
     *        
     * @throws DirectoryServiceException
     *         Thrown if a path is invalid, or there's a problem getting or modifying the permissions information.
     * @throws InvalidPermissionException 
     *         Thrown if a scope specification is incorrect for the path, or a principal
     *         does not exist in the global authentication domain.
     *         
     * @see com.sonicsw.mf.common.security.IManagementPermission
     * @see com.sonicsw.mf.common.security.IConfigureScopeBits
     * @see com.sonicsw.mf.common.security.IConfigurePermissionBits
     * @see com.sonicsw.mf.common.security.IManageScopeBits
     * @see com.sonicsw.mf.common.security.IManagePermissionBits
     * 
     */
    public void setManagementPermissions(String[] paths, String type, IManagementPermission[][] permissions)
    throws DirectoryServiceException, InvalidManagementPermissionException;

    /**
     * Return the management permissions for the given paths of the particular type ("configure" or "manage").
     * <p>
     * Will throw a runtime ConfigurePermissionDeniedException
     * if the calling user does not have IConfigurePermissionBit.ALLOW_READ permission on any of the paths.
     * 
     * @param paths
     *        The logical names of folders or configurations (for "configure" permissions) or folders, containers or
     *        components for "manage" permissions.
     * @param type
     *        "manage" or "configure", for manage or configure permissions.
     *        
     * @return An array of defined permissions for each path in paths.
     * 
     * @throws DirectoryServiceException
     *         Thrown if a path is invalid, or there's a problem getting the permissions information.
     *         
     * @see com.sonicsw.mf.common.security.IManagementPermission
     * @see com.sonicsw.mf.common.security.IConfigureScopeBits
     * @see com.sonicsw.mf.common.security.IConfigurePermissionBits
     * @see com.sonicsw.mf.common.security.IManageScopeBits
     * @see com.sonicsw.mf.common.security.IManagePermissionBits
     */
    public IManagementPermission[][] getManagementPermissions(String[] paths, String type)
    throws DirectoryServiceException;

    /**
     * Remove the manage or configure permissions in the corresponding principals array for each path in paths.
     * <p>
     * Will throw a runtime ConfigurePermissionDeniedException if the caller does not have IConfigurePermissionBits.ALLOW_SET_PERMISSIONS
     * permission on any of the paths.
     * 
     * @param paths
     *        The logical names of folders or configurations (for "configure" permissions) or folders, containers or components for "manage" permissions.
     * @param type
     *        "manage" or "configure", for manage or configure permissions.
     * @param principals
     *        An array of array of principals for each path in paths. This method removes the permissions set for each principal.
     * 
     * @throws DirectoryServiceException Thrown if a path is invalid, or there's a problem getting the permissions information.
     * @throws InvalidManagementPermissionException Thrown if a given principal does not have a permission on the associated path.
     * 
     * @see com.sonicsw.mf.common.security.IManagementPermission
     * @see com.sonicsw.mf.common.security.IConfigureScopeBits
     * @see com.sonicsw.mf.common.security.IConfigurePermissionBits
     * @see com.sonicsw.mf.common.security.IManageScopeBits
     * @see com.sonicsw.mf.common.security.IManagePermissionBits
     */
    public void removeManagementPermissions(String[] paths, String type, String[][] principals)
    throws DirectoryServiceException, InvalidManagementPermissionException;
    
    /**
     * Remove all manage or configure permissions for each path in paths.
     * <p>
     * Will throw a ConfigurePermissionDeniedException if the caller does not have IConfigurePermissionBits.ALLOW_SET_PERMISSIONS
     * permission on any of the paths.
     * 
     * @param paths
     *        The logical names of folders or configurations (for "configure" permissions) or folders, containers or components for "manage" permissions.
     * @param type
     *        "manage" or "configure", for manage or configure permissions.
     *        
     * @throws DirectoryServiceException Thrown if a path is invalid, or there's a problem getting the permissions information.
     * 
     * @see com.sonicsw.mf.common.security.IManagementPermission
     * @see com.sonicsw.mf.common.security.IConfigureScopeBits
     * @see com.sonicsw.mf.common.security.IConfigurePermissionBits
     * @see com.sonicsw.mf.common.security.IManageScopeBits
     * @see com.sonicsw.mf.common.security.IManagePermissionBits
     */
    public void removeManagementPermissions(String[] paths, String type)
    throws DirectoryServiceException;

    /**
     * Remove all management permissions.
     * 
     * @throws DirectoryServiceException Thrown if there's a problem updating the permissions information.
     */
    public void removeAllManagementPermissions()
    throws DirectoryServiceException;

    /**
     * Set the global default manage and configure permissions for the Administrators group.
     * 
     * @throws DirectoryServiceException Thrown if there's a problem updating the permissions information.
     */
    public void setDefaultManagementPermissions()
    throws DirectoryServiceException;
}
