package com.sonicsw.mf.common.security.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;

import com.sonicsw.mf.common.security.IManagementPermission;

public class ManagementPermission
implements IManagementPermission, Serializable
{
    private static final long serialVersionUID = 8192287077613976467L;
    private static final short m_serialVersion = 0;

    private String m_principal;
    private short m_principalType;
    private int m_scope;
    private int m_permissions;

    public ManagementPermission(String principal, short principalType)
    {
        this(principal, principalType, 0, 0);
    }

    public ManagementPermission(String principal, short principalType, int scope, int permissions)
    {
        m_principal = principal;
        m_principalType = principalType;
        setScope(scope);
        setPermissions(permissions);
    }

    @Override
    public String getPrincipal()
    {
        return m_principal;
    }

    @Override
    public short getPrincipalType()
    {
        return m_principalType;
    }

    @Override
    public int getScope()
    {
        return m_scope;
    }

    @Override
    public final void setScope(int scope)
    {
        if (scope < 0)
        {
            throw new IllegalArgumentException("Permission scope must be >= 0");
        }
        
        m_scope = scope;
    }

    @Override
    public int getPermissions()
    {
        return m_permissions;
    }

    @Override
    public final void setPermissions(int permissions)
    {
        if (permissions < 0)
        {
            throw new IllegalArgumentException("Permissions value must be >= 0");
        }
        
        // since permission bits are patterned with DENY as << 1 ALLOW we can validate that
        // a sensible value has been specified (i.e. you can specify both ALLOW and DENY)
        int testPermissions = permissions;
        int count = 0;
        while (testPermissions > 0)
        {
            boolean allowBitSet = false;
            boolean denyBitSet = false;
            int temp = (testPermissions >> 1) << 1;
            if (testPermissions - temp > 0)
            {
                allowBitSet = true;
            }

            testPermissions = testPermissions >> 1;

            temp = (testPermissions >> 1) << 1;
            if (testPermissions - temp > 0)
            {
                denyBitSet = true;
            }
            
            if (allowBitSet && denyBitSet)
            {
                throw new IllegalArgumentException("Permission value is invalid; it contains conflicting allow and deny bits [" + (int)Math.pow(2, count) + " + " + (int)Math.pow(2, ++count) + "]");
            }

            testPermissions = testPermissions >> 1;
            count = count + 2;
        }
        
        
        m_permissions = permissions;
    }

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(4);

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("principal");
        stream.writeObject(m_principal);

        stream.writeUTF("scope");
        stream.writeObject(new Integer(m_scope));

        stream.writeUTF("permissions");
        stream.writeObject(new Integer(m_permissions));
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore

        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                try { m_principal = (String)map.get("principal"); } catch(Exception e) {}
                try { m_scope = ((Integer)map.get("scope")).intValue(); } catch(Exception e) {}
                try { m_permissions = ((Integer)map.get("permissions")).intValue(); } catch(Exception e) {}
                break;
            }
        }
    }
}
