package com.sonicsw.mf.common.url;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.StringTokenizer;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.BooleanExpression;
import com.sonicsw.mf.common.config.query.EqualExpression;
import com.sonicsw.mf.common.config.query.From;
import com.sonicsw.mf.common.config.query.FromDirectory;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.config.query.Where;
import com.sonicsw.mf.common.dirconfig.DirectoryDoesNotExistException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.view.ILogicalNameSpace;

/**
 * The generalized form of the "sonicrn:///" URL is:

 sonicrn:///<nodename>[?<parameter>=<value>[&<parameter>=<value>[&....]]]

.. where:

    "sonicrn" indicates that resolution is required based on a combination of node (broker or cluster members) configuration and broker runtime location information
    "<nodename>" is a valid Sonic routing node name specified for a standalone broker or cluster of brokers
    "<parameter>" & "<value>" are parameters used to limit the resolution and/or specify information on how to connect to the DS in order request resolution

 * @author davila
 *
 */
public class SonicRNURL
{
    public static final String SONICRN_URL_PREFIX = "sonicrn:///";
    public static final int SONICRN_URL_PREFIX_LENGTH = SONICRN_URL_PREFIX.length();
    
    private String m_url = "";
    private String m_nodename = "";
    private String m_visibility = "all";
    private String m_acceptorNames = "";
    private static final String VISIBILITY_PARAMETER_NAME = "visibility";
    private static final String ACCEPTOR_PARAMETER_NAME = "acceptor";
    private static boolean m_debug = Boolean.parseBoolean(System.getProperty("DEBUG_SONICRNHANDLER", "false"));
    public static SimpleDateFormat SIMPLE_DATE_FORMAT;		//NOSONAR change is not required.
    private static final AttributeName CLUSTER_NODE_NAME_ATTRIBUTE = new AttributeName("ROUTING_NODE_NAME");
    private static final AttributeName BROKER_NODE_NAME_ATTRIBUTE = (new AttributeName("BROKER_ROUTING_PARAMETERS")).setNextComponent("ROUTING_NODE_NAME");
    private static final String CLUSTER_MEMBERS_ATTRIBUTE = "CLUSTER_MEMBERS";
    public static final String CLUSTER_CONFIG_ELEMENT_REF_ATTRIBUTE = "CLUSTER_CONFIG_ELEMENT_REF";
    public static final String BROKER_NODENAME_DIRECTORY = "/_MQRuntime/location/";
    public static final String USE_DYNAMIC_HOST_BINDING_ATTRIBUTE = "USE_DYNAMIC_HOST_BINDING";
    public static final String CONFIG_ELEMENT_REFERENCES_ATTRIBUTE = "CONFIG_ELEMENT_REFERENCES";
    private static final String PRIMARY_CONFIG_ATTRIBUTE = "PRIMARY_CONFIG_ELEMENT_REF";
    public static final String BROKER_NAME_ATTRIBUTE = "BROKER_NAME";
    public static final String PRIMARY_KEYWORD = "primary";
    public static final String BACKUP_KEYWORD = "backup";
    public static final String ACCEPTORS_REF_ATTRIBUTE = "ACCEPTOR_CONFIG_ELEMENT_REF";
    public static final String ACCEPTORS_ATTRIBUTE = "ACCEPTORS";
    public static final String ACCEPTOR_URL_ATTRIBUTE = "ACCEPTOR_URL";
    private static final String ACCEPTOR_EXTERNAL_URL_ATTRIBUTE = "ACCEPTOR_EXTERNAL_URL";
    private static final String VISIBILITY_ALL = "all";
    private static final String VISIBILITY_INTERNAL = "internal";
    private static final String VISIBILITY_EXTERNAL = "external";
    private static final String ACCEPTOR_NAME_ATTRIBUTE = "ACCEPTOR_NAME";
    public static final String PUBLIC_HOSTNAME_ATTRIBUTE = "PUBLIC_HOSTNAME";
    public static final String PRIVATE_HOSTNAME_ATTRIBUTE = "PRIVATE_HOSTNAME";
    public static final String PRIMARY_CONFIG_ELEMENT_REF_ATTRIBUTE = "PRIMARY_CONFIG_ELEMENT_REF";
    public static final String BACKUP_CONFIG_ELEMENT_REF_ATTRIBUTE = "BACKUP_CONFIG_ELEMENT_REF";
    public static final String REPLICATION_CONNECTIONS_ELEMENT_REF_ATTRIBUTE = "REPLICATION_CONNECTIONS_ELEMENT_REF";
    public static final String REPLICATION_CONNECTIONS_ATTRIBUTE = "REPLICATION_CONNECTIONS";
    public static final String PRIMARY_ADDR_ATTRIBUTE = "PRIMARY_ADDR";
    public static final String BACKUP_ADDR_ATTRIBUTE = "BACKUP_ADDR";
    public static final String BROKER_CONFIG_ELEMENT_REF_ATTRIBUTE = "BROKER_CONFIG_ELEMENT_REF";
    
    // MQ broker location runtime element schema items
    public static final String MQ_RUNTIME_LOCATION_TYPE = "MQ_RUNTIME_LOCATION";
    
    // Connection factory element schema items
    public static final String REFERENCE_OBJECT_TYPE = "ReferenceObject";
    public static final String CLASSNAME_ATTRIBUTE = "classname";
    public static final String REF_ADDRESSES_ATTRIBUTE = "RefAdresses";
    public static final String BROKER_LIST_ATTRIBUTE = "brokerList";
    public static final String RESOLVED_BROKER_LIST_ATTRIBUTE = "RESOLVED_BROKER_LIST";
    
    
    static
    {
        SIMPLE_DATE_FORMAT = new SimpleDateFormat();
        SIMPLE_DATE_FORMAT.applyPattern("yy/MM/dd HH:mm:ss");
    }

    public static boolean isValid(String url)
    {
        return isValid(url, false);
    }
    
    public static boolean isValid(String url, boolean strict)
    {
        if(url == null || url.trim().isEmpty())
        {
            return false;
        }
        url = url.trim();
        if(! url.startsWith(SONICRN_URL_PREFIX))
        {
            return false;
        }
        if(! strict)
        {
            return true;
        }
        // Check for valid node name
        SonicRNURL sonicrnUrl = new SonicRNURL(url);
        String nodeName = sonicrnUrl.getNodename();
        if(nodeName == null || nodeName.trim().isEmpty())
        {
            return false;
        }
        return true;
    }
    
    public SonicRNURL(URL url)
    {
        // fix up what a regular URL would have done to the sonicrn:/// URL
        this(url.toExternalForm().startsWith(SONICRN_URL_PREFIX) ? url.toExternalForm() : url.toExternalForm().replace(":/", ":///"));
    }
    
    public SonicRNURL(String url)
    {
        if(! isValid(url, false))
        {
            throw new IllegalArgumentException("Invalid sonicrn:/// url: " + url);
        }
        m_url = url.trim();
        parseURL();
    }
    
    private void parseURL()
    {
        StringTokenizer urlTokenizer = new StringTokenizer(m_url.substring(SONICRN_URL_PREFIX_LENGTH), "?&=");
        
        m_nodename = urlTokenizer.nextToken();
        if(m_nodename != null)
        {
            m_nodename = m_nodename.trim();
        }
        
        while (urlTokenizer.hasMoreTokens())
        {
            String parameterName = urlTokenizer.nextToken();
            
            // remember there can be more than one name specified in the URL
            if (parameterName.equals(SonicRNURL.ACCEPTOR_PARAMETER_NAME))
            {
                m_acceptorNames += (m_acceptorNames.length() > 0 ? "," : "") + urlTokenizer.nextToken();
            }
            
            if (parameterName.equals(SonicRNURL.VISIBILITY_PARAMETER_NAME))
            {
                m_visibility = urlTokenizer.nextToken();
            }
        }
        debug("SonicRNURL.parseURL parsed " + m_url + " into nodename=" + m_nodename + " acceptor(s)=" + m_acceptorNames + " visibility=" + m_visibility);
    }
    
    public String getURL()
    {
        return m_url;
    }
    
    public String getNodename()
    {
        return m_nodename;
    }
    
    public String getVisibility()
    {
        return m_visibility;
    }
    
    public String getAcceptorName()
    {
        return m_acceptorNames;
    }
    
    /**
     * 
     * With as few trips to the DS as possible: 
     * 1. Find all brokers with node name = node name in url. 
     * 2. Find the node/broker mapping element in the DS for the brokers in 1. 
     * 3. Find all named acceptors in those brokers, using the ?acceptor value given, if any. 
     * If none given, take all acceptors for all brokers found in 1. 
     * 4. For each acceptor in 3, find the mapping element that defines the internal/external hostnames for its broker. 
     * 4a. If ?include is not specified, or ?include == all, create a full acceptor url with the internal hostname of the
     * broker, and one acceptor url with the external hostname of the broker 
     * 4b. If ?include is specified and its == internal, create a full acceptor url with the internal hostname for the corresponding broker 
     * 4c. If ?include is specified and its == external, create a full acceptor url with the external hostname for the corresponding broker
     * or the external URL provided by the acceptor, if there is one
     * 
     * unless an explicit external URL is provided .. in which case use that note the external urls can only be added if
     * there the container system property sets the external hostname to something other than "" or there is an explicit
     * external URL configured don't forget that if a broker is FT then you have to include its peers acceptors .. not
     * sure how we would order those relative to other members in the cluster
     */
    
    /**
     * The nodename/brokername element looks like this:
     * <Directory name="/_MQRuntime">
      <Directory name="/_MQRuntime/location">
        <Directory name="/_MQRuntime/location/MgmtBroker">
          <Directory name="/_MQRuntime/location/MgmtBroker/MgmtBroker">
            <ConfigElement>
              <ElementID name="/_MQRuntime/location/MgmtBroker/MgmtBroker/primary" creationTimestamp="1350412684814" type="MQ_RUNTIME_LOCATION" releaseVersion="1.0" version="9" />
              <AttributeSet>
                <Attribute name="EXTERNAL_HOSTNAME" value="" type="string" />
                <Attribute name="INTERNAL_HOSTNAME" value="pcbeddavila.bedford.progress.com" type="string" />
                <Attribute name="BROKER_CONFIG_ELEMENT_REF" value="/mq/brokers/1350411868794_7" type="string" />
              </AttributeSet>
            </ConfigElement>
          </Directory>
        </Directory>
      </Directory>
    </Directory>
     */

    public String resolveURL(IDirectoryAdminService ds) throws IOException
    {
        ArrayList<IDirElement> brokerElementList = findBrokersWithMatchingNodename((IDirectoryAdminService)ds);
        debug("SonicRNURL.resolveURL found " + brokerElementList.size() + " brokers with nodename == " + m_nodename);
        
        String resolved = "";
        for (IDirElement brokerElement : brokerElementList)
        {
            resolved = resolveAcceptorURLs(ds, brokerElement, resolved);
        }
        
        debug("SonicRNURL.resolveURL returning \"" + resolved + "\"");
        return resolved;
    }
    
    static void debug(String message)
    {
        if (m_debug)
        {
            System.out.print("[");
            synchronized (SIMPLE_DATE_FORMAT)
            {
                System.out.print(SIMPLE_DATE_FORMAT.format(new Date(System.currentTimeMillis())));
            }
            System.out.print("]");
            System.out.println(" " + message);
        }
    }
    
    private ArrayList<IDirElement> findBrokersWithMatchingNodename(IDirectoryAdminService ds) throws IOException
    {
        // location transparency depends upon nodenames being unique amongst brokers and clusters, so as an optimization
        // we will just handle the first matching cluster and if no cluster then the first matching broker (note: a
        // cluster with no members will still override a matching standalone broker resulting in an empty list)
        try
        {
            // first get any clustered brokers where the cluster has the required nodename
            ArrayList<IDirElement> brokerElementList = findClusteredBrokersWithMatchingNodename(ds);
            
            // if no clustered brokers, look for an individual broker (replicated or not)
            if (brokerElementList == null)
            {
                brokerElementList = findNonClusteredBrokerWithNodename(ds);
            }
            
            // if no matching brokers at all, then get out of here with an empty list
            if (brokerElementList == null)
            {
                return new ArrayList<IDirElement>();
            }
            
            // go through the brokers to see if any have backups (of course usually it will be none or all)
            for (int i = 0, initialBrokerCount = brokerElementList.size(); i < initialBrokerCount; i++)
            {
                IDirElement brokerElement = brokerElementList.get(i);
                IAttributeSet brokerAttributes = brokerElement.getAttributes();
                IAttributeSet brokerFTAttributes = (IAttributeSet)brokerAttributes.getAttribute("CONFIG_ELEMENT_REFERENCES");
                Reference backupBrokerReference = (Reference)brokerFTAttributes.getAttribute("BACKUP_CONFIG_ELEMENT_REF");
                if (backupBrokerReference != null)
                {
                    IDirElement backupBrokerElement = ds.getElement(backupBrokerReference.getElementName(), false);
                    if (backupBrokerElement != null)
                    {
                        brokerElementList.add(backupBrokerElement);
                    }
                }
            }
            
            return brokerElementList;
        }
        catch (Throwable t)
        {
            IOException ioE = new IOException("Unable to find brokers with nodename " + m_nodename);
            ioE.initCause(t);
            throw ioE;
        }
    }
    
    private ArrayList<IDirElement> findClusteredBrokersWithMatchingNodename(IDirectoryAdminService ds) throws IOException
    {
        ArrayList<IDirElement> matchingBrokers = new ArrayList<IDirElement>();
        
        try
        {
            Query query = new Query();
            From from = new FromDirectory("/mq/clusters");
            BooleanExpression nodenameEqualsExpression = new EqualExpression(CLUSTER_NODE_NAME_ATTRIBUTE, m_nodename);
            Where where = new Where(new BooleanExpression[]{ nodenameEqualsExpression });
            query.setFrom(from);
            query.setWhere(where);
            
            IDirElement[] clustersWithMatchingNodename = ds.getElements(query, false);
            if (clustersWithMatchingNodename.length == 0)
            {
                return null;
            }
            
            // location transparency depends upon nodenames being unique amongst brokers and clusters, so as an optimization
            // we will just handle the first matching cluster
            IDirElement clusterElement = clustersWithMatchingNodename[0];
            
            IAttributeSet clusterAttributes = clusterElement.getAttributes();
            IAttributeSet clusterMemberAttributes = (IAttributeSet)clusterAttributes.getAttribute(CLUSTER_MEMBERS_ATTRIBUTE);
            for (Object clusterMember : clusterMemberAttributes.getAttributes().values())
            {
                Reference clusteredMemberReference = (Reference)clusterMember;
                IDirElement brokerElement = ds.getElement(clusteredMemberReference.getElementName(), false);
                // its possible the broker cannot be found because its just been deleted but the trigger to remove it from the
                // has not yet fired
                if (brokerElement != null)
                {
                    matchingBrokers.add(brokerElement);
                }
            }
        }
        catch (Throwable t)
        {
            IOException ioE = new IOException("Unable to find clusters with nodename " + m_nodename);
            ioE.initCause(t);
            throw ioE;
        }
        
        return matchingBrokers;
    }
    
    private ArrayList<IDirElement> findNonClusteredBrokerWithNodename(IDirectoryAdminService ds) throws IOException
    {
        ArrayList<IDirElement> matchingBrokers = new ArrayList<IDirElement>();
        
        try
        {
            Query query = new Query();
            From from = new FromDirectory("/mq/brokers");
            BooleanExpression nodenameEqualsExpression = new EqualExpression(BROKER_NODE_NAME_ATTRIBUTE, m_nodename);
            Where where = new Where(new BooleanExpression[]{ nodenameEqualsExpression });
            query.setFrom(from);
            query.setWhere(where);
            
            IDirElement[] brokersWithMatchingNodename = ds.getElements(query, false);
            if (brokersWithMatchingNodename.length == 0)
            {
                return null;
            }
            
            // location transparency depends upon nodenames being unique amongst brokers and clusters, so as an optimization
            // we will just handle the first matching broker
            IDirElement brokerElement = brokersWithMatchingNodename[0];
            
            // if one was found, verify that its not clustered
            IAttributeSet brokerAttributes = brokerElement.getAttributes();
            IAttributeSet referenceAttributes = (IAttributeSet)brokerAttributes.getAttribute(CONFIG_ELEMENT_REFERENCES_ATTRIBUTE);
            Reference clusterReference = (Reference)referenceAttributes.getAttribute(CLUSTER_CONFIG_ELEMENT_REF_ATTRIBUTE);
            if (clusterReference == null)
            {
                // then broker is not part of a cluster and we can use it
                matchingBrokers.add(brokerElement);
            }
        }
        catch (Throwable t)
        {
            IOException ioE = new IOException("Unable to find clusters with nodename " + m_nodename);
            ioE.initCause(t);
            throw ioE;
        }
        
        return matchingBrokers;
    }
    
    private IDirElement getBrokerHostMappingElement(IDirectoryAdminService ds, IDirElement broker) throws IOException
    {
        try
        {
            boolean isPrimary = ((IAttributeSet)broker.getAttributes().getAttribute(CONFIG_ELEMENT_REFERENCES_ATTRIBUTE)).getAttribute(PRIMARY_CONFIG_ATTRIBUTE) == null;
            String mappingElementName = BROKER_NODENAME_DIRECTORY + m_nodename + IMFDirectories.MF_DIR_SEPARATOR + broker.getAttributes().getAttribute(BROKER_NAME_ATTRIBUTE) + IMFDirectories.MF_DIR_SEPARATOR
                                        + (isPrimary ? PRIMARY_KEYWORD : BACKUP_KEYWORD);
            try
            {
                IDirElement map = ds.getElement(mappingElementName, false);
                return map;
            }
            catch (DirectoryDoesNotExistException dirE) {return null;} //ok, element does not exist
            
        }
        catch (DirectoryServiceException dirE)
        {
            IOException ioE = new IOException("Unable to get the nodename/brokername element for " + broker.getIdentity().getName());
            ioE.initCause(dirE);
            throw ioE;
        }
    }
    
    private IDirElement[] getQualifiedAcceptors(IDirectoryAdminService ds, IDirElement broker) throws IOException
    {
        ArrayList<IDirElement> acceptorsList = new ArrayList<IDirElement>();

        Reference acceptorsReference = (Reference)((IAttributeSet)broker.getAttributes().getAttribute(CONFIG_ELEMENT_REFERENCES_ATTRIBUTE)).getAttribute(ACCEPTORS_REF_ATTRIBUTE);
        if (acceptorsReference != null)
        {
            try
            {
                String acceptorsElementName = acceptorsReference.getElementName();
                
                // on config bean import such a condition is possible .. so all we can do is ignore and the right thing
                // will be done when the reference is actually correctly set
                if (acceptorsElementName.startsWith(ILogicalNameSpace.NO_STORAGE_LABEL))
                {
                    return new IDirElement[0];
                }

                IDirElement acceptorsElement = ds.getElement(acceptorsElementName, false);
                if (acceptorsElement != null)
                {
                    IAttributeSet acceptorRefs = (IAttributeSet)acceptorsElement.getAttributes().getAttribute(ACCEPTORS_ATTRIBUTE);
                    for (Reference acceptorRef : (Collection<Reference>)acceptorRefs.getAttributes().values())
                    {
                        IDirElement acceptor = ds.getElement(acceptorRef.getElementName(), false);
                        if (acceptor != null)
                        {
                            String acceptorType = acceptor.getIdentity().getType();
                            // location transparency is not interested in direct protocols
                            if (!(acceptorType.equals("MQ_ACCEPTOR_TCPS") || acceptorType.equals("MQ_ACCEPTOR_TUNNELING")))
                            {
                                continue;
                            }
                            if (m_acceptorNames.length() > 0)
                            {
                                String currentAcceptorName = (String)acceptor.getAttributes().getAttribute(ACCEPTOR_NAME_ATTRIBUTE);
                                StringTokenizer acceptorNames = new StringTokenizer(m_acceptorNames, ",");
                                while (acceptorNames.hasMoreTokens())
                                {
                                    if (currentAcceptorName.equals(acceptorNames.nextToken()))
                                    {
                                        acceptorsList.add(acceptor);
                                    }
                                }
                            }
                            else
                            {
                                acceptorsList.add(acceptor);
                            }
                        }
                    }
                }
            }
            catch (DirectoryServiceException dirE)
            {
                IOException ioE = new IOException("Unable to get the acceptors for " + broker.getIdentity().getName());
                ioE.initCause(dirE);
                throw ioE;
            }
        }
        
        IDirElement[] acceptors = new IDirElement[acceptorsList.size()];
        acceptorsList.toArray(acceptors);
        debug("SonicRNURL.getQualifiedAcceptors after looking for acceptor names found " + acceptors.length + " acceptors for broker " + broker.getIdentity().getName());
        return acceptors;
    }
    
    private String resolvePublic(IDirElement acceptor, IDirElement nodeMapping) throws MalformedURLException
    {
        String publicHost = (String)nodeMapping.getAttributes().getAttribute(PUBLIC_HOSTNAME_ATTRIBUTE);
        if (publicHost.length() !=0)
        {
            BrokerURL url = new BrokerURL((String)acceptor.getAttributes().getAttribute(ACCEPTOR_URL_ATTRIBUTE), true);
            return url.getBrokerProtocol() + "://" + publicHost + ":" + url.getBrokerPort();
        }
        else
        {
            return null;
        }
    }
    
    private String resolveAcceptorURLs(IDirectoryAdminService ds, IDirElement broker, String resolved)
    throws IOException
    {
        IDirElement hostMapping = getBrokerHostMappingElement(ds, broker);
        // get the acceptors
        IDirElement[] acceptors = getQualifiedAcceptors(ds, broker);
        // TODO: possibly resolve the acceptors here if they have not been resolved. For now assume
        // the DS has resolved them if the nodename/brokername element has been set
        //
        for (IDirElement acceptor : acceptors)
        {
            debug("SonicRNURL.resolveAcceptorURLs broker == " + broker.getIdentity().getName() + " acceptor url == " + acceptor.getAttributes().getAttribute(ACCEPTOR_URL_ATTRIBUTE) + 
                      " acceptor external url == " + acceptor.getAttributes().getAttribute(ACCEPTOR_EXTERNAL_URL_ATTRIBUTE) );
            String externalURL = (String)acceptor.getAttributes().getAttribute(ACCEPTOR_EXTERNAL_URL_ATTRIBUTE);
            if (m_visibility.equals(VISIBILITY_ALL) || m_visibility.equals(VISIBILITY_EXTERNAL))
            {
                if (externalURL != null)
                {
                    resolved += (resolved.length() > 0 ? "," : "") + externalURL;
                }
                else if (hostMapping != null)
                {
                    String resolvedExternal = resolvePublic(acceptor, hostMapping);
                    if (resolvedExternal != null)
                    {
                        resolved += (resolved.length() > 0 ? "," : "") + resolvedExternal;
                    }
                }
            }
            if (m_visibility.equals(VISIBILITY_ALL) || m_visibility.equals(VISIBILITY_INTERNAL))
            {
                String acceptorURLString = (String)acceptor.getAttributes().getAttribute(ACCEPTOR_URL_ATTRIBUTE);
                if (acceptorURLString.contains("://:"))
                 {
                    // equates to no hostname
                    continue; // ignore because this implies dynamic host binding and the host has not yet been set
                }
                resolved += (resolved.length() > 0 ? "," : "") + acceptorURLString;
            }
        }
        debug("SonicRNURL.resolveAcceptorURLs broker == " + broker.getIdentity().getName() + " returning solved ==  " + resolved);
        return resolved;
    }
}
