/* Copyright (c) 2007 Progress Software Corporation.  All Rights Reserved. */

package com.sonicsw.mf.common.util;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.xml.ElementBuilder;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

/**
 * Container configuration Utilities
 */

public class Container
{

    public static final String SONICFS_PROTOCOL  = "sonicfs:///";
    public static final String SONICHOME_PROTOCOL = "sonichome:///";

    private static final String CONNECTION_PREFIX = "Connection.";
    private static final String CENTRAL_CONNECTION_PREFIX = "CentralConnection.";
    private static final String CACHE_PASSWORD_ATTR = "CACHE_PASSWORD";         //NOSONAR field change is not required.
    private static final String CENTRAL_CONNECTION_SETUP_PROP = "CENTRAL_CONNECTION";

    private static Set xmlBootAttributes = new HashSet();
    static
    {
        xmlBootAttributes.add(IContainerConstants.CLASSNAME_ATTR);
        xmlBootAttributes.add(IContainerConstants.DOMAIN_NAME_ATTR);
        xmlBootAttributes.add(IContainerConstants.CONTAINER_NAME_ATTR);
        xmlBootAttributes.add(IContainerConstants.HOSTS_DIRECTORY_SERVICE_ATTR);
        xmlBootAttributes.add(IContainerConstants.COMMAND_LINE_ATTR);
        xmlBootAttributes.add(IContainerConstants.CACHE_ATTR);
        xmlBootAttributes.add(IContainerConstants.CONNECTION_ATTR);
        xmlBootAttributes.add(IContainerConstants.CENTRAL_CONNECTION_ATTR);
        xmlBootAttributes.add(IContainerConstants.LOG_TO_CONSOLE_ATTR);
        xmlBootAttributes.add(IContainerConstants.LOG_TO_FILE_ATTR);
        xmlBootAttributes.add(IContainerConstants.LOG_FILE_ATTR);
        xmlBootAttributes.add(IContainerConstants.LOG_FILE_SIZE_THRESHOLD_ATTR);
        xmlBootAttributes.add(IContainerConstants.TRACE_MASK_ATTR);
    }

    private static Set iniBootAttributes = new HashSet();
    static
    {
        iniBootAttributes.add(IContainerConstants.DOMAIN_NAME_ATTR);
        iniBootAttributes.add(IContainerConstants.HOSTS_DIRECTORY_SERVICE_ATTR);
        iniBootAttributes.add(IContainerConstants.CACHE_ATTR);
        iniBootAttributes.add(IContainerConstants.CONNECTION_ATTR);
        iniBootAttributes.add(IContainerConstants.CENTRAL_CONNECTION_ATTR);
        iniBootAttributes.add(IContainerConstants.LOG_FILE_ATTR);
        iniBootAttributes.add(IContainerConstants.TRACE_MASK_ATTR);
    }

   /**
    * Returns the container boot information. exportContainerBootConfiguration exports the element to XML including
    * only attributes needed for the container boot process.
    *
    * @param containerConfigElement the container configuration element. The element is typically returned
    * from a IDirectoryAdminService.getElement("/containers/XXX" ...) call.
    *
    * @param domainName the name of the domain where is container is defined
    *
    * @return the XML boot file content.
    *
    */
    public static String exportContainerBootConfiguration(IElement containerConfigElement, String domainName)
    {
        try
        {
            return ElementBuilder.exportElement(filterBootConfiguration(containerConfigElement, xmlBootAttributes), domainName);
        }
        catch (Exception e) // Should never happen
        {
            e.printStackTrace();
            throw new Error(e.toString(), e);
        }
    }

   /**
    * Returns the container boot information as an container.ini string including
    * only attributes needed for the container boot process.
    *
    * @param containerConfigElement the container configuration element. The element is typically returned
    * from a IDirectoryAdminService.getElement("/containers/XXX" ...) call.
    *
    * @return the container.ini file content.
    *
    */
    public static String exportContainerBootConfigurationToINI(IElement containerConfigElement)
    {
        try
        {
            Properties props = new Properties();
            bootConfigurationToProp(containerConfigElement, props);

            Enumeration propItems = props.keys();
            StringBuffer sb = new StringBuffer();
            sb.append("");
            while (propItems.hasMoreElements())
            {
                String key = (String)propItems.nextElement();
                String value = props.getProperty(key);
                if (value.length() == 0)
                {
                    continue;
                }

                sb.append(key).append("=").append(value).append("\n");
            }

            return sb.toString();
        }
        catch (Exception e) // Should never happen
        {
            e.printStackTrace();
            throw new Error(e.toString(), e);
        }
    }

    //Find out whether central connection is configured
    private static boolean containsCentralConnection(HashMap attributeTable)
    {
        Iterator iterator = attributeTable.keySet().iterator();
        while (iterator.hasNext())
        {
            String attributeName = (String)iterator.next();
            if (attributeName.equalsIgnoreCase(IContainerConstants.CENTRAL_CONNECTION_ATTR))
            {
                return true;
            }
        }
        return false;
    }

   /**
    * Transforms the configured attributes of a container to the Properties format. The caller passes a Properties object
    * that contains non configured properties and the method adds into the Properties object the configured properties. The
    * most common non configured property is WINDOWS_SERVICE_NAME and system properties in the form of SytemProperty.property=value.
    * See the Launcher Install spec for the full list.
    *
    * @param containerConfigElement the container configuration element
    * @param properties initially contains the non configured, the method adds the configured properties
    */
    public static void bootConfigurationToSetupProp(IElement containerConfigElement, Properties properties)
    {
        configurationToPropInternal(containerConfigElement, properties, true);
    }

    public static void bootConfigurationToProp(IElement containerConfigElement, Properties props)
    {
        configurationToPropInternal(containerConfigElement, props, false);
    }

    public static boolean iniFileAttribute(String attribute)
    {
        return iniBootAttributes.contains(attribute);
    }

    private static void configurationToPropInternal(IElement containerConfigElement, Properties props, boolean forSetup)
    {
        IElement filteredElement = filterBootConfiguration(containerConfigElement, iniBootAttributes);
        IAttributeSet containerAttrs = filteredElement.getAttributes();
        HashMap attributeTable = containerAttrs.getAttributes();

        boolean hasCentral = containsCentralConnection(attributeTable);

        Iterator iterator = attributeTable.keySet().iterator();
        boolean domainNamePresent = false;
        while (iterator.hasNext())
        {
            String attributeName = (String)iterator.next();
            if (attributeName.equals(IContainerConstants.DOMAIN_NAME_ATTR))
            {
                domainNamePresent = true;
            }

            Object value = attributeTable.get(attributeName);
            if (value == null)
            {
                continue;
            }

            if (attributeName.equalsIgnoreCase(IContainerConstants.CACHE_ATTR))
            {
                IAttributeSet cacheAttrs = (IAttributeSet)value;
                String password = (String)cacheAttrs.getAttribute(IContainerConstants.PASSWORD_ATTR);
                if (password != null)
                {
                    props.setProperty(CACHE_PASSWORD_ATTR, password);
                }
                String cacheDir = (String)cacheAttrs.getAttribute(IContainerConstants.CACHE_DIRECTORY_ATTR);
                if (cacheDir != null)
                {
                    props.setProperty(IContainerConstants.CACHE_DIRECTORY_ATTR, cacheDir);
                }

            }
            else if (attributeName.equalsIgnoreCase(IContainerConstants.CONNECTION_ATTR) && !hasCentral)
            {
                addConnectionAtts((IAttributeSet)value, props, CONNECTION_PREFIX, forSetup);
            }
            else if (attributeName.equalsIgnoreCase(IContainerConstants.CENTRAL_CONNECTION_ATTR))
            {
                addConnectionAtts((IAttributeSet)value, props, CENTRAL_CONNECTION_PREFIX, forSetup);
            }
            else
            {
                props.setProperty(attributeName, value.toString());
            }
        }
        if (forSetup && hasCentral)
        {
            props.setProperty(CENTRAL_CONNECTION_SETUP_PROP, "true");
        }

        if (!domainNamePresent)
        {
            props.setProperty(IContainerConstants.DOMAIN_NAME_ATTR, IContainerConstants.DOMAIN_NAME_DEFAULT);
        }

    }

    private static void addConnectionAtts(IAttributeSet connAtts, Properties props, String prefix, boolean forSetup)
    {
        //Setup Properties are not in the form of Connection.ConnectURLs=<urls> but rather ConnectURLs=<urls>
        if (forSetup)
        {
            prefix = "";
        }

        HashMap attributeTable = connAtts.getAttributes();
        Iterator iterator = attributeTable.keySet().iterator();
        boolean connectionUrlsPresent = false;
        while (iterator.hasNext())
        {
            String attributeName = (String)iterator.next();
            if (attributeName.equalsIgnoreCase(IContainerConstants.CONNECTIONURLS_ATTR))
            {
                connectionUrlsPresent = true;
            }
            props.setProperty(prefix+attributeName, attributeTable.get(attributeName).toString());
        }
        if (!connectionUrlsPresent)
        {
            props.setProperty(prefix+IContainerConstants.CONNECTIONURLS_ATTR, IContainerConstants.CONNECTIONURLS_DEFAULT);
        }
    }

    private static IElement filterBootConfiguration(IElement containerConfigElement, Set bootAttributes)
    {
        try
        {
            IElement writableCopy = containerConfigElement.createWritableClone();
            IAttributeSet writableAttributes = writableCopy.getAttributes();

            IAttributeSet containerAttrs = containerConfigElement.getAttributes();
            HashMap attributeTable = containerAttrs.getAttributes();
            Iterator iterator = attributeTable.keySet().iterator();
            while (iterator.hasNext())
            {
                String attributeName = (String)iterator.next();
                if (!bootAttributes.contains(attributeName))
                {
                    writableAttributes.deleteAttribute(attributeName);
                }
            }

            // remove a deprecated connection attribute
            IAttributeSet connectionAttrs = (IAttributeSet)writableAttributes.getAttribute(IContainerConstants.CONNECTION_ATTR);
            if (connectionAttrs != null)
            {
                if (connectionAttrs.getAttribute("TRY_DS_AM_BACKUPS_ON_FAILURE") != null)
                {
                    connectionAttrs.deleteAttribute("TRY_DS_AM_BACKUPS_ON_FAILURE");
                }
            }

            return writableCopy;

        }
        catch (Exception e) // Should never happen
        {
            e.printStackTrace();
            throw new Error(e.toString(), e);
        }
    }

    public static IDeltaDirElement initializeNewContainer(IDirElement containerConfigElement,
                                              Hashtable defaultContainerParams,
                                              Hashtable defaultConnectionParams) throws Exception
    {
        IAttributeSet containerAtts = containerConfigElement.getAttributes();

        IAttributeSet cacheAtts = (IAttributeSet)containerAtts.getAttribute(IContainerConstants.CACHE_ATTR);
        IAttributeSet connectAtts = (IAttributeSet)containerAtts.getAttribute(IContainerConstants.CONNECTION_ATTR);

        String cacheDir = (String)defaultContainerParams.remove(IContainerConstants.CACHE_DIRECTORY_ATTR);
        String cachePassword = (String)defaultContainerParams.remove("CACHE_PASSWORD");

        Enumeration keys = defaultContainerParams.keys();
        while (keys.hasMoreElements())
        {
            String key = (String)keys.nextElement();
            containerAtts.setObjectAttribute(key, defaultContainerParams.get(key));
        }

        if (cacheDir != null)
        {
            cacheAtts.setStringAttribute(IContainerConstants.CACHE_DIRECTORY_ATTR, cacheDir);
        }
        if (cachePassword != null)
        {
            cacheAtts.setStringAttribute(IContainerConstants.PASSWORD_ATTR, cachePassword);
        }

        keys = defaultConnectionParams.keys();
        while (keys.hasMoreElements())
        {
            String key = (String)keys.nextElement();
            connectAtts.setObjectAttribute(key, defaultConnectionParams.get(key));
        }

        return (IDeltaDirElement)containerConfigElement.doneUpdate();

    }

}
