package com.sonicsw.mf.common.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * This class manages a lock file. The lock file is created if necessary and is either in a state of
 * locked or unlocked.
 * <p>
 * Locking ensures the file is exclusively locked by the caller. Unlocking removes the lock file.
 * <p>
 * Lock files cannot be locked if the JRE is < 1.4 and the OS is not Windows and a file of the same
 * name already exists.
 */
public class LockFile
{
    private File m_lockFile;
    private File m_processFile;
    private RandomAccessFile m_raf;
    private Object m_fileLock;

    private static boolean m_canUseNIO = false;
    private static boolean m_isWindowsOS;

    static
    {
        try
        {
            Class fileLockClass = Class.forName("java.nio.channels.FileLock");
            m_canUseNIO = true;
        }
        catch(Throwable e) { }
        m_isWindowsOS = (File.pathSeparatorChar == ';');
    }

    public LockFile(String path)
    {
        m_lockFile = new File(path);
        m_processFile = new File(path + ".tmp");
    }

    /**
     * Exclusively locks the lock file.
     *
     * @return true if the file was successfully locked, false if not.
     */
    public synchronized boolean lock()
    {
        if (m_raf != null)
        {
            return true;
        }

        // NOTE: THE WINDOWS MECHANISM FOR USE WITH jdk1.3 DOES NOT WORK WITH ANOTHER
        //       PROCESS USING THE jdk1.4 (NIO) MECHANISM - THUS WE ALWAYS USE THE
        //       WINDOWS MECHANISM FOR USE WITH jdk1.3 IRRESPECTIVE OF WHETHER NIO IS
        //       AVAILABLE

        if (m_isWindowsOS) // is it a Windows OS?
        {
            // NOTE: WE USE THIS MECHANISM ON WINDOWS EVEN IF NIO IS AVAILABLE AS THERE SEEMS TO BE
            //       SOME ISSUE BETWEEN CLOCKS WITH DIFFERENT VM VERSIONS (IBM VS. SUN)
            try
            {
                // if a process file exists for the lock then some other process is in the process
                // of locking, trying to get a lock or unlocking
                if (m_processFile.exists())
                {
                    return false;
                }
                // if it did not exist when the exist call was made, it might exist now (another process
                // created it just after our call) .. so try to create it exclusively
                if (!m_processFile.createNewFile())
                {
                    return false;
                }

                m_lockFile.delete();

                // if the delete failed on Windows its because the file is in use by another process
                if (m_lockFile.exists())
                {
                    return false;
                }

                // create the lock file
                try
                {
                    if (!m_lockFile.createNewFile())
                    {
                        return false;
                    }
                }
                catch (IOException e)
                {
                    return false;
                }

                // now use it
                try
                {
                    m_raf = new RandomAccessFile(m_lockFile, "rw");
                    return true;
                }
                catch (FileNotFoundException e)
                {
                    return false;
                }
            }
            catch(IOException e)
            {
                return false;
            }
            finally
            {
                m_processFile.delete();
            }
        }
        else if (m_canUseNIO) // is this >= jdk1.4 (so that we can use NIO?)
        {
            try
            {
                m_raf = new RandomAccessFile(m_lockFile, "rw");
                // have to do the rest using reflection so we can still compile with jdk1.3
                Class rafClass = RandomAccessFile.class;
                try
                {
                    Method getChannelMethod = rafClass.getMethod("getChannel", new Class[0]);
                    Object channel = getChannelMethod.invoke(m_raf, new Object[0]);
                    Class channelClass = channel.getClass();
                    Method getLockMethod = channelClass.getMethod("tryLock", new Class[0]);
                    m_fileLock = getLockMethod.invoke(channel, new Object[0]);
                    return m_fileLock != null;
                }
                catch(InvocationTargetException e)
                {
                    if (!(e.getTargetException() instanceof IOException))
                    {
                        // HP-UX 1.4 throws this!
                        e.getTargetException().printStackTrace();
                    }
                    return false;
                }
                catch(Exception e)
                {
                    // should never happen
                    e.printStackTrace();
                    return false;
                }
            }
            catch(FileNotFoundException e)
            {
                e.printStackTrace();
                return false;
            }
            finally
            {
                if (m_fileLock == null)
                {
                    cleanup();
                }
            }
        }
        else // if the lock file does not exist we can create it
        {
            try
            {
                return m_lockFile.createNewFile();
            }
            catch (IOException e)
            {
                return false;
            }
        }
    }

    /**
     * Unlocks and deletes the lock file.
     */
    public synchronized void unlock()
    {
        if (m_isWindowsOS)
        {
            while (true)
            {
                try
                {
                    // if a process file exists for the lock then some other process is in the process
                    // of locking, trying to get a lock or unlocking else try to create it (exclusively)
                    if (!m_processFile.exists() && m_processFile.createNewFile())
                    {
                        break;
                    }
                }
                catch (IOException e)
                {
                    // eat the exception since were trying to cleanup
                    e.printStackTrace();
                }
                // backoff and try again
                try { this.wait(250); } catch (InterruptedException e) { }
            }

            try
            {
                cleanup();
            }
            finally
            {
                // it is possible this delete might fail because in between the time we released the RAF
                // another process may opened the file
                m_lockFile.delete();

                m_processFile.delete();
            }
        }
        else
        {
            try
            {
                cleanup();
            }
            finally
            {
                m_lockFile.delete();
            }
        }
    }

    private void cleanup()
    {
        try
        {
            if (m_fileLock != null)
            {
                Class fileLockClass = m_fileLock.getClass();
                try
                {
                    Method releaseMethod = fileLockClass.getMethod("release", new Class[0]);
                    releaseMethod.invoke(m_fileLock, new Object[0]);
                }
                catch(Exception e)
                {
                    // should never happen
                    e.printStackTrace();
                }
            }
        }
        finally
        {
            try
            {
                if (m_raf != null)
                {
                    m_raf.close();
                    m_raf = null;
                }
            } catch(IOException e) { /*e.printStackTrace();*/ } // HP-UX get IOException!!
            m_fileLock = null;
        }
    }
}