/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.util;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeMetaData;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.ITriggerDSContext;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.FromDirectory;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.mgmtapi.config.constants.IComponentCollectionConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

/**
 * Utilities for keeping MF referential integrity
 */
public class ReferentialIntegrity
{
    // this is a copy of IContainer.CONTAINERS_DIR, but we can't ref that constant from this file
    private static final String CONTAINERS_DIR = "/containers";

    // remove the component from any containers that host its configuration and if we find any, remove
    // the runtime ids from any component collections that might include them
    public static void removeComponent(ITriggerDSContext ds, String configID) throws DirectoryServiceException
    {
        if (!ds.directoryExists(CONTAINERS_DIR))
        {
            return;
        }

        IElementIdentity[] ids = ds.listElements(CONTAINERS_DIR);

        try
        {
            for (int i = 0; i < ids.length; i++)
            {
                IDirElement containerConfig = ds.getElement(ids[i].getName());
                boolean elementModified = false;
                IAttributeSet containerAtts = containerConfig.getAttributes();
                String containerName = (String)containerAtts.getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
                String domainName = (String)containerAtts.getAttribute(IContainerConstants.DOMAIN_NAME_ATTR);
                IAttributeSet componentsAtt = (IAttributeSet)containerAtts.getAttribute(IContainerConstants.COMPONENTS_ATTR);
                if (componentsAtt == null)
                {
                    continue;
                }

                // If the component list is from a template then we should fix only the template
                IAttributeMetaData compMeta = containerAtts.getAttributeMetaData(IContainerConstants.COMPONENTS_ATTR);
                if (compMeta.isFromTemplate())
                {
                    continue;
                }

                HashMap componentsTable = componentsAtt.getAttributes();
                Iterator iterator = componentsTable.keySet().iterator();
                while (iterator.hasNext())
                {
                    String componentName = (String)iterator.next();
                    IAttributeSet compDefinition = (IAttributeSet)componentsTable.get(componentName);
                    Reference configRef = (Reference)compDefinition.getAttribute(IContainerConstants.CONFIG_REF_ATTR);

                    if (configRef == null)
                    {
                        continue;
                    }

                    if (configRef.getElementName().equals(configID))
                    {
                        elementModified = true;
                        componentsAtt.deleteAttribute(componentName);
                        removeComponentFromComponentCollections(ds, new CanonicalName(domainName, containerName, componentName));
                        if (componentName.equals("DIRECTORY SERVICE"))
                        {
                            containerAtts.deleteAttribute(IContainerConstants.HOSTS_DIRECTORY_SERVICE_ATTR);
                        }
                    }
                }
                if (elementModified)
                {
                    ds.setElement((IDeltaElement)containerConfig.doneUpdate());
                }
            }
        }
        catch (ConfigException e)
        {
            e.printStackTrace(); // Should never happen
            throw new Error(e.toString(), e);
        }
    }

    // Remove the given component from all the component collections that refer to it
    private static void removeComponentFromComponentCollections(ITriggerDSContext ds, CanonicalName componentName) throws DirectoryServiceException
    {
        if (!ds.directoryExists("/componentCollections"))
        {
            return;
        }

        try
        {

            AttributeName attrName = new AttributeName(IComponentCollectionConstants.COMPONENTS_ATTR);
            FromDirectory f = new FromDirectory("/componentCollections");
            IDirElement[] elements = ds.getElements(new Query().setFrom(f), true);

            for (int i = 0; i < elements.length; i++)
            {
                IAttributeSet collectionAtts = elements[i].getAttributes();

                // If the set comes from a template then we'll modify the template
                if (collectionAtts.getAttributeMetaData(IComponentCollectionConstants.COMPONENTS_ATTR).isFromTemplate())
                {
                    continue;
                }

                IAttributeSet set = (IAttributeSet)collectionAtts.getAttribute(IComponentCollectionConstants.COMPONENTS_ATTR);
                Iterator iterator = set.getAttributes().entrySet().iterator();
                while (iterator.hasNext())
                {
                    Map.Entry entry = (Map.Entry)iterator.next();
                    IAttributeSet itemSet = (IAttributeSet)entry.getValue();
                    String id = (String)itemSet.getAttribute(IComponentCollectionConstants.COMPONENT_RUNTIME_ID_ATTR);
                    if (id.equals(componentName.getCanonicalName()))
                    {
                        set.deleteAttribute((String)entry.getKey());
                        ds.setElement((IDeltaElement)elements[i].doneUpdate());
                        break;
                    }
                }
            }
        }
        catch (ConfigException e)
        {
            e.printStackTrace(); // Should never happen
            throw new Error(e.toString(), e);
        }
    }
}
