/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.view;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.ReadOnlyException;

/**
 * An IFolder object is a view element which contains sub folders and link elements
 */
public interface IFolder extends IViewElement
{

 /**
  * Creates a new folder.
  *
  * @param folderName the name of the new folder
  *
  * @return the new folder
  *
  * @exception ViewException if there is already a view element with this name
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if the folderName is illegal
  *
  */
  public IFolder createFolder(String folderName) throws ViewException, ReadOnlyException, ConfigException;

 /**
  * Deletes a folder and the view elements it contains.
  *
  * @param folderName the name of the folder
  *
  * @exception ViewException if the folder does not exist
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if the name is illegal
  *
  */
  public void deleteFolder(String folderName) throws ViewException, ReadOnlyException, ConfigException;

 /**
  * Creates a new link.
  *
  * @param linkName the name of the link
  * @param linkedObjectName the name the object the link is created to represent
  *
  * @return the new link
  *
  * @exception ViewException if there is already a view element with this name
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if linkName is an illegal name
  *
  */
  public ILink link(String linkName, String linkedObjectName) throws ViewException, ReadOnlyException, ConfigException;

 /**
  * Creates a new link.
  *
  * @param linkName the name of the link
  * @param linkedObjectName the name the object the link is created to represent
  * @param complex if true then the link is interpreted as a link to an entire DS subtree
  *
  * @return the new link
  *
  * @exception ViewException if there is already a view element with this name
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if linkName is an illegal name
  *
  */
  public ILink link(String linkName, String linkedObjectName, boolean complex) throws ViewException, ReadOnlyException, ConfigException;


 /**
  * Deletes a link.
  *
  * @param linkName the link to be deleted
  *
  * @exception ViewException if the link does not exist
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if linkName is an illegal name
  *
  */
  public void unlink(String linkName) throws ViewException, ReadOnlyException, ConfigException;

 
 /**
  * Returns a view element (a link or a folder)
  *
  * @param viewElementName the name of the view element
  *
  * @return the view element
  *
  * @exception ViewException if the view element does not exist
  * @exception ConfigException if the name is illegal
  *
  */
  public IViewElement getViewElement(String viewElementName) throws ViewException, ConfigException;

 /**
  * Lists all the view elements names under this folder.
  *
  * @return a list of view elements names
  *
  */
  public String[] list();
}
