/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.view;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.ReadOnlyException;


/**
 * The IView object defines a logical organization of a directory domain. There is one view per domain. The view
 * is a tree structure with folders as tree nodes and logical links for leaves. A link typically represents a domain's 
 * directory or domain's element. While directory and element names cannot be modified, the view can be easily reorganized
 * with folder and link renaming. Each view element has a flat set of attributes. There are two types of attributes,
 * (com.sonicsw.mf.common.config.IAttributeSetType) - one for folders and the other for links.
 * An administration application typically retrieves
 * a snapshot of the view through the IDirectoryAdminService.getView() call and potentially modifies it. It applies the
 * modified version of the view by calling the doneUpdate() method which returns the delta between the original
 * version of the snapshot and the modified version. Then the administration tool calls
 * the IDirectoryAdminService.setView(IDeltaView) method to apply the modification. When
 * the view modification corresponds to a directory modification (such as the addition of a new element), the 
 * related IDirectoryadminService method allows to pass the new version of the view as input, so that the
 * directory and view modifications can be performed as one atomic operation. The view is implemented as a configuration
 * element, hence allowing the same functionality provided by configuration elements, such as the propagation of incremental
 * updates. Link and folder names cannot contain the '/', '\' and ':' characters.
 *
 * @see com.sonicsw.mf.common.IDirectoryAdminService#getView
 * @see com.sonicsw.mf.common.IDirectoryAdminService#setView
 * @see com.sonicsw.mf.common.IDirectoryAdminService#deleteElement
 * @see com.sonicsw.mf.common.IDirectoryadminService#setElement
 * @see com.sonicsw.mf.common.IDirectoryAdminService#importViewFromXML
 * @see com.sonicsw.mf.common.view.ViewFactory
 */
public interface IView 
{
  public static String VIEW_ELEMENT = com.sonicsw.mf.common.view.impl.ViewConstants.VIEW_ELEMENT;
 
 /**
  * Returns the identity of the element underlying the view implementation.
  *
  * @return the identity of the element underlying the view implementation
  */
  public IElementIdentity getIdentity();

/**
  * Returns true if root folder exists.
  *
  * @return true if root folder exists
  *
  */
 public boolean hasRootFolder();



 /**
  * Returns the root folder.
  *
  * @return the root folder
  *
  */
  public IFolder getRootFolder();

 /**
  * Returns the attribute set type for folders. When a new view is initiated, the attribute set type is defined
  * and cannot be modified again after folders are created.
  *
  * @return the attribute set type for folders
  *
  */
  public IAttributeSetType getFolderAttributeSetType();

 /**
  * Returns the attribute set type for links. When a new view is initiated, the attribute set type is defined
  * and cannot be modified again after links are created.
  *
  * @return the attribute set type for links
  *
  */
  public IAttributeSetType getLinkAttributeSetType();

  /**
  * Renames a link or a folder. 
  *
  * @param oldName the old full name of the view element
  * @param newName the new full name of the view element
  *
  * @exception ViewException if the oldName link does not exist or there is already a view element with newName
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if one of the names is illegal
  *
  */
  public void rename(String oldName, String newName) throws ViewException, ReadOnlyException, ConfigException;


 /**
  * Locates links by the name of the linked object.
  *
  * @param linkedObjectName the name of the linked object
  *
  * @return the list of links object fullnames , that are linked to the specified linked object (the list can be empty)
  *
  */
  public String[] lookupLink(String linkedObjectName);

 /**
  * Finishes a sequence of updates to the view and return the delta to the original view
  *
  * @return the delta view object
  *
  * @exception ReadOnlyException if it is a read-only IView object
  *
  */
  public IDeltaView doneUpdate() throws ReadOnlyException;

  /**
  *
  * Returns true if view was modified and doneUpdate() was not called yet.
  *
  * @return true if this element was modified.
  */
  public boolean isDirty();
  
  /**
  *
  * Returns View Element(folder or link) specified by name.
  * @param name the full DS name of the view element, such as /brokers/broker1
  * @return IViewElement 
  * @exception ViewException if a view element does not exist.
  * @exception ConfigException if the name is illegal.
  */
  public IViewElement getElement(String name) throws ViewException, ConfigException;

 /**
  * Called on a just updated read-only view, to create an updatepble next version view. the token is returned
  * from most of the DS calls that modify the view:
  *
  * Example:
  * 
  * elements[0] = element0.doneUpdate();
  * elements[1] = element1.doneUpdate();
  * INextVersionToken token = ds.setElements(elements, null, view.doneUpdate())
  * element0 = element0.setNextVersion(token);
  * element1 = element1.setNextVersion(token);
  * view = view.setNextVersion(token);
  *
  * @param the next version token returned from the Directory Service
  *
  * @return an updatable next version element
  *
  */
  public IView getNextVersion(INextVersionToken token);

 /**
  * Creates a new link.
  *
  * @param linkName the full path of the link
  * @param linkedObjectName the name the object the link is created to represent
  * @param complex true if complex configuration
  *
  * @return the new link
  *
  * @exception ViewException if there is already a view element with this name
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if linkName is an illegal name
  *
  */
  public void link(String path, String linkedObjectName, boolean complex) throws ViewException, ReadOnlyException, ConfigException;

 /**
  * Delete a link or a folder
  *
  * @param path the path of the link or folder
  *
  * @return the new link
  *
  * @exception ViewException if there is already a view element with this name
  * @exception ConfigException if linkName is an illegal name
  *
  */
  public void delete(String path) throws ConfigException, ViewException;

 /**
  * Creates a new folder.
  *
  * @param folderPath the path of the new folder
  *
  * @return the new folder
  *
  * @exception ViewException if there is already a view element with this name
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if the folderName is illegal
  *
  */
  public void createFolder(String folderPath) throws ViewException, ReadOnlyException, ConfigException;

 /**
  * List all the names under a folder
  *
  * @param folderPath the path of the folder
  *
  * @return the the name list
  *
  * @exception ViewException if a the folder does not exist.
  * @exception ConfigException if the name is illegal.
  *
  */
  public String[] list(String folderPath) throws ViewException, ConfigException;



}
