package com.sonicsw.mf.common.view.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.view.IFolder;
import com.sonicsw.mf.common.view.ILink;
import com.sonicsw.mf.common.view.IViewElement;
import com.sonicsw.mf.common.view.IViewElementAttributes;
import com.sonicsw.mf.common.view.ViewException;

final class Folder implements IFolder
{
  IAttributeSet m_attrSet = null; //IAttributeSet mapped to this Folder
  IAttributeSetType m_folderAttrType = null;
  IAttributeSetType m_linkAttrType = null;
  ViewElementAttributes m_attributes = null;
  private String m_name = null;

  Folder( IAttributeSet parent, IAttributeSetType folderAttrs, IAttributeSetType linkAttrs, String name)
   throws AttributeSetTypeException, ReadOnlyException, ConfigException
  {
    String folderLowerCase = name;
    if (name.compareTo(ViewConstants.ROOT_FOLDER) != 0)
    {
        folderLowerCase = name.toLowerCase();
    }
    m_attrSet = (IAttributeSet)parent.getAttribute(folderLowerCase);
    if ((m_attrSet == null) && (System.getProperty("MQ_UPGRADE", "false").equals("true")))
    {
        m_attrSet = (IAttributeSet)parent.getAttribute(name);
    }
    if(m_attrSet == null)
    {
        m_attrSet = parent.createAttributeSet(folderLowerCase);
        m_attrSet.createAttributeList(ViewConstants.ATTRS_VALUES);
        m_attrSet.setIntegerAttribute(ViewConstants.TYPE_ATTR, ViewConstants.FOLDER_TYPE);
        m_attrSet.setStringAttribute(ViewConstants.ORIGINAL_NAME, name);
    }
    m_name = name;
    m_folderAttrType = folderAttrs;
    m_linkAttrType = linkAttrs;
    setAttributes();
  }


  ////**Implementation of IFolder
  /////////////////////////////////////

    /**
  * Creates a new folder.
  *
  * @param folderName the name of the new folder
  *
  * @return the new folder
  *
  * @exception ViewException if there is already a view element with this name
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if the folderName is illegal
  *
  */
  @Override
public IFolder createFolder(String folderName) throws ViewException, ReadOnlyException, ConfigException
  {
    validateElementName(folderName);
    if(checkExistance(folderName))
     {
        throw new ViewException(folderName + " already exists."); //can't create folder, folder already exist
    }
    Folder subFolder = new Folder(m_attrSet, m_folderAttrType, m_linkAttrType, folderName);
    return subFolder;
  }

 /**
  * Deletes a folder and the view elements it contains.
  *
  * @param folderName the name of the folder
  *
  * @exception ViewException if the folder does not exist
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if the name is illegal
  *
  */
  @Override
public void deleteFolder(String folderName) throws ViewException, ReadOnlyException, ConfigException
  {
    if(!checkExistance(folderName))
     {
        throw new ViewException("Can't create folder.Folder: " + folderName + " does not exist");//can't delete folder, folder does not exist
    }
    deleteViewElement(folderName);
  }

 /**
  * Creates a new link.
  *
  * @param linkName the name of the link
  * @param linkedObjectName the name the object the link is created to represent
  *
  * @return the new link
  *
  * @exception ViewException if there is already a view element with this name
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if linkName is an illegal name
  *
  */
  @Override
public ILink link(String linkName, String linkedObjectName, boolean complex) throws ViewException, ReadOnlyException, ConfigException
  {
     validateElementName(linkName);
     View.validateFullName(linkedObjectName);
     if(checkExistance(linkName))
     {
        throw new ViewException(linkName + " already exists."); //can't create link, link already exist
    }
     ILink link = new Link(m_attrSet, m_linkAttrType, linkName, linkedObjectName, complex);
     //ViewAttributesList.setReadOnly(true, this.m_linkAttrType);
     return link;
  }

  @Override
public ILink link(String linkName, String linkedObjectName) throws ViewException, ReadOnlyException, ConfigException
  {
      return link(linkName, linkedObjectName, false);
  }


 /**
  * Deletes a link.
  *
  * @param linkName the link to be deleted
  *
  * @exception ViewException if the link does not exist
  * @exception ReadOnlyException if it is a read-only IView object
  * @exception ConfigException if linkName is an illegal name
  *
  */
  @Override
public void unlink(String linkName) throws ViewException, ReadOnlyException, ConfigException
  {
    if(!checkExistance(linkName))
     {
        throw new ViewException("Can't unlink link: " + linkName + ". Link doesn't exist.");//can't delete link, link does not exist
    }
    deleteViewElement(linkName);
  }


 /**
  * Returns a view element (a link or a folder)
  *
  * @param viewElementName the name of the view element
  *
  * @return the view element
  *
  * @exception ViewException if the view element does not exist
  * @exception ConfigException if the name is illegal
  *
  */
  @Override
public IViewElement getViewElement(String viewElementName) throws ViewException, ConfigException
  {
    IAttributeSet element = (IAttributeSet)m_attrSet.getAttribute(viewElementName.toLowerCase());
    if( (element == null) && (System.getProperty("MQ_UPGRADE", "false").equals("true")))
    {
        element = (IAttributeSet)m_attrSet.getAttribute(viewElementName);
    }
    if (element == null)
     {
        throw new ViewException("Element or folder " + viewElementName + " doesn't exist"); //element does not exist
    }
    Integer type = (Integer)element.getAttribute(ViewConstants.TYPE_ATTR);
    String name = (String)element.getAttribute(ViewConstants.ORIGINAL_NAME);
    if (name == null && (System.getProperty("MQ_UPGRADE", "false").equals("true")))
    {
        name = viewElementName;
    }
    if (name == null) {
    	throw new ViewException("Element name for " + viewElementName + " is null"); //element name is null    	
    }
    if(type.intValue() == ViewConstants.FOLDER_TYPE.intValue())
    {
        return new Folder(m_attrSet, m_folderAttrType, m_linkAttrType, name);
    }
    return new Link(m_attrSet, m_linkAttrType, name, (String)element.getAttribute(ViewConstants.ATTR_LINKOBJECT), false);
  }

 /**
  * Lists all the view elements names under this folder.
  *
  * @return a list of view elements
  *
  */
  @Override
public String[] list()
  {
    ArrayList list = new ArrayList();
    HashMap map = m_attrSet.getAttributes();
    Iterator iterator = map.keySet().iterator();

    while(iterator.hasNext())
    {
        String name = (String)iterator.next();
        Object value = map.get(name);
        if(value instanceof IAttributeSet)
        {
            String originalName = (String)((IAttributeSet)value).getAttribute(ViewConstants.ORIGINAL_NAME);
            if (originalName != null)
            {
                list.add(originalName);
            }
            else
                if (System.getProperty("MQ_UPGRADE", "false").equals("true"))
                {
                    list.add(name);
                }
        }
    }

    if(list.isEmpty())
    {
        return IEmptyArray.EMPTY_STRING_ARRAY;
    }

    return (String [])list.toArray(new String[list.size()]);
  }


  ///**Implementation of IViewElement
  /////////////////////////////////////

 /**
  * Returns this element's attributes
  *
  * @return this element's attributes
  *
  */
  @Override
public IViewElementAttributes getAttributes()
  {
    return m_attributes;
  }

  @Override
public String getName()
  {
    return m_name;
  }

  //*** Helper methods ***//

  //Set Attributes for this Folder
  private void setAttributes()
  {
    m_attributes = new ViewElementAttributes();
    m_attributes.setAttributeNamesList(m_folderAttrType.getAttributeNames());
    m_attributes.setAttributeValuesList((IAttributeList)m_attrSet.getAttribute(ViewConstants.ATTRS_VALUES));
  }

  private boolean checkExistance(String name)
  {
      return m_attrSet.getAttributes().get(name.toLowerCase()) != null;
  }

  void deleteViewElement(String name)throws ReadOnlyException, ConfigException
  {
    validateElementName(name);
    m_attrSet.deleteAttribute(name.toLowerCase()); //deletes child AttributeSet from the mapped AttributeSet
  }

  private void validateElementName(String name) throws ConfigException
  {
    if (EntityName.containsAnyChar(name, ViewConstants.RESERVED_CHARACTERS))
    {
        throw new ConfigException("Invalid name '" + name + "'. " +
                                  "Folder and link names cannot contain the characters: " + ViewConstants.RESERVED_CHARACTERS);
    }

    if (name.startsWith(ViewConstants.MF_PREFIX))
    {
        throw new ConfigException("Folder and link names cannot start with the '_MF' prefix.");
    }
  }

}//end of class
