/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.view.impl;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryCacheService;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.INamingNotification;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.impl.ElementCreateNotification;
import com.sonicsw.mf.common.config.impl.ElementDeleteNotification;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.config.impl.FolderCreateNotification;
import com.sonicsw.mf.common.config.impl.FolderDeleteNotification;
import com.sonicsw.mf.common.config.impl.MetaAttributesChangeNotification;
import com.sonicsw.mf.common.config.impl.RenameNotification;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementInPathException;
import com.sonicsw.mf.common.view.IFolder;
import com.sonicsw.mf.common.view.ILink;
import com.sonicsw.mf.common.view.ILogicalNameSpace;
import com.sonicsw.mf.common.view.INamingListener;
import com.sonicsw.mf.common.view.IView;
import com.sonicsw.mf.common.view.IViewElement;
import com.sonicsw.mf.common.view.ViewException;

public final class LogicalNameSpace implements ILogicalNameSpace
{
    public static final String HINT_DIRECTORY_ATT = "HINT_DIRECTORY";
    public static final String HINT_COMPLEX_ATT = "HINT_COMPLEX";
    private static final String INCONSISTENT_ERROR_MESSAGE = "The view was updated by an old (admin. API) client and is now inconsistent.";

    private IView m_view;
    IDirectoryAdminService m_ds;
    private StorageToLogical m_storageToLogical;
    private static final String HINT_SEPARATOR = "_$$_";
    private static int m_counter = 0;
    private static String MISC_DIR_NAME = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_MISC_DIR + IMFDirectories.MF_DIR_SEPARATOR;
    private IAttributeSet m_hints;
    private NamingNotificationManager m_notificationManager;
    private String m_hintsElementName;
    private String m_hintsAttribute;
    private HashMap m_dirMap;
    private HashMap m_complexMap;
    private HashMap m_tempBlobStorageNames;
    private boolean m_viewIsDirty;
    private boolean m_inconsistent;
    private List m_auditRecords;

    public void storeUpdates() throws DirectoryServiceException
    {

        if (!m_viewIsDirty)
        {
            return;
        }

        // m_ds is allowed to be null only in test environment
        if (m_ds == null)
        {
            return;
        }

        m_view = ((IDirectoryCacheService)m_ds).storeViewInternal(m_view);
        m_viewIsDirty = false;
        m_auditRecords.clear();
    }

    //FOR TESTING ONLY
    public void init(IView view) throws ViewException
    {
        m_inconsistent = false;
        m_view = view;
        m_viewIsDirty = false;
        m_dirMap = null;
        m_notificationManager = new NamingNotificationManager();
        m_hints = null;
        m_storageToLogical = new StorageToLogical();
        m_tempBlobStorageNames = new HashMap();
        if (m_view.hasRootFolder())
        {
            m_storageToLogical.updateFromView(m_view.getRootFolder(), "/");
        }
        m_auditRecords = new LinkedList();
    }

    public void init(IDirectoryAdminService ds, String hintsElementName, String hintsAttribute)
    {
        m_hintsElementName = hintsElementName;
        m_hintsAttribute = hintsAttribute;
        m_dirMap = null;
        m_ds = ds;
        m_notificationManager = new NamingNotificationManager();
        m_auditRecords = new LinkedList();
        reset();
    }

    public void setNotConsistent()
    {
        // That happens is testing mode only - we don't care about this deployment consistency issue
        if (m_ds == null)
        {
            return;
        }
        m_inconsistent = true;
    }

    public HashMap getStorageToLogicalMap()
    {
        return m_storageToLogical.getMap();
    }

    public void reset()
    {
        try
        {
            m_view = m_ds.getView();
            m_viewIsDirty = false;
            m_storageToLogical = new StorageToLogical();
            m_tempBlobStorageNames = new HashMap();
            if (m_view.hasRootFolder())
            {
                m_storageToLogical.updateFromView(m_view.getRootFolder(), "/");
            }
            m_notificationManager.reset();
            resetStorageHints();
            m_inconsistent = false;
            m_auditRecords.clear();
        }
        catch (RuntimeException e)
        {
        	throw e;
        }
        catch (Exception e)
        {
        	throw new RuntimeException("Exception during reset, see cause", e);
        }
    }

    public String complexDirForType(String type)
    {
        if (m_complexMap == null)
        {
            createTypeDirMap();
        }
        return (String)m_complexMap.get(type);
    }

    public ArrayList directoriesForType(String type)
    {
        if (m_dirMap == null)
        {
            createTypeDirMap();
        }
        return (ArrayList)m_dirMap.get(type);
    }

    // Create a map from element types to their storage directories. Each element type
    // could be stored in several directories according the name postfix.
    void createTypeDirMap()
    {
        m_dirMap = new HashMap();
        m_complexMap = new HashMap();
        if( m_hints == null)
        {
            return;
        }

        Iterator hintsIterator = m_hints.getAttributes().keySet().iterator();
        while (hintsIterator.hasNext())
        {
            String hintKey = (String)hintsIterator.next();
            HintID hintType = new HintID(hintKey);
            ArrayList dirList = (ArrayList)m_dirMap.get(hintType.m_type);

            if (dirList == null)
            {
                dirList = new ArrayList();
                m_dirMap.put(hintType.m_type, dirList);
            }
            else if (dirList.isEmpty()) //An empty list signifies a type of a complex configuration
            {
                continue; //Keep it empty
            }

            IAttributeSet hint = (IAttributeSet)m_hints.getAttribute(hintKey);

            Boolean complex = (Boolean)hint.getAttribute(HINT_COMPLEX_ATT);
            if (complex != null && complex.booleanValue())
            {
                m_complexMap.put(hintType.m_type, hint.getAttribute(HINT_DIRECTORY_ATT));
                continue;
            }

            dirList.add(hint.getAttribute(HINT_DIRECTORY_ATT));
        }

    }

    private void resetStorageHints() throws DirectoryServiceException
    {
        if (m_hintsElementName == null)
        {
            return;
        }
        IElement storageHints = m_ds.getElement(m_hintsElementName, false);
        if (storageHints != null)
        {
            resetStorageHints((IAttributeSet)storageHints.getAttributes().getAttribute(m_hintsAttribute));
        }
    }

    // Must be called every time the hints are modified - from xml or from hint calls
    public void resetStorageHints(IAttributeSet hints)
    {
        m_dirMap = null; // Needs to be re-populated
        m_hints = hints;
    }

    @Override
    public void createFolder(String path) throws DirectoryServiceException
    {
        try
        {
            m_view.createFolder(path);
            m_viewIsDirty = true;
            String canonicalPath = new EntityName(path).getName();
            m_notificationManager.notifyNamingEvent(new FolderCreateNotification(canonicalPath));
            m_auditRecords.add(new AuditRecord(AuditRecord.CREATE_FOLDER, null, canonicalPath, null));
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }
    }
    
    //Returns the storage name of the created element
    // If the hint suggests that a complex configuration is needed and non
    // was created yet then create one.
    // Its the responsibility of the caller to create the parent directory if does not exist yet
    
    @Override
    public String createElement(String path1, String elementType) throws DirectoryServiceException
    {
        return createElement(path1, elementType, true);
    }

    // 10/17/08 createElement expanded to pass a boolean to 
    // addSimpleEntry that determines whether the view element
    // should be modified or not.
    public String createElement(String path1, String elementType, boolean modView) throws DirectoryServiceException
    {
        try
        {
            EntityName pathE = new EntityName(path1);
            if (modView)
            {
                m_notificationManager.notifyNamingEvent(new ElementCreateNotification(pathE.getName()));
            }
            String path = pathE.getName();

            String storageName = null;
            try
            {
                storageName = storageFromLogical(path);
            }
            catch (Exception e){}

            IViewElement viewElement = null;
            try
            {
                viewElement = m_view.getElement(path);
            }
            catch (Exception e){}

            // Its not explicitly in the view but it has a storage name - its under a complex configuration
            if (storageName != null && viewElement == null)
            {
                return storageName;
            }
            else if (viewElement != null)
            {
                throw new ViewException(path + " already exists.");
            }

            String postfix = null;
            String generatedName = createUniqueID();

            int postfixIndex = path.lastIndexOf('.');
            if (postfixIndex != -1)
            {
                postfix = path.substring(postfixIndex+1).toLowerCase();
            }

            IAttributeSet hint = null;
            if (m_hints != null)
            {
                hint = (IAttributeSet)m_hints.getAttribute(createHintID(elementType, postfix));
            }
            if (hint == null && m_hints != null)
            {
                hint = (IAttributeSet)m_hints.getAttribute(elementType);
            }

            if (hint == null)
            {
                return addSimpleEntry(path, MISC_DIR_NAME + generatedName, modView);
            }
            else
            {
                boolean complexHint = false;
                Boolean complex = (Boolean)hint.getAttribute(HINT_COMPLEX_ATT);
                if (complex != null && complex.booleanValue())
                {
                    complexHint = true;
                }

                if (!complexHint)
                {
                    return addSimpleEntry(path, (String)hint.getAttribute(HINT_DIRECTORY_ATT) +
                                          IMFDirectories.MF_DIR_SEPARATOR + generatedName, modView);
                }
                else
                {
                    String complexDir = (String)hint.getAttribute(HINT_DIRECTORY_ATT) +
                                         IMFDirectories.MF_DIR_SEPARATOR +
                                         generatedName;
                    EntityName logicalDirE = pathE.getParentEntity();

                    // If the base name of the logical name and its parent are the same
                    // then we store under name of the form hinst-dir/122222-1//122222-1
                    // for example the element /na/policies/policy1/policy1 will be stored
                    // under /policies/122222-1/122222-1
                    // That is in order be be backwards compatible with version 5
                    String storageBaseName = pathE.getBaseName().equalsIgnoreCase(logicalDirE.getBaseName()) ?
                                             generatedName : pathE.getBaseName();

                    storageName = complexDir +
                                         IMFDirectories.MF_DIR_SEPARATOR +
                                         storageBaseName;

                    String logicalDir = logicalDirE.getName();

                    // It's also an implicit folder creation
                    m_notificationManager.notifyNamingEvent(new FolderCreateNotification(logicalDir));
                    // so far the modView=false is only used for files,
                    // so it wouldn't apply to complex elements, so the
                    // view is modified every time here.
                    m_view.link(logicalDir, storageName, true);
                    m_viewIsDirty = true;
                    m_storageToLogical.setComplex(new EntityName(complexDir), getCaseSensitiveName(logicalDir));
                    return storageName;
                }
            }

        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }

    }
    
    // Called from DirectoryService.appendFSBlob and appendFSBlobInternal
    // when it is the last chunk so that the view element will finally get 
    // the link. Previously addSimpleEntry always added the link, but that resulted in
    // links associated with storage elements that were never saved in the
    // DS because the last chunk of the file was never written - for instance
    // from an aborted ESBAdmin session
    public void addViewLink(String path, String storageName) throws DirectoryServiceException
    {
        try
        {
            m_view.link(path, storageName, false);
            m_viewIsDirty = true;
            m_notificationManager.notifyNamingEvent(new ElementCreateNotification(path));
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }
    }
    
    // added 10/17/08 to add an entry to the logical map, with the ability of
    // not adding it to the view
    private String addSimpleEntry(String path, String storageName, boolean modView) throws ViewException, ConfigException
    {
        if (modView)
        {
            m_view.link(path, storageName, false);
            m_viewIsDirty = true;
            m_tempBlobStorageNames.remove(path);
        }
        m_storageToLogical.setElement(new EntityName(storageName), getCaseSensitiveName(path, modView));
        if (!modView)
        {
            m_tempBlobStorageNames.put(path, storageName);
        }
        return storageName;
    }
    
    public String tempBlobStorageName(String path) throws ConfigException
    {
    	EntityName eName = new EntityName(path);
        return (String)m_tempBlobStorageNames.get(eName.getName());
    }

    public String addSimpleEntry(String path, String storageName) throws ViewException, ConfigException, DirectoryServiceException
    {
        return addSimpleEntry(path, storageName, true);
    }

    @Override
    public String logicalFromStorage(String storagePath)  throws DirectoryServiceException
    {
        if (m_inconsistent)
        {
            throw new DirectoryServiceException(INCONSISTENT_ERROR_MESSAGE);
        }
        try
        {
            return m_storageToLogical.getElementLogicalName(new EntityName(storagePath));
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }
    }

    @Override
    public String storageFromLogical(String path) throws DirectoryServiceException
    {
        if (m_inconsistent)
        {
            throw new DirectoryServiceException(INCONSISTENT_ERROR_MESSAGE);
        }
        try
        {
            return logicalToStorage(m_view, path);
        }
        catch (DirectoryServiceException dirEx)
        {
            throw dirEx;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }
    }

    @Override
    public void deleteElement(String path) throws DirectoryServiceException
    {
        deleteElement(path, false);
    }

    @Override
    public void deleteElement(String path, boolean elementUnderComplexConfig) throws DirectoryServiceException
    {
        try
        {
            m_notificationManager.notifyNamingEvent(new ElementDeleteNotification(new EntityName(path).getName()));
            if (!elementUnderComplexConfig)
            {
                delete(path, false);
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }
    }

    @Override
    public void deleteFolder(String path) throws DirectoryServiceException
    {
        try
        {
            delete(path, true);
            String canonicalPath = new EntityName(path).getName();
            m_notificationManager.notifyNamingEvent(new FolderDeleteNotification(canonicalPath));
            m_auditRecords.add(new AuditRecord(AuditRecord.DELETE_FOLDER, canonicalPath, null, null));
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }
    }

    @Override
    public String[] list(String folderPath) throws DirectoryServiceException
    {
        try
        {
            return m_view.list(folderPath);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString(), e);
        }
    }

    @Override
    public int getNameSpaceType(String path)
    {
        IViewElement element = null;
        try
        {
            element = m_view.getElement(path);
        }
        catch (Exception e)
        {
            return DOES_NOT_EXIST_TYPE;
        }
        if (element == null)
        {
            return DOES_NOT_EXIST_TYPE;
        }

        if (element instanceof IFolder)
        {
            return FOLDER_TYPE;
        }
        else
        {
            if (((ILink)element).isComplex())
            {
                return COMPLEX_FOLDER_TYPE;
            }
            else
            {
                return ELEMENT_TYPE;
            }
        }
    }

    private void delete(String path, boolean folder) throws ConfigException, ViewException, DirectoryServiceException
    {
        IViewElement element = m_view.getElement(path);
        if (element instanceof IFolder)
        {
            m_storageToLogical.deleteFromView((IFolder)element);
        }
        else
        {
            String storageName = ((ILink)element).getLinkedObjectName();
            if (((ILink)element).isComplex())
            {
                m_storageToLogical.delete(new EntityName(storageName).getParentEntity());
            }
            else
            {
                if (folder)
                {
                    throw new ViewException(path + " is not a folder.");
                }
                m_storageToLogical.delete(new EntityName(storageName));
            }
        }

        m_view.delete(path);
        m_viewIsDirty = true;
    }

    @Override
    public void rename(String oldPath, String newPath) throws DirectoryServiceException
    {
        try
        {
            String canonicalOldPath = new EntityName(oldPath).getName();
            String canonicalNewPath = new EntityName(newPath).getName();
            m_notificationManager.notifyNamingEvent(new RenameNotification(canonicalOldPath, canonicalNewPath));
            m_view.rename(oldPath, newPath);
            m_viewIsDirty = true;
            IViewElement renamedElement = m_view.getElement(newPath);
            if (renamedElement instanceof IFolder)
            {
                m_storageToLogical.updateFromView((IFolder)renamedElement, newPath);
                m_auditRecords.add(new AuditRecord(AuditRecord.RENAME, canonicalOldPath, canonicalNewPath, null));
            }
            else // renamedElement is an ILink
            {
                ILink link = (ILink)renamedElement;
                String storageName = link.getLinkedObjectName();
                if (!link.isComplex())
                {
                    m_storageToLogical.setElement (new EntityName(storageName), newPath);
                }
                else
                {
                    m_storageToLogical.setComplex (new EntityName(storageName).getParentEntity(), newPath);
                }
                m_auditRecords.add(new AuditRecord(AuditRecord.RENAME, canonicalOldPath, canonicalNewPath, storageName));
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString(), e);
        }

    }

    @Override
    public String folderToDirectory (String logical) throws DirectoryServiceException
    {
        try
        {
            return logicalFolderToStorage(logical);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString(), e);
        }

    }

    // If the folder corresponds to a directory (in a complex configuration) returns the directory name.
    // null is returned if is it's regular folder.
    private String logicalFolderToStorage(String logical) throws ViewException, ConfigException
    {
        IFolder currentFolder = m_view.getRootFolder();
        EntityName logicalE = new EntityName(logical);
        if (logicalE.isRoot())
        {
            return null;
        }
        String[] nameComponents = logicalE.getNameComponents();

        for (int i = 0; i < nameComponents.length; i++)
        {
            IViewElement viewElement =  currentFolder.getViewElement(nameComponents[i]);

            if (viewElement == null)
            {
                throw new ViewException(logical + " was not found");
            }
            else if (i+1 == nameComponents.length)
            {
                if (viewElement instanceof IFolder)
                {
                    return null;
                }
                ILink link = (ILink)viewElement;
                if (link.isComplex())
                {
                    return new EntityName(link.getLinkedObjectName()).getParent();
                }
                else
                {
                    throw new ViewException (logical + " is not a folder.");
                }
            }
            else
            {
                if (viewElement instanceof IFolder)
                {
                    currentFolder = (IFolder)viewElement;
                }
                else
                {
                    ILink link = (ILink)viewElement;
                    if (!link.isComplex())
                    {
                        throw new ViewException(logical + " is not a folder.");
                    }
                    else
                    {
                        String subPath = "";
                        for (int j = i+1; j < nameComponents.length; j++)
                        {
                            subPath += IMFDirectories.MF_DIR_SEPARATOR + nameComponents[j];
                        }
                        return new EntityName(link.getLinkedObjectName()).getParent() + subPath;
                    }
                }
            }
        }
        throw new Error("logical path " + logical); // Should never get here
    }

    public static String logicalToStorage(IView view, String logical) throws ViewException, ConfigException, ElementInPathException
    {
        IFolder currentFolder = view.getRootFolder();
        EntityName logicalE = new EntityName(logical);
        String[] nameComponents = logicalE.getNameComponents();
        String searchedPath = "";
        if (nameComponents.length == 0)
        {
            throw new ViewException("Bad argument: storageFromLogical() or logicalToStorage() was called with the '/' logical name.");
        } 
        for (int i = 0; i < nameComponents.length; i++)
        {
            IViewElement viewElement =  currentFolder.getViewElement(nameComponents[i]);
            searchedPath = searchedPath + "/" + nameComponents[i];
            if (viewElement == null)
            {
                throw new ViewException(logical + " was not found");
            }
            else if (i+1 == nameComponents.length)
            {
                if (viewElement instanceof IFolder)
                {
                    throw new ViewException(logical + " is a folder name");
                }
                ILink link = (ILink)viewElement;
                if (link.isComplex())
                {
                    return new EntityName(link.getLinkedObjectName()).getParent();
                }
                else
                {
                    return link.getLinkedObjectName();
                }
            }
            else
            {
                if (viewElement instanceof IFolder)
                {
                    currentFolder = (IFolder)viewElement;
                }
                else
                {
                    ILink link = (ILink)viewElement;
                    // remember this link. This might be a blob, and the logical name
                    // could be a file under this directory
                    if (!link.isComplex())
                    {
                        throw new ElementInPathException(logical + " is not a valid element name.", searchedPath, link.getLinkedObjectName());
                    }
                    else
                    {
                        String subPath = "";
                        EntityName linkedObjectName = new EntityName(link.getLinkedObjectName());
                        for (int j = i+1; j < nameComponents.length; j++)
                        {
                            String compName = nameComponents[j];

                            // the case where /policy1/policy1 is translated to /12333/-1/12333-1
                            if (j+1 == nameComponents.length && j > 0 && compName.equalsIgnoreCase(nameComponents[j-1]))
                            {
                                compName = linkedObjectName.getBaseName();
                            }

                            subPath += IMFDirectories.MF_DIR_SEPARATOR + compName;
                        }
                        return linkedObjectName.getParent() + subPath;
                    }
                }
            }
        }
        throw new Error("logical path " + logical); // Should never get here
    }
    
    // The original getCaseSensitiveName assumed the view had already been modified
    public String getCaseSensitiveName(String logical) throws ViewException, ConfigException
    {
        return getCaseSensitiveName(logical, true);
    }

    // alreadyAdded will be false when blob element names are getting added to the logical 
    // namespace during the first file chunk. The names are not put in the view until the last
    // chunk is written to storage; however, this method still gets called to get the
    // case sensitive name to add it to the logical map. In that case, find the case sensitive
    // name of the folder, and use the last component of the logical name as is, as it has been
    // provided by the caller and it will be the case sensitive name of the element when it is
    // added.
    public String getCaseSensitiveName(String logical, boolean alreadyAdded) throws ViewException, ConfigException
    {
        IFolder currentFolder = m_view.getRootFolder();
        EntityName logicalE = new EntityName(logical);
        String[] nameComponents = logicalE.getNameComponents();
        String csLogicalName = "";

        for (int i = 0; i < nameComponents.length; i++)
        {
            IViewElement viewElement =  null;
            try
            {
                viewElement = currentFolder.getViewElement(nameComponents[i]);
            }
            catch (ViewException e)
            {
                // couldn't be found. If the logical name has not been added to the view,
                // but we get here and we know it hasn't been added to the view (alreadyAdded = false),
                // we know the sensitivity of the last name component is what we want, as it is
                // the name entered by the caller for a new element. Use that as is as the
                // last component of the name.
                if (!alreadyAdded && i+1 == nameComponents.length)
                {
                    return csLogicalName + IMFDirectories.MF_DIR_SEPARATOR +  nameComponents[i];
                }
                throw e;
            }

            if (viewElement == null)
            {
                throw new ViewException(logical + " was not found");
            }
            else if (i+1 == nameComponents.length)
            {
                return csLogicalName + IMFDirectories.MF_DIR_SEPARATOR +  viewElement.getName();
            }
            else
            {
                if (viewElement instanceof IFolder)
                {
                    currentFolder = (IFolder) viewElement;
                    csLogicalName = csLogicalName + IMFDirectories.MF_DIR_SEPARATOR + viewElement.getName();
                }
                else
                {
                    ILink link = (ILink)viewElement;
                    if (!link.isComplex())
                    {
                        throw new ViewException(logical + " is not a valid element name.");
                    }
                    else
                    {
                        csLogicalName = csLogicalName + IMFDirectories.MF_DIR_SEPARATOR +  viewElement.getName();
                        for (int j=i+1; j < nameComponents.length; j++)
                        {
                            csLogicalName = csLogicalName + IMFDirectories.MF_DIR_SEPARATOR + nameComponents[j];
                        }
                        return csLogicalName;
                    }
                }
            }
        }
        if ((logical.length() == 1) &&  (logical.charAt(0) == IMFDirectories.MF_DIR_SEPARATOR))
         {
            return logical; //it was the root folder
        }
        throw new Error("logical path " + logical); // Should never get here
    }

    public static String createHintID(String type, String postfix)
    {
        String hintID = type;
        if (postfix != null)
        {
            hintID += HINT_SEPARATOR + postfix.toLowerCase();
        }
        return hintID;
    }

    String createUniqueID()
    {
        if (m_counter + 1 == Integer.MAX_VALUE)
        {
            m_counter = 0;
        }
        return System.currentTimeMillis() + "_" + m_counter++;
    }

    public NameReplacer createNameReplacer(boolean fromLogical)
    {
        return new NameReplacer(fromLogical, this);
    }

    private class HintID
    {
        String m_type;
        String m_postfix;

        HintID (String idString)
        {
            m_postfix = null;
            int sepIndex = idString.indexOf(HINT_SEPARATOR);
            if (sepIndex == -1)
            {
                m_type = idString;
            }
            else
            {
                m_type = idString.substring(0, sepIndex);
                m_postfix = idString.substring(sepIndex+HINT_SEPARATOR.length());
            }
        }

    }

    public void doNotify()
    {
        m_notificationManager.doNotify();
    }

    @Override
    public void subscribe(INamingListener listener)
    {
        m_notificationManager.subscribe(listener);
    }

    @Override
    public void unsubscribe()
    {
        m_notificationManager.unsubscribe();
    }

    private final class NamingNotificationManager
    {

        INamingListener m_listener;
        ArrayList m_notifications;

        NamingNotificationManager()
        {
            m_listener = null;
            m_notifications = new ArrayList();
        }

        void reset()
        {
            m_notifications = new ArrayList();
        }

        void doNotify()
        {
            if (m_listener != null && !m_notifications.isEmpty())
            {
                m_listener.onNotifications(m_notifications);
            }

            reset();
        }

        void notifyNamingEvent(INamingNotification notification)
        {
            m_notifications.add(notification);
        }

        void notifyNamingEvents(ArrayList notifications)
        {
            for (int i = 0; i < notifications.size(); i++)
            {
                m_notifications.add(notifications.get(i));
            }
        }

        void subscribe(INamingListener listener)
        {
            m_listener = listener;
        }

        void unsubscribe()
        {
            m_listener = null;
        }

    }

    @Override
    public void defineFolderMetaAttribute(String attributeName) throws DirectoryServiceException
    {
        try
        {
            m_view.getFolderAttributeSetType().addAttributeName(attributeName);
            m_viewIsDirty = true;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }
    }

    @Override
    public void defineElementMetaAttribute(String attributeName) throws DirectoryServiceException
    {
        try
        {
            m_view.getLinkAttributeSetType().addAttributeName(attributeName);
            m_viewIsDirty = true;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }

    }

    @Override
    public void setMetaAttributes(String elementName, HashMap newAttributes) throws DirectoryServiceException
    {
        try
        {
            // We have to cleanup first because we don't store the qualifiers, but we send thme in the MetaAttributesChangeNotification
            // below
            HashMap newAttributesCopy = (HashMap)newAttributes.clone();
            newAttributesCopy.remove(ELEMENT_IDENTITY);
            newAttributesCopy.remove(FOLDER_NAME);
            newAttributesCopy.remove(IS_COMPLEX);

            IViewElement element = m_view.getElement(elementName);
            if (element instanceof ILink && ((ILink)element).isComplex() )
            {
                ((ILink)element).setComplexAttributes(newAttributesCopy);
            }
            else
            {
                Iterator iterator = newAttributesCopy.keySet().iterator();
                while (iterator.hasNext())
                {
                  String key = (String)iterator.next();
                  Object value = newAttributesCopy.get(key);
                  if (!(value instanceof java.lang.String))
                {
                    throw new DirectoryServiceException("Only string attribute are supported " + key +
                                                          " type is " + value.getClass().getName());
                }
                  element.getAttributes().setAttribute(key, (String)value);
                }
            }
            m_notificationManager.notifyNamingEvent(new MetaAttributesChangeNotification(elementName, newAttributes));
            m_viewIsDirty = true;
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString(), e); // Should never happen
        }
        catch (Exception other)
        {
            throw new DirectoryServiceException(other.getMessage());
        }
    }

    @Override
    public String[] getDefinedFolderMetaAttributes()
    {
        return m_view.getFolderAttributeSetType().getAttributeNames();

    }

    @Override
    public String[] getDefinedElementMetaAttributes()
    {
        return m_view.getLinkAttributeSetType().getAttributeNames();
    }

    @Override
    public HashMap getMetaAttributes(String elementName) throws DirectoryServiceException
    {
        try
        {
            IViewElement element = m_view.getElement(elementName);
            if (element instanceof ILink && ((ILink)element).isComplex() )
            {
                return ((ILink)element).getComplexAttributes();
            }
            else
            {
                return element.getAttributes().getAttributes();
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage(), e);
        }
    }
    
    public List getAuditRecords()
    {
        return m_auditRecords;
    }
    
    public class AuditRecord
    {
        public static final int CREATE_FOLDER = 1;
        public static final int DELETE_FOLDER = 2;
        public static final int RENAME = 3;
        
        private int m_action = 0;
        private String m_oldPath = null;
        private String m_newPath = null;
        private String m_storagePath = null;
        
        private AuditRecord(int action, String oldPath, String newPath, String storagePath)
        {
            m_action = action;
            m_oldPath = oldPath;
            m_newPath = newPath;
            m_storagePath = storagePath;
        }
        
        public int getAction()
        {
            return m_action;
        }
        
        public String getOldPath()
        {
            return m_oldPath;
        }
        
        public String getNewPath()
        {
            return m_newPath;
        }
        
        public String getStoragePath()
        {
            return m_storagePath;
        }
    }
}
