package com.sonicsw.mf.common.view.impl;

import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.view.IFolder;
import com.sonicsw.mf.common.view.ILink;
import com.sonicsw.mf.common.view.IViewElement;
import com.sonicsw.mf.common.view.ViewException;

// Translates storage names to logical (view) names
public final class StorageToLogical 
{
    // For stand-alone testing
    public static void main(String[] args) throws Exception
    {
        StorageToLogical st = new StorageToLogical();
        st.setElement(new EntityName("/a1/b2/E1"), "xE1");
        st.setElement(new EntityName("/a1/b2/E2"), "xE2");
        st.setElement(new EntityName("/a1/b2/E3"), "xE3");
        st.setElement(new EntityName("/a1/b2/c3/E3"), "xcE3");

        HashMap map = st.getMap();
        Iterator iter = map.keySet().iterator();
        while(iter.hasNext())
        {
            String name = (String)iter.next();
            System.out.println("MAP PAIR " + name + " " + map.get(name));
        }


        System.out.println(st.getElementLogicalName(new EntityName("/a1/b2/E1")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/b2/E2")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/b2/E3")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/b2/c3/E3")));

        st.setElement(new EntityName("/a1/b2/E1"), "yE1");
        st.setElement(new EntityName("/a1/b2/E2"), "yE2");
        st.setElement(new EntityName("/a1/b2/E3"), "yE3");
        st.setElement(new EntityName("/a1/b2/c3/E3"), "ycE3");

        System.out.println(st.getElementLogicalName(new EntityName("/a1/b2/E1")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/b2/E2")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/b2/E3")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/b2/c3/E3")));

        st.setComplex(new EntityName("/a1/domains/0"), "/l/m/n");
        System.out.println(st.getElementLogicalName(new EntityName("/a1/domains/0/descriptor")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/domains/0/_MFUsers/udi1")));

        st.setComplex(new EntityName("/a1/domains/0"), "/r/s");
        System.out.println(st.getElementLogicalName(new EntityName("/a1/domains/0/descriptor")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/domains/0/_MFUsers/udi1")));
 
        st.delete(new EntityName("/a1/domains/0"));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/domains/0/descriptor")));
        System.out.println(st.getElementLogicalName(new EntityName("/a1/domains/0/_MFUsers/udi1")));
    }

    private HashMap m_root;

    public StorageToLogical() 
    {
        m_root = new HashMap();
    }

    // Return the storage-to-logical list as a simple string-to-string map (complex links are excluded)
    HashMap getMap()
    {
        HashMap map = new HashMap();
        getMap(m_root, "", map); 
        return map;
    }

    private void getMap(HashMap dir, String dirName, HashMap result)
    {
        Iterator iter = dir.keySet().iterator();
        while(iter.hasNext())
        {
            String name = (String)iter.next();
            Object value = dir.get(name);
            if (value instanceof String)
            {
                result.put(dirName + "/" + name, value);
            }
            else  if (value instanceof HashMap)
            {
                getMap((HashMap)value, dirName + "/" + name, result);
            }
        }
    }

    public void updateFromView(IFolder folder, String folderName1) throws ViewException
    {
        try
        {
            String folderName = folderName1;
            if (folderName1.equals("/"))
            {
                folderName = "";
            }
    
            String[] kids = folder.list();
            for (int i = 0; i < kids.length; i++)
            {
                String kidsName = folderName + "/" + kids[i];
                IViewElement viewElement =  folder.getViewElement(kids[i]);
                if (viewElement instanceof IFolder)
                {
                    updateFromView((IFolder)viewElement, kidsName);
                }
                else // it's a link
                {
                    String linkedObject = ((ILink)viewElement).getLinkedObjectName();
                    if (((ILink)viewElement).isComplex())
                    {
                        setComplex((new EntityName(linkedObject)).getParentEntity(), kidsName);
                    }
                    else
                    {
                        setElement(new EntityName(linkedObject), kidsName);
                    }
                }
            }
        }
        catch (Exception e) // Should never happen
        {
            e.printStackTrace();
            throw new ViewException(e.toString(), e); 
        }
    }

    // Delete all storage name map entries appeared under a view folder
    public void deleteFromView(IFolder folder) throws ViewException
    {
        try
        {
            String[] kids = folder.list();
            for (int i = 0; i < kids.length; i++)
            {
                IViewElement viewElement =  folder.getViewElement(kids[i]);
                if (viewElement instanceof IFolder)
                {
                    deleteFromView((IFolder)viewElement);
                }
                else // it's a link
                {
                    ILink link = (ILink)viewElement;
                    String linkedObject = link.getLinkedObjectName();
                    if (link.isComplex())
                    {
                        delete(new EntityName(linkedObject).getParentEntity());
                    }
                    else
                    {
                        delete(new EntityName(linkedObject));
                    }
                }
            }
        }
        catch (Exception e) // Should never happen
        {
            e.printStackTrace();
            throw new ViewException(e.toString(), e);
        }
    }


    public void setElement (EntityName elementName, String logicalName) throws ViewException
    {
        HashMap parentDir = getParentDir(elementName, true, true);
 
        // parentDir is null when this element is under complex configuration
        if (parentDir != null)
        {
            parentDir.put(elementName.getBaseName(), logicalName);
        }
    }

    public void setComplex(EntityName dirName, String logicalName) throws ViewException
    {
        HashMap parentDir = getParentDir(dirName, false, true);
        parentDir.put(dirName.getBaseName(), new ComplextConfig(logicalName));
    }

    public String getElementLogicalName(EntityName storageName) throws ViewException
    {
        String[] components = storageName.getNameComponents();
        HashMap currentDir = m_root;
        for (int i = 0; i < components.length; i++)
        {
            Object nextLevel =  currentDir.get(components[i]);
            if (nextLevel == null)
            {
                return null;
            }
            else if (nextLevel instanceof ComplextConfig)
            {
                // Concatinate the logical part of the path with the storage part under the complex subtree
                String logicalName = ((ComplextConfig)nextLevel).m_logicalName;
                String subDir = "";
                if (i + 1 == components.length)
                {
                    throw new ViewException(storageName.getName() + " is a complex directory, not an element.");
                }
                for (int j = i + 1; j < components.length; j++)
                {
                    String compName = components[j];
                    if (j + 1 == components.length && compName.equals(components[i]))
                    {
                        // Last component is the same as parent
                        try
                        {
                            compName = new EntityName(logicalName).getBaseName();
                        }
                        catch(ConfigException e)
                        {
                            e.printStackTrace();
                            throw new Error(e.toString(), e); // Should never happen
                        }
                    }
                    subDir += "/" + compName;
                }
                return logicalName + subDir;
            }
            else if (nextLevel instanceof String)
            {
                if (i + 1 != components.length)
                {
                    throw new ViewException("Not parent directory for " + storageName.getName());
                }
                return (String)nextLevel;
            }
            else if (nextLevel instanceof HashMap)
            {
                if (i + 1 == components.length)
                {
                    throw new ViewException(storageName.getName() + " is a directory, not an element.");
                }
                currentDir = (HashMap)nextLevel;
            }
        }
        throw new Error("Should never get here");
    }

    public void delete(EntityName storageName) throws ViewException
    {
        HashMap parentDir = getParentDir(storageName, true, false);
 
        // parentDir is null when this element is under complex configuration or the parent dir does not exist
        if (parentDir != null)
        {
            parentDir.remove(storageName.getBaseName());
            if (parentDir.isEmpty()) // Remove the parent as well if empty
            {
                EntityName parentEntity = storageName.getParentEntity();
                if (!parentEntity.isRoot())
                {
                    delete(parentEntity);
                }
            }
        }
    }

    // Can return null if under a complex configuration (so we don't keep a parent dir) or
    // the parent dir doesn't exist (and ensureExistence is false);
    private HashMap getParentDir(EntityName storageName, boolean forElement, boolean ensureExistence) throws ViewException
    {
        String[] parentComponents = storageName.getParentComponents();

        HashMap currentDir = m_root;
        for (int i = 0; i < parentComponents.length; i++)
        {
            Object nextLevel = currentDir.get(parentComponents[i]);

            if (nextLevel != null && nextLevel instanceof ComplextConfig)
            {
                if (!forElement)
                {
                    throw new ViewException("Do not support nested complex configuration.");
                }
                return null;
            }

            HashMap nextDir = (HashMap)nextLevel;

            if (nextDir == null)
            {
                if (ensureExistence)
                {
                    nextDir = new HashMap();
                    currentDir.put(parentComponents[i], nextDir);
                }
                else
                {
                    return null;
                }
            }
            currentDir = nextDir;
        }
        return currentDir;
    }

    private class ComplextConfig
    {
        String m_logicalName;

        ComplextConfig(String logicalName)
        { 
            m_logicalName = logicalName;
        }

    }

}
