package com.sonicsw.mf.common.view.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.view.IDeltaView;
import com.sonicsw.mf.common.view.IFolder;
import com.sonicsw.mf.common.view.ILink;
import com.sonicsw.mf.common.view.IView;
import com.sonicsw.mf.common.view.IViewElement;
import com.sonicsw.mf.common.view.ViewException;

public final class View implements IView, java.io.Serializable
{
    private static final long serialVersionUID = 0L;
    private IDirElement m_viewDirElemnt = null;
    private transient IAttributeSet m_topAttributeSet = null;
    private transient Folder m_rootFolder = null;

    public View(IDirElement viewDirElmnt)
    {
        if(viewDirElmnt == null)
        {
            return;
        }
        m_viewDirElemnt = viewDirElmnt;
        m_topAttributeSet = m_viewDirElemnt.getAttributes();
    }

   public IDirElement getConfigElement()
   {
       return m_viewDirElemnt;
   }

   @Override
public IView getNextVersion(INextVersionToken token)
   {
       return new View( ((Element)m_viewDirElemnt).getNextVersion(token));
   }


    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
        s.writeInt(ViewConstants.VERSION);
        s.writeObject(m_viewDirElemnt);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException, ViewException
    {
        int version = s.readInt();
        if (version != ViewConstants.VERSION)
         {
            throw new ViewException("Serialization version mismatch. Serialized object version: " + version +
                                        " class version: " + ViewConstants.VERSION); //Version mismatch error
        }
        m_viewDirElemnt = (IDirElement)s.readObject();
        m_topAttributeSet = m_viewDirElemnt.getAttributes();
        m_rootFolder = null;
    }


    private void createRootFolder() throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        m_rootFolder = new Folder(m_topAttributeSet, getFolderAttributeSetType(), getLinkAttributeSetType(), ViewConstants.ROOT_FOLDER);
        setReadOnly();
    }

    @Override
    public IElementIdentity getIdentity()
    {
        return m_viewDirElemnt.getIdentity();
    }

    /**
    * Returns the root folder.
    *
    * @return the root folder
    *
    */
    @Override
    public IFolder getRootFolder()
    {
        if(m_rootFolder == null)
        {
            try {
                createRootFolder();
            }
            catch(Exception ex)
            {
                System.out.println(ex.getMessage());
            }
        }
        return m_rootFolder;
    }

    @Override
    public boolean hasRootFolder()
    {
        return m_topAttributeSet.getAttribute(ViewConstants.ROOT_FOLDER) != null;
    }

    /**
    * Returns the attribute set type for folders. When a new view is initiated, the attribute set type is defined
    * and cannot be modified again after folders are created.
    *
    * @return the attribute set type for folders
    *
    */
    @Override
    public IAttributeSetType getFolderAttributeSetType()
    {
        return new ViewAttributesList(m_topAttributeSet, ViewConstants.FOLDER_ATTRIBUTESSETTYPE);
    }

    /**
    * Returns the attribute set type for links. When a new view is initiated, the attribute set type is defined
    * and cannot be modified again after links are created.
    *
    * @return the attribute set type for links
    *
    */
    @Override
    public IAttributeSetType getLinkAttributeSetType()
    {
        return new ViewAttributesList(m_topAttributeSet, ViewConstants.LINK_ATTRIBUTESSETTYPE);
    }


    /**
    * Locates links by the name of the linked object.
    *
    * @param linkedObjectName the full name of the linked object
    *
    * @return the list of links to that linked object (the list can be empty)
    *
    */
    @Override
    public String[] lookupLink(String linkedObjectName)
    {
        ArrayList list = new ArrayList();
        IAttributeSet set = (IAttributeSet)m_topAttributeSet.getAttribute(ViewConstants.ROOT_FOLDER);
        traverseForLinks(set, list, "" + IMFDirectories.MF_DIR_SEPARATOR, linkedObjectName);
        if(list.isEmpty())
         {
            return IEmptyArray.EMPTY_STRING_ARRAY; //empty list
        }
        return (String[])list.toArray(new String [list.size()]);
    }

    protected void traverseForLinks(IAttributeSet set, ArrayList list, String fullName, String linkedObjName)
    {
        HashMap map = set.getAttributes();
        Iterator i = map.keySet().iterator();
        while(i.hasNext())
        {
            String name = (String)i.next();
            Object obj = map.get(name);
            if(obj instanceof IAttributeSet)
            {
                Integer type = (Integer)((IAttributeSet)obj).getAttribute(ViewConstants.TYPE_ATTR);
                if(type.intValue() == ViewConstants.LINK_TYPE.intValue())
                {
                    String linkedObject = (String)((IAttributeSet)obj).getAttribute(ViewConstants.ATTR_LINKOBJECT);
                    if(linkedObject.compareTo(linkedObjName) == 0)
                    {
                        list.add(fullName + (String)((IAttributeSet)obj).getAttribute(ViewConstants.ORIGINAL_NAME));
                    }
                }
                else if(type.intValue() == ViewConstants.FOLDER_TYPE.intValue())
                {
                    traverseForLinks((IAttributeSet)obj, list, fullName +
                        (String)((IAttributeSet)obj).getAttribute(ViewConstants.ORIGINAL_NAME) + IMFDirectories.MF_DIR_SEPARATOR, linkedObjName);
                }
            }
        }
    }

    @Override
    public IViewElement getElement(String name)
        throws ViewException, ConfigException
    {
        EntityName entity = new EntityName(name);

        if (entity.isRoot())
        {
            return getRootFolder();
        }

        IAttributeSet parentSet = lookupForAttributeSet(entity.getParentComponents());
        IAttributeSet elementSet = (IAttributeSet)parentSet.getAttribute(entity.getBaseName().toLowerCase());
        if ((elementSet == null) && (((String)System.getProperty("MQ_UPGRADE", "false")).equals("true")))
        {
            elementSet = (IAttributeSet)parentSet.getAttribute(entity.getBaseName());
        }
        if(elementSet == null)
        {
            throw new ViewException("Element or folder " + name + " doesn't exist.");
        }

        Integer type = (Integer)elementSet.getAttribute(ViewConstants.TYPE_ATTR);

        if(type.intValue() == ViewConstants.FOLDER_TYPE.intValue())
        {
            return new Folder(parentSet, getFolderAttributeSetType(), getLinkAttributeSetType(), entity.getBaseName());
        }
        return new Link(parentSet, getLinkAttributeSetType(), entity.getBaseName(), (String)elementSet.getAttribute(ViewConstants.ATTR_LINKOBJECT), false);
    }

    @Override
    public void rename(String oldName, String newName)
        throws ViewException, ReadOnlyException, ConfigException
    {
        boolean useOriginalName = false;
        EntityName oldNameEntity = new EntityName(oldName);
        EntityName newNameEntity = new EntityName(newName);
        IAttributeSet setOld = lookupForAttributeSet(oldNameEntity.getParentComponents());
        IAttributeSet oldComponent = (IAttributeSet)setOld.getAttribute(oldNameEntity.getBaseName().toLowerCase());
        if ((oldComponent == null) && (System.getProperty("MQ_UPGRADE", "false").equals("true")))
        {
            oldComponent = (IAttributeSet) setOld.getAttribute(oldNameEntity.getBaseName());
            useOriginalName = true;
        }
        IAttributeSet setNew = lookupForAttributeSet(newNameEntity.getParentComponents());
        String newBase = newNameEntity.getBaseName().toLowerCase();
        String oldBase = oldNameEntity.getBaseName().toLowerCase();
        IAttributeSet newComponent = (IAttributeSet)setNew.createAttributeSet(newBase);//DS API should throw an error if exist
        buildViewElementSet(oldComponent, newComponent);
        newComponent.setStringAttribute(ViewConstants.ORIGINAL_NAME, newNameEntity.getBaseName());

        // We just renamed - since we use only lower case, a deletion of the old name will delete the new name - no need to delete
        if (!useOriginalName && setNew == setOld && newBase.equals(oldBase))
        {
            return;
        }

        if (useOriginalName)
        {
            setOld.deleteAttribute(oldNameEntity.getBaseName());
        }
        else {
            setOld.deleteAttribute(oldBase);//removes AttributeSet mapped to the ViewElement with name 'oldName'
        }
    }


    private IAttributeSet lookupForAttributeSet(String [] nameParentComponents)
        throws ViewException
    {
        IAttributeSet set = (IAttributeSet)m_topAttributeSet.getAttribute(ViewConstants.ROOT_FOLDER);
        IAttributeSet previousSet = null;
        for(int i = 0; i < nameParentComponents.length; i++)
        {
            if(set == null)
             {
                throw new ViewException("Folder " + nameParentComponents[i-1] +  " doesn't exist.");//level doesn't exist
            }
            previousSet = set;
            set = (IAttributeSet)set.getAttribute(nameParentComponents[i].toLowerCase());
            if ((set == null) && (((String)System.getProperty("MQ_UPGRADE", "false")).equals("true")))
            {
                set = (IAttributeSet)previousSet.getAttribute(nameParentComponents[i]);
            }
        }
        return set;
    }

    private void buildViewElementSet(IAttributeSet oldSet, IAttributeSet newSet) throws ReadOnlyException, ConfigException, AttributeSetTypeException
    {
        HashMap map = oldSet.getAttributes();
        Iterator i = map.keySet().iterator();
        while(i.hasNext())
        {
            String name = (String)i.next();
            Object obj = map.get(name);
            if(obj instanceof IAttributeSet)
            {
                IAttributeSet set = newSet.createAttributeSet(name); //mapped to Folder or Link
                buildViewElementSet((IAttributeSet)obj, set);
            }
            else{
                if(obj instanceof IAttributeList)//set Attribute's values list
                {
                    IAttributeList list = newSet.createAttributeList(name);
                    addListItems(((IAttributeList)obj).getItems(), list);
                }
                else{
                    if(obj instanceof Integer)//set type
                    {
                        newSet.setIntegerAttribute(name, (Integer)obj);
                    }
                    else if(obj instanceof String)//set Attribute mapped to the linked object name
                    {
                        newSet.setStringAttribute(name, (String)obj);
                    }
                    else if(obj instanceof Boolean)
                    {
                        newSet.setBooleanAttribute(name, (Boolean)obj);
                    }
                }
            }

        }
    }


    private void addListItems(ArrayList items, IAttributeList newList)throws ReadOnlyException
    {
        for (int i = 0; i < items.size(); i++)
        {
            newList.addStringItem((String)items.get(i));
        }
    }

    //on first call getRootFolder() sets Boolean item in list, to set AttributesListType to read-only
    private void setReadOnly() throws ReadOnlyException
    {
        IAttributeList folderAttrs = (IAttributeList)m_topAttributeSet.getAttribute(ViewConstants.FOLDER_ATTRIBUTESSETTYPE);
        if(folderAttrs.getCount() > 0)
        {
            if(folderAttrs.getItem(folderAttrs.getCount()-1) instanceof Boolean)
            {
                return;
            }
        }
        //adds BooleanItem once to the end of each list, indicates that list is read-only
        folderAttrs.addBooleanItem(new Boolean(true));
        IAttributeList linkAttrs = (IAttributeList)m_topAttributeSet.getAttribute(ViewConstants.LINK_ATTRIBUTESSETTYPE);
        linkAttrs.addBooleanItem(new Boolean(true));
    }

    /**
    * Finishes a sequence of updates to the view and return the delta to the original view
    *
    * @return the delta view object
    *
    * @exception ReadOnlyException if it is a read-only IView object
    *
    */
    @Override
    public IDeltaView doneUpdate() throws ReadOnlyException
    {
        IBasicElement delta = m_viewDirElemnt.doneUpdate();
        if(delta  == null)
        {
            return null;
        }
        return new DeltaView(delta);
    }

    @Override
    public boolean isDirty()
    {
        return m_viewDirElemnt.isDirty();
    }

    public static void validateFullName(String name) throws ConfigException
    {
        new EntityName(name);
    }



    public static boolean validateView(IDirElement viewElement)
    {
        if(viewElement.getIdentity().getType().compareTo(ViewConstants.VIEW_TYPE) != 0)
        {
            return false;
        }
        IAttributeSet top = viewElement.getAttributes();
        if(!isRootFolder(top))
        {
            return false;
        }
        return validate(top.getAttributes(), top, true);
    }

    private static boolean isRootFolder(IAttributeSet top)
    {
        if(top.getAttribute(ViewConstants.ROOT_FOLDER) == null)
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    private static boolean validate(Object collection, Object parent, boolean result)
    {
        if( result == false )
        {
            return false;
        }

        Object item = null;

        if(collection instanceof ArrayList)
        {
            for(int i= 0; i < ((ArrayList)collection).size(); i++) //order is important
            {
            item = ((ArrayList)collection).get(i);
                if( item instanceof Long ||
                    item instanceof Integer ||
                    item instanceof BigDecimal ||
                    item instanceof byte[] ||
                    item instanceof Reference)
                {
                result = false; //only String, Boolean types are allowed for IAtrributeList in View
                break;
                }
            }
        }
        else if(collection instanceof HashMap)
        {
            Set set = ((HashMap)collection).keySet();
            Iterator i = set.iterator();
            while (i.hasNext())
            {
                String name = (String)i.next();
                item = ((HashMap)collection).get(name);
                if(item instanceof IAttributeList)
                {
                    if(name.compareTo(ViewConstants.FOLDER_ATTRIBUTESSETTYPE)== 0 ||
                        name.compareTo(ViewConstants.LINK_ATTRIBUTESSETTYPE) ==0 ||
                        name.compareTo(ViewConstants.ATTRS_VALUES) == 0){
                            result = validate(((IAttributeList)item).getItems(),item, result);
                    }
                    else{
                        result =false;
                        break;
                    }

                }
                else
                {
                    if(item instanceof IAttributeSet)
                    {
                        result = validate(((IAttributeSet)item).getAttributes(), item, result);
                    }
                    else
                    {
                        if(item instanceof Integer && name.compareTo(ViewConstants.TYPE_ATTR) != 0){
                            result = false;
                            break;
                        }
                        if( item instanceof Long ||
                            item instanceof Boolean ||
                            item instanceof BigDecimal ||
                            item instanceof byte[] ||
                            item instanceof Reference){
                            result = false;//only String, Integer types allowed for IAttributeSet in View
                            break;
                        }
                    }
                }
            }
        }
        return result;
    }

    @Override
    public void link(String name, String linkedObjectName, boolean complex) throws ConfigException, ViewException
    {
        EntityName nameE = new EntityName(name);
        if (nameE.isRoot())
        {
            throw new ViewException("Cannot create link to the root");
        }

        IFolder parentFolder = getParentFolder(nameE);

        // name is under a complex configuration path
        if (parentFolder == null)
        {
            if (complex)
            {
                throw new ViewException("Nested complexs configurations are not supported.");
            }
            else {
                return; // An explicit link is not needed
            }
        }

        parentFolder.link(nameE.getBaseName(), linkedObjectName, complex);
    }

    @Override
    public void delete(String name) throws ConfigException, ViewException
    {
        EntityName nameE = new EntityName(name);
        if (nameE.isRoot())
        {
            throw new ViewException("Cannot delete the root");
        }

        IFolder parentFolder = getParentFolder(nameE);

        // name is under a complex configuration path - there is no explicit link to delete
        if (parentFolder == null)
        {
            return;
        }

        IViewElement viewElement = parentFolder.getViewElement(nameE.getBaseName());
        if (viewElement == null)
        {
            throw new ViewException(nameE.getName() + " does not exist.");
        }

        ((Folder)parentFolder).deleteViewElement(nameE.getBaseName());

    }

    @Override
    public void createFolder(String path) throws ViewException, ReadOnlyException, ConfigException
    {
        EntityName pathE = new EntityName(path);
        if (pathE.isRoot())
        {
            throw new ViewException("Cannot delete the root");
        }

        IFolder parentFolder = getParentFolder(pathE);

        // name is under a complex configuration path - no need to create an explicit folder
        if (parentFolder == null)
        {
            return;
        }

        parentFolder.createFolder(pathE.getBaseName());
    }

    private IFolder getParentFolder(EntityName nameE) throws ConfigException, ViewException
    {
        IFolder parentFolder = getRootFolder();

        String[] nameComponents = nameE.getNameComponents();

        for (int i = 0; i < nameComponents.length-1; i++)
        {
            IViewElement viewElement =  parentFolder.getViewElement(nameComponents[i]);
            if (viewElement == null)
            {
                throw new ViewException(nameE.getParent() + " does not exist.");
            }

            // Under a complex link - no parent folder
            if (viewElement instanceof ILink && ((ILink)viewElement).isComplex())
            {
                return null;
            }

            if (!(viewElement instanceof IFolder))
            {
                throw new ViewException(nameE.getParent() + " is not a folder.");
            }

            parentFolder = (IFolder)viewElement;
        }

        return parentFolder;
    }

    @Override
    public String[] list(String folderPath) throws ViewException, ConfigException
    {
        IFolder folder = null;
        if (new EntityName(folderPath).isRoot())
        {
            folder = getRootFolder();
        }
        else
        {
            folder = (IFolder)getElement(folderPath);
        }
        return folder.list();
    }


}//end of class
