/* CopyrightVersion 1.0
 *
 * Change Log:
 *    Last Modified By: $Author: Irene $
 *    Last Modified On: $Date: 05/19/01 3:59p $
 */

package com.sonicsw.mf.common.xml;

import java.util.HashMap;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryCacheService;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;


/**
 * ElementBuilder class traverse DS tree, map DS objects to XML Elements.
 * Returns XML formatted string, that defines XML config file for the specified component.
 *
 */

public class DirectoryBuilder
{
    XMLStringWriter m_xmlWriter = null;
    String m_dirName = null;
    IDirectoryAdminService m_dirService = null;

    /** Constructs a <code>DirectoryBuilder</code> object.
    * <p>
    * @param  dirName <code>String</code> value specifies MF component name, represented as 'Directory'
    *         in XML config file.
    *
    */
    public DirectoryBuilder(String dirName)
    {
        this.m_dirName = dirName;
        m_xmlWriter = new XMLStringWriter();
    }

    /** Set reference to the Directory Service
    * <p>
    * @param  dirService <code>IDirectoryAdminService</code> value specifies directory service
    *
    */
    public void setDirectoryService(IDirectoryAdminService dirService)
    {
        this.m_dirService = dirService;
    }


    /**
    * Returns XML presentation of the specified DS Directory branch processed by DirectoryBuilder.
    * @return The string in XML format that represents DS Directory branch processed by DirectoryBuilder.
    *
    */
    public String getXMLString()
    {
        return m_xmlWriter.getXMLString().toString();
    }

    /**
    * Initializes the DirectoryBuilder, must be called after method setDirectoryService() called.
    */
    public void init() throws DirectoryServiceException
    {
        if(m_dirService == null)
        {
            return;
        }

        m_xmlWriter.setProcInstruction(XMLConstants.DEFAULT_XML_VERSION, XMLConstants.DEFAULT_XML_ENCODE);
        createDomainElement();

    }

    private void traverseBranch(IIdentity[] list) throws DirectoryServiceException
    {
        HashMap alreadyExportedTable = new HashMap();
        for(int i = 0; i < list.length; i++)
        {

            String entityName = list[i].getName();
            if (alreadyExportedTable.get(entityName) != null)
            {
                continue;
            }

            if(list[i] instanceof IElementIdentity)
            {
                // Retrieve efficiently the element and any other element that can be retrieved without extra I/O
                IDirElement[] elements = ((IDirectoryCacheService)m_dirService).getElements(entityName, false);

                for (int j = 0; j < elements.length; j++)
                {
                    String elementName = elements[j].getIdentity().getName();

                    // We don't want to export this element since re-importing will fail - MF_REFRESH_ELEMENT cannot be imported.
                    if (elementName.endsWith(XMLConstants.MF_REFRESH_ELEMENT))
                    {
                        continue;
                    }

                    alreadyExportedTable.put(elementName, Boolean.TRUE);
                    ElementBuilder builder = new ElementBuilder(elements[j], m_xmlWriter);
                    builder.setDirectoryService(m_dirService);
                    builder.init();
                }
            }
            else if (list[i] instanceof IDirIdentity)
            {
                if(!isSystemFolder(((IIdentity)list[i]).getName()))//laminate '/MF_System' directory from export
                {
                    m_xmlWriter.setTabCounter();
                    createDirectoryElement(((IIdentity)list[i]).getName());
                    IIdentity [] subList = m_dirService.listAll(((IIdentity)list[i]).getName());
                    if(subList.length > 0)
                    {
                        traverseBranch(subList);
                    }
                    m_xmlWriter.endElement(XMLConstants.DIRECTORY_ELEMENTNAME);
                    m_xmlWriter.resetTabCounter();
                }
            }
        }
    }


    private void createDirectoryElement(String dirName)
    {
      HashMap attrPair = new HashMap();
      attrPair.put(XMLConstants.NAME_ATTR, dirName);
      m_xmlWriter.startElement(XMLConstants.DIRECTORY_ELEMENTNAME, attrPair, false);
    }

    private void createDomainElement() throws DirectoryServiceException
    {
        HashMap attrPair = new HashMap();
        attrPair.put(XMLConstants.XMLNS_ATTR, XMLConstants.XMLNS_TARGET);
        attrPair.put(XMLConstants.XSI_ATTR, XMLConstants.XSI_TARGET);
        attrPair.put(XMLConstants.SCHEMALOCATION_ATTR, XMLConstants.XMLNS_TARGET + " " + XMLConstants.XSD_FILE);
        attrPair.put(XMLConstants.NAME_ATTR, m_dirService.getDomain());
        m_xmlWriter.startElement(XMLConstants.DOMAIN_ELEMENTNAME, attrPair, false);
        processDirBranch(m_dirName);
        m_xmlWriter.endElement(XMLConstants.DOMAIN_ELEMENTNAME);
    }

    private void processDirBranch(String dirName) throws DirectoryServiceException
    {
        m_xmlWriter.setTabCounter();
        createDirectoryElement(dirName);
        IIdentity [] listBranch = m_dirService.listAll(dirName);
        if(listBranch.length > 0)
        {
            traverseBranch(listBranch);
        }
        m_xmlWriter.endElement(XMLConstants.DIRECTORY_ELEMENTNAME);
        m_xmlWriter.resetTabCounter();
    }

    private boolean isSystemFolder(String name)
    {
        boolean ret = false;
        if(name.compareTo(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SYSTEM_DIR) == 0)
        {
            ret = true;
        }
        return ret;
    }
}//end of class
