/* CopyrightVersion 1.0
 *
 * Change Log:
 *    Last Modified By: $Author: Irene $
 *    Last Modified On: $Date: 05/03/01 3:59p $
 */

package com.sonicsw.mf.common.xml;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;

/**
 * ElementBuilder class traverse DS tree, map DS objects to XML Elements.
 * Returns XML formatted string, that defines XML config file for the specified component.
 *
 */

public class ElementBuilder
{
    IDirElement m_configElement = null;
    XMLStringWriter m_xmlWriter = null;
    IDirectoryAdminService m_dirService = null;
    String m_elmntName = null;
    String m_domainName = null;
    boolean isDSRequired = false;

    /** Constructs a <code>ElementBuilder</code> object.
    * <p>
    * @param  elmntName <code>String</code> value specifies MF component name, represented as 'ConfigElement'
    *         in XML config file.
    * @param  reference to XML string writer  <code>XMLStringWriter</code> set by DirecetoryBuilder, otherwise is null
    */
    public ElementBuilder(String elmntName, XMLStringWriter writer)
    {
        this.m_elmntName = elmntName;
        this.m_xmlWriter = writer;
        isDSRequired = true;
    }

    public ElementBuilder(IDirElement configElement, XMLStringWriter writer)
    {

        m_configElement = configElement;
        this.m_xmlWriter = writer;
        isDSRequired = true;
    }


    public ElementBuilder(IElement element, String domainName)
    {
        this.m_configElement = (IDirElement)element;
        this.m_domainName = domainName;
    }

    // Returns the XML presentation of the element
    public static String exportElement(IElement element, String domainName) throws DirectoryServiceException
    {
        ElementBuilder elementBuilder = new ElementBuilder(element, domainName);
        elementBuilder.init();
        return elementBuilder.getXMLString();
    }

    /** reference to DirectoryService, must be called after ElementBuilder is constructed.
    */
    public void setDirectoryService(IDirectoryAdminService dirService)
    {
        this.m_dirService = dirService;
    }

    private void checkDSElement() throws DirectoryServiceException
    {

        if (m_configElement != null &&
            (m_configElement.getIdentity().getType().equals("MF_DIRECTORY_SERVICE") ||
            m_configElement.getIdentity().getType().equals("MF_BACKUP_DIRECTORY_SERVICE")))
        {
            // check that HOST_DIRECTORY is set
            String hostDir = (String) m_configElement.getAttributes().getAttribute("HOST_DIRECTORY");
            if (hostDir == null || hostDir.length() == 0)
            {
                // set HOST_DIRECTORY in memory so it will get exported
                m_configElement = (IDirElement)m_configElement.createWritableClone();
                try
                {
                    m_configElement.getAttributes().setStringAttribute("HOST_DIRECTORY", ".");
                }
                catch (ConfigException e)
                {
                    throw new DirectoryServiceException(e.toString(), e);
                }
            }
        }
    }

    /**
    * Initializes the ElementBuilder, must be called after method setDirectoryService() called.
    */

    public void init() throws DirectoryServiceException
    {
        if(isDSRequired)
        {
            if(m_dirService == null)
            {
                return;
            }

            if (m_configElement == null)
            {
                m_configElement = m_dirService.getElement(m_elmntName, false);
            }
            m_domainName = m_dirService.getDomain();
        }
        checkDSElement();
        if(m_xmlWriter != null) //called by DirectoryBuilder
        {
            createConfigElement();
        }
        else{
            m_xmlWriter = new XMLStringWriter();
            m_xmlWriter.setProcInstruction(XMLConstants.DEFAULT_XML_VERSION, XMLConstants.DEFAULT_XML_ENCODE);
            createDomainElement();
        }
    }

    /**
    * Returns XML presentation of the specified DS Element processed by ElementBuilder.
    * @return The string in XML format that represents DS Element processed byElementBuilder.
    *
    */
    public String getXMLString()
    {
        return m_xmlWriter.getXMLString().toString();
    }


    /**Creates 'Domain' XML element
    */
    private void createDomainElement()
    {
        HashMap attrPair = new HashMap();
        attrPair.put(XMLConstants.XMLNS_ATTR, XMLConstants.XMLNS_TARGET);
        attrPair.put(XMLConstants.XSI_ATTR, XMLConstants.XSI_TARGET);
        attrPair.put(XMLConstants.SCHEMALOCATION_ATTR, XMLConstants.XMLNS_TARGET + " " + XMLConstants.XSD_FILE);
        attrPair.put(XMLConstants.NAME_ATTR, m_domainName);
        m_xmlWriter.startElement(XMLConstants.DOMAIN_ELEMENTNAME, attrPair, false);
        createConfigElement();
        m_xmlWriter.endElement(XMLConstants.DOMAIN_ELEMENTNAME);
    }

    /**Creates 'ConfigElement' XML element
    */
    private void createConfigElement()
    {
       m_xmlWriter.setTabCounter();
       m_xmlWriter.startElement(XMLConstants.CONFIGELMNT_ELEMENTNAME, null, false);
       createElementID();
       processConfigElemAttribs();
       m_xmlWriter.endElement(XMLConstants.CONFIGELMNT_ELEMENTNAME);
       m_xmlWriter.resetTabCounter();
    }

    /**Creates 'ElementID' XML element
    */
    private void createElementID()
    {
        String type;
        m_xmlWriter.setTabCounter();
        HashMap attrPair = new HashMap();
        IElementIdentity identity = m_configElement.getIdentity();
        attrPair.put(XMLConstants.NAME_ATTR, identity.getName());
        attrPair.put(XMLConstants.VERSION_ATTR , new Long(identity.getVersion()));
        attrPair.put(XMLConstants.RELEASEVERSION_ATTR, identity.getReleaseVersion());
        attrPair.put(XMLConstants.TIMESTAMP_ATTR, new Long(identity.getCreationTimestamp()));
        //check if optional attribute 'type' is set
        if(( type = identity.getType())!= null)
        {
            attrPair.put(XMLConstants.TYPE_ATTR, type);
        }

        m_xmlWriter.startElement(XMLConstants.ELEMENTID_ELEMENTNAME, attrPair, true);
        m_xmlWriter.endElement(null);//empty element
        m_xmlWriter.resetTabCounter();
    }

    /** Creates top 'AttributeSet' XML element
    */
    private void processConfigElemAttribs()
    {
        IAttributeSet elementAttributes = m_configElement.getAttributes();
        m_xmlWriter.setTabCounter();
        addAttributeSetElement(null, null, elementAttributes);
        traverse(elementAttributes.getAttributes(), elementAttributes);
        m_xmlWriter.endElement(XMLConstants.ATTRSET_ELEMENTNAME);
        m_xmlWriter.resetTabCounter();
    }

    /** Traverse DS Element tree
    */
    private void traverse(Object collection, Object parent)
    {
        Object item = null;
        String name = null;
        String type = null;
        m_xmlWriter.setTabCounter();
        if(collection instanceof HashMap)
        {
            Set set = ((HashMap)collection).keySet();
            Iterator i = set.iterator();
            while (i.hasNext())
            {
                name = (String)i.next();
                item = ((HashMap)collection).get(name);
                if(item instanceof IAttributeSet)
                {
                    addAttributeSetElement(name, ((IAttributeSet)item).getTypeName(), item);
                    traverse(((IAttributeSet)item).getAttributes(), item);
                    m_xmlWriter.endElement(XMLConstants.ATTRSET_ELEMENTNAME);
                }
                else
                {
                    if(item instanceof IAttributeList)
                    {
                        addAttributeListElement(name, item);
                        traverse(((IAttributeList)item).getItems(),item);
                        m_xmlWriter.endElement(XMLConstants.ATTRLIST_ELEMENTNAME);
                    }
                    else
                    {
                        processJavaPrimitives(item,name,parent);
                    }
                }

            }
        }
        else if (collection instanceof ArrayList)
        {
            for(int i= 0; i < ((ArrayList)collection).size(); i++) //order is important
            {
                item = ((ArrayList)collection).get(i);
                if(item instanceof IAttributeSet)
                {
                    addAttributeSetElement(name, ((IAttributeSet)item).getTypeName(), item);
                    traverse(((IAttributeSet)item).getAttributes(), item);
                    m_xmlWriter.endElement(XMLConstants.ATTRSET_ELEMENTNAME);
                }
                else
                {
                    if(item instanceof IAttributeList)
                    {
                        addAttributeListElement(name, item);
                        traverse(((IAttributeList)item).getItems(),item);
                        m_xmlWriter.endElement(XMLConstants.ATTRLIST_ELEMENTNAME);
                    }
                    else
                    {
                        processJavaPrimitives(item,name,parent);
                    }
                }

            }

        }
        m_xmlWriter.resetTabCounter();
    }//end of traverse

    /**Convert Java primitives to the String
    */
    private void processJavaPrimitives(Object attr, String name, Object parent)
    {
        HashMap attrPair = new HashMap();
        if(attr instanceof String)
        {
           addJavaObject(name, (String)attr, XMLConstants.STRING);
        }
        else
        {
            if(attr instanceof Integer)
            {
                addJavaObject(name, ((Integer)attr).toString(), XMLConstants.INTEGER);
            }
            else
            {
                if(attr instanceof Long)
                {
                    addJavaObject(name, ((Long)attr).toString(), XMLConstants.LONG);
                }
                else
                {
                    if(attr instanceof Boolean)
                    {
                        addJavaObject(name, ((Boolean)attr).toString(), XMLConstants.BOOLEAN);
                    }
                    else
                    {
                        if(attr instanceof Date)
                        {
                            addJavaObject(name, ((Date)attr).toString(), XMLConstants.DATE);
                        }
                        else
                        {
                            if(attr instanceof BigDecimal)
                            {
                                addJavaObject(name, ((BigDecimal)attr).toString(), XMLConstants.BIGDECIMAL);
                            }
                            else
                            {
                                if(attr instanceof Reference)
                                {
                                    addJavaObject(name, ((Reference)attr).getElementName(), XMLConstants.REFERENCE);
                                }
                                else if(attr instanceof byte[])
                                {
                                    String encodedString = ByteStringConverter.getStringFromBytes((byte[])attr);
                                    addJavaObject(name, encodedString, XMLConstants.BYTES);
                                }

                            }
                        }
                    }
                }
            }
        }
    }//end of processJavaPrimitives

    /**Create 'AttributeSet' XML element
    */
    private void addAttributeSetElement(String name, String Type, Object item)
    {
        HashMap attrPair = null;

        if(Type != null)
        {
            attrPair = new HashMap();
            attrPair.put(XMLConstants.TYPE_ATTR, Type);
        }
        m_xmlWriter.startElement(XMLConstants.ATTRSET_ELEMENTNAME, attrPair, false);
        if(name != null)
        {
            addAttributeNameElement(name);
        }
        String [] types = ((IAttributeSet)item).getAllTypeNames();
        for(int i = 0; i < types.length; i++)
        {
            addAttributeSetType(((IAttributeSet)item).getAttributeSetType(types[i]), types[i]);
        }
    }

    /**Create 'AttributeList' XML element
    */
    private void addAttributeListElement(String name, Object item)
    {
        m_xmlWriter.startElement(XMLConstants.ATTRLIST_ELEMENTNAME, null, false);
        if(name != null)
        {
            addAttributeNameElement(name);
        }
        String [] types = ((IAttributeList)item).getAllTypeNames();
        for(int i = 0; i < types.length; i++)
        {
            addAttributeSetType(((IAttributeList)item).getAttributeSetType(types[i]), types[i]);
        }
    }

    /**Create 'AttributeSetType' XML element
    */
    private void addAttributeSetType(Object setType, String name)
    {
        HashMap attrPair = null;
        if(name != null)
        {
            attrPair = new HashMap();
            attrPair.put(XMLConstants.NAME_ATTR, name);
        }
        m_xmlWriter.setTabCounter();
        m_xmlWriter.startElement(XMLConstants.ATTRSETTYPE_ELEMENTNAME, attrPair, false);
        String [] attrList = ((IAttributeSetType)setType).getAttributeNames();
        for(int i=0; i < attrList.length; i++)
        {
            addAttributeNameElement(attrList[i]);
        }
        m_xmlWriter.endElement(XMLConstants.ATTRSETTYPE_ELEMENTNAME);
        m_xmlWriter.resetTabCounter();
    }

    /**Create 'AttributeName' XML element
    */
    private void addAttributeNameElement(String Name)
    {
        m_xmlWriter.setTabCounter();
        HashMap attrPair = new HashMap();
        attrPair.put(XMLConstants.NAME_ATTR, Name);
        m_xmlWriter.startElement(XMLConstants.ATTRIBUTENAME_ELEMENTNAME, attrPair, true);
        m_xmlWriter.endElement(null);//empty element
        m_xmlWriter.resetTabCounter();
    }

    /**Create 'Attribute' / 'ListItem' XML elements. Depends on value of the param 'name'
    */
    private void addJavaObject(String name, String value, String type)
    {
        String elementName;
        boolean emptyTag = true;
        String markupValue = null;
        HashMap attrPair = new HashMap();
        if(name != null)
        {
           attrPair.put(XMLConstants.NAME_ATTR, name);
           elementName = XMLConstants.ATTRIBUTE_ELEMENTNAME;
        }
        else
        {
           elementName = XMLConstants.LISTITEM_ELEMENTNAME;
        }
        //for CDATA support
        if(type.equalsIgnoreCase(XMLConstants.STRING) && m_xmlWriter.isMarkup(value))
        {
            markupValue = value;
            attrPair.put(XMLConstants.TYPE_ATTR, type);
            emptyTag=false;
        }
        else{
            attrPair.put(XMLConstants.VALUE_ATTR, value);
            attrPair.put(XMLConstants.TYPE_ATTR, type);
        }
        m_xmlWriter.startElement(elementName, attrPair, emptyTag);
        if(!emptyTag)
        {
            m_xmlWriter.createCDATA(markupValue);
            m_xmlWriter.endElement(elementName); //only in case of #PCDATA
        }
        else {
            m_xmlWriter.endElement(null);//empty element
        }
    }


}//end of ElementBuilder
