
/* CopyrightVersion 1.0
 *                                                                          
 * Change Log:
 *    Last Modified By: $Author: Irene $
 *    Last Modified On: $Date: 05/17/01 3:59p $
 */
 
package com.sonicsw.mf.common.xml;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Stack;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ext.LexicalHandler;
import org.xml.sax.helpers.DefaultHandler;

import com.sonicsw.mf.common.IDSImport;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.ITypeCollection;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.InvalidXMLException;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;

/**
 * Imports data from XML configuration file. 
 * SAX API used for validation and document processing. 
 * Provides SAX2 ContentHandler events handling for parsed XML data.
 *
 */
 
public class XMLDocumentHandler extends DefaultHandler implements ContentHandler,LexicalHandler
{
    /** Data members */
    protected Stack elements = null; //contains non-leaf objects
    
    protected boolean topAttributeSet = false;
    protected IDirectoryAdminService m_directoryService = null;
    protected Locator m_locator = null; 
    protected String m_strDomain = null;
    protected IDirElement m_dirElement = null; // used by ElementFactory
    protected ArrayList m_elements = new ArrayList(); // Used by ElementFactory to return multiple elements
    
    //support for CDATA
    protected StringBuffer m_contentCDATA = new StringBuffer();
    protected boolean m_isCData = false;
    protected ArrayList m_stringElements = new ArrayList(XMLConstants.MAX_SIZE);//!can contain only one element!
    
    // usually we only want to keep elements when reading the DS boot file,
    // but, for instance, not when reading a large number of elements, like
    // when loading a dumped DS.
    boolean m_keepElements = false;
    /** Default constructor. */
    public XMLDocumentHandler() {
        //System.out.println("Event handler constructed");
        
    }
    
    
    public void setDirectoryService(IDirectoryAdminService directoryService)
    {
        this.m_directoryService = directoryService;
    }
    
    public void setDomainName(String name)
    {
        this.m_strDomain = name;
    }
    
    public void setKeepElements(boolean keep)
    {
    	m_keepElements = keep;
    }
    
    public IDirElement getElement()
    {
        return m_dirElement;
    }
    
    public IDirElement[] getElements()
    {
    	IDirElement[] elements = new IDirElement[m_elements.size()];
    	m_elements.toArray(elements);
    	return elements;
    }
    
    // SAX2 Events hanler, uses 
    // ContentHandler Interface methods
    //

    @Override
    public void setDocumentLocator(Locator locator) {
        this.m_locator = locator;
    }

    @Override
    public void startDocument() throws SAXException {
        elements = new Stack(); //initialize the Stack
    }
    
    @Override
    public void characters(char ch[], int start, int length)
    {         
        if(m_isCData){
            m_contentCDATA.append(ch,start,length);
        } 
    }
    
    
    @Override
    public void startElement(String uri, String localpart, String rawname, Attributes attributes) throws SAXException 
    {
        try 
        {
            if(localpart.equals(XMLConstants.DOMAIN_ELEMENTNAME))
            {   
                checkDomain(attributes.getValue(XMLConstants.NAME_ATTR)); 
            }
            if(localpart.equals(XMLConstants.DIRECTORY_ELEMENTNAME))
            {
                createDirectory(attributes);
            }
            if(localpart.equals(XMLConstants.CONFIGELMNT_ELEMENTNAME))
            {
                topAttributeSet = true; //element encountered
            }
            else{
               
                if(m_stringElements.size() == XMLConstants.MAX_SIZE)
                {
                    //to handle Attribute and LIstItem elements without CDATA content
                    addStringAttribute(null);
                }
                    
                if(localpart.equals(XMLConstants.ELEMENTID_ELEMENTNAME))
                {
                    createElement(attributes);
                }
                else{
                    if(localpart.equals(XMLConstants.ATTRSET_ELEMENTNAME))
                    {
                        createAttributeSet(attributes); 
                    }
                    else{
                        if(localpart.equals(XMLConstants.ATTRLIST_ELEMENTNAME))
                        {
                            createAttributeList(attributes); 
                        }
                        else{
                            if(localpart.equals(XMLConstants.ATTRSETTYPE_ELEMENTNAME))
                            {
                                createAttributeSetType(attributes);
                            }
                            else //only leaf elements, no needs to store them on stack
                            { 
                                if(localpart.equals(XMLConstants.ATTRIBUTENAME_ELEMENTNAME))
                                {
                                    createAttributeName(attributes);
                                }
                                else
                                {
                                    if(localpart.equals(XMLConstants.ATTRIBUTE_ELEMENTNAME))
                                    {
                                        createAttribute(attributes);
                                    }
                                    else if (localpart.equals(XMLConstants.LISTITEM_ELEMENTNAME))
                                    {
                                        createListItem(attributes);
                                    }
                                }
                            }
                        }
                        
                    }
                }
            } 
        }
        catch(Exception e)
        {
            throw new SAXParseException( e.getMessage(), m_locator, e);  //used SAXParserException to report error location
        }   
    }

    @Override
    public void endElement(String uri, String localpart, String rawname) throws SAXException 
    {
        if(m_stringElements.size() == XMLConstants.MAX_SIZE)
        {
            //to handle last element in  Attribute/ListItem list
            addStringAttribute(null);
        }

        if(localpart.equals(XMLConstants.ATTRSET_ELEMENTNAME))
        {
            
            elements.pop();
        }
        else
        {
            if(localpart.equals(XMLConstants.ATTRLIST_ELEMENTNAME))
            {
                  elements.pop();      
            }
            else
            {
                if(localpart.equals(XMLConstants.ATTRSETTYPE_ELEMENTNAME))
                {
                     elements.pop();       
                }
                else if(localpart.equals(XMLConstants.CONFIGELMNT_ELEMENTNAME)){
                    m_dirElement = (IDirElement)elements.peek(); //used by ElementFactory
                    if (m_keepElements)
                     {
                        m_elements.add(m_dirElement); // used by ElementFactory
                    }
                    if(m_directoryService != null)
                    {
                        addElementToDS(m_dirElement);
                    }
                    elements.pop();
                }
            }
        }
    }
    
    @Override
    public void endDocument() throws SAXException {
        //System.out.println("endDocument()");
    }
     
   //LexicalHandler interface impl
    
    @Override
    public void startCDATA() throws SAXException {
        m_isCData=true;
    }

    @Override
    public void endCDATA() throws SAXException {
        m_isCData = false;
        String tmp = XMLStringWriter.replaceAll(m_contentCDATA.toString(),
                                                XMLConstants.MF_START_CDATA, XMLConstants.START_CDATA);
        String data = XMLStringWriter.replaceAll(tmp, XMLConstants.MF_END_CDATA, XMLConstants.END_CDATA);
        addStringAttribute(data);
        m_contentCDATA.setLength(XMLConstants.ZERO_INDEX); //cleanup buffer    
    }
    
    //current implementation isn't required to hadle the following callbacks in LexicalHandler
    @Override
    public void comment(char[] ch,int start,int length) throws SAXException {}

    @Override
    public void startDTD(java.lang.String name,java.lang.String publicId,java.lang.String systemId) throws SAXException{}
    
    @Override
    public void endDTD()throws SAXException {}
            
    @Override
    public void startEntity(java.lang.String name) throws SAXException {}
    
    @Override
    public void endEntity(java.lang.String name) throws SAXException {}
                  
    
    /////////////////////////////////////////////////////////////////////////////////////
    //Building directory branch
    //Create new directory branch, if branch exist will ignore it.
    //////////////////////////////////////////////
    
    private void addElementToDS(IDirElement dirElement) throws SAXException
    {
        try{
                ((IDSImport)m_directoryService).importedElement(dirElement.doneUpdate());
        }
        catch(Exception e){
            cleanup();
            throw new SAXParseException(e.getMessage(),m_locator, e);
        }
    }
    
    private void createDirectory(Attributes attributes) throws SAXException
    {
        if(m_directoryService == null)
        {
            throw new SAXException("Directory element can't be specified.");
        }
        String dirName = attributes.getValue(XMLConstants.NAME_ATTR);
        //To do: Add support for the optional attributes, when DS API will be ready
        try{
            if(!checkDirectoryExistance(dirName))
            {
                buildParentDirectories(dirName);
                m_directoryService.createDirectory(dirName);    
            }
        }
        catch(Exception ex)
        {
            throw new SAXException(ex.getMessage(),ex);
        }
    }
    
    private void buildParentDirectories(String dirName) throws DirectoryServiceException, VersionOutofSyncException, ConfigException
    {
        EntityName name = new EntityName(dirName); //validate name
        String [] parentsList = name.getNameComponents();
        String fullName = "";
        if(parentsList.length == 0)
        {
            return;
        }
        for(int i = 0; i < parentsList.length -1 ; i++)
        {
            fullName += "/" + parentsList[i];
            if(!checkDirectoryExistance(fullName))
            {
                m_directoryService.createDirectory(fullName);
            }
        }
    }
    
    private boolean checkDirectoryExistance(String name) throws DirectoryServiceException, VersionOutofSyncException
    {
        boolean exist = false;
        IIdentity identity = m_directoryService.getIdentity(name);
        if(identity instanceof IElementIdentity)
        {
            throw new DirectoryServiceException("Can't create directory, domain contain element with same name");
        }
        if(identity != null)
        {
            exist = true;
        } 
        return exist;
    }
    
    /////////////////////////////////////////////////////////////////////////////////////
    //Building Element
    //////////////////////////////////////////////
    
    //Attributes contains ElementID attributes map
    private void createElement(Attributes attributes)
    {
        String name =  attributes.getValue(XMLConstants.NAME_ATTR);
        String type = attributes.getValue(XMLConstants.TYPE_ATTR);
        String releaseVersion = attributes.getValue(XMLConstants.RELEASEVERSION_ATTR);
        IDirElement configElement = ElementFactory.createElement(name, type, releaseVersion); 
        elements.push(configElement);
    }
    
    private void createAttributeSet(Attributes attributes) throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        IAttributeSet attrSet = null;

        if(topAttributeSet)
        {
            attrSet = ((IDirElement)elements.peek()).getAttributes(); 
            topAttributeSet = false;
        }
        else{
            Object parent = elements.peek();
           
            if(parent instanceof IAttributeSet)
            {
                NamedCollection attrs = new NamedCollection(XMLConstants.SET);
                attrs.setAttributes(attributes);
                elements.push(attrs);
            }
            else {
                if(parent instanceof IAttributeList)
                {
                    IAttributeSetType setType = null;
                    String type = attributes.getValue(XMLConstants.TYPE_ATTR);
                    if(type != null)
                    {
                        setType = getAttributeSetType(type);
                    }
                    attrSet = ((IAttributeList)parent).addNewAttributeSetItem(setType); 
                }
                else if(parent instanceof NamedCollection)//user error, AttributeName isn't specified for parent 
                {
                    cleanup();
                    throw new ConfigException(XMLConstants.NAMING_COLLECTION_ERROR);
                }
            }
           
        }
        if(attrSet != null)
        {
            elements.push(attrSet);
        }
    }
    
    private void createAttributeList(Attributes attributes) throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        IAttributeList attrList = null;

        Object parent = elements.peek();    
        if(parent instanceof IAttributeSet)
        {
            NamedCollection attrs = new NamedCollection(XMLConstants.LIST);
            attrs.setAttributes(attributes);
            elements.push(attrs);
        }
        else{
            if(parent instanceof IAttributeList)
            {
                attrList = ((IAttributeList)parent).addNewAttributeListItem();
            }
            else if(parent instanceof NamedCollection)//user error, AttributeName isn't specified for parent 
            {
                 cleanup();
                 throw new ConfigException(XMLConstants.NAMING_COLLECTION_ERROR);   
            }
        }
       
        if(attrList != null)
        {
            elements.push(attrList);
        }
    }
    
   
    private void createAttributeSetType(Attributes attributes) throws ReadOnlyException, ConfigException, AttributeSetTypeException
    {
        if(elements.peek()instanceof NamedCollection){//user error, AttributeName isn't specified for parent
           cleanup();
           throw new ConfigException(XMLConstants.NAMING_COLLECTION_ERROR);
        }
        IAttributeSetType attrSetType = getAttributeSetType(attributes.getValue(XMLConstants.NAME_ATTR));
        if(attrSetType != null)
        {
            elements.push(attrSetType);
        }
    }
    
    private IAttributeSetType getAttributeSetType(String type) throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        ITypeCollection parent = (ITypeCollection)elements.peek();
        IAttributeSetType attrSetType = parent.getAttributeSetType(type);
        if(attrSetType == null)
        {
            attrSetType = parent.createAttributeSetType(type); 
        }
        
        return attrSetType;
    }
    
    private void createAttribute(Attributes attributes) throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        if(elements.peek()instanceof NamedCollection){//user error, AttributeName isn't specified for parent
           cleanup();
           throw new ConfigException(XMLConstants.NAMING_COLLECTION_ERROR);
        }
        String name = attributes.getValue(XMLConstants.NAME_ATTR);
        String value = attributes.getValue(XMLConstants.VALUE_ATTR);
        String type = attributes.getValue(XMLConstants.TYPE_ATTR);
        
        IAttributeSet parent = (IAttributeSet)elements.peek();
        if(type.equalsIgnoreCase(XMLConstants.INTEGER))
        {
           parent.setIntegerAttribute(name, new Integer(value == null ? XMLConstants.DEFAULT_DIGIT : value));
        }
        else{
            if(type.equalsIgnoreCase(XMLConstants.LONG))
            {
                parent.setLongAttribute(name,new Long(value == null ? XMLConstants.DEFAULT_DIGIT : value)); 
            }
            else{
                if(type.equalsIgnoreCase(XMLConstants.BOOLEAN))
                {
                    parent.setBooleanAttribute(name, new Boolean(value == null ? XMLConstants.DEFAULT_BOOLEAN : value)); 
                }
                else{
                    if(type.equalsIgnoreCase(XMLConstants.BIGDECIMAL)) 
                    {
                        parent.setDecimalAttribute(name, new BigDecimal(value == null ? XMLConstants.DEFAULT_DIGIT : value));
                    }
                    else{
                        if(type.equalsIgnoreCase(XMLConstants.STRING)) 
                        {
                            String strValue = (value == null ? XMLConstants.DEFAULT_STRING : value);
                            m_stringElements.add(new StringAttributeInfo(name, strValue, parent));
                        }
                        else{
                            if(type.equalsIgnoreCase(XMLConstants.REFERENCE)) 
                            {
                                parent.setReferenceAttribute(name, new Reference(value == null ? XMLConstants.DEFAULT_STRING : value));
                            }
                            else{
                                if(type.equalsIgnoreCase(XMLConstants.BYTES))
                                {
                                    byte[] bytes = ByteStringConverter.getBytesFromString(value == null ? XMLConstants.DEFAULT_STRING : value); 
                                    parent.setBytesAttribute(name, bytes);
                                }
                                else if(type.equalsIgnoreCase(XMLConstants.DATE))
                                {
                                    parent.setDateAttribute(name, new Date(value == null ? XMLConstants.DEFAULT_DATE : value));
                                }
                            }
                        }
                     
                   }
               }
            }
        }
    }
    
    private void createListItem(Attributes attributes) throws ReadOnlyException, ConfigException
    {
        if(elements.peek()instanceof NamedCollection){//user error, AttributeName isn't specified for parent
           cleanup();
           throw new ConfigException(XMLConstants.NAMING_COLLECTION_ERROR);
        }
        String value = attributes.getValue(XMLConstants.VALUE_ATTR);
        String type = attributes.getValue(XMLConstants.TYPE_ATTR);
        IAttributeList parent = (IAttributeList)elements.peek();
        if(type.equalsIgnoreCase(XMLConstants.INTEGER))
        {
           parent.addIntegerItem(new Integer(value == null ? XMLConstants.DEFAULT_DIGIT : value));
        }
        else{
            if(type.equalsIgnoreCase(XMLConstants.LONG))
            {
                parent.addLongItem(new Long(value == null ? XMLConstants.DEFAULT_DIGIT : value)); 
            }
            else{
                if(type.equalsIgnoreCase(XMLConstants.BOOLEAN))
                {
                    parent.addBooleanItem(new Boolean(value == null ? XMLConstants.DEFAULT_BOOLEAN : value)); 
                }
                else{
                    if(type.equalsIgnoreCase(XMLConstants.BIGDECIMAL)) 
                    {
                        parent.addDecimalItem( new BigDecimal(value == null ? XMLConstants.DEFAULT_DIGIT : value));
                    }
                    else{
                        if(type.equalsIgnoreCase(XMLConstants.STRING)) 
                        {
                            String strValue = (value == null ? XMLConstants.DEFAULT_STRING : value);
                            m_stringElements.add(new StringAttributeInfo(XMLConstants.EMPTY_STRING, strValue, parent));
                        }
                        else{
                            if(type.equalsIgnoreCase(XMLConstants.REFERENCE)) 
                            {
                                parent.addReferenceItem(new Reference(value == null ? XMLConstants.DEFAULT_STRING : value));
                            }
                            else{
                                if(type.equalsIgnoreCase(XMLConstants.BYTES))
                                {
                                    byte[] bytes = ByteStringConverter.getBytesFromString(value == null ? XMLConstants.DEFAULT_STRING : value); 
                                    parent.addBytesItem(bytes);
                                }
                                else if(type.equalsIgnoreCase(XMLConstants.DATE))
                                {
                                    parent.addDateItem(new Date(value == null ? XMLConstants.DEFAULT_DATE : value));
                                }
                            }
                        }
                     
                   }
               }
            }
        }
        
    }
    
    private void createNewListItem(Object parent, Attributes attributes)
    {
        //no support in current DS API
       
    }
    
    private void createDeletedListItem(Object parent, Attributes attributes)
    {
        //no support in current DS API 
    }
    
    private void createAttributeName(Attributes attributes) throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        Object parent = elements.peek();
        if(parent instanceof NamedCollection) //replace it with AttributeSet or AttributeList
        {
          NamedCollection attrs = (NamedCollection)elements.pop(); 
          createNamedCollection(attributes.getValue(XMLConstants.NAME_ATTR), attrs);
        }
        else if (parent instanceof IAttributeSetType)
        {
           ((IAttributeSetType)parent).addAttributeName(attributes.getValue(XMLConstants.NAME_ATTR));
        }
    }
    
    
    //helper for creation of named AttributeSet/AttributeList 
    private void createNamedCollection(String name, NamedCollection attrs) throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        Object parent = elements.peek(); //reset parent
        if(attrs.getType() == 0)
        {
            IAttributeList attrList = ((IAttributeSet)parent).createAttributeList(name);
            elements.push(attrList);
        }
        else
        {
            IAttributeSetType setType = null;
            String type = attrs.getValue(XMLConstants.TYPE_ATTR);
            if(type != null)
            {
                setType = getAttributeSetType(type);
            }
            IAttributeSet attrSet = ((IAttributeSet)parent).createAttributeSet(name, setType);
            elements.push(attrSet);
        }
        
    }
    
    private void checkDomain(String domainName)throws com.sonicsw.mf.common.dirconfig.InvalidXMLException{
        if(m_strDomain == null)
        {
            return;
        }
        if(!domainName.equals(m_strDomain))
        {
            throw new InvalidXMLException(XMLConstants.FOREIGNDOMAIN_ERROR);
        }
    }
    
    private void addStringAttribute(String value)throws SAXException
    {
        if(m_stringElements.size() == XMLConstants.ZERO_INDEX)
        {
            throw new SAXException("(#PCDATA) can be entered for 'string' type only.");
        }
        StringAttributeInfo attrInfo = (StringAttributeInfo)m_stringElements.get(XMLConstants.ZERO_INDEX);
        Object parent = attrInfo.getParent();
        if(value == null)
        {
            value = attrInfo.getValue();
        }
        try{
            if(parent instanceof IAttributeSet)
            {
                ((IAttributeSet)parent).setStringAttribute(attrInfo.getName(), value);
            }
            else if(parent instanceof IAttributeList)
            {
                ((IAttributeList)parent).addStringItem( value);
            }
        }
        catch(Exception ex)
        {
            throw new SAXException(ex);
        }
        m_stringElements.clear();
    }
    
    private void cleanup()
    {
        elements.removeAllElements();
        elements = null;
        if(m_directoryService != null)
        {
            m_directoryService = null;
        }
        m_dirElement = null;    
        m_locator = null;
    }
    
  
    //Inner helper class to handle AttributeSet / AttributeList XML elements 
    //that have 'AttributeName' child element 
    
    class NamedCollection
    {  
        private HashMap m_attrs = null;
        private int m_collectionType; // '0' is LIST; '1' is SET;
        
        NamedCollection(int type)
        {
            m_collectionType = type;
        }
        
        private void setAttributes(Attributes attrs)
        {
            m_attrs = new HashMap(attrs.getLength());
            for(int i=0; i <attrs.getLength(); i++)
            {
                m_attrs.put(attrs.getLocalName(i),attrs.getValue(i));
            }
        }
        
        private String getValue(String attrName)
        {
           return (String)m_attrs.get(attrName);
        }
        
        private int getType()
        {
          return m_collectionType;  
        }
    }
    
   //Inner helper class to handle #PCDATA 
   //in Attribute/ListItem element of string type
   
    class StringAttributeInfo{
        
        private String m_attrName = null;
        private String m_attrValue = null;
        private Object m_attrParent = null; //IAttributeSet / IAttributeList
        
        StringAttributeInfo(String name, String value, Object parent)
        {
            m_attrName = name;
            m_attrValue = value;
            m_attrParent = parent;
        }
        
         protected String getName()
         {
            return m_attrName;
         }
         
         protected String getValue()
         {
            return m_attrValue;
         }
       
         protected Object getParent()
         {
            return m_attrParent;
         }
    }
    
}//end of class
