package com.sonicsw.mf.dstriggers;

import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeMetaData;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.ITriggerDSContext;
import com.sonicsw.mf.common.config.IValidationDSContext;
import com.sonicsw.mf.common.config.IValidationElementChange;
import com.sonicsw.mf.common.config.IValidator;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.FromDirectory;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.runtime.impl.ContainerIdentity;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.common.util.ReferentialIntegrity;
import com.sonicsw.mf.mgmtapi.config.constants.IActivationDaemonConstants;
import com.sonicsw.mf.mgmtapi.config.constants.ICollectionsMonitorConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IComponentCollectionConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerCollectionConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDomainConstants;

/*
    A trigger to cleanup after configuration deletetion of an MF configuration element (MF_CONTAINER for example).
    The following is the XML defintion of this trigger plug-in in the DS.

        <Directory name="/">

          <Directory name="/_MFLibrary">
            <Directory name="/_MFLibrary/validators">
              <ConfigElement>
                <ElementID version="1" releaseVersion="2.0" name="/_MFLibrary/validators/MF_2.0" creationTimestamp="0" type="MF_VALIDATOR" />
                <AttributeSet>
                  <Attribute name="RELEASE_VERSION" type="string" value="2.0" />
                  <Attribute name="VALIDATION_CLASS" type="string" value="com.sonicsw.mf.framework.triggers.DeleteTrigger" />
                </AttributeSet>
              </ConfigElement>
            </Directory>
          </Directory>

        </Directory>

*/

public final class DeleteTrigger86
implements IValidator
{
    private static final String[] TYPES_VALIDATED = new String[]
    {
        "MF_DOMAIN",
        "MF_MANAGEMENT_PERMISSIONS",
        "MF_AUTHENTICATION_GROUP",
        "MF_AUTHENTICATION_USER",
        "MF_AUTHENTICATION_DOMAIN",
        "MF_CONTAINER",
        "MF_COMPONENT_COLLECTION",
        "MF_ACTIVATION_DAEMON",
        "MF_DIRECTORY_SERVICE",
        "MF_BACKUP_DIRECTORY_SERVICE",
        "MF_AGENT_MANAGER",
        "MF_BACKUP_AGENT_MANAGER",
        "MF_COLLECTION_MONITOR",
        "MF_GENERIC_COMPONENT",
        "MF_LOGGER"
    };

    private static final String RELEASE_VERSION = "106";

    @Override
    public String getReleaseVersion()
    {
        return RELEASE_VERSION;
    }

    @Override
    public String[] getElementTypesValidated()
    {
    	
        return TYPES_VALIDATED;
    }

    @Override
    public void onDelete(ITriggerDSContext dsContext, IDirElement[] beforeImages)
    throws DirectoryServiceException
    {
        for (int i = 0; i < beforeImages.length; i++)
        {
            String type = beforeImages[i].getIdentity().getType();
            if (type.equals("MF_CONTAINER"))
            {
                ContainerIdentity containerID = new ContainerIdentity(beforeImages[i]);

                removeContainerFromActivationDaemons(dsContext, beforeImages[i].getIdentity().getName());
                removeContainerFromContainerCollections(dsContext, containerID.getCanonicalName());
                removeContainerComponentsFromComponentCollections(dsContext, containerID.getCanonicalNamePrefix());
            }
            else if (type.equals("MF_COMPONENT_COLLECTION"))
            {
                removeCollectionFromCollectionsMonitors(dsContext, beforeImages[i].getIdentity().getName());
                removeCollectionFromLoggers(dsContext, beforeImages[i].getIdentity().getName());
            }
            else if (type.equals("MF_AUTHENTICATION_DOMAIN"))
                onDeleteOfAuthDomain(dsContext, beforeImages[i].getIdentity().getName(), beforeImages[i].getAttributes());
            else
            // assume its from the component list
            {
                for (int j = 6; j < TYPES_VALIDATED.length; j++) // iterate thru all but the container
                {
                    if (type.equals(TYPES_VALIDATED[j]))
                    {
                        ReferentialIntegrity.removeComponent(dsContext, beforeImages[i].getIdentity().getName());
                        break;
                    }
                }
            }
        }
    }

    @Override
    public void validate(IValidationDSContext dsContext, IValidationElementChange[] changes)
    throws Exception
    {
  
        for (int i = 0; i < changes.length; i++)
        {
   
            if (changes[i].skipValidation())
                continue;

            // validate deletes
            if (changes[i].getChangeType() == IValidationElementChange.ELEMENT_DELETED)
            {
            	
                IDirElement beforeImage = (IDirElement)changes[i].getBeforeImage();

                if (beforeImage == null)
                    return; // due to import which we can't check

                IElementIdentity identity = beforeImage.getIdentity();
                String type = identity.getType();

                if (type.equals("MF_DOMAIN") || type.equals("MF_MANAGEMENT_PERMISSIONS"))
                    onValidateDeleteOfSecurityElements(identity);
                else if (type.equals("MF_AUTHENTICATION_GROUP"))
                    onValidateDeleteOfGroup(beforeImage);
                else if (type.equals("MF_AUTHENTICATION_USER"))
                    onValidateDeleteOfUser(beforeImage);
            }
            else if (changes[i].getChangeType() == IValidationElementChange.ELEMENT_UPDATED)
            {
            
                // TODO when we have support for update triggers we should remove component manage permissions
                //      when a component is removed from a container
                //if (changes[i].getElement().getIdentity().getType().equals("MF_CONTAINER")) { TODO }
            }
                      	 
        }
    }

    private void onValidateDeleteOfSecurityElements(IElementIdentity identity)
    throws DirectoryServiceException
    {
        throw new DirectoryServiceException("Failed to delete element \"" + identity.getName() + "\"; deleting of this element is not allowed");
    }

    private void onValidateDeleteOfGroup(IDirElement beforeImage)
    throws DirectoryServiceException
    {
        IAttributeSet groupAttrs = beforeImage.getAttributes();
        String groupName = (String)groupAttrs.getAttribute("GROUP_NAME");
        if (groupName.equals("Administrators"))
            throw new DirectoryServiceException("Failed to delete \"Administrators\" group; deleting of this group is not allowed");
    }

    private void onValidateDeleteOfUser(IDirElement beforeImage)
    throws DirectoryServiceException
    {
        IAttributeSet userAttrs = beforeImage.getAttributes();
        String userName = (String)userAttrs.getAttribute("USER_NAME");
        if (userName.equals(IManagementPermission.SUPER_USER_NAME))
            throw new DirectoryServiceException("Failed to delete user \"Administrator\"; deleting of this user is not allowed");
    }

    //Verifies that the auth domain is not used by management config security and not by any broker or policy domain
    public static void onDeleteOfAuthDomain(ITriggerDSContext ds, String authDescriptorID, IAttributeSet authAtts) throws DirectoryServiceException
    {
        String domainName = null;
        try
        {
            domainName = (String)authAtts.getAttribute("DOMAIN_NAME");
            if (domainName == null)
                domainName = authDescriptorID;
        }
        catch (Exception e)
        {
            domainName = authDescriptorID;
        }

        //Validate this auth domain is not used by any broker
        IElementIdentity[] ids = ds.listElements("/mq/brokers");
        for (int i = 0; i < ids.length; i++)
        {
            IDirElement broker = ds.getElement(ids[i].getName());
            IAttributeSet brokerAtts = broker.getAttributes();
            try
            {
                IAttributeSet brokerRefs = (IAttributeSet)brokerAtts.getAttribute("CONFIG_ELEMENT_REFERENCES");
                Reference authRef = (Reference)brokerRefs.getAttribute("AUTHENTICATION_DOMAIN_CONFIG_ELEMENT_REF");
                String brokerAuth = authRef != null ? authRef.getElementName() : null;

                if (brokerAuth != null && brokerAuth.equals(authDescriptorID))
                    throw new DirectoryServiceException("Cannot delete " + domainName + " - used by broker " +
                                                        brokerAtts.getAttribute("BROKER_NAME"));
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("Delete trigger failed on deletion of " + domainName + ": " + e.toString());
            }
        }

        //Validate this auth domain is not used by any policy domain
        IDirIdentity[] ids1 = ds.listDirectories("/policies");
        for (int i = 0; i < ids1.length; i++)
        {
            String descName = null;
            String policyAuth = null;
            String policyName = null;
            try
            {
                EntityName policyDir = new EntityName(ids1[i].getName());
                EntityName policyDesc = policyDir.createChild("_MQPolicyDescriptor");
                descName = policyDesc.getName();
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("Delete trigger failed on deletion of " + domainName + ": " + e.toString());
            }
            try
            {
                IDirElement descElement = ds.getElement(descName);
                IAttributeSet policyDescAtts = descElement.getAttributes();
                Reference policyAuthRef = (Reference)policyDescAtts.getAttribute("DEFAULT_DOMAIN");
                policyAuth = policyAuthRef != null ? policyAuthRef.getElementName() : null;
                policyName = (String)policyDescAtts.getAttribute("POLICY_NAME");
            }
            catch (Exception e) //Ok if there is no _MQPolicyDescriptor element at all
            {
            }

            if (policyAuth != null && policyAuth.equals(authDescriptorID))
                throw new DirectoryServiceException("Cannot delete " + domainName + " - used by policy " + policyName);
        }

        //Validate this auth domain is not used by management security
        String authDomain = null;
        try
        {
            IDirElement domain = ds.getElement("/domain/domain");
            IAttributeSet domainAtts = domain.getAttributes();
            Reference domainAuthRef = (Reference)domainAtts.getAttribute(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR);
            authDomain = domainAuthRef != null ? domainAuthRef.getElementName() : null;
        }
        catch (Exception e) //If domain/domain doesn't exist then ok
        {
        }

        if (authDomain != null && authDomain.equals(authDescriptorID))
            throw new DirectoryServiceException("Cannot delete " + domainName + " - used by management security");
    }

    // remove the container from daemons that launch it
    public static void removeContainerFromActivationDaemons(ITriggerDSContext ds, String containerConfigID)
    throws DirectoryServiceException
    {
        if (!ds.directoryExists("/activation"))
            return;

        try
        {

            FromDirectory f = new FromDirectory("/activation");
            IDirElement[] elements = ds.getElements(new Query().setFrom(f), true);

            for (int i = 0; i < elements.length; i++)
            {
                IAttributeSet daemonAtts = elements[i].getAttributes();
                IAttributeMetaData daemonAttsMeta = daemonAtts.getAttributeMetaData(IActivationDaemonConstants.CONTAINERS_ATTR);

                // If the value is from a template we'll modify the template
                if (daemonAttsMeta.isFromTemplate())
                    continue;

                IAttributeSet containers = (IAttributeSet)daemonAtts.getAttribute(IActivationDaemonConstants.CONTAINERS_ATTR);
                Iterator iterator = containers.getAttributes().entrySet().iterator();
                while (iterator.hasNext())
                {
                    Map.Entry entry = (Map.Entry)iterator.next();
                    IAttributeSet containerAttrs = (IAttributeSet)entry.getValue();
                    if (((Reference)containerAttrs.getAttribute(IActivationDaemonConstants.CONTAINER_REF_ATTR)).getElementName().equals(containerConfigID))
                    {
                        containers.deleteAttribute((String)entry.getKey());
                        ds.setElement((IDeltaElement)elements[i].doneUpdate());
                        break;
                    }
                }
            }
        }
        catch (ConfigException e)
        {
            e.printStackTrace(); // Should never happen
            throw new Error(e.toString(), e);
        }
    }

    // remove the collection from CMs that reference it
    public static void removeCollectionFromCollectionsMonitors(ITriggerDSContext ds, String collectionConfigID)
    throws DirectoryServiceException
    {
        if (!ds.directoryExists("/monitors"))
            return;

        try
        {

            FromDirectory f = new FromDirectory("/monitors");
            IDirElement[] elements = ds.getElements(new Query().setFrom(f), true);

            for (int i = 0; i < elements.length; i++)
            {
                IAttributeSet monitorAtts = elements[i].getAttributes();
                IAttributeMetaData monitorAttsMeta = monitorAtts.getAttributeMetaData(ICollectionsMonitorConstants.COLLECTIONS_ATTR);

                // If the value is from a template we'll modify the template
                if (monitorAttsMeta.isFromTemplate())
                    continue;

                IAttributeList collections = (IAttributeList)monitorAtts.getAttribute(ICollectionsMonitorConstants.COLLECTIONS_ATTR);
                for (int j = collections.getCount(); j > 0; j--)
                {
                    if (((Reference)collections.getItem(j - 1)).getElementName().equals(collectionConfigID))
                    {
                        collections.deleteAttributeItem(j - 1);
                        ds.setElement((IDeltaElement)elements[i].doneUpdate());
                        break;
                    }
                }
            }
        }
        catch (ConfigException e)
        {
            e.printStackTrace(); // Should never happen
            throw new Error(e.toString());
        }
    }

    // remove the collection from EM Loggers that reference it
    public static void removeCollectionFromLoggers(ITriggerDSContext ds, String collectionConfigID)
    throws DirectoryServiceException
    {
        if (!ds.directoryExists("/loggers"))
            return;

        try
        {

            FromDirectory f = new FromDirectory("/loggers");
            IDirElement[] elements = ds.getElements(new Query().setFrom(f), true);

            for (int i = 0; i < elements.length; i++)
            {
                IAttributeSet monitorAtts = elements[i].getAttributes();
                IAttributeMetaData monitorAttsMeta = monitorAtts.getAttributeMetaData("COMPONENT_COLLECTIONS");

                // If the value is from a template we'll modify the template
                if (monitorAttsMeta.isFromTemplate())
                    continue;

                IAttributeSet collections = (IAttributeSet)monitorAtts.getAttribute("COMPONENT_COLLECTIONS");
                Iterator iterator = collections.getAttributes().entrySet().iterator();
                while (iterator.hasNext())
                {
                    Map.Entry entry = (Entry)iterator.next();
                    Reference reference = (Reference)entry.getValue();
                    if (reference.getElementName().equals(collectionConfigID))
                    {
                        String key = (String)entry.getKey();
                        collections.deleteAttribute(key);
                        ds.setElement((IDeltaElement)elements[i].doneUpdate());
                        break;
                    }
                }
            }
        }
        catch (ConfigException e)
        {
            e.printStackTrace(); // Should never happen
            throw new Error(e.toString());
        }
    }

    // remove the container from container collections that refer to it
    private static void removeContainerFromContainerCollections(ITriggerDSContext ds, String canonicalName)
    throws DirectoryServiceException
    {
        if (!ds.directoryExists("/containerCollections"))
            return;

        try
        {
            FromDirectory f = new FromDirectory("/containerCollections");
            IDirElement[] elements = ds.getElements(new Query().setFrom(f), true);

            for (int i = 0; i < elements.length; i++)
            {
                IAttributeSet collectionAtts = elements[i].getAttributes();

                // If the set comes from a template then we'll modify the template
                if (collectionAtts.getAttributeMetaData(IContainerCollectionConstants.CONTAINERS_ATTR).isFromTemplate())
                    continue;

                IAttributeSet set = (IAttributeSet)collectionAtts.getAttribute(IContainerCollectionConstants.CONTAINERS_ATTR);
                Iterator iterator = set.getAttributes().entrySet().iterator();
                while (iterator.hasNext())
                {
                    Map.Entry entry = (Map.Entry)iterator.next();
                    IAttributeSet itemSet = (IAttributeSet)entry.getValue();
                    String id = (String)itemSet.getAttribute(IContainerCollectionConstants.CONTAINER_RUNTIME_ID_ATTR);
                    if (id.equals(canonicalName))
                    {
                        set.deleteAttribute((String)entry.getKey());
                        ds.setElement((IDeltaElement)elements[i].doneUpdate());
                        break;
                    }
                }
            }
        }
        catch (ConfigException e)
        {
            e.printStackTrace(); // Should never happen
            throw new Error(e.toString());
        }
    }

    // remove all the components hosted by the given container from component collections that refer to them
    private static void removeContainerComponentsFromComponentCollections(ITriggerDSContext ds, String containerNamePrefix)
    throws DirectoryServiceException
    {
        if (!ds.directoryExists("/componentCollections"))
            return;

        try
        {
            FromDirectory f = new FromDirectory("/componentCollections");
            IDirElement[] elements = ds.getElements(new Query().setFrom(f), true);

            for (int i = 0; i < elements.length; i++)
            {
                IAttributeSet collectionAtts = elements[i].getAttributes();

                // If the set comes from a template then we'll modify the template
                if (collectionAtts.getAttributeMetaData(IComponentCollectionConstants.COMPONENTS_ATTR).isFromTemplate())
                    continue;

                IAttributeSet set = (IAttributeSet)collectionAtts.getAttribute(IComponentCollectionConstants.COMPONENTS_ATTR);
                Iterator iterator = set.getAttributes().entrySet().iterator();

                boolean hasChanges = false;
                while (iterator.hasNext())
                {
                    Map.Entry entry = (Map.Entry)iterator.next();
                    IAttributeSet itemSet = (IAttributeSet)entry.getValue();
                    String id = (String)itemSet.getAttribute(IComponentCollectionConstants.COMPONENT_RUNTIME_ID_ATTR);
                    if (id.startsWith(containerNamePrefix))
                    {
                        set.deleteAttribute((String)entry.getKey());
                        hasChanges = true;
                    }
                }
                if (hasChanges)
                    ds.setElement((IDeltaElement)elements[i].doneUpdate());
            }
        }
        catch (ConfigException e)
        {
            e.printStackTrace(); // Should never happen
            throw new Error(e.toString());
        }
    }

    @Override
    public void onUpdate(ITriggerDSContext dsContext, IDirElement[] beforeImages, IDeltaElement[] afterImages)
    throws DirectoryServiceException
    {
        // TODO Auto-generated method stub
        
    }

    @Override
    public void onCreate(ITriggerDSContext dsContext, IDirElement[] afterImages)
    throws DirectoryServiceException
    {
        // TODO Auto-generated method stub
        
    }
}
