// Copyright (c) 2009 Progress Software Corporation.  All Rights Reserved.
package com.sonicsw.mf.eclipse;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.util.SonicFSException;
import com.sonicsw.mx.config.util.SonicFSFileSystem;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.Version;
import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IHandlerConstants;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.query.FromElementType;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.common.util.MonitoredContainer;
import com.sonicsw.mf.common.util.MonitoredContainerList;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;
import com.sonicsw.mf.mgmtapi.config.constants.IActivationDaemonConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;
import com.sonicsw.mf.mgmtapi.runtime.IDirectoryServiceProxy;

/**
 * This class adds the EclipseDSHandler to the DS.
 *
 * Amongst others it uploads the implementation jar, registers the workspace
 * mapping and registers the DSHandler. Last it also sets up monitored
 * containers.
 *
 * This class is a complete refactoring of
 * \vobs_install\install\src\AddToDS.java.
 *
 * @author Thomas Steinborn
 */
public class AddToDS
{

    private static String                 AUTO_START_DAEMON_NAME        = "/Framework Components/WorkbenchAutoStartContainers";
    private static String                 DAEMON_ARCHIVE_VALUE          = "MF/" + Version.getMajorVersion() + "." + Version.getMinorVersion() + "/MFdaemon.car";
    private static volatile String        m_DMContainerConfigID;

    private static final String VERSION_MAJOR_MINOR = Version.getMajorVersion()
            + "." + Version.getMinorVersion();

    /** default upload directory. */
    private static final String SYSTEM_IW_MAJOR_MINOR_LIB = "/System/IW/"
            + VERSION_MAJOR_MINOR + "/lib";

    /** upload location in the DS. */
    private static final String ECLIPSE_HANDLER_JAR = SYSTEM_IW_MAJOR_MINOR_LIB
            + "/EclipseHandler.jar";

    /** class path URL. */
    private static final String ECLIPSE_HANDLER_JAR_URL = "sonicfs://"
            + ECLIPSE_HANDLER_JAR;

    /** name of the DSHandler registration element. */
    private static final String MFLIBRARY_DS_HANDLERS_ECLIPSE_HANDLER = IHandlerConstants.MF_HANDLERS_DIR
            + "/EclipseHandler";

    private static final String WORKSPACE_HANDLER_NAME_PREFIX = "/workspace";

    private static boolean DEBUG = true;

    private static boolean firstDebug = true;

    private volatile static String m_url = null;

    private static volatile String m_user = null;

    private static volatile String m_password = null;

    private static volatile String m_domain = null;

    private String m_importFile = null;

    private String m_eclipseWorkspace = null;

    private static volatile ConfigServerUtility m_csu;

    private static volatile IDirectoryAdminService m_dsAdmin;

    private static volatile IDirectoryFileSystemService m_dsFS;

    private static volatile SonicFSFileSystem m_fs;

    static volatile IConfigServer configServer;

    public static void main(String[] args) throws Exception
    {
        new AddToDS(args);
    }

    public AddToDS(String[] args) throws Exception
    {
        parseArgs(args);

        connect();

        try
        {

            unregisterEclipseHandler();
            unregisterWorkspaceMapping();
            // create the folder after unregistering so that the call does not
            // end up in the (old) handler, which does not support folder
            // creation...
            // disconnect and reconnect first. This will reinitialize the
            // handlers for the Directory Service, so the unregistered handler
            // will be really gone. Necessary in an upgrade scenario.
            disconnect();
            connect();
            initializeLauncherDaemon();
            m_dsFS.createFolder(WORKSPACE_HANDLER_NAME_PREFIX, true);

            uploadEclipseHandlerJar();
            registerWorkspaceMapping();
            registerEclipseHandler();

            setupMonitoredContainers();

        } catch (Exception e)
        {
            System.out
                    .println("*Error*: while seeding the Eclipse handler due to "
                            + e.getMessage());
            e.printStackTrace();
            throw e;
        } finally
        {
            if (m_csu != null)
            {
                try
                {
                   disconnect();
                } catch (ConfigServiceException e)
                {
                    // ignore disconnect problems
                }
                m_csu = null;
                m_dsAdmin = null;
                m_dsFS = null;
                m_fs = null;
            }
        }
    }

    private void unregisterWorkspaceMapping()
    {
        try
        {
            IDirElement mappingEl = m_dsAdmin.getElement(EclipseDSHandler.MAPPING_STORAGE_NAME,
                    false, false);
            if (mappingEl != null)
            {
                // Remember what the mapping was
                m_eclipseWorkspace = (String)mappingEl.getAttributes().getAttribute(EclipseDSHandler.ECLIPSE_WORKSPACE_DISK_LOCATION_ATTR);
                m_dsAdmin.deleteElement(EclipseDSHandler.MAPPING_STORAGE_NAME,
                        null);
            }
        } catch (DirectoryServiceException e)
        {
            // ignore that e.g. the parent directory does not exist
        }
    }

    /**
     * Seed the mapping of the workspace disk location into the DS.
     *
     * Call this BEFORE registering the handler.
     *
     * @throws ConfigException
     * @throws DirectoryServiceException
     */
    private void registerWorkspaceMapping() throws ConfigException,
            DirectoryServiceException
    {
        try
        {
            m_dsAdmin
                    .createDirectory(EclipseDSHandler.MAPPING_STORAGE_DIRECTORY);
        } catch (DirectoryServiceException e)
        {
            // ignore if parent dir already exists
        }

        IDirElement handlerMap = ElementFactory.createElement(
                EclipseDSHandler.MAPPING_STORAGE_NAME, "Eclipse_Map", "1.0");
        IAttributeSet topSet = handlerMap.getAttributes();

        topSet.setStringAttribute(
                EclipseDSHandler.ECLIPSE_WORKSPACE_DISK_LOCATION_ATTR,
                m_eclipseWorkspace);
        debug("Set string attribute for ECLIPSE_WORKSPACE_DISK_LOCATION");

        m_dsAdmin.setElement(handlerMap.doneUpdate(), null);
    }

    private void registerEclipseHandler() throws ReadOnlyException,
            AttributeSetTypeException, ConfigException,
            DirectoryServiceException, VersionOutofSyncException
    {
        IDirElement handlerEl = ElementFactory.createElement(
                MFLIBRARY_DS_HANDLERS_ECLIPSE_HANDLER,
                IHandlerConstants.MF_DS_HANDLER_TYPE, VERSION_MAJOR_MINOR);
        IAttributeSet topSet = handlerEl.getAttributes();
        topSet.setStringAttribute(IHandlerConstants.DS_HANDLER_NAME,
                WORKSPACE_HANDLER_NAME_PREFIX);
        topSet.setStringAttribute(IHandlerConstants.DS_HANDLER_CLASS,
                EclipseDSHandler.class.getName());
        IAttributeList archiveList = topSet
                .createAttributeList(IHandlerConstants.DS_HANDLER_ARCHIVES);
        archiveList.addStringItem(ECLIPSE_HANDLER_JAR_URL);

        m_dsAdmin.setElement(handlerEl.doneUpdate(), null);
    }

    private void unregisterEclipseHandler() throws DirectoryServiceException,
            VersionOutofSyncException
    {
        IDirElement handlerElement = m_dsAdmin.getElement(
                MFLIBRARY_DS_HANDLERS_ECLIPSE_HANDLER, false);
        debug("handlerElement = " + handlerElement);
        if (handlerElement != null)
        {
            m_dsAdmin
                    .deleteElement(MFLIBRARY_DS_HANDLERS_ECLIPSE_HANDLER, null);
        }
    }

    private void uploadEclipseHandlerJar() throws SonicFSException
    {
        m_fs.createDirectoryPath(SYSTEM_IW_MAJOR_MINOR_LIB);
        m_fs.updateFile(ECLIPSE_HANDLER_JAR, new File(m_importFile), true);
    }

    private void parseArgs(String[] args)
    {
        if (args.length < 5*2)
        {
            printUsage();
        }

        for (int i = 0; i < args.length; i++)
        {
            if (args[i].startsWith("-"))
            {
                if (args[i].equalsIgnoreCase("-url"))
                {
                    m_url = args[++i];
                } else if (args[i].equalsIgnoreCase("-user"))
                {
                    m_user = args[++i];
                } else if (args[i].equalsIgnoreCase("-password"))
                {
                    m_password = args[++i];
                } else if (args[i].equalsIgnoreCase("-domain"))
                {
                    m_domain = args[++i];
                } else if (args[i].equalsIgnoreCase("-importFile"))
                {
                    m_importFile = args[++i];
                } else if (args[i].equalsIgnoreCase("-eclipseWorkspace"))
                {
                    m_eclipseWorkspace = args[++i];
                } else
                {
                    printUsage();
                }
            } else
            {
                printUsage();
            }
        }
        if (m_url == null || m_user == null || m_domain == null
                || m_importFile == null || m_eclipseWorkspace == null)
        {
            printUsage();
        }
    }

    private void setupMonitoredContainers() throws Exception
    {
        try
        {
            MonitoredContainerList
                    .setMonitoredContainerList(m_dsFS,
                            new MonitoredContainer[]
                            {
                                    // Logical Path Auto Launch
                                    new MonitoredContainer(
                                            "/Containers/dev_ESBCore", true),
                                    new MonitoredContainer(
                                            "/Containers/dev_ESBTest", true)});
            if (DEBUG)
            {
                MonitoredContainer[] mcList = MonitoredContainerList
                        .getMonitoredContainerList(m_dsFS);
                for (int i = 0; i < mcList.length; i++)
                {
                    debug("MonitoredContainerList[" + i + "] = ("
                            + mcList[i].getContainerLogicalPath() + ", "
                            + mcList[i].isAutoLaunch() + ")");
                }
            }
        } catch (Exception e)
        {
            System.out
                    .println("Error: while setting monitored container list due to "
                            + e.getMessage());
            e.printStackTrace();
            throw e;
        }
    }

    private static void printUsage()
    {
        System.out
                .println("(-url <url>) (-user <user>) (-password <password>) (-domain <domain>) (-importFile <import file>) (-eclipseWorkspace <eclipse workspace>)");
        System.out
                .println("The pairs can be in any order, but all must be used");
        System.out
                .println("Example:\n-url tcp://localhost:2506 -user Administrator -password Administrator -domain Domain1 -importFile C:\\Sonic\\Workbench8.0\\lib\\EclipseHandler.jar -eclipseWorkspace C:\\Sonic\\Workbench8.0\\workspace");
        System.exit(1);
    }

    protected static void debug(String message)
    {
        if (DEBUG && firstDebug)
        {
            System.out.println("AddToDS:");
            firstDebug = false;
        }
        if (DEBUG)
        {
            System.out.println("\tAddToDS: " + message);
        }
    }

    private static void connect() throws ConfigServiceException
    {
        m_csu = new ConfigServerUtility();
        m_csu.connect(m_domain, m_url, m_user, m_password, true);

        m_dsAdmin = (IDirectoryAdminService) m_csu.getDirectoryService();
        m_dsFS = m_csu.getDirectoryService();
        m_fs = new SonicFSFileSystem(m_csu.getDirectoryService(), m_user);
        configServer = m_csu.getConfigServer();
    }

    private static void disconnect() throws ConfigServiceException
    {
        m_csu.disconnect();
        m_csu = null;
        m_dsAdmin = null;
        m_dsFS = null;
        m_fs = null;
        configServer = null;
    }

    /**
     * Method for creating Launcher Daemon configuration. Can be called by anyone.
     * @param args
     * @throws Exception
     */
    public static void configureLauncherDaemon(String[] args) throws Exception {
        try{
            parseLDArgs(args);
            connect();
            initializeLauncherDaemon();
        }catch(Exception e){
            System.out.println("*Error*: while configuring Launcher Daemon " + e.getMessage());
            e.printStackTrace();
            throw e;
        }finally
        {
            if (m_csu != null)
            {
                try
                {
                   disconnect();
                } catch (ConfigServiceException e)
                {
                    // ignore disconnect problems
                }
                m_csu = null;
                m_dsAdmin = null;
                m_dsFS = null;
                m_fs = null;
                configServer = null;
            }
        }
    }

    private static void parseLDArgs(String[] args)
    {
        if (args.length < 3*2)
        {
            printLDUsage();
        }

        for (int i = 0; i < args.length; i++)
        {
            if (args[i].startsWith("-"))
            {
                if (args[i].equalsIgnoreCase("-url"))
                {
                    m_url = args[++i];
                } else if (args[i].equalsIgnoreCase("-user"))
                {
                    m_user = args[++i];
                } else if (args[i].equalsIgnoreCase("-password"))
                {
                    m_password = args[++i];
                } else if (args[i].equalsIgnoreCase("-domain"))
                {
                    m_domain = args[++i];
                } else
                {
                    printLDUsage();
                }
            } else
            {
                printLDUsage();
            }
        }
        if (m_url == null || m_user == null || m_domain == null)
        {
            printLDUsage();
        }
    }

    private static void printLDUsage()
    {
        System.out
                .println("(-url <url>) (-user <user>) (-password <password>) (-domain <domain>)");
        System.out
                .println("The pairs can be in any order, but all must be used");
        System.out
                .println("Example:\n-url tcp://localhost:2506 -user Administrator -password Administrator -domain Domain1 ");
        System.exit(1);
    }



    /**
     * Initialize method which will check for occurence of AutoLaunchDaemon if not will start creating one.
     * @throws Exception
     */
    private static void initializeLauncherDaemon() throws Exception {

        findDomainManagerName();
        if (configServer.loadConfigElement(AUTO_START_DAEMON_NAME) == null && m_DMContainerConfigID != null)
        {
            createAutoLaunchDaemon();
        }
    }

    /**
     * Creates the activation daemon.
     *
     * @throws Exception
     */
    private static IConfigElement createAutoLaunchDaemon() throws Exception {
        IConfigElement daemon = null;
        try {
            daemon = configServer.createConfigBean(AUTO_START_DAEMON_NAME, IActivationDaemonConstants.DS_TYPE, IActivationDaemonConstants.DS_C_VERSION, false);
            daemon.setAttribute(IActivationDaemonConstants.CLASSNAME_ATTR, IActivationDaemonConstants.CLASSNAME_FIXED);
            daemon.setAttribute(IActivationDaemonConstants.ARCHIVE_NAME_ATTR, DAEMON_ARCHIVE_VALUE);
            daemon.createAttributeMap(IActivationDaemonConstants.CONTAINERS_ATTR);

            Map metaAttributes = configServer.getMetaAttributes(AUTO_START_DAEMON_NAME);
            if (metaAttributes == null)
            {
                metaAttributes = new HashMap();
            }

            metaAttributes.put(ConfigServerUtility.TYPE, IActivationDaemonConstants.DS_TYPE);
            metaAttributes.put(ConfigServerUtility.CONFIG_VERSION, IActivationDaemonConstants.DS_C_VERSION);
            metaAttributes.put(ConfigServerUtility.PRODUCT_VERSION, IActivationDaemonConstants.DS_P_VERSION);

            // add it to the domain manager container
            IConfigElement domainManager = configServer.loadConfigElement(m_DMContainerConfigID);
            IAttributeMap componentsMap = (IAttributeMap) domainManager.getAttribute(IContainerConstants.COMPONENTS_ATTR);
            IAttributeMap daemonMap = componentsMap.createAttributeMap("AutoLaunchDaemon");
            daemonMap.setAttribute(IContainerConstants.CONFIG_REF_ATTR, daemon);
            componentsMap.setAttribute("AutoLaunchDaemon", daemonMap);
            // the caller commits this object in the transaction
            configServer.storeConfigElement(domainManager);
            configServer.storeConfigElement(daemon);
            configServer.setMetaAttributes(AUTO_START_DAEMON_NAME, metaAttributes);
            debug(AUTO_START_DAEMON_NAME);
        } catch (Exception e) {
            System.out.println("Error: while creating Auto Launch Daemon due to " + e.getMessage());
            e.printStackTrace();
            throw e;
        }
        return daemon;
    }


    private static void findDomainManagerName() throws MalformedObjectNameException, ConfigServiceException, ReflectionException, InstanceNotFoundException, MBeanException {
            IRemoteMBeanServer mBeanServer = m_csu.getMBeanServer();
            if (mBeanServer != null && mBeanServer.isConnected())
            {
                ObjectName id = new ObjectName(m_csu.getDomain() + "." + IDirectoryServiceProxy.GLOBAL_ID + ":ID=" + IAgentProxy.ID);
                IContainerState state = (IContainerState) m_csu.getMBeanServer().invoke(id, "getContainerState", new Object[] {}, new String[] {});
                m_DMContainerConfigID = configServer.storageToLogical(state.getRuntimeIdentity().getConfigIdentity().getName());
            }
            else
            {
                // we'll have to find a directory service configuration in a container. If there's
                // more than one in a container, we won't know the the domain manager container name
                // is
                Set containers = listAllOfType(IContainerConstants.DS_TYPE);
                if(containers != null){
                    Iterator containersIT = containers.iterator();
                    String domainManagerID = null;
                    boolean multipleContainers = false;
                    while (containersIT.hasNext() && !multipleContainers)
                    {
                        String containerName = (String) containersIT.next();
                        IConfigElement[] dsComponents = getComponents(containerName, IDirectoryServiceConstants.DS_TYPE);
                        if (dsComponents != null && dsComponents.length == 1)
                        {
                            if (domainManagerID == null)
                            {
                                domainManagerID = containerName;
                            }
                            else
                            {
                                multipleContainers = true;
                            }
                        }
                        if (!multipleContainers)
                        {
                            m_DMContainerConfigID = domainManagerID;
                        }
                    }
                }
            }
        }

    private static Set listAllOfType(String type) throws ConfigServiceException {
        Query typeQuery = new Query();
        typeQuery.setFrom(new FromElementType(type));
         return configServer.listConfigElements(typeQuery);
    }

    private static IConfigElement[] getComponents(String containerName, String componentType) throws ConfigServiceException {
        ArrayList result = new ArrayList();
        IConfigElement container = configServer.loadConfigElement(containerName);
        if (container == null)
        {
            throw new ConfigServiceException("getComponents failed: container " + containerName + " does not exist");
        }

        IAttributeMap components = (IAttributeMap) container.getAttribute(IContainerConstants.COMPONENTS_ATTR);
        Collection compCollection = components.values();
        Iterator compIterator = compCollection.iterator();
        while (compIterator.hasNext())
        {
            Object x = compIterator.next();
            IAttributeMap componentMap = (IAttributeMap) x;
            IConfigBean component = (IConfigBean) componentMap.getAttribute(IContainerConstants.CONFIG_REF_ATTR);

            if (component.getConfigType().getName().equals(componentType))
            {
                // Get the component from the DS, instead of using the cached one
                IConfigElement configEl = configServer.loadConfigElement(component.getName());
                result.add(configEl);
            }
        }
        IConfigElement[] resultArray = new IConfigElement[result.size()];
        result.toArray(resultArray);
        return resultArray;
    }

}
