package com.sonicsw.mf.eclipse;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.AbstractDSHandler;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IChunkedBlobStreamer;
import com.sonicsw.mf.common.config.IDSHandlerContext;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IHandlerConstants;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.INamingNotification;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.impl.Blob;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.ElementCreateNotification;
import com.sonicsw.mf.common.config.impl.ElementIdentity;
import com.sonicsw.mf.common.config.impl.FolderCreateNotification;
import com.sonicsw.mf.common.config.impl.MetaAttributesChangeNotification;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.view.ILogicalNameSpace;
import com.sonicsw.mf.common.view.INamingListener;
import com.sonicsw.mf.framework.IContainer;


/**
 * <p>Title: </p>
 * <p>Description: Implements DS methods that do lookups in the Eclipse workspace. There is a TIDE test in the mgmt
 * sandbox for this class. The way it works is that if you modify this class, you also modify
 * vobs_qa_mgmt/testresources/qa/mgmt/mf/directory/EclipseHandler.jar to contain the modified class. This class
 * doesn't exist in vobs_qa_mgmt otherwise. The TIDE test is in vobs_qa_mgmt/src/qa/mgmt/mf/directory/EclipseHandlerTest</p>
 * <p>Copyright: Copyright (c) 2005</p>
 * <p>Company: </p>
 * @author Mari Davila
 * @version 1.0
 */

public class EclipseDSHandler extends AbstractDSHandler implements IChunkedBlobStreamer
{
    /*
     * The URI Prefix is introduced in Eclipse3.2. Imported project's .location file will 
     * carry a URI_PREFIX infront of its URL entry for Eclipse version >=3.2. 
     */
	private static final String URI_PREFIX = "URI//";
    private static final String FILE_SEPARATOR = System.getProperties().getProperty("file.separator");
    private static final char FILE_SEPARATOR_CHAR = FILE_SEPARATOR.charAt(0);
    private static final String ECLIPSE_ELEMENT_TYPE = "MF_FILE";
    private static final String ECLIPSE_ELEMENT_RELEASE_VERSION = "100"; // that's what's in the .xsd  ARGGGH!
    
    /** attribute name storing the workspace disk location */
    public static final String ECLIPSE_WORKSPACE_DISK_LOCATION_ATTR = "ECLIPSE_WORKSPACE_DISK_LOCATION"; // workspace disk location
    /**
     * marker postfix for element creation to trigger an update of the workspace location.
     * 
     * @see #createFSElement(IDirElement)
     */
    public static final String MAP_ELEMENT_NAME = "EclipseMap";
    /** storage directory for the mapping element. */
    public static final String MAPPING_STORAGE_DIRECTORY = IHandlerConstants.MF_HANDLERS_DIR + "/eclipse";
    /** storage location for the mapping element. */
    public static final String MAPPING_STORAGE_NAME = MAPPING_STORAGE_DIRECTORY + "/_mapping";
    
    private static final String TOOL_ATTRIBUTES_ATTR = "TOOL_ATTRIBUTES";
    private static final String SIZE_ATTR = "SIZE";
    private static final String LAST_MODIFIED_TIME_ATTR = "LAST_MODIFIED_TIME";
    private static final String PERMISSIONS_ATTR = "PERMISSIONS";
    private static final String CREATION_TIME_ATTR = "CREATION_TIME";
    private static final String CREATED_BY_ATTR = "CREATED_BY";
    private static final String HIDDEN_ATTR = "HIDDEN";
    
    private static final String PROJECT_METADATA_DIR = ".metadata" + FILE_SEPARATOR + ".plugins" + FILE_SEPARATOR +
                                               "org.eclipse.core.resources" + FILE_SEPARATOR + ".projects";
    private static final String PROJECT_SAFETABLE_PATH = ".metadata" + FILE_SEPARATOR + ".plugins" + FILE_SEPARATOR + 
                                               "org.eclipse.core.resources" + FILE_SEPARATOR + ".safetable" + FILE_SEPARATOR + "org.eclipse.core.resources";
    private static final String PROJECT_DESCRIPTION_FILE = ".project";

    private static final String DS_FILE_SEPARATOR = new String(new byte[] {IMFDirectories.MF_DIR_SEPARATOR});
    
    private String m_workspaceDiskLocation;
    private String m_metadataFolder;
    private NamingNotificationManager m_notificationManager;
    
    boolean DEBUG = false;
    
    public EclipseDSHandler()
    {
        super();
        String prop = System.getProperty("EclipseHandlerDebug", "false");
        DEBUG = prop.equals("true");
        if (DEBUG)
        {
            System.out.println("EclipseDSHandler constructor called");
        }
        m_notificationManager = new NamingNotificationManager();
    }
    
    @Override
    public void setMetaAttributes(String name, HashMap newAttributes) throws DirectoryServiceException
    {
        
        // do nothing. Just implement so that a ClientTransaction creating a file, which includes
        // setMetaAttributes, will not fail.
    }
    

    /**
     * Crafts a set of meta attributes for the Eclipse folder or element, based on the defined meta attributes in the DS
     * @param logicalName The name of the Eclipse file or folder
     * @return Meta attributes for the file or folder, null if the name is not found in the Eclipse workspace
     * @throws DirectoryServiceException
     */
    
    @Override
    public HashMap getMetaAttributes(String logicalName) throws DirectoryServiceException
    {
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.getMetaAttributes for logicalName " + logicalName);
        }
        HashMap metaAttrs = new HashMap();
        String[] definedMetaAttributes;
        EclipseFile eFile = new EclipseFile(logicalName);
        String onDiskName = eFile.m_diskLocation;
        // special case when the workspace doesn't exist - as when a new SUI with a 
        // new Eclipse is installed
        if (onDiskName == null)
        {
        	if (DEBUG)
            {
                System.out.println("EclipseDSHandler.getMetaAttributes onDiskName was null");
            }
            String withSlash, withoutSlash;
            if (logicalName.endsWith(DS_FILE_SEPARATOR))
            {
                withSlash = logicalName;
                withoutSlash = logicalName.substring(0, logicalName.length() - 1);
            }
            else
            {
                withoutSlash = logicalName;
                withSlash = logicalName + DS_FILE_SEPARATOR;
            }
        	if (withSlash.equals(m_handlerName) || withoutSlash.equals(m_handlerName))
	        {
	        	definedMetaAttributes = m_dsContext.getDefinedFolderMetaAttributes();
	        	for (int i=0; i<definedMetaAttributes.length; i++)
                {
                    metaAttrs.put(definedMetaAttributes[i], null);
                }
	        	metaAttrs.put(ILogicalNameSpace.FOLDER_NAME, logicalName);
	        	return metaAttrs;
	        }
            else {
                return null;		// DS returns null if the logical name doesn't exist
            }
        }
               
        File onDiskFile = new File(onDiskName);
        if (onDiskFile.exists())
        {
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.getMetaAttributes file exists: " + onDiskName);
            }
            if (onDiskFile.isFile())
            {
                definedMetaAttributes = m_dsContext.getDefinedElementMetaAttributes();
            }
            else
            {
                definedMetaAttributes = m_dsContext.getDefinedFolderMetaAttributes();
            }
            for (int i=0; i<definedMetaAttributes.length; i++)
            {
                metaAttrs.put(definedMetaAttributes[i], null);
            }
            if (onDiskFile.isFile())
            {
                if (DEBUG)
                {
                    System.out.println("EclipseDSHandler.getMetaAttributes logicalName is a file\n");
                }
                metaAttrs.put(ILogicalNameSpace.ELEMENT_IDENTITY, createElementIdentity(logicalName));
                metaAttrs.put(TOOL_ATTRIBUTES_ATTR, "TYPE=MF_FILE");
            }
            else
            {
                if (DEBUG)
                {
                    System.out.println("EclipseDSHandler.getMetaAttributes logicalName is a folder, setting _FOLDER_NAME to " + logicalName + "\n");
                }
                metaAttrs.put(ILogicalNameSpace.FOLDER_NAME, logicalName);
            }
            return metaAttrs;
        }
        else
        {
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.getMetaAttributes file does not exist: " + onDiskName + "\n");
            }
            return null;
        }
    }

    /**
     *
     * @param folderPath Eclipse workspace path
     * @return An array of foler meta attributes for the subfolders of folderPath
     * @throws DirectoryServiceException If folderPath is not a path in the Eclipse workspace
     */

    @Override
    public HashMap[] listFolders(String folderPath) throws DirectoryServiceException
    {
    	if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.listFolders folderPath == " + folderPath);
        }

        ArrayList returnFolders = listFoldersInternal(folderPath);
        HashMap[] all = new HashMap[returnFolders.size()];
        return (HashMap[])returnFolders.toArray(all);
    }

    /**
     *
     * @param folderPath A folder in the Eclipse workspace
     * @return An array of meta attributes for all the files and folders in folderPath
     * @throws DirectoryServiceException If folderPath does not exist in the Eclipse workspace
     */
    @Override
    public HashMap[] listFSAll(String folderPath) throws DirectoryServiceException
    {
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.listFSAll with folder "+ folderPath);
        }
        ArrayList returnAll = listFSAll(folderPath, true, true, null); 
        HashMap[] all = new HashMap[returnAll.size()];
        return (HashMap[])returnAll.toArray(all);
    }
    
    private void  addLinkedResources(EclipseFile eFile, ArrayList returnAll)throws DirectoryServiceException
    {
    	if (DEBUG)
        {
            System.out.println("\nEclipseDSHandler.addLinkedResources, file.m_diskLocation == " + eFile.m_diskLocation);
        }
    	if (eFile.m_logicalName.endsWith(eFile.m_projectName) && eFile.m_linkedResources != null)
    	{ 
    		HashMap linkedResources = eFile.m_linkedResources;
    		Collection linkNames = linkedResources.keySet();
    		Iterator linkNamesIterator = linkNames.iterator();
    		while (linkNamesIterator.hasNext())
    		{
    			String linkName = (String)linkNamesIterator.next();
    			LinkDescription descr = (LinkDescription)linkedResources.get(linkName);
    			// we only know how to deal with linked folder type (haven't seen an example of other types)
    			if (descr.getType() == LinkDescription.FOLDER_TYPE)
                {
                    returnAll.add(getMetaAttributes(eFile.m_logicalName + IMFDirectories.MF_DIR_SEPARATOR + linkName));
                }
    		}
    	}
    }
    
    @Override
    public ArrayList listFSAll(String folderPath, boolean getFolders, boolean getElements, String extension) throws DirectoryServiceException
    {
        ArrayList retList = new ArrayList();
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.listFSAll folderPath == " + folderPath);
        }

        if (getFolders)
        {
            retList.addAll(listFoldersInternal(folderPath));
        }
        if (getElements)
        {
            retList.addAll(listFSElementsInternal(folderPath, extension));
        }
        return retList;
    }

    /**
     *
     * @param folderPath A path in the Eclipse workspace
     * @return An array of meta attributes for the files in folderPath
     * @throws DirectoryServiceException If folder path is not a folder in the Eclipse workspace
     */

    @Override
    public HashMap[] listFSElements(String folderPath) throws DirectoryServiceException
    {
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.listFSElements with folder "+ folderPath);
        }
        ArrayList returnElements = new ArrayList();
        EclipseFile eFile = new EclipseFile(folderPath);
        String mappedPath = eFile.m_diskLocation;
        //special case when the workspace doesn't exist - as when a new SUI with a 
        // new Eclipse is installed
        if (mappedPath == null)
        {
            String withSlash, withoutSlash;
            if (folderPath.endsWith(DS_FILE_SEPARATOR))
            {
                withSlash = folderPath;
                withoutSlash = folderPath.substring(0, folderPath.length() - 1);
            }
            else
            {
                withoutSlash = folderPath;
                withSlash = folderPath + DS_FILE_SEPARATOR;
            }
        	if (withSlash.equals(m_handlerName) || withoutSlash.equals(m_handlerName))
            {
                return new HashMap[0];
            }
            else
            {
                throw new DirectoryServiceException("Folder " + folderPath + " does not exist");
            }
        }
        File mappedFile = new File(mappedPath);
        if (!mappedFile.exists())
        {
            throw new DirectoryServiceException("Folder " + folderPath + " does not exist");
        }
        if (mappedFile.isFile())
        {
            throw new DirectoryServiceException(folderPath + " is not a folder");
        }
        File[] children = mappedFile.listFiles();
        String projectName = eFile.m_projectName;
        try
        {
            for (int i=0; i< children.length; i++)
            {
                if (children[i].isFile())
                {
                    returnElements.add(getMetaAttributes(getLogicalName(children[i].getCanonicalPath(), projectName)));
                }
            }
        }
        catch (Exception e)
        {
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.listFSElements caught exception, trace follows");
                e.printStackTrace(System.out);
            }
            throw new DirectoryServiceException("Could not list files for " + folderPath + ": " + e.toString(), e);
        }
        HashMap[] all = new HashMap[returnElements.size()];
        return (HashMap[])returnElements.toArray(all);
    }

    /**
     *
     * @param blobName The path to a file in the Eclipse workspace
     * @return A Blob object corresponding to the file in the Eclipse workspace, null if the file doesn't exist
     * @throws DirectoryServiceException If blobName is the name of a folder, not a file, or if there is an error while fetching the file.
     */
    @Override
    public IBlob getBlobByLogicalName(String blobName) throws DirectoryServiceException
    {
    	if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.getBlobByLogicalName blobName == " + blobName);
        }

        return getFSBlob(blobName, false, 0);
    }

    /**
     * This is here to satisfy the IChunkedBlobStreamer interface, although it would never get called for two reasons:
     * 1) The blobs returned by this handler always have logical names
     * 2) The DS getBlob method is not written to call a ds handler in this case (it could, if we had to)
     * @param blobName
     * @param forUpdate
     * @param offset
     * @return
     * @throws DirectoryServiceException
     */
    @Override
    public IBlob getBlob(String blobName, boolean forUpdate, int offset) throws DirectoryServiceException
    {
    	if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.getBlob blobName == " + blobName);
        }

        return getFSBlob(blobName, forUpdate, offset);
    }

    /**
     *
     * @param blobName The path to a file in the Eclipse workspace.
     * @param forUpdate This flag is passed on but will have no effect for Eclipse files. Behaves like false all the time.
     * @return A Blob object corresponding to the file in the Eclipse workspace, null if the file doesn't exist
     * @throws DirectoryServiceException If blobName is the name of a folder, not a file, or if there is an error while fetching the file.
     */
    @Override
    public IBlob getFSBlob(String blobName, boolean forUpdate) throws DirectoryServiceException
    {
    	if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.getFSBlob(2 arguments) blobName == " + blobName);
        }

        return getFSBlob(blobName, forUpdate, 0);
    }
    // throws exception if file is found but something happens while fetching it
    /**
     *
     * @param blobName The path to a file in the Eclipse workspace
     * @return A Blob object corresponding to the file in the Eclipse workspace, null if the file doesn't exist
     * @throws DirectoryServiceException If blobName is the name of a folder, not a file, or if there is an error while fetching the file.
     */

    @Override
    public IBlob getFSBlob(String blobName, boolean forUpdate, int offset) throws DirectoryServiceException
    {
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.getFSBlob (3 arguments) getting called");
        }
        EclipseFile eFile = new EclipseFile(blobName);
        String fileOnDisk = eFile.m_diskLocation;
        if (fileOnDisk == null)
        {
            return null;
        }
        File diskFile = new File(fileOnDisk);
        if (!diskFile.exists())
        {
            return null;
        }
        if (diskFile.isDirectory())
        {
            throw new DirectoryServiceException(blobName + " is not a file");
        }
        IBlob returnBlob = null;
        byte[] fileChunk = null;
        Integer blobTransferState;
        try
        {
            IDirElement blobEl = craftNewElement(blobName, diskFile);
            RandomAccessFile fStream = new RandomAccessFile(diskFile, "r");
            fStream.seek(offset);
            if ( (offset + IBlob.BLOB_CHUNK_SIZE) >= diskFile.length())
            {
                fileChunk = new byte[ (int) (diskFile.length() - offset)];
                blobTransferState = IBlob.END;
            }
            else
            {
                fileChunk = new byte[IBlob.BLOB_CHUNK_SIZE];
                blobTransferState = IBlob.PARTIAL;
            }
            fStream.read(fileChunk);
            fStream.close();
            returnBlob = new Blob(blobEl, fileChunk, this);
            Blob.markBlobState((Element)blobEl, blobTransferState, IBlob.BLOB_TRANSFER_STATE);
            if (DEBUG)
            {
                System.out.println("Setting the blob logical to true\n");
            }
            returnBlob.setLogical(true);
        }
        catch (Exception e)
        {
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.getFSBlob caught exception, stack trace follows");
                e.printStackTrace();
            }
            throw new DirectoryServiceException("Could not fetch blob " + blobName + ": " + e.toString(), e);
        }
        return returnBlob;
    }

    // 
    @Override
    public void appendFSBlob(IBasicElement element, byte[] blob, int offset, boolean last) throws DirectoryServiceException
    {
        String logicalFileName = element.getIdentity().getName();
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.appendFSBlob logicalFileName == " + logicalFileName);
        }
        EclipseFile eFile = new EclipseFile(logicalFileName, offset == 0);
        String mappedPath = eFile.m_diskLocation;
        if (mappedPath == null)
        {
            throw new DirectoryServiceException("Folder for " + logicalFileName + " does not exist");
        }
        File blobFile = new File(mappedPath);
        if (offset == 0 && blobFile.exists())
        {
            blobFile.delete();
        }
        if ((offset != 0) && (blobFile.length() != offset))
        {
            throw new DirectoryServiceException("Cannot append to a blob in the middle of an existing file: " + logicalFileName);
        }
        try
        {
            FileOutputStream fileOutStream = new FileOutputStream(mappedPath, true);
            fileOutStream.write(blob);
            fileOutStream.close();
            if (last && element instanceof Element)
            {
                newFileNotifications(logicalFileName);
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Error while appending to file " + logicalFileName + ": " + e.toString(), e);
        }
    }
    
    private void newFileNotifications(String logicalName)
    {
    	HashMap metaAttrs = new HashMap();
    	metaAttrs.put(ILogicalNameSpace.ELEMENT_IDENTITY, createElementIdentity(logicalName));
        metaAttrs.put(TOOL_ATTRIBUTES_ATTR, "TYPE=MF_FILE");
        m_notificationManager.notifyNamingEvent(new ElementCreateNotification(logicalName));
        m_notificationManager.notifyNamingEvent(new MetaAttributesChangeNotification(logicalName, metaAttrs));
    }
    
    @Override
    public void createFolder(String folderName) throws DirectoryServiceException
    {
        createFolder(folderName, false);
    }
       

    @Override
    public void createFolder(String folderName, boolean existingOk)
    throws DirectoryServiceException
    {

        if (DEBUG)
        {
            System.out.println("\nAPI call EclipseDSHandler.createFolder folderName " + folderName + " existingOk == " + existingOk);
        }
        EclipseFile eFile = new EclipseFile(folderName, true);
        String mappedPath = eFile.m_diskLocation;
        if (mappedPath == null)
        {
            throw new DirectoryServiceException("Parent folder for " + folderName + " does not exist");
        }
        File newFolder = new File(mappedPath);
        if (newFolder.exists() && !existingOk)
        {
            throw new DirectoryServiceException("Folder " + folderName + " cannot be created because it exists");
        }
        
        if (!newFolder.exists())
        {
        	// don't make folders at the top level where we're expecting projects
        	// can only make folders as subfolders of the projects
        	/*if (newFolder.getParentFile().getCanonicalPath().equals(m_workspaceDiskLocation))
        		throw new DirectoryServiceException("Only subfolders of projects can be created; folders under the workspace root directory cannot be created: " + newFolder.toString());*/
            boolean madeDir = newFolder.mkdir();
            if (!madeDir)
            {
                throw new DirectoryServiceException("Unable to create folder " + folderName + ": mkdir returned false");
            }
            m_notificationManager.notifyNamingEvent(new FolderCreateNotification(folderName));
        
        }        
    }
    @Override
    public void attachFSBlob(IBasicElement fileElement, byte[]blob) throws DirectoryServiceException, VersionOutofSyncException
    {
        String logicalFileName = fileElement.getIdentity().getName();
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.attachFSBlob logicalFileName == " + logicalFileName);
        }
        EclipseFile eFile = new EclipseFile(logicalFileName, true);
        String mappedPath = eFile.m_diskLocation;
        if (mappedPath == null)
        {
            throw new DirectoryServiceException("Folder for " + logicalFileName + " does not exist");
        }
        try
        {
            FileOutputStream fileStream = new FileOutputStream(mappedPath);
            // if it's there we overwrite it
            fileStream.write(blob);
            fileStream.close();
            if (fileElement instanceof Element)
            {
                newFileNotifications(logicalFileName);
            }
        }
        catch (Exception e)
        {
        	if (DEBUG)
            {
                e.printStackTrace();
            }
            throw new DirectoryServiceException("Unable to attach blob " + logicalFileName + ": " + e.toString(), e);
        }
    }

    @Override
    public void attachFSBlob(IBasicElement fileElement, java.io.InputStream blobStream) throws DirectoryServiceException, VersionOutofSyncException
    {
        String logicalFileName = fileElement.getIdentity().getName();
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.attachFSBlob logicalFileName == " + logicalFileName);
        }
        EclipseFile eFile = new EclipseFile(logicalFileName, true);
        String mappedPath = eFile.m_diskLocation;
        if (mappedPath == null)
        {
            throw new DirectoryServiceException("Folder for " + logicalFileName + " does not exist");
        }
        File eclipseFile = new File(mappedPath);

        try
        {
            ByteArrayOutputStream oStream;
            BufferedInputStream bStream = new BufferedInputStream(blobStream, IBlob.BLOB_CHUNK_SIZE);
            int src = 0;

            byte[] blobPiece;
            int readIn = 0;
            FileOutputStream eclipseStream = new FileOutputStream(eclipseFile);
            while (readIn != -1)
            {
                int chunkIndex = 0;
                oStream = new ByteArrayOutputStream();
                while ( (chunkIndex < IBlob.BLOB_CHUNK_SIZE) && (readIn != -1))
                {
                    readIn = bStream.read();
                    if (readIn != -1)
                    {
                        oStream.write(readIn);
                        chunkIndex = chunkIndex + 1;
                    }
                }
                oStream.close();
                blobPiece = oStream.toByteArray();
                if (src > 0)
                {
                    //reopen the file for the next chunk. The first time around it is opened (above)
                    // it overwrites whatever was there
                    eclipseStream = new FileOutputStream(eclipseFile, true);
                }
                eclipseStream.write(blobPiece);
                eclipseStream.close();
                src = src + blobPiece.length;
            }
            if (fileElement instanceof Element)
            {
                newFileNotifications(logicalFileName);
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to attach blob " + logicalFileName + ": " + e.toString(), e);
        }
    }

    /**
     *
     * @param elementPath The path to an Eclipse workspace file
     * @param forUpdate Ignored, it behaves as if it was false all the time
     * @return The element that represent the file in the Eclipse workspace, or null if the file doesn't exist
     * @throws DirectoryServiceException If elementPath is a folder, not a file.
     */
    @Override
    public IDirElement getFSElement(String elementPath, boolean forUpdate) throws DirectoryServiceException
    {
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.getFSElement " + elementPath);
        }
        EclipseFile eFile = new EclipseFile(elementPath);
        String nameOnDisk = eFile.m_diskLocation;
        if (nameOnDisk == null)
         {
            return null;		// the DS returns null if it cannot find an element
        }
        File fileOnDisk = new File(nameOnDisk);
        if (!fileOnDisk.exists())
        {
            return null;
        }
        if (fileOnDisk.isDirectory())
        {
            throw new DirectoryServiceException(elementPath + " is a folder, not an element");
        }
        try
        {
            return craftNewElement(elementPath, fileOnDisk);
        }
        catch (Exception e)
        {
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.getFSElement caught exception, trace follows");
                e.printStackTrace();
            }
            throw new DirectoryServiceException(e.toString(), e);
        }
    }

    /**
     *
     * @param The path of an Eclipse workspace folder
     * @param forUpdate Ignored, behaves as if it was false always
     * @return An array of elements for all the files in a workspace folder
     * @throws DirectoryServiceException If folderPath doesn't exist, or if it's a file, not a folder
     */
    @Override
    public IDirElement[] getFSElements(String folderPath, boolean forUpdate) throws DirectoryServiceException
    {
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.getFSElements with folder "+ folderPath);
        }
        ArrayList returnList = new ArrayList();
        EclipseFile eFile = new EclipseFile(folderPath);
        String nameOnDisk = eFile.m_diskLocation;
        
        if (nameOnDisk == null )
        {
            String withSlash, withoutSlash;
            if (folderPath.endsWith(DS_FILE_SEPARATOR))
            {
                withSlash = folderPath;
                withoutSlash = folderPath.substring(0, folderPath.length() - 1);
            }
            else
            {
                withoutSlash = folderPath;
                withSlash = folderPath + DS_FILE_SEPARATOR;
            }
        	if (withSlash.equals(m_handlerName) ||withoutSlash.equals(m_handlerName))
            {
                return new IDirElement[0]; // special case for non-initialized workspace
            }
            else
            {
                throw new DirectoryServiceException("Folder "+ folderPath + " does not exist");
            }
        }
        File fileOnDisk = new File(nameOnDisk);
        if (!fileOnDisk.exists())
        {
            throw new DirectoryServiceException("Folder "+ folderPath + " does not exist");
        }
        if (fileOnDisk.isFile())
        {
            throw new DirectoryServiceException(folderPath + " is a file, not an element");
        }
        try
        {
            File[] children = fileOnDisk.listFiles();
            for (int i=0; i< children.length; i++)
            {
                if (children[i].isFile())
                {
                    returnList.add(craftNewElement(getLogicalName(children[i].getCanonicalPath(), eFile.m_projectName), children[i]));
                }
            }
        }
        catch (Exception e)
        {
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.getFSElements caught exception, trace follows");
                e.printStackTrace();
            }
            throw new DirectoryServiceException("Couldn't get all elements for folder "+ folderPath + ": " + e.toString(), e);
        }
        IDirElement[] elements = new IDirElement[returnList.size()];
        return (IDirElement[])returnList.toArray(elements);
    }


    /**
     *
     * @param elementPath The path to an Eclipse workspace file
     * @return The DS identity of the file.
     * @throws DirectoryServiceException If elementPath doesn't exist, or if it's a folder, not an element
     */
    @Override
    public IElementIdentity getFSIdentity (String elementPath) throws DirectoryServiceException
    {
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDSHandler.getFSIdentity for element "+ elementPath);
        }
        EclipseFile eFile = new EclipseFile(elementPath);
        String nameOnDisk = eFile.m_diskLocation;
        if (nameOnDisk == null)
        {
            String withSlash, withoutSlash;
            if (elementPath.endsWith(DS_FILE_SEPARATOR))
            {
                withSlash = elementPath;
                withoutSlash = elementPath.substring(0, elementPath.length() - 1);
            }
            else
            {
                withoutSlash = elementPath;
                withSlash = elementPath + DS_FILE_SEPARATOR;
            }
        	if (withSlash.equals(m_handlerName) || withoutSlash.equals(m_handlerName))
            {
                throw new DirectoryServiceException(elementPath + " is a folder, not an element");
            }
            else
            {
                throw new DirectoryServiceException("Element "+ elementPath + " does not exist");
            }
        }
        File fileOnDisk = new File(nameOnDisk);
        if (!fileOnDisk.exists())
        {
            throw new DirectoryServiceException("Element "+ elementPath + " does not exist");
        }
        if (fileOnDisk.isDirectory())
        {
            throw new DirectoryServiceException(elementPath + " is a folder, not an element");
        }
        return createElementIdentity(elementPath);
    }

    /**
     *
     * @param elementPath The path to an Eclipse workspace file
     * @param forUpdate Ignored, behaves as if it was false always
     * @param getSubclassingDelta Ignored, behaves as if it was false always
     * @return The element representing the file in the Eclipse workspace or null if it doesn't exist
     * @throws DirectoryServiceException If elementPath is a folder, not a file.
     */
    @Override
    public IDirElement getFSElement(String elementPath, boolean forUpdate, boolean getSubclassingDelta) throws DirectoryServiceException
    {
        return getFSElement(elementPath, forUpdate);
    }

    /**
     * Create an element in the DS. This method will only create the mapping element for the DS handler and throws
     * DirectoryServiceException if called for any other element.
     * @param element The Eclipse mapping element with a name that starts with the DS_HANDLER_NAME of the handler and ends in "EclipseMap"
     * @return Always returns null
     * @throws DirectoryServiceException For all elements other than the mapping element
     * @throws VersionOutofSyncException
     */
    @Override
    public INextVersionToken createFSElement(IDirElement element) throws DirectoryServiceException, VersionOutofSyncException
    {
        if (DEBUG)
        {
            System.out.println("\nAPI CALL EclipseDHandler.createFSElement");
        }
        String elName = element.getIdentity().getName();
        if (elName.endsWith(MAP_ELEMENT_NAME))
        {
            m_workspaceDiskLocation = (String)element.getAttributes().getAttribute(ECLIPSE_WORKSPACE_DISK_LOCATION_ATTR);
            try
            {
                m_workspaceDiskLocation = (new File(m_workspaceDiskLocation)).getCanonicalPath();
                m_metadataFolder = (new File(m_workspaceDiskLocation, ".metadata")).getCanonicalPath();
            }
            catch (IOException e)
            {
                throw new DirectoryServiceException(e.toString(), e);
            }
            if (DEBUG)
            {
                System.out.println("Eclipse handler setting disk location to " + m_workspaceDiskLocation + "\n");
            }
            // store it under _MFLibrary/ds_handlers/eclipse/_mapping
            ((ElementIdentity)element.getIdentity()).setName(MAPPING_STORAGE_NAME);
            if (!m_dsContext.directoryExists(MAPPING_STORAGE_DIRECTORY))
            {
                m_dsContext.createDirectory(MAPPING_STORAGE_DIRECTORY);
            }
            if (m_dsContext.getElement(MAPPING_STORAGE_NAME) != null)
            {
                m_dsContext.deleteElement(MAPPING_STORAGE_NAME);
            }
            m_dsContext.setElement(element);
        }
        else
        {
            throw new DirectoryServiceException("Operation not supported: createFSElement");
        }
        return null;
    }


    /**
     * When this handler is instantiated, it will look in the DS for a mapping element. The handler is instantiated when
     * the handler element is created in the DS and every time the DS is restarted.
     * @param dsContext Handed over by the DS when it's instantiating the handler. It is used by the handler to communicate with the DS
     * @throws DirectoryServiceException
     */
    @Override
    public void setDSContext(IDSHandlerContext dsContext) throws DirectoryServiceException
    {
        super.setDSContext(dsContext);
        IDirElement mapping = null;
        if (m_dsContext.directoryExists(MAPPING_STORAGE_DIRECTORY))
        {
            if ( (mapping = m_dsContext.getElement(MAPPING_STORAGE_NAME)) != null)
            {
                 IAttributeSet topSet = mapping.getAttributes();
                 String diskMap = (String) topSet.getAttribute(ECLIPSE_WORKSPACE_DISK_LOCATION_ATTR);
                 try
                 {
                     m_workspaceDiskLocation = (new File(diskMap)).getCanonicalPath();
                     m_metadataFolder = (new File(m_workspaceDiskLocation, ".metadata")).getCanonicalPath();
                 }
                 catch (IOException e)
                {
                    throw new DirectoryServiceException(e.toString(), e);
                }
            }
        }
    }

    private String linkedLocation(String projectName) throws Exception
    {
        String locationPath = m_workspaceDiskLocation + FILE_SEPARATOR +
          PROJECT_METADATA_DIR + FILE_SEPARATOR + projectName + FILE_SEPARATOR + ".location";
        if (DEBUG)
        {
            System.out.println("\nEclipseDSHandler.linkedLocation opening location file : " + locationPath);
        }
        File locationFile = new File(locationPath);
        if (DEBUG && locationFile.exists())
        {
            System.out.println("EclipseDSHandler.linkedLocation location file exists and is of size " +
                           locationFile.length());
        }
        if (locationFile.exists())
        {
		    DataInputStream inStream = new DataInputStream(new SafeChunkyInputStream(locationFile));
		    String location = inStream.readUTF();
		    if (DEBUG)
            {
                System.out.println("EclipseDSHandler.linkedLocation read location from file: " + location);
            }
		    inStream.close();
		    location = getLocation(location);
		    if (location == null)
            {
                return null;
            }
		    return (new File(location)).getCanonicalPath();
        }
        else
        {
            return null;
        }
    }

    /**
     * The location corresponds to the entry in the .location file for an imported project.
     * To get the actual location from it, we remove the URI_PREFIX and convert the URL path 
     * to a file path and return it to make file manipulations based on this.
     * Comment extracted from Eclipse class LocalMetaArea
     * //location format < 3.2 was a local file system OS path
	 * //location format >= 3.2 is: URI_PREFIX + uri.toString()
     * 
     * For more details Refer to Eclise file as below. 
     * Method: readPrivateDescription(...) ||  writePrivateDescription(...) 
     * Class: org.eclipse.core.internal.resources.LocalMetaArea 
     * 
     * @param location
     * @return fileLocation
     */
	public static String getLocation(String location) {
		// if we cannot recognize the location String as a linked location
		// for the project, return null. The .location file can also contain
		// linked projects for local projects.
		if(!location.startsWith(URI_PREFIX))
        {
            return null;
        }
		if(location.startsWith(URI_PREFIX)) {
			location = location.substring(URI_PREFIX.length());
		}

//		MAHESH.S : Using a URI instead of URL for getting scheme(protocol) specific parts.
//		For ex: "%20" gets auto resolved to "Spaces" for a file scheme.  

		URI uri = URI.create(location);
		String file = uri.getSchemeSpecificPart();

		return file;
		
	}

    // projectName can be null for local projects
    private String getLogicalName(String diskName, String projectName) throws Exception
    {
        if (DEBUG)
        {
            System.out.println("\nEclipseDSHandler.getLogicalName diskName "+ diskName + " m_workspaceDiskLocation " +
                                          m_workspaceDiskLocation + " projectName == " + projectName);
        }
        String replacement = null;
        if (diskName.startsWith(m_workspaceDiskLocation)) //local project
        {
            replacement = m_handlerName + diskName.substring(m_workspaceDiskLocation.length());
            replacement = replacement.replace(FILE_SEPARATOR_CHAR, IMFDirectories.MF_DIR_SEPARATOR);
            //String replacement = diskName.replaceFirst(m_workspaceDiskLocation, m_handlerName);
        }
        else
        {
        	if (DEBUG)
            {
                System.out.println("EclipseDSHandler.getLogicalName name could be a linked project or local with linked resource");
            }
            replacement = m_handlerName + IMFDirectories.MF_DIR_SEPARATOR + projectName;
            String location = linkedLocation(projectName);
            String projectSubdir = "";
            if (location != null)
            {
            	if (DEBUG)
                {
                    System.out.println("EclipseDSHandler.getLogicalName it's a linked project, location == " + location);
                }
            	if (diskName.indexOf(location) > -1) // local resource in a linked project
            	{
            		
            		if (DEBUG)
                    {
                        System.out.println("EclipseDSHandler.getLogicalName > -1 test succeeds");
                    }
	                projectSubdir = diskName.substring(location.length() + 1);
            	}
            	else
            	{
            		// linked resource in a linked project
            		projectSubdir = linkedResourceSubDir(location, diskName);
            		if (DEBUG)
                    {
                        System.out.println("EclipseDSHandler.getLogicalName > -1 test did not succeed");
                    }
            	}
            		
            }
            else
            {
            	// it is a linked resource in a workspace project, find the .project file
            	if (DEBUG)
                {
                    System.out.println("EclipseDSHandler.getLogicalName it's a linked resource in a local project");
                }
            	projectSubdir = linkedResourceSubDir(m_workspaceDiskLocation + FILE_SEPARATOR + projectName, diskName);
            }
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.getLogicalName projectSubDir == " + projectSubdir);
            }
            replacement = replacement + IMFDirectories.MF_DIR_SEPARATOR +  projectSubdir;
         }
        
        if (replacement != null)
        {
            replacement = replacement.replace(FILE_SEPARATOR_CHAR, IMFDirectories.MF_DIR_SEPARATOR);
        }
        if (DEBUG)
        {
            System.out.println("EclipseDSHandler.getLogicalName " + diskName + " returning " + replacement + "\n");
        }
        return replacement;
    }
    
    // returns the end of the logical name of a linked resource, taken from
    // the disk name, prepending the link name to the part of the path after the
    // link location.
    private String linkedResourceSubDir(String projectFileLocation, String diskName) throws IOException
    {
    	String subDir = "";
    	File projectFile = new File(projectFileLocation + FILE_SEPARATOR + PROJECT_DESCRIPTION_FILE);
    	if (projectFile.exists())
    	{
    		if (DEBUG)
            {
                System.out.println("\nEclipseDSHandler.linkedResourceLogicalName found the .project file " + projectFile.getCanonicalPath());
            }
    		HashMap linkedDescriptions = findLinkedResources(projectFile.getCanonicalPath());
    		LinkDescription linkedResDescription = findResourceInLinks(diskName, linkedDescriptions);
    		if (linkedResDescription != null)
    		{
    			String location = (new File(linkedResDescription.getLocation())).getCanonicalPath();
    			subDir =  linkedResDescription.getName() + IMFDirectories.MF_DIR_SEPARATOR + 
    							diskName.substring(location.length() + 1);
    			if (DEBUG)
                {
                    System.out.println("EclipseDShandler.linkedResourceLogicalName found linked description, subDir = " + subDir + "\n");
                }
    		}
    		else
    		{
    			if (DEBUG)
                {
                    System.out.println("EclipseDSHandler.linkedResourceLogicalName could not find the linked resource directory for " + diskName + "\n");
                };
    			// this could be an error
    		}
    	}
    	else
    	{
    		if (DEBUG)
             {
                System.out.println("\nEclipseDSHandler.linkedResourceLogicalName could not find the .project file " + projectFile.getAbsolutePath() + "\n");
                // this could be an error
            }
    	}
    	return subDir;
    }
    // returns the LinkDescription corresponding to the resource disk name
    // always compare canonical paths
    private LinkDescription findResourceInLinks(String resourceDiskName, HashMap linkedResources)
    	throws IOException
    {
    	String canonicalDiskName = (new File(resourceDiskName)).getCanonicalPath();
    	Collection linkResCol = linkedResources.values();
    	Iterator resIt = linkResCol.iterator();
    	while (resIt.hasNext())
    	{
    		LinkDescription linkDescr = (LinkDescription)resIt.next();
    		String linkLocation = linkDescr.getLocation();
    		String canonicalLinkLocation = (new File(linkLocation)).getCanonicalPath();
    		if (canonicalDiskName.startsWith(canonicalLinkLocation))
            {
                return linkDescr;
            }
    	}
    	return null;
    }

    private IElementIdentity createElementIdentity(String logicalName)
    {
        // assumes we only create elements for files
        ElementIdentity id = new ElementIdentity(logicalName, ECLIPSE_ELEMENT_TYPE, ECLIPSE_ELEMENT_RELEASE_VERSION);
        id.setCreationTimestamp(System.currentTimeMillis());
        return id;
    }

    // make up a new file element
    // assumes the file exists - that has been tested before we try to craft a new element
    private IDirElement craftNewElement(String elName, File onDiskFile) throws Exception
    {
        if (DEBUG)
        {
            System.out.println("\nEclipseDSHandler.craftNewElement " + elName + ", on disk file: " + onDiskFile);
        }
        //Because we're creating an element for a file that is "already in the DS",
        // it's ok to mark this element as isNew == false.
        //This is necessary because the combination of isNew = true and doneUpdateCalled = false
        // makes it so that we get an exception when the SMC connects directly
        // to the DS and we browse workspace files.
        // When the SMC connects to the DS through JMS, and the element is serialized,
        // the isNew flag is set to false when the element is reconstructed by readObject,
        // and so we don't have the same problem.
        IDirElement newElement = new Element(createElementIdentity(elName),false);
        IAttributeSet topSet = newElement.getAttributes();
        IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
        // test this - is it always null?
        if (systemAttrs == null)
        {
            systemAttrs = topSet.createAttributeSet(IBlob.SYSTEM_ATTRIBUTES);
        }
        if (DEBUG)
        {
            System.out.println("EclipseDSHandler.craftNewElement about to set DO_NOT_CACHE");
        }
        systemAttrs.setBooleanAttribute(IContainer.DO_NOT_CACHE_ATTR, Boolean.TRUE);
        if (DEBUG)
        {
            System.out.println("EclipseDSHandler.craftNewElement about to set LAST_MODIFIED_TIME");
        }
        topSet.setLongAttribute(LAST_MODIFIED_TIME_ATTR, new Long(onDiskFile.lastModified()));
        topSet.setLongAttribute(CREATION_TIME_ATTR, new Long(0));
        topSet.setStringAttribute(CREATED_BY_ATTR, "");
        topSet.setBooleanAttribute(HIDDEN_ATTR, new Boolean(onDiskFile.isHidden()));
        topSet.setStringAttribute(PERMISSIONS_ATTR, "");
        FileInputStream fileStream = new FileInputStream(onDiskFile);
        if (DEBUG)
        {
            System.out.println("EclipseDSHandler.craftNewElement about to set SIZE");
        }
        topSet.setIntegerAttribute(SIZE_ATTR, new Integer(fileStream.available()));
        fileStream.close();
        if (DEBUG)
        {
            System.out.println("EclipseDSHandler.craftNewElement about to return element\n");
        }
        
        return newElement;
    }

    private void  testMapInit() throws DirectoryServiceException
    {
        if (m_handlerName == null || m_workspaceDiskLocation == null)
        {
            throw new DirectoryServiceException("Eclipse handler map has not been initialized");
        }
    }
    
    // assumes an initialized workspace
    private String[] getProjects()
    {
    	if (DEBUG)
        {
            System.out.println("EclipseDSHandler.getProjects ");
        }
    	String projectsDirName = m_workspaceDiskLocation + FILE_SEPARATOR + PROJECT_METADATA_DIR;
    	File projectsDir = new File(projectsDirName);
    	if (projectsDir.isDirectory())
    	{
    		return projectsDir.list();
    	}
        else
        {
            return new String[] {};
        }
    }
    
    private boolean isAProject(String projectName)
    {
    	String[] projects = getProjects();
    	for (int i=0; i<projects.length; i++)
    	{
    		if (projectName.equalsIgnoreCase(projects[i]))
    		{
    			return true;
    		}
    	}
    	return false;
    }
    
    private ArrayList listFoldersInternal(String folderPath) throws DirectoryServiceException
    {
    	if (DEBUG)
        {
            System.out.println("listFoldersInternal " + folderPath);
        }
        ArrayList returnFolders = new ArrayList();
        EclipseFile eFile = new EclipseFile(folderPath);
        String mappedPath = eFile.m_diskLocation;
        if (mappedPath == null)
        {
            String withSlash, withoutSlash;
            if (folderPath.endsWith(DS_FILE_SEPARATOR))
            {
                withSlash = folderPath;
                withoutSlash = folderPath.substring(0, folderPath.length() - 1);
            }
            else
            {
                withoutSlash = folderPath;
                withSlash = folderPath + DS_FILE_SEPARATOR;
            }
        	if (!withSlash.equals(m_handlerName) && !withoutSlash.equals(m_handlerName))
            {
                throw new DirectoryServiceException("Folder " + folderPath + " does not exist");
            }
            else
            {
                // special case when the workspace has not been initialized, do not throw an exception
                return new ArrayList();
            }
        }
        File mappedFile = new File(mappedPath);
        if (!mappedFile.exists())
        {
            if (DEBUG)
            {
                System.out.println("\nEclipseDSHandler.listFoldersInternal "+ mappedPath + " doesn't exist");
            }
            throw new DirectoryServiceException("Folder " + folderPath + " does not exist");
        }
        if (mappedFile.isFile())
        {
            if (DEBUG)
            {
                System.out.println("\nEclipseDSHandler.listFoldersInternal " + mappedPath + " is a file, not a folder");
            }
            throw new DirectoryServiceException(folderPath + " is not a folder");
        }
        if (DEBUG)
        {
            System.out.println("\nEclipseDSHandler.listFoldersInternal about to call java File.listFiles\n");
        }
        File[] children = mappedFile.listFiles();
        if (DEBUG)
        {
            System.out.println("\nEclipseDSHandler.listFoldersInternal File.listFiles returned " + children.length + " items");
        }
        String projectName = eFile.m_projectName;
        try 
        {
			/*if (mappedFile.getCanonicalPath().equals(m_workspaceDiskLocation)) 
			{
				if (DEBUG) 
					System.out.println("EclipseDSHandler.listFoldersInternal is listing project folders");
				String[] projects = getProjects();
				for (int i=0; i<projects.length; i++)
					returnFolders.add(getMetaAttributes(m_handlerName
							+ IMFDirectories.MF_DIR_SEPARATOR + projects[i]));
			}
			else */
			for (int i = 0; i < children.length; i++) 
			{
				if (DEBUG)
                {
                    System.out.println("\nEclipseDSHandler.listFoldersInternal processing child "
                    					+ children[i].toString());
                }
				if (children[i].isDirectory() && !children[i].getCanonicalPath().equals(m_metadataFolder))
                {
                    returnFolders.add(getMetaAttributes(getLogicalName(children[i].getCanonicalPath(), projectName)));
                }				
			}	
		} 
        catch (Exception ex) 
        {
			if (DEBUG)
            {
                ex.printStackTrace();
            }
			throw new DirectoryServiceException("Could not list folders for "
					+ folderPath + ": " + ex.toString(), ex);
		}
        
        //check for linked resources, if at the top of a project that has linked resources
        addLinkedResources(eFile, returnFolders);
        //now get the linked projects, if at the top of the workspace
        addLinkedProjects(mappedPath, returnFolders);

        return returnFolders;
    }


    private void addLinkedProjects(String mappedPath, ArrayList folderList) throws DirectoryServiceException
    {
        // now get the linked projects, if at the top of the workspace
        
        if (DEBUG)
        {
            System.out.println("\nEclipseDSHandler.addLinkedProjects mappedPath == " + mappedPath + " m_workspaceDiskLocation == " + m_workspaceDiskLocation);
        }
        if (mappedPath.equals(m_workspaceDiskLocation))
        {
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.addLinkedProjects passed the test");
            }
            // check the .projects directory for lnked projects
            File projectsDir = new File(m_workspaceDiskLocation + FILE_SEPARATOR + PROJECT_METADATA_DIR);
            if (projectsDir.exists())
            {
	            File[] projects = projectsDir.listFiles();
	            for (int i = 0; i < projects.length; i++)
	            {
	                if (isLinkedDirectory(projects[i]))
	                {
	                    if (DEBUG)
                        {
                            System.out.println("EclipseDSHandler.addLinkedProjects found linked project, adding folder "+ m_handlerName + IMFDirectories.MF_DIR_SEPARATOR + projects[i].getName());
                        }
	                    folderList.add(getMetaAttributes(m_handlerName + IMFDirectories.MF_DIR_SEPARATOR + projects[i].getName()));
	                }
	            }
            }
        }
        if (DEBUG)
        {
            System.out.println();
        }        
    }

    private boolean isLinkedDirectory(File projectDir)
    {
        if (DEBUG)
        {
            System.out.println("\nEclipseDShandler.isLinkedDirectory for file " + projectDir + " will return " + (new File(projectDir, ".location")).exists() + "\n");
        }
        return (new File(projectDir, ".location")).exists();
    }

    private ArrayList listFSElementsInternal(String folderPath, String extension) throws DirectoryServiceException
    {
        if (DEBUG)
        {
            System.out.println("\nEclipseDSHandler.listFSElementsInternal with folder "+ folderPath + "\n");
        }
        ArrayList returnElements = new ArrayList();
        EclipseFile eFile = new EclipseFile(folderPath);
        String mappedPath = eFile.m_diskLocation;
        //special case when the workspace doesn't exist - as when a new SUI with a 
        // new Eclipse is installed
        if (mappedPath == null)
        {
            String withSlash, withoutSlash;
            if (folderPath.endsWith(DS_FILE_SEPARATOR))
            {
                withSlash = folderPath;
                withoutSlash = folderPath.substring(0, folderPath.length() - 1);
            }
            else
            {
                withoutSlash = folderPath;
                withSlash = folderPath + DS_FILE_SEPARATOR;
            }
        	if (withSlash.equals(m_handlerName) || withoutSlash.equals(m_handlerName))
            {
                return new ArrayList();
            }
            else
            {
                throw new DirectoryServiceException("Folder " + folderPath + " does not exist");
            }
        }
        File mappedFile = new File(mappedPath);
        if (!mappedFile.exists())
        {
            throw new DirectoryServiceException("Folder " + folderPath + " does not exist");
        }
        if (mappedFile.isFile())
        {
            throw new DirectoryServiceException(folderPath + " is not a folder");
        }
        File[] children = mappedFile.listFiles();
        String projectName = eFile.m_projectName;
        try
        {
            for (int i=0; i< children.length; i++)
            {
                if (children[i].isFile())
                {
                    if (extension == null || children[i].getName().endsWith(extension))
                    {
                        returnElements.add(getMetaAttributes(getLogicalName(children[i].getCanonicalPath(), projectName)));
                    }
                }
            }
        }
        catch (Exception e)
        {
            if (DEBUG)
            {
                System.out.println("EclipseDSHandler.listFSElementsInternal caught exception, trace follows");
                e.printStackTrace();
            }
            throw new DirectoryServiceException("Could not list files for " + folderPath + ": " + e.toString(), e);
        }
        return returnElements;
    } 
    
    HashMap findLinkedResources(String descrFile ) throws IOException
    {
    	return (new ProjectDescriptionReader().read(descrFile));
    }
    

    class EclipseFile
    {
        String m_projectName;
        String m_diskLocation;
        boolean m_inWorkspace;
        String m_logicalName;
        String m_strippedOfMap;
        String m_nameAfterProjectName = "";
        String m_linkedProjectLocation;
        String m_linkedResourceLocation;
        HashMap m_linkedResources;

        EclipseFile(String logicalName)  throws DirectoryServiceException
        {
            this(logicalName, false);
        }

        EclipseFile(String logicalName, boolean createFile)  throws DirectoryServiceException
        {
            m_logicalName = logicalName;
            try
            {
                parse(createFile);
            }catch ( DirectoryServiceException ex)
            {
                if (DEBUG)
                {
                    ex.printStackTrace();
                }
                throw ex;
            }catch (Exception ex)
            {
                if (DEBUG)
                {
                    ex.printStackTrace();
                }
                throw new DirectoryServiceException("EclipseFile constructor: Unable to parse logical name of eclipse file " + logicalName + ": " + ex.toString(), ex);
            }

        }
        
        HashMap parseLinkedResources() throws IOException
        {
        	File descrFile;
        	if (DEBUG)
            {
                System.out.println("\nEclipseFile.parseLinkedResources getting called");
            }
        	if (m_inWorkspace)
            {
                descrFile = new File(m_workspaceDiskLocation + FILE_SEPARATOR + m_projectName +
        							FILE_SEPARATOR + PROJECT_DESCRIPTION_FILE);
            }
            else
            {
                // assume we're getting here after the linked project location has been computed
        		descrFile = new File(m_linkedProjectLocation, PROJECT_DESCRIPTION_FILE);
            }
        	if (DEBUG)
            {
                System.out.println("EclipseFile.parseLinkedResources descrFile == " + descrFile.toString() + "\n");
            }
        	if (descrFile.exists())
            {
                return findLinkedResources(descrFile.getCanonicalPath());
            }
            else
            {
                return null;
            }
        }

        // return the disk location of the linked resource being parsed
        String checkLinkedResources(boolean createFile) throws IOException
        {
            if (DEBUG)
            {
                System.out.println("\nEclipseFile.checkLinkedResources called");
            }
            if (DEBUG)
            {
                System.out.println("\n m_linkedResources= "+m_linkedResources+" :: m_nameAfterProjectName= ["+m_nameAfterProjectName +"]");
            }
    		if (m_linkedResources != null && (m_nameAfterProjectName.length() > 0))
    		{
    			if (DEBUG)
                {
                    System.out.println("EclipseFile.checkLinkedResources resources are not null\n");
                }
	    		Collection descriptionsCollection = m_linkedResources.values();
	    		Iterator descrIterator = descriptionsCollection.iterator();
	    		while (descrIterator.hasNext())
	    		{
	    			LinkDescription linkDescr = (LinkDescription)descrIterator.next();
	    			String linkLocation = linkDescr.getLocation();
	    			String linkName = linkDescr.getName();
	    			int linkType = linkDescr.getType();
	    			File possibleFile;
	    			int nameAfterLinkedLocation = -1;
	    			// the link name has to be in the path
	    			// we only know how to deal with folder linked resources
	    			if (m_nameAfterProjectName.indexOf(linkName) > -1 && linkType == LinkDescription.FOLDER_TYPE)
	    			{
		    			if ((nameAfterLinkedLocation = m_nameAfterProjectName.indexOf(IMFDirectories.MF_DIR_SEPARATOR)) > -1)
                        {
                            possibleFile = new File(linkLocation, m_nameAfterProjectName.substring(nameAfterLinkedLocation + 1));
                        }
                        else
                        {
                            possibleFile = new File(linkLocation);
                        }
		    			if (possibleFile.exists() ||
		    					(createFile && possibleFile.getParentFile().exists()))
		    			{
		    				m_linkedResourceLocation = (new File(linkLocation)).getCanonicalPath();
		    				return possibleFile.getCanonicalPath();
		    			}
	    			}
	    		}
    		}
    		else
    			if (DEBUG)
                {
                    System.out.println("EclipseFile.checkLinkedResources did not find a linked resource\n");
                }
        	return null;
        }

        final void parse(boolean createFile) throws Exception
        {
            testMapInit();
            if (DEBUG)
            {
                System.out.println("\nEclipseFile.parse createFile == " + createFile);
            }
            if (DEBUG)
            {
                System.out.println("EclipseFile.parse m_logicalName == " + m_logicalName);
            }
            m_strippedOfMap = stripEclipseMap();
            if (DEBUG)
            {
                System.out.println("EclipseFile.parse m_strippedOfMap == " + m_strippedOfMap);
            }
            m_projectName = projectName(createFile);
            if (DEBUG)
            {
                System.out.println("EclipseFile.parse m_projectName == " + m_projectName);
            }
            m_inWorkspace = projectInWorkspace();
            if (DEBUG)
            {
                System.out.println("EclipseFile.parse m_inWorkspace == " + m_inWorkspace);
            }
            if (DEBUG)
            {
                System.out.println("EclipseFile.parse m_nameAfterProjectName == " + m_nameAfterProjectName);
            }
           
            
			if (m_inWorkspace)
            {
            	m_diskLocation = m_workspaceDiskLocation + FILE_SEPARATOR + m_strippedOfMap;
                if (DEBUG)
                {
                    System.out.println("EclipseFile.parse m_diskLocation in m_inWorkspace block " + m_diskLocation);
                }
            	boolean localParentFileExists = (new File(m_diskLocation)).getParentFile().exists();
            	// check for a local file first
            	m_linkedResources = parseLinkedResources();
            	if (!(new File(m_diskLocation)).exists())
            	{
                    // try finding the resource in the linked resources
                    String linkedDiskLocation = checkLinkedResources(createFile);
                    if (linkedDiskLocation != null && (new File(linkedDiskLocation)).exists())
                    {
                        m_diskLocation = linkedDiskLocation;
                    }
                    else
                        // The file was not found locally or linked. If createFile == true,
                        // and the parent path exists locally, create locally, else if
                        // createFile == true and the linked parent path exists linked, create
                        // the path in the linked area.
                    {
                        // m_diskLocation is currently set to a local path that's not found
                        if (!localParentFileExists)
                        {
                            if (linkedDiskLocation != null)
                            {
                                boolean linkedParentFileExists = (new File(linkedDiskLocation)).getParentFile().exists();
                                if (linkedParentFileExists && createFile)
                                {
                                    m_diskLocation = linkedDiskLocation;
                                }
                                else
                                {
                                    m_diskLocation = null;
                                }
                            }
                            else
                            {
                                m_diskLocation = null;
                            }
                        }
                        else if (!createFile)
                        {
                            m_diskLocation = null;
                        }
                      
                    }
            	}
            }
            /*else if (!isAProject(m_projectName))// assume m_projectName is not null at this point
            {
            	if (DEBUG) System.out.println("EclipseFile.parse did not find folder in projectInfos so returning");
            	return;			// we have a logical name that doesn't correspond to a file we know.
            } */
            else
            {
                if (DEBUG)
                {
                    System.out.println("EclipseFile.parse logicalName could be linked");
                }
                m_linkedProjectLocation = linkedLocation(m_projectName);
                m_linkedResources = parseLinkedResources();
                if(DEBUG)
                {
                    System.out.println("parsing... m_linkedResources = " + m_linkedResources);
                }
                if(DEBUG)
                {
                    System.out.println("parsing... m_linkedProjectLocation = " + m_linkedProjectLocation);
                }
                if(DEBUG)
                {
                    System.out.println("parsing... m_strippedOfMap = "+m_strippedOfMap);
                }
               
                if(DEBUG)
                {
                    System.out.println("parsing... m_strippedOfMap.indexOf(IMFDirectories.MF_DIR_SEPARATOR) = "+m_strippedOfMap.indexOf(IMFDirectories.MF_DIR_SEPARATOR));
                }
                try
                {
                    if (m_strippedOfMap.indexOf(IMFDirectories.MF_DIR_SEPARATOR) > -1)
                    {
                        m_diskLocation = m_linkedProjectLocation + FILE_SEPARATOR +
                            m_strippedOfMap.substring(m_strippedOfMap.indexOf(IMFDirectories.MF_DIR_SEPARATOR) + 1);
                    }
                    else {
                        m_diskLocation = m_linkedProjectLocation;
                    // The linked project can have linked sources. The current disk location
                    // so far is local to the linked project. Test if we need to go find the linked
                    // resource
                    }
                    
                    if(DEBUG)
                    {
                        System.out.println("parsing.... m_diskLocation = "+m_diskLocation);
                    }
                    if (m_diskLocation != null && !(new File(m_diskLocation)).exists())
                    {
                        // try finding the resource in the linked resources
                        String linkedDiskLocation = checkLinkedResources(createFile);
                        if (linkedDiskLocation != null && (new File(linkedDiskLocation)).exists())
                        {
                            m_diskLocation = linkedDiskLocation;
                        }
                        else
                            // The file was not found locally or linked in the linked project. 
                            // If createFile == true,
                            // and the parent path exists locally, create locally, else if
                            // createFile == true and the linked parent path exists linked, create
                            // the path in the linked area.
                        {
                            boolean localParentFileExists = (new File(m_diskLocation)).getParentFile().exists();
                            // m_diskLocation is currently set to a local path that's not found (in the linked project)
                            if (!localParentFileExists)
                            {
                                if (linkedDiskLocation != null)
                                {
                                    boolean linkedParentFileExists = (new File(linkedDiskLocation)).getParentFile().exists();
                                    if (linkedParentFileExists && createFile)
                                    {
                                        m_diskLocation = linkedDiskLocation;
                                    }
                                    else
                                    {
                                        m_diskLocation = null;
                                    }
                                }
                                else
                                {
                                    m_diskLocation = null;
                                }
                            }
                            else if (!createFile)
                            {
                                m_diskLocation = null;
                            }
                          
                        }
                    }
                }
                catch (Exception ioE)
                {
                    if (DEBUG) 
                    {
                    	System.out.println("EclipseFile.parse throwing exception, trace follows");
                    	ioE.printStackTrace();
                    }
                    throw new DirectoryServiceException("Unable to find path for linked project " + m_logicalName + ": " +
                                                        ioE.toString(), ioE);
                }
            }
            // if the workspace hasn't been created, as in the case of a new SUI install
            // with a new eclipse install, don't do the following
            if (m_diskLocation != null)
            {
	            if (m_diskLocation.endsWith(FILE_SEPARATOR))
                {
                    m_diskLocation = m_diskLocation.substring(0, m_diskLocation.length() - 1);
                }
	            m_diskLocation = m_diskLocation.replace(IMFDirectories.MF_DIR_SEPARATOR,
	                                            FILE_SEPARATOR_CHAR);
	            m_diskLocation = (new File(m_diskLocation)).getCanonicalPath();
            }
            if (DEBUG)
            {
                System.out.println("EclipseFile.parse disk location ==  " + m_diskLocation +"\n");
            }
        }

        // assumes m_logicalName is set
        private String stripEclipseMap()
        {
            // strips off m_eclipseMap and the trailing "/"
            String tempStripped = m_logicalName.substring(m_handlerName.length());
            if (tempStripped.length() == 0)
            {
                // special case when we're just looking at the workbench root
                return tempStripped;
            }
            return tempStripped.substring(1);
        }

        // assumes m_strippedOfMap has been set (and that depends on m_logicalName being set)
        private String projectName(boolean createFile) throws DirectoryServiceException
        {
            if (m_strippedOfMap.length() == 0)
            {
               // special case when looking at the top of the workbench
            	m_nameAfterProjectName = "";
                return "";
            }
            // or there's no traling "/"
            else if (m_strippedOfMap.indexOf(IMFDirectories.MF_DIR_SEPARATOR) == -1)
            {
                // need to figure out if it's a file at the top (without a project) or whether it's a folder (project name)
                File testFile = new File(m_workspaceDiskLocation + FILE_SEPARATOR + m_strippedOfMap);
                if (createFile || testFile.isFile()) // we could be creating a file to be saved to the top of the workspace- always a file,
                                                             // we don't create folders
                {
                	m_nameAfterProjectName = m_strippedOfMap;
                    return "";
                }
                else
                {
                	m_nameAfterProjectName = "";
                    return m_strippedOfMap;
                }
            }
            m_nameAfterProjectName = m_strippedOfMap.substring(m_strippedOfMap.indexOf(IMFDirectories.MF_DIR_SEPARATOR)+ 1);
            return m_strippedOfMap.substring(0, m_strippedOfMap.indexOf(IMFDirectories.MF_DIR_SEPARATOR));
        }

        //assumes m_projectName has been set with the name of a possible project - and that depends on 
        // m_logicalName and m_strippedOfMap being set
        private boolean projectInWorkspace()
        {
            if (m_projectName.length() == 0)
            { // special case, maybe we want to return the top level workspace files
                if (DEBUG)
                {
                    System.out.println("\nEclipseFile.projectInWorkspace m_projectName length == 0\n");
                }
                return true;
            }
           /* if (!isAProject(m_projectName))
            	return false; // it's not a project*/
            File root = new File(m_workspaceDiskLocation);
            String[] projectsInRoot = root.list();
            if (projectsInRoot != null) // for instance, a bogus folder name
            {
	            for (int i = 0; i < projectsInRoot.length; i++)
	            {
	                if (projectsInRoot[i].equalsIgnoreCase(m_projectName))
                    {
	                    // stays the same on Unix
                        return true;
                    }
	            }
            }
            return false;
        }
    }
    
    @Override
    public void subscribe(INamingListener listener)
    {
        m_notificationManager.subscribe(listener);
    }
    
    @Override
    public void doNotify()
    {
    	m_notificationManager.doNotify();
    }
    
    private final class NamingNotificationManager
    {

        INamingListener m_listener;
        ArrayList m_notifications;

        NamingNotificationManager()
        {
            m_listener = null;
            m_notifications = new ArrayList();
        }

        void reset()
        {
            m_notifications = new ArrayList();
        }

        void doNotify()
        {
            if (m_listener != null && !m_notifications.isEmpty())
            {
                m_listener.onNotifications(m_notifications);
            }

            reset();
        }

        void notifyNamingEvent(INamingNotification notification)
        {
            m_notifications.add(notification);
        }

        void notifyNamingEvents(ArrayList notifications)
        {
            for (int i = 0; i < notifications.size(); i++)
            {
                m_notifications.add(notifications.get(i));
            }
        }

        void subscribe(INamingListener listener)
        {
            m_listener = listener;
        }

        void unsubscribe()
        {
            m_listener = null;
        }

    }
}