package com.sonicsw.mf.eclipse;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.Stack;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

// Handler to parse the .project files. Initially tailored to read the
// linked resources section only. Taken from
// \org\eclipse\core\internal\resources\ProjectDescriptionReader
public class ProjectDescriptionReader extends DefaultHandler {

	public static final String LINKED_RESOURCES = "linkedResources"; //$NON-NLS-1$
	public static final String LINK = "link"; //$NON-NLS-1$
	public static final String LOCATION = "location"; //$NON-NLS-1$
	public static final String NAME = "name"; //$NON-NLS-1$
	public static final String TYPE = "type"; //$NON-NLS-1$
	public static final String PROJECT_DESCRIPTION = "projectDescription"; //$NON-NLS-1$
	
	//states
	
	protected static final int S_INITIAL = 8;
	protected static final int S_LINK = 9;
	protected static final int S_LINK_LOCATION = 10;
	protected static final int S_LINK_NAME = 11;
	protected static final int S_LINK_TYPE = 12;
	protected static final int S_LINKED_RESOURCES = 13;
	protected static final int S_PROJECT_DESC = 17;
	
	protected final StringBuffer charBuffer = new StringBuffer();

	protected Stack objectStack;
	protected HashMap m_linkedResources = null;

	protected int state = S_INITIAL;
	
	private boolean DEBUG = false;

	/**
	 * @see ContentHandler#characters(char[], int, int)
	 */
	@Override
    public void characters(char[] chars, int offset, int length) {
		//accumulate characters and process them when endElement is reached
		charBuffer.append(chars, offset, length);
	}

	/**
	 * @see ContentHandler#endElement(String, String, String)
	 */
	@Override
    public void endElement(String uri, String elementName, String qname) {
		if (DEBUG)
        {
            System.out.println("endElement " + elementName + " state == " + state);
        }
		switch (state) {
			case S_LINK :
				endLinkElement(elementName);
				break;
			case S_LINKED_RESOURCES :
				endLinkedResourcesElement(elementName);
				return;
			case S_LINK_NAME :
				endLinkName(elementName);
				break;
			case S_LINK_TYPE :
				endLinkType(elementName);
				break;
			case S_LINK_LOCATION :
				endLinkLocation(elementName);
				break;
		}
		charBuffer.setLength(0);
	}

	/**
	 * End this group of linked resources and add them to the project description.
	 */
	private void endLinkedResourcesElement(String elementName) {
		if (elementName.equals(LINKED_RESOURCES)) {
			if (DEBUG)
            {
                System.out.println("endLinkedresourcesElement " + elementName);
            }
			m_linkedResources = (HashMap) objectStack.pop();
			state = S_PROJECT_DESC;
			if (m_linkedResources.isEmpty())
            {
                return;
            }
            else
			{
				Set keySet = m_linkedResources.keySet();
				Iterator keySetIterator = keySet.iterator();
				if (DEBUG)
                {
                    System.out.println("Printing linked resources");
                }
				while (keySetIterator.hasNext())
				{
					Object key = keySetIterator.next();
					Object value = m_linkedResources.get(key);
					if (DEBUG)
                    {
                        System.out.println("key == " + key + " value == " + value);
                    }
				}
			}
		}
	}

	/**
	 * End a single linked resource and add it to the HashMap.
	 */
	private void endLinkElement(String elementName) {
		if (elementName.equals(LINK)) {
			if (DEBUG)
            {
                System.out.println("endLinkElement");
            }
			state = S_LINKED_RESOURCES;
			// Pop off the link description
			LinkDescription link = (LinkDescription) objectStack.pop();
			// Make sure that you have something reasonable
			String name = link.getName();
			int type = link.getType();
			String location = link.getLocation();
			if ((name == null) || name.length() == 0) {
				if (DEBUG)
                {
                    System.out.println("The name of the link is not defined");
                }
				return;
			}
			if (type == -1) {
				if (DEBUG)
                {
                    System.out.println("The type of the link is not defined");
                }
				return;
			}
			if ((location == null) || location.length() == 0) {
				if (DEBUG)
                {
                    System.out.println("The location of the link is not defined");
                }
				return;
			} 

			// The HashMap of linked resources is the next thing on the stack
			((HashMap) objectStack.peek()).put(link.getName(), link);
		}
	}

	private void endLinkLocation(String elementName) {
		if (elementName.equals(LOCATION)) {
			if (DEBUG)
            {
                System.out.println("endLinkLocation");
            }
			
			String newLocation = charBuffer.toString().trim();
			((LinkDescription) objectStack.peek()).setLocation(newLocation);
			
			state = S_LINK;
		}
	}

	private void endLinkName(String elementName) {
		if (elementName.equals(NAME)) {
			if (DEBUG)
            {
                System.out.println("endLinkName");
            }
			
			String newName = charBuffer.toString().trim();
			
			String oldName = ((LinkDescription) objectStack.peek()).getName();
			if (oldName.length() != 0) {
				if (DEBUG)
                {
                    System.out.println("Expected to find oldName of length 0, but found " +  oldName);
                }
			} else {
				((LinkDescription) objectStack.peek()).setName(newName);
			} 
			state = S_LINK;
		}
	}

	private void endLinkType(String elementName) {
		if (elementName.equals(TYPE)) {
			if (DEBUG)
            {
                System.out.println("endLinkType");
            }
			
			int newType = 1;
			try {
				
				newType = Integer.parseInt(charBuffer.toString().trim());
			} catch (NumberFormatException e) {
				log(e);
			}
			// objectStack has a LinkDescription on it. Set the type
			// on this LinkDescription.
			int oldType = ((LinkDescription) objectStack.peek()).getType();
			if (oldType != -1) {
				if (DEBUG)
                {
                    System.out.println("Expected to find old type == -1, found " + oldType);
                }
			} else {
				((LinkDescription) objectStack.peek()).setType(newType);
			}
			state = S_LINK;
		}
	}

	/**
	 * @see ErrorHandler#error(SAXParseException)
	 */
	@Override
    public void error(SAXParseException error) {
		log(error);
	}

	/**
	 * @see ErrorHandler#fatalError(SAXParseException)
	 */
	@Override
    public void fatalError(SAXParseException error) throws SAXException {
		
		String message = error.getMessage();
		if (DEBUG)
        {
            System.out.println("ERROR: " + message);
        }
		
		throw error;
	}

	protected void log(Exception ex) {
		
		String message = ex.getMessage();
		if (DEBUG)
        {
            System.out.println("LOG: " + message);
        }
		
	}

	private void parseProjectDescription(String elementName) {
		if (DEBUG)
        {
            System.out.println("parseProjectDescription " + elementName);
        }
		
		if (elementName.equals(LINKED_RESOURCES)) {
			// Push a HashMap to collect all the links.
			objectStack.push(new HashMap());
			state = S_LINKED_RESOURCES;
			return;
		}
	}

	public String read(InputSource input) {
		objectStack = new Stack();
		if (DEBUG)
        {
            System.out.println("read InputSource");
        }
		state = S_INITIAL;
		try {
			SAXParserFactory factory = SAXParserFactory.newInstance();
			factory.setNamespaceAware(true);
			try {
				factory.setFeature("http://xml.org/sax/features/string-interning", true); //$NON-NLS-1$
			} catch (SAXException e) {
				// In case support for this feature is removed
			}
			SAXParser parser = factory.newSAXParser();
			parser.parse(input, this);
		} catch (ParserConfigurationException e) {
			log(e);
		} catch (IOException e) {
			log(e);
		} catch (SAXException e) {
			log(e);
		} catch(RuntimeException e) {
            e.printStackTrace();
            throw e;
        } catch(Error e) {
            e.printStackTrace();
            throw e;
        }
		return "Finished parsing";
	}

	/**
	 * Reads and returns a project description stored at the given location
	 */
	public HashMap read(String location) throws IOException {
		if (DEBUG)
        {
            System.out.println("read String");
        }
		BufferedInputStream file = null;
		try {
			file = new BufferedInputStream(new FileInputStream(location));
			read(new InputSource(file));
			return m_linkedResources;
		} finally {
			if (file != null)
            {
                file.close();
            }
		}
	}
	/*
	static public void main(String[] args)
	{
		try
		{
		    new ProjectDescriptionReader().read("C:\\eclipse\\workspace\\mgmt7.0\\.project");
		    Thread.sleep(15000);
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
	} */

	/**
	 * @see ContentHandler#startElement(String, String, String, Attributes)
	 */
	@Override
    public void startElement(String uri, String elementName, String qname, Attributes attributes) throws SAXException {
		//clear the character buffer at the start of every element
		charBuffer.setLength(0);
		if (DEBUG)
        {
            System.out.println("startElement " + elementName + " state == "+ state);
        }
		switch (state) {
		case S_INITIAL :
			if (elementName.equals(PROJECT_DESCRIPTION)) {
				state = S_PROJECT_DESC;
			} else {
				throw (new SAXException("Element name does not start a project description: " +  elementName));
			}
			break;
			case S_PROJECT_DESC :
				parseProjectDescription(elementName);
				break;
			case S_LINKED_RESOURCES :
				if (elementName.equals(LINK)) {
					state = S_LINK;
					// Push place holders for the name, type and location of
					// this link.
					objectStack.push(new LinkDescription());
				}
				break;
			case S_LINK :
				if (elementName.equals(NAME)) {
					state = S_LINK_NAME;
				} else if (elementName.equals(TYPE)) {
					state = S_LINK_TYPE;
				} else if (elementName.equals(LOCATION)) {
					state = S_LINK_LOCATION;
				}
				break;
		}
	}

	/**
	 * @see ErrorHandler#warning(SAXParseException)
	 */
	@Override
    public void warning(SAXParseException error) {
		log(error);
	}
}
