/* Copyright (c) 2007 Progress Software Corporation.  All Rights Reserved. */

package com.sonicsw.mf.framework;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.comm.IGlobalComponentListener;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.common.runtime.IContainerState;

/**
 * This interface contains the following:
 * <ul>
 * <li>Container wide constants commonly used within this package and those public
 *     constants pertaining to container/delegate code that are used on the framework side
 * <li>Definitions of public methods exposed by the container implementation (generally
 *     intended for framework side usage).
 * </ul>
 */
public interface IContainer
{
    //
    // System property constants
    //

    /**
     * The system property name used to define the password for PBE decryption of bootfiles and the cache.
     * <p>
     * As an example : -Dsonicsw.mf.password=foo
     */
    public static final String MF_PASSWORD_PROPERTY = "sonicsw.mf.password";        //NOSONAR field change is not required.
    /**
     * The system property name used to define whether special (non-printable) characters will be sent to the trace log streams
     * on startup and shutdown. This allows applications monitoring the trace logs for such conditions to not have to
     * depend on particular strings.
     * <p>
     * As an example : -Dsonicsw.mf.signal=true
     * <p>
     * The default is false.
     */
    public static final String MF_SIGNAL_PROPERTY = "sonicsw.mf.signal";
    /**
     * The system property name used to indicate that the parent shell or process will restart the container if it detects the
     * restart exit code.
     * <p>
     * As an example : -Dsonicsw.mf.allowRestart=true
     * <p>
     * The default is false.
     */
    public static final String MF_ALLOW_RESTART_PROPERTY = "sonicsw.mf.allowRestart";
    /**
     * Set sonicsw.mf.configureFromCache to true to cause the container to initially configure from cache rather than
     * from the DS. That will allow for quick container startup when the DS is not available. After startup, the container will
     * keep trying to connect to the DS and reconcile the cache with any DS modifications. That feature was implemented
     * to resolve Sonic00025844
     * <p>
     * As an example : -Dsonicsw.mf.configureFromCache=true
     * <p>
     * The default is false.
     */
    public static final String CONFIGURE_FROM_CACHE_PROPERTY = "sonicsw.mf.configureFromCache";
    /**
     * The system property name used to define local classes (classpath) that should be pre-pended to every components
     * private class loader. For development/test use only.
     * <p>
     * As an example : -Dsonicsw.mf.devPrivateClasspath=..\classes
     */
    public static final String MF_DEV_PRIVATE_CLASSPATH_PROPERTY = "sonicsw.mf.devPrivateClasspath";
    /**
     * The system property name used to define local classes (classpath) that should be pre-pended to each
     * shared library class loader. For development/test use only.
     * <p>
     * As an example : -Dsonicsw.mf.devSharedClasspath=..\classes
     */
    public static final String MF_DEV_SHAREDLIBRARY_CLASSPATH_PROPERTY = "sonicsw.mf.devSharedLibraryClasspath";
    /**
     * The system property name used to define local classes (classpath) that should be pre-pended to each
     * shared library class loader. For development/test use only.
     * <p>
     * As an example : -Dsonicsw.mf.devGlobalClasspath=..\classes
     */
    public static final String MF_DEV_GLOBAL_CLASSPATH_PROPERTY = "sonicsw.mf.devGlobalClasspath";
    /**
     * The system property name used to define whether special internal functionality will be used to load multiple
     * containers (colocate) in a single executable for large scale deployment testing.
     * <p>
     * As an example : -Dsonicsw.mf.lsd.colocate=true
     * <p>
     * The default is false which means no special behavior.
     */
    public static final String MF_LSD_COLOCATE_PROPERTY = "sonicsw.mf.lsd.colocate";
    /**
     * The system property name used to used to indicate the count of multiple
     * containers (colocate) in a single executable for large scale deployment testing.
     * <p>
     * As an example : -Dsonicsw.mf.lsd.colocate.count=25
     * <p>
     * The value is only significant if MF_LSD_COLOCATE_PROPERTY is set and it is for
     * internal use only.
     */
    public static final String MF_LSD_COLOCATE_COUNT_PROPERTY = "sonicsw.mf.lsd.colocate.count";
    /**
     * The system property name used to define whether special internal functionality will be exposed for development/QA
     * purposes only.
     * <p>
     * As an example : -Dsonicsw.mf.qa=true
     */
    public static final String MF_QA_PROPERTY = "sonicsw.mf.qa";
    /**
     * The system property name used to define whether special internal functionality will be exposed for development/QA
     * purposes only. The value should indicate the number of seconds for which a new shutdown thread will be allowed to execute
     * before aborting the container run when a file name <domain>.<container>.abort (in the current working directory) is detected.
     * <p>
     * As an example : -Dsonicsw.mf.qa.abort=10
     * <p>
     * The default is 0 which means no special shutdown activity should occur.
     */
    public static final String MF_QA_ABORT_PROPERTY = "sonicsw.mf.qa.abort";
    /**
     * The system property name used to define whether the container command line should be forced on irrespective of
     * what the container configuration indicates. This is used by the Activation Daemon and the Windows Service
     * executable to allow them to issue commands to a container via a pipe.
     * <p>
     * As an example : -Dsonicsw.mf.cli=true
     * <p>
     * The default is false.
     */
    public static final String MF_CLI_PROPERTY = "sonicsw.mf.cli";
   /**
     * used by the startcontainer script to pass the container the roor directory of the installation. The container
     * uses that to resolve sonichome:/// urls.
     * <p>
     * sonichome:/// will be typically used in the container's configuration archive search path.
     * <p>
     * There is not default - runtime error occurs if sonichome:/// was specified in the configuration and "sonicsw.home" is not set
     */
    public static final String SONICSW_HOME_PROPERTY = "sonicsw.home";
    /**
     * The system property name used to define whether the container should redirect all output sent directly to
     * System.out/err should be sent to all enabled trace log destinations.
     * <p>
     * This property is used when there is volumous output due to developer tracing that would significantly decrease
     * performance due to the output being written to bottleneck destinations (e.g. the disk based trace log).
     * <p>
     * As an example : -Dsonicsw.mf.teeOutput=false
     * <p>
     * The default is true.
     */
    public static final String MF_TEEOUPUT_PROPERTY = "sonicsw.mf.teeOutput";
    /**
     * The system property name used to define the non-default location for dynamically loaded
     * native libraries. Such libraries are dynamically loaded by the component prior to use. An
     * example are native libraries loaded from JCA RARs.
     * <p>
     * The default is <cwd>\libX. The value is specified in the startcontainer.{bat|sh} script.
     * <p>
     * As an example : -Dsonicsw.mf.libX=/Sonic/Container1/libX
     */
    public static final String MF_LIBX_PROPERTY = "sonicsw.mf.libX";
    /**
     * The system property name used relax the check made in >= v7.5 for external clients spoofing
     * internal communications. Only required when permission checking is enabled in an environment
     * where there are both >= v7.5 and < v7.5 containers and spoofing is considered a risk.
     * <p>
     * The default is false. The value is specified in the startcontainer.{bat|sh} script or in
     * container "Environment" system properties.
     * <p>
     * As an example : -Dsonicsw.mf.preV75Security=true
     */
    public static final String MF_PRE75_SECURITY_PROPERTY = "sonicsw.mf.preV75Security";

    /**
     * The system property name used to enable and specify a location for the native
     * library (DLL) used on Windows as an alernative to -Xrs to prevent spurious
     * container shutdowns when the container is launched via a Windows service.
     * sonicmf.exe is responsible for setting this.
     */
    public static final String WINSVC_UTILDIR_PROPERTY = "sonicsw.mf.winsvc.utilDir";

    /**
     * Tell the launcher that was called from a SH script and should generate a SH command line even on Windows
     */
    public static final String GENERATE_SH_ON_WINDOWS = "sonicsw.mf.generate.sh";

    /**
      * Sanity check properties from the launcher to the DS component: Make sure the agree on FT role
      */
    public static final String SANITY_DS_FT_ROLE_BACKUP = "sonicsw.mf.ds.backup";
    public static final String SANITY_DS_FT_ROLE_PRIMARY = "sonicsw.mf.ds.primary";

    /**
     * Used for passing the container's classpath cached and local list from the launcher to the container
     */
    public static final String CONTAINER_CLASSPATH_PROPERTY = "sonicsw.mf.containerClasspath";

    /**
     * Used for passing cache generation properties from the script to the launcher and ContainerImpl
     */
    public static final String CACHE_GENERATION_URL_PROPERTY = "sonicsw.mf.cacheGeneration.url";
    public static final String CACHE_GENERATION_USER_PROPERTY = "sonicsw.mf.cacheGeneration.user";
    public static final String CACHE_GENERATION_PASSWORD_PROPERTY = "sonicsw.mf.cacheGeneration.password";  //NOSONAR field change is not required.

    public static final String DS_START_ACTIVE_PROPERTY = "sonicsw.mf.DS.startactive";


    /**
     * The default native library directory.
     *
     * @see #MF_LIBX_PROPERTY
     */
    public static final String DEFAULT_LIBX_DIR = "." + File.separatorChar + "libX";

    /**
     * The DS directory that holds container configurations.
     *
     * @see #MF_LIBX_PROPERTY
     */
    public static final String CONTAINERS_DIR = "/containers";

    //
    // Constants containing values of system properties
    //

    /**
     * The value of the password property.
     *
     * @see #MF_PASSWORD_PROPERTY
     */
    public static final String PASSWORD = System.getProperty(MF_PASSWORD_PROPERTY);     //NOSONAR field change is not required.
    /**
     * A flag to indicate that startup and shutdown signalling should be performed.
     *
     * @see #MF_SIGNAL_PROPERTY
     */
    public static final boolean SIGNAL_MODE = System.getProperty(MF_SIGNAL_PROPERTY, "false").equals("true");
    /**
     * The char used on the stdout to indicate the container has started.
     *
     * @see #MF_SIGNAL_PROPERTY
     */
    public static final char STARTUP_SIGNAL_CHAR = (char)8;
    /**
     * The char used on the stdout to wrap the exit code of the container and indicate the container has shutdown.
     *
     * @see #MF_SIGNAL_PROPERTY
     */
    public static final char SHUTDOWN_SIGNAL_CHAR = (char)7;
    /**
     * A flag to indicate that QA behavior is in effect.
     *
     * @see #MF_QA_PROPERTY
     */
    public static final boolean QA_MODE = (System.getProperty(MF_QA_PROPERTY, "false").equals("true") || !System.getProperty(MF_QA_ABORT_PROPERTY, "0").equals("0"));
    /**
     * The native library directory.
     *
     * @see #MF_LIBX_PROPERTY
     */
    public static final File LIBX_DIR = new File(System.getProperty(MF_LIBX_PROPERTY, DEFAULT_LIBX_DIR));
    /**
     * A flag to indicate if preV75 security should be used.
     *
     * @see #MF_PRE75_SECURITY_PROPERTY
     */
    public static final boolean PRE75_SECURITY = System.getProperty(MF_PRE75_SECURITY_PROPERTY, "false").equals("true");

    //
    // Other constants
    //

    /**
     * URL protocol for classpaths stored in  DS.
     * <p>
     * NOTE: This should be moved to a client side constants class so that the GUI can eventually use it.
     */
    public static final String DS_CLASSPATH_PROTOCOL = "sonicfs:///";
    /**
     * Delimiter between component classpath tokens stored in DS base configurations.
     * <p>
     * NOTE: This should be moved to a client side constants class so that the GUI can eventually use it.
     */
    public static final String DS_CLASSPATH_DELIMITER = ";";
    /**
     * The value of the system property "line.separator" (i.e. how the new line is formed on the execution platform).
     */
    public static final String NEWLINE = System.getProperty("line.separator");
    /**
     * The comms type is specified as one of the tokens in the subjects we use for management comms. The token used
     * for internal (container to container) comms is defined by this type.
     */
    public static final String INTERNAL_COMMS_TYPE = "mf";
    /**
     * The container startup thread name will always start with this.
     */
    public static final String CONTAINER_BOOT_THREAD_PREFIX = "Container boot thread";
    /**
     * The container startup thread name will always start with this.
     */
    public static final String CONTAINER_CACHE_GENERATION_THREAD_PREFIX = "Cache generation thread";

    /**
     * The container signal to the launcher thru this file that it just updated the cache from central management
     * connection and the launcher can configure from cache
     */
    public static final String CONFIGURE_FROM_CACHE_FILE = "configure_from_cache";

    /**
     * The container signal to the launcher thru this file that cache must be recreated before starting the container
     */
    public static final String RESET_CACHE_FILE = "reset_cache_file";

    /**
     * The file is locked while the container is running. Used to implement a method to detect shutdown that doesn't rely on unreliable log entries.
     */
    public static final String CONTAINER_RUNNING_LOCK_FILE = "container_running.lock";

    /**
     * The file contains the startup timestamp of the container. Used as an ID by shutdown to prevent an old shutdown file from shutting down the currently running container
     */
    public static final String START_TIMESTAMP_FILE = "start_timestamp";


    /**
     * The file signals to the contaiber to shutdown.
     */
    public static final String SHUTDOWN_CONTAINER_FILE = "shutdown_sonic_container";

    /**
     * The container signal to the launcher thru this file that all the runtime files in the working directory (cache, broker files etc.) must be deleted
     * before the container is restarted (the log, scripts and container.ini are not deleted).
     */
    public static final String CLEAN_RESTART_FILE = "clean_restart_file";

    /**
     * The launcher tells the container to generate cache and then restart
     */
    public static final String GENERATE_CACHE_FROM_CENTRAL_CONNECTION_PROPERTY = "sonicsw.mf.generateCacheFromConnection";

    /**
     * Connection parameters in container.ini are prefixed with "connection." or "CentralConnection."
     */
    public static final String CONNECTION_PROP_PREFIX = "connection.";

    /**
     * Connection parameters in container.ini are prefixed with "connection." or "CentralConnection."
     */
    public static final String CENTRAL_CONNECTION_PROP = "CentralConnection";

    /**
     * Connection parameters in container.ini are prefixed with "connection." or "CentralConnection."
     */
    public static final String CENTRAL_CONNECTION_PROP_PREFIX = CENTRAL_CONNECTION_PROP + ".";

    /**
     * System property parameters in container.ini are prefixed with "SystemProperty." e.g. SystemProperty.sonicsw.mq.startactive=true
     */
    public static final String SYSTEM_PROP_PREFIX = "SystemProperty.";

    /**
     * Sets by framework.agent.ci.Agent so that the expanded archived cache can be accessed later
     */
    public static final String MF_CONTAINER_CAR_ROOT_PROPERTY = "sonicsw.mf.container_archive_root";

    /**
     * Sets the SONIC_CONTAINERS directory to be used by remote clients who call the getJVMProperties
     */
    public static final String SONIC_CONTAINERS_DIR_PROPERTY = "sonicsw.mf.containers_directory";

    /**
     * The launcher version is passed as a property from the launcher scripts to the container
     */
    public  static final String CURRENT_LAUNCHER_VERSION = System.getProperty("sonicsw.launcher_version");

    /**
     * Launcher directory name is LAUNCHERS_ROOT_DIR<version>
     */
    public static final String LAUNCHERS_ROOT_DIR = "Launcher/";

    /**
     * The name of the root directory of the launcher
     */
    public static final String LAUNCHER_DIR = LAUNCHERS_ROOT_DIR+CURRENT_LAUNCHER_VERSION;

    /**
     * The name of a working directory file that points to the container log
     */
    public static final String LOG_LOCATION_FILE = "log_location";

    /**
     * The default name of the ds xml boot file
     */
    public static final String DEFAULT_DS_BOOTFILE_NAME = "ds.xml";

    /**
     * Added to Java command-line by StartContainerCommand to identify
     * the container running in the VM (mainly to assist sys admins rather
     * than for Sonic's own use?)
     */
    public static final String CONTAINER_ID_PROPERTY = "sonicsw.mf.id";

    /**
     * Added to Java command-line by StartContainerCommand to identify
     * the private hostname to be used for location transparency
     */
    public static final String CONTAINER_PRIVATE_HOST_PROPERTY = "sonicsw.mf.privateHost";

    /**
     * Added to Java command-line by StartContainerCommand to identify
     * the public hostname to be used for location transparency .. note
     * this will only be set if one can be found
     */
    public static final String CONTAINER_PUBLIC_HOST_PROPERTY = "sonicsw.mf.publicHost";

    /**
     * User can add this to a container's environment to avoid attempts
     * to query EC2 metadata (via http://169.254.169.254/latest/meta-data)
     * for details such as internal/external host info'.  The assumption
     * is then that the container is not running in EC2.
     */
    public static final String CONTAINER_SKIP_EC2_CHECKS_PROPERTY = "sonicsw.mf.skipEC2Checks";

    /**
     * Log debug tracing when querying EC2 metadata
     */
    public static final String CONTAINER_TRACE_EC2_CHECKS_PROPERTY = "sonicsw.mf.traceEC2Checks";
    
    /**
     * User can add this to a container's environment to override the
     * default timeout for EC2 metadata queries.  The value is in milliseconds.
     */
    public static final String CONTAINER_TIMEOUT_EC2_CHECKS_PROPERTY = "sonicsw.mf.timeoutEC2Checks";

    /**
     * Default timeout for EC2 checks, in milliseconds.
     */
    public static final int CONTAINER_TIMEOUT_EC2_CHECKS_DEFAULT = 2000;    
    
    //Centralized install constants - some will be eventually moved to the schema and will be
    // Generetd as mgmt api contants
    public static final String _MF_CI_CONFIG_ID_TYPE = "_MF_CI_CONFIG_ID_TYPE";
    public static final String CONFIG_ID_ELEMENT_PATH = "/" + "_MF_CONFIG_ID_ELEMENT";
    public static final String CREATE_IF_DOES_NOT_EXIST_ATTR =  "CREATE_IF_DOES_NOT_EXIST";
    public static final String WORKING_DIRECTORY_ATTR = "WORKING_DIRECTORY";
    public static final String CACHE_PASSWORD_ATTR = "CACHE_PASSWORD";      //NOSONAR field change is not required.
    public static final String CONTAINER_PATH_ATTR = "CONTAINER_PATH";
    public static final String ACTIVATION_DAEMON_PATH_ATTR = "ACTIVATION_DAEMON_PATH";
    public static final String HOST_MANAGER_PATH_ATTR = "HOST_MANAGER_PATH";
    public static final String DS_ADDRESS = "DIRECTORY SERVICE:ID=DIRECTORY SERVICE";
    public static final String CONNECTIONURLS_DEFAULT = "tcp://localhost:2506";
    public static final String _MF_CONFIG_ID_ATTR = "_MF_CONFIG_ID";
    public static final String DO_NOT_CACHE_ATTR = "DO_NOT_CACHE";


    public boolean isBooted();
    public boolean isClosing();
    public boolean isHostingComponent(String id);

    public IContainerState getContainerState();
    public IContainerIdentity getContainerIdentity();
    public ITaskScheduler getTaskScheduler();
    public INotificationPublisher getNotificationPublisher();

    /**
     * Global components are those MF components that can be globally
     * addressed within the domain and do not require definition of
     * the container in which they reside. Examples of global components
     * are the Directory Service and the Agent Manager.
     *
     * This method should be called to have the JMSConnectorServer listen for
     * requests for for the given global component instance.
     *
     * @param globalID    The globally unique ID for the component
     * @param instanceID  Optional - global components that have primary and backversions will supply
     *                    a value of PRIMARY or BACKUP (otherwise null should be supplied)
     * @param globalComponentListener Optional - callback object to handle duplicate conditions
     */
    public void addGlobalComponentSupport(String globalID, String instanceID, IGlobalComponentListener globalComponentListener)
    throws Exception;

    /**
     * This method should be called to have the JMSConnectorServer stop listening
     * for requests for for the given global component instance.
     *
     * @see #addGlobalComponentSupport(String, String)
     */
    public void removeGlobalComponentSupport(String globalID, String instanceID);

    /**
     * Used by the DS to set whether the active DS is collocated with this container.
     * Set to null to unset the local DS.
     */
    public void setLocalDS(IFrameworkComponent ds);

    /**
     * Synchronous shutdown of the container using the given exit code.
     *
     * @param exitCode A valid MF exit code
     *
     * @see com.sonicsw.mf.common.runtime.IContainerExitCodes
     */
    public void shutdown(int exitCode);

    /**
     * Synchronous shutdown of the container using the given exit code.
     *
     * @param exitCode A valid MF exit code
     * @param cleanRestart true if a clean restart follows the shutdown
     *
     * @see com.sonicsw.mf.common.runtime.IContainerExitCodes
     */
    public void shutdown(int exitCode, boolean cleanRestart);

    /**
     * Log a message to the container log(s).
     *
     * @param id May be null if you dont want any component ID identified as the source of the message
     * @param message The text message to be logged
     * @param severityLevel The severity level (as defined by Level)
     *
     * @see com.sonicsw.mf.common.runtime.Level
     */
    public void logMessage(String id, String message, int severityLevel);

    /**
     *
     * @param id May be null if you dont want any component ID identified as the source of the message
     * @param exception The exception whose stack trace will be logged
     * @param severityLevel The severity level (as defined by Level)
     *
     * @see com.sonicsw.mf.common.runtime.Level
     */
    public void logMessage(String id, Throwable exception, int severityLevel);

    /**
     *
     * @param id May be null if you dont want any component ID identified as the source of the message
     * @param message The text message to be logged
     * @param exception The exception whose stack trace will be logged
     * @param severityLevel The severity level (as defined by Level)
     *
     * @see com.sonicsw.mf.common.runtime.Level
     */
    public void logMessage(String id, String message, Throwable exception, int severityLevel);

    public IConnectorServer getConnectorServer();

    public IElement getConfigurationFromDS(String configID);

    public ArrayList prepareCacheForActivatedContainer(String adName, String containerName, String containerID, String containerWorkDirDirectoryName,
                                                  String cacheHostDirectoryName, String cachePassword, HashMap activationProps) throws Exception;
    /**
     * Make the container's cache directory available for components to
     * use to write temporary files. The assumption is that this is
     * called by components in a container and that the container cache
     * has been initialized.
     * @return The root directory to the container's cache directory
     */
    public File getCacheDirectory();
}
