package com.sonicsw.mf.framework;

import java.io.IOException;
import java.io.InputStream;
import java.util.Date;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.ILogger;
import com.sonicsw.mf.common.runtime.ICollectiveOpStatus;

/**
 * The IFrameworkComponentContext interface context for any MF framework component.
 *
 * It provides access to information and services the framework component needs
 * to run. It also acts as a factory for certain kinds of objects.
 */
public interface IFrameworkComponentContext
extends IComponentContext, ILogger
{
    /**
     * Gets the underlying container so short cuts can be done from framework components.
     */
    public IContainer getContainer();

    /**
     * Load a component.
     *
     * @param id        The unique name of the component within the container.
     * @param configID  The name of the configuration for this component as known by the
     *                  DirectoryService.
     * @param start     Start the component.
     * @param debugMask The initial setting for the components debug mask.
     */
    public void loadComponent(String id, String configID, boolean start, int debugMask)
    throws Exception;

    /**
     * Unload a component.
     *
     * @param id       The unique name of the component within the container.
     */
    public void unloadComponent(String id)
    throws Exception;

    /**
     * Framework components can invoke operations on other components in
     * the deployment (both local and remote).
     *
     * @param target        The canonical form of the target component's runtime identity
     * @param operationName The operation to be executed on the component
     * @param params        The parameters to the operation
     * @param signatue      The signature of the operation
     * @param synchronous   If true, then the operation will be invoked asynchronously and null
     *                      will be returned
     * @param timeout       The timeout for the request. Use a value of < 1 to get the default timeout value.
     *                      The timeout will only apply if the request is for a remote target (different container)
     */
    public Object invoke(String target, String operationName, Object[] params, String[] signature, boolean synchronous, long timeout)
    throws Exception;

    /**
     * Framework components can invoke operations on groups of components in
     * the deployment (both local and remote).
     *
     * @param target        An array of the canonical form of the target components runtime identities
     * @param operationName The operation to be executed on the components
     * @param params        The parameters to the operation
     * @param signatue      The signature of the operation
     * @param synchronous   If true, then the operation will be invoked asynchronously and null
     *                      will be returned
     * @param timeout       The timeout for the collection of all results
     *
     * @return If asynchronous, return null. If synchronous, the collective result of all successful
     *         invocations and/or exceptions
     */
    public ICollectiveOpStatus invoke(String[] targets, String operationName, Object[] params, String[] signature, boolean synchronous, long timeout)
    throws Exception;

    /**
     * Framework components can set attributes on other components in
     * the deployment (both local and remote).
     *
     * @param target          The canonical form of the target component's runtime identity
     * @param attributeNames  The names of the attributes to be set
     * @param attributeValues The corresponding attribute values we require
     * @param synchronous     If true, then the setting of attributes will occur asynchronously and null
     *                        will be returned
     *
     * @return The list of attributes that were successfully set
     */
    public String[] setAttributes(String target, String[] attributeNames, Object[] attributeValues, boolean synchronous)
    throws Exception;

    /**
     * Framework components can get attribute values on other components in
     * the deployment (both local and remote).
     *
     * @param source          The canonical form of the source component's runtime identity
     * @param attributeNames  The names of the attributes to be get
     *
     * @return The corresponding set of attribute values.
     */
    public Object[] getAttributeValues(String source, String[] attributeNames)
    throws Exception;

    /**
     * Schedule a task to be executed at the given date/time. This is an
     * inaccurate version of the same method in the super class, since
     * tasks will be executed from the pool of management threads (and
     * task execution can be delayed if no pooled thread is available).
     */
    @Override
    public void scheduleTask(Runnable task, Date startTime);

    /**
     * Set the value used to determine how often a notification subscription should be renewed, in milliseconds.
     *
     * @param notificationSubscriptionRenewalInterval The interval that renewals will be made.
     */
    public void setNotificationSubscriptionRenewalInterval(Long notificationSubscriptionRenewalInterval);

    /**
     * Returns the value used to determine how often a notification subscription should be renewed, in milliseconds.
     *
     * @returns The interval that renewals will be made.
     *
     * @see com.sonicsw.mf.framework.IFrameworkComponentContext#setNotificationSubscriptionRenewalInterval(Long)
     */
    public Long getNotificationSubscriptionRenewalInterval();

    /**
     * Adds an internal notification handler for the given target component. Repeated
     * calls for the same target/handler combination will replace the previous notification
     * subscription list.
     *
     * @param notificationSource The canonical form of the runtime identity for the component
     *                           from which notifications should be subscribed to.
     * @param handler            A handler to which received notifications should be passed.
     * @param notificationTypes  An array of notification types to be subscribed to (e.g.
     *                           { "system.state.online", "system.state.offline" })
     */
    public void addNotificationHandler(String notificationSource, INotificationHandler handler, String[] notificationTypes);

    /**
     * Remove an internal notification handler associated with the given target component.
     *
     * @param notificationSource The canonical form of the runtime identity for the component
     *                           from which notifications should be subscribed to.
     * @param handler            A handler to which received notifications should be passed.
     */
    public void removeNotificationHandler(String notificationSource, INotificationHandler handler);

    public void makeGlobalComponentUniquenessCall(String globalComponentID, String callID)
    throws Exception;

    public void makeGlobalComponentUniquenessCall(String globalComponentID, String callID, String instanceID)
    throws Exception;

    /**
     * Adds the classname to the list of classnames which may be
     * loaded by the framework shared class loader.
     */
    public void addSharedClassname(String classname);

    /**
     * Stores the native library (accessed via the given stream) to the given
     * filename in the container's native library directory.
     * <p>
     * If the file already exists, attempts will be made to overwrite the
     * existing file. If the overwrite fails (due to the file being in use) an
     * IOException will be thrown.
     *
     * @param filename The OS specific filename for the library.
     * @param filename A stream from which to read the library bytecode.
     */
    public void writeLibrary(String filename, InputStream stream)
    throws IOException;

    /**
     * Gets the logical name associated with the given storage name.
     *
     * @param storageName
     * @return The matching logical name, otherwise null
     */
    public String storageToLogical(String storageName);

    /**
     * Gets the fine grained security PermissionsManager for this container
     */
    public IPermissionsManager getPermissionsManager();

    /**
     * Gets the fine grained security AuditManager for this container
     */
    public IAuditManager getAuditManager();

    /**
     * Gets the Host Manager of this container
     */
    public IHostManager getHostManager();
}
