// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.agent;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Properties;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenDataException;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;

import com.sonicsw.mx.util.IEmptyArray;
import com.sonicsw.mf.comm.IConnectorClient;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IFSElementChange;
import com.sonicsw.mf.common.config.impl.AttributeSet;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.metrics.IAlert;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IValueType;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.metrics.manager.ISampledStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatisticProvider;
import com.sonicsw.mf.common.metrics.manager.StatisticsFactory;
import com.sonicsw.mf.common.runtime.ICanonicalName;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.common.runtime.IFaultTolerantState;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.AbstractFrameworkComponent;
import com.sonicsw.mf.framework.IAuditManager;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.IHostManager;
import com.sonicsw.mf.framework.IPermissionsManager;
import com.sonicsw.mf.framework.directory.DSComponent;
import com.sonicsw.mf.framework.security.AuditManager;
import com.sonicsw.mf.framework.security.PermissionsManager;
import com.sonicsw.mf.framework.util.RollingFileLogger;
import com.sonicsw.mf.interceptor.actional.soniclog.SonicLogInterceptor;
import com.sonicsw.mf.jmx.client.MFNotification;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;

public final class Agent
extends AbstractFrameworkComponent
{
    public static final String FILELOG_SUFFIX = ".log";
    static final String SONICSW_HOME = System.getProperty(IContainer.SONICSW_HOME_PROPERTY);

    private static final String DOMAIN_CONFIG_ID = "/domain/domain";
    private static final boolean m_forceCLI = (new Boolean(System.getProperty(IContainer.MF_CLI_PROPERTY, "false"))).booleanValue();
    private static final String SET_ENC_PWD_SH_SCRIPT_NAME = "set_encryption_pwd.sh";

    // life cycle notifications
    private static final String STARTUP_NOTIFICATION_TYPE = "Startup";
    private static final String SHUTDOWN_NOTIFICATION_TYPE = "Shutdown";
    private static final String LOAD_NOTIFICATION_TYPE = "Load";
    private static final String UNLOAD_NOTIFICATION_TYPE = "Unload";
    private static final String CONTAINER_STATE_NOTIFICATION_TYPE = "ContainerState";
    public static final String FAILOVER_NOTIFICATION_TYPE = "Failover";
    public static final String MANAGE_PERMISSION_DENIED_NOTIFICATION_TYPE = "ManagePermissionDenied";

    // log notifications
    private static final String LOGFAILURE_NOTIFICATION_TYPE = "Failure";
    private static final String LOGTHRESHOLD_NOTIFICATION_TYPE = "Threshold";
    public static final String LOG_MESSAGE_NOTIFICATION_TYPE = "LogMessage";

    private static final String LOG_MESSAGE_OPERATION_NAME = "logMessage";
    private static final String[] LOG_MESSAGE_OPERATION_SIGNATURE = new String[] { String.class.getName() };

    private ContainerImpl m_containerImpl;
    private String m_agentManager;

    public static final OutputStream m_stdout = System.out;

    private boolean m_commandLineEnabled = false;

    private boolean m_logToConsole = true;
    private boolean m_logToFile = true;
    private String m_defaultLogFile = null;
    private String m_logDir = null;
    private long m_logFileThreshold = IContainerConstants.LOG_FILE_SIZE_THRESHOLD_DEFAULT;
    private long m_logFileRolloverThreshold = IContainerConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_DEFAULT;
    private int m_logFileRolloverTimeInterval = IContainerConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_DEFAULT;
    private Boolean m_centrallyLogMessages = null;
    private boolean m_cliInitialized = false;
    private Boolean m_defaultActionalLogInterceptor = Boolean.FALSE;
    private boolean m_actionalLogInterceptor = m_defaultActionalLogInterceptor.booleanValue();

    private static volatile ConsoleLogger m_consoleLogger = null;
    private static volatile RollingFileLogger m_fileLogger = null;

    private TaskScheduler m_taskScheduler;
    private ContainerStateNotifier m_containerStateNotifier;
    private NotificationPublisher m_notificationPublisher;
    
    // used as mutex to serialize async configuration updates we receive from the DS
    private Object m_asyncDSUpdateReceiverLock = new Object();
    private ArrayList m_asyncDSUpdateReceivers = new ArrayList();

    // metrics
    private IMetricsRegistrar m_metricsRegistrar;
    private static final String REPLACE_ENABLED_METRICS_OPERATION_NAME = "replaceEnabledMetrics";
    private static final String[] REPLACE_ENABLED_METRICS_OPERATION_SIGNATURE = new String[] { IMetricIdentity[].class.getName() };
    private static final String REPLACE_ENABLED_ALERTS_OPERATION_NAME = "replaceEnabledAlerts";
    private static final String[] REPLACE_ENABLED_ALERTS_OPERATION_SIGNATURE = new String[] { IAlert[].class.getName() };

    // statistics
    private IStatistic m_memoryUsageStatistic;
    private IStatistic m_maxMemoryUsageStatistic;
    private IStatistic m_totalThreadsStatistic;

    // fine grained management security
    private PermissionsManager m_permissionsManager;
    private AuditManager m_auditManager;

    private boolean m_initialized = false;

    private String m_configID;
    private long m_initialConfigVersion = -1;
    private ICanonicalName m_agentName;

    public static final String AGENT_TRACE_MASK_VALUES = IConnectorClient.TRACE_MASK_VALUES + ",256=notification subscription expirations,512=configuration cache,1024=class loading failures,2048=container FT,4096=container launch,8192=permission denied";
    public static final int TRACE_DETAIL = IConnectorClient.TRACE_DETAIL;
    public static final int TRACE_SUBSCRIPTION_EXPIRATIONS = IConnectorClient.TRACE_CONNECTION_STATE_CHANGES << 1;
    public static final int TRACE_CONFIGURATION_CACHE = TRACE_SUBSCRIPTION_EXPIRATIONS << 1;
    public static final int TRACE_CLASS_LOADING_FAILURES = TRACE_CONFIGURATION_CACHE << 1;
    public static final int TRACE_CONTAINER_FT = TRACE_CLASS_LOADING_FAILURES << 1;
    public static final int TRACE_CONTAINER_LAUNCH = TRACE_CONTAINER_FT << 1;
    public static final int TRACE_PERMISSION_DENIED = TRACE_CONTAINER_LAUNCH << 1;

    private static final ArrayList ATTRIBUTE_INFOS = new ArrayList();
    private static final ArrayList OPERATION_INFOS = new ArrayList();
    private static final ArrayList NOTIFICATION_INFOS = new ArrayList();

    private static CompositeType LOG_COMPOSITE_TYPE;
    private static String[] LOG_COMPOSITE_TYPE_ITEM_NAMES;

    private boolean DEBUG = false;
    private SdfMFTracingIntegration m_SdfMFTracingIntegration;
    private HostManager m_hostManager;
    private HashSet m_nonConfiguredPropSet;

    private static Object FILE_LOCK_OBJ = new Object();
    private static Object CONSOLE_LOCK_OBJ = new Object();
    static
    {
        Method method = null;
        //
        // Attributes
        //


        // Local CLI
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("CommandLine", Boolean.class.getName(), "Flag indicating if a local command line interface is enabled.", true, true, false));
        // Log
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogToConsole", Boolean.class.getName(), "Flag indicating if messages logged via the framework should be sent to the container console.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogToFile", Boolean.class.getName(), "Flag indicating if messages logged via the framework should be sent to the log file.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFile", String.class.getName(), "The directory pathname of the container log file. Defaults to {<domain>.<container>.log} in the container's working directory.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFileSize", Long.class.getName(), "The number of bytes written to the container log file. This value gets reset to 0 when the log is cleared or archived.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFileSizeThreshold", Long.class.getName(),"Once the size of the container log file reaches this threshold or for each time the size exceeds this threshold by a further 10%, the Agent will send a warning notification.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFileRolloverSizeThreshold", Long.class.getName(),"If the current log file size equals or exceeds this threshold at midnight, then the container will roll over the old contents to an archive file and create a new empty container log file.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFileRolloverTimeInterval", Integer.class.getName(),"Time interval to rollover the log file contents.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("ActionalLogInterceptor", Boolean.class.getName(),"Flag indicating if the Sonic Logging Interceptor is enabled.", true, true, false));
        // MF Task Scheduler Threads
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("MaxThreads", Integer.class.getName(),"The maximum number of threads that the container can create to service transient management tasks.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("MinThreads", Integer.class.getName(),"The minimum number of threads that the container will cache for reuse to service transient management tasks.", true, false, false));
        // Hostname
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("Hostname", String.class.getName(),"The hostname for the container host machine.", true, false, false));
        // Container FT
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("FaultToleranceRole", String.class.getName(), "The fault tolerant role of the container (primary or backup) or null if not fault tolerant.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("FaultTolerantState", Short.class.getName(), "The current fault tolerant state of this container.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("FaultTolerantStateString", String.class.getName(), "The description of the current fault tolerant state of this container.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("AllowFailover", Boolean.class.getName(), "When this container is in a standby state and this attribute is set to 'false', fault tolerant failover to active will not occur.", true, true, false));

        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("StartupComplete", Boolean.class.getName(), "Set to true when all components have been loaded and started (if required).", true, false, false));


        //
        // Operations
        //
        MBeanParameterInfo[] mbParamInfos = null;

        //Execute diagnostics instructions
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("instructions", String.class.getName(), "One or more diagnostic instructions - an instruction per line")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("diagnose", "Execute diagnostics instructions", mbParamInfos, String.class.getName(), MBeanOperationInfo.INFO));

        // container ping
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("pingMessage", String.class.getName(), "Any string.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("ping", "Checks the container for accessibility. Returns the given string.",
                                                   mbParamInfos, String.class.getName(), MBeanOperationInfo.INFO));
        // container state
        OPERATION_INFOS.add(new MBeanOperationInfo("getContainerState", "Gets the execution state of the container and its hosted components.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, IContainerState.class.getName(), MBeanOperationInfo.INFO));
        // shutdown the container
        OPERATION_INFOS.add(new MBeanOperationInfo("shutdown", "Shutdown the Agent/Container.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));
        // restart the container
        OPERATION_INFOS.add(new MBeanOperationInfo("restart", "Restart the Agent/Container (requires support from parent shell/process).",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));

        // restart the container after deleting all the files in the working directory of the container (other than scripts, container.ini and the log)
        OPERATION_INFOS.add(new MBeanOperationInfo("cleanRestart", "Restart the Agent/Container after deleting runtime files from the container's working directory (requires support from parent shell/process).",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));

        // reload component
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("id", String.class.getName(), "The runtime identity of the component within the container to be reloaded.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("reloadComponent", "Asynchronously reload the given component ID.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));
        // File Log utility
        OPERATION_INFOS.add(new MBeanOperationInfo("clearLogFile", "Clear the log file.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("filename", String.class.getName(), "A valid filename (including path) to save the log file to.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("saveLogFile", "Saves the contents of the log file to the given file name.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("fromPosition", Long.class.getName(), "The starting position from which to read the log for the given date. If null, then the total log length minus the given readLength is assumed (i.e. the tail of the logging output). The total log length is calculated by summing the size of all log files (current + previous versions."),
            new MBeanParameterInfo("readLength", Long.class.getName(), "The maximum amount of bytes to read from the given (or assumed) starting position. The max size for this value is 1Mb. If null, 200Kb is assumed.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getLogExtract", "Reads the requested bytes from the log file and returns them as a String.",
                                                   mbParamInfos, String.class.getName(), MBeanOperationInfo.INFO));
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("fromPosition", Long.class.getName(), "The starting position from which to read the log for the given date. If null, then the total log length minus the given readLength is assumed (i.e. the tail of the logging output). The total log length is calculated by summing the size of all log files (current + previous versions."),
            new MBeanParameterInfo("readLength", Long.class.getName(), "The maximum amount of bytes to read from the given (or assumed) starting position. The max size for this value is 1Mb. If null, 200Kb is assumed.")
        };
        try
        {
            LOG_COMPOSITE_TYPE_ITEM_NAMES = new String[] { "LogExtract", "LogFileSize" };
            LOG_COMPOSITE_TYPE = new CompositeType("LogExtract", "An extract from and the length of the current container log",
                                                   LOG_COMPOSITE_TYPE_ITEM_NAMES,
                                                   new String[] { "Log extract String", "Total log size" },
                                                   new OpenType[] { SimpleType.STRING, SimpleType.LONG });
        } catch(OpenDataException e) { e.printStackTrace(); } // should not happen
        OPERATION_INFOS.add(new MBeanOperationInfo("getLogExtractAndLogFileSize", "Reads the requested bytes from the log file and returns them as a String encapsulated with the current log size. The current log size is the size of the current log file plus the sum of all available prior versions. When evaluating log output to be read, the current log file and previous versions (due to rollover activity) will be effectively viewed as a single file.",
                                                   mbParamInfos, CompositeData.class.getName(), MBeanOperationInfo.INFO));

        // Container FT: relinquish the active role to the current standby
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("seconds", Integer.class.getName(), "The minimum time the active role will be suspended.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("suspendActiveRole", "Relinquish the current active role to the the standby. For an active container, this operation attempts to relinquish the active role to the current standby. The active container will wait up to the given number of seconds for a standby to takeover the active role; if a standby does not takeover the active role in this period, the container will continue its active role.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));

        OPERATION_INFOS.add(new MBeanOperationInfo("getJVMProperties", "Returns the JVM's system properties.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Properties.class.getName(), MBeanOperationInfo.INFO));

        OPERATION_INFOS.add(new MBeanOperationInfo("downloadArchives", "Makes sure all the latest versions of archives used by components of this container are cached in the container's cache.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));

        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("archivesRootDSpath", String.class.getName(), "The Directory Service path of the root of the archives - the \"sonicfs://\" prefix can be omitted. Example: \"/Archives\""),
            new MBeanParameterInfo("archivesVersion", String.class.getName(), "The version of the archive(s). Example: \"8.0\".")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("downloadArchives", "Makes sure all the archives from the specified Directory Service path and the specified version and used by components of this container are cached in the container's cache.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));

        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("dsFilePath", String.class.getName(), "The Directory Service path of the file.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("downloadFile", "Makes sure the latest version of the specified Directory Service file is cached in the container's cache.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));


        // when internal usage flag set
        if (IContainer.QA_MODE)
        {
            // load component
            mbParamInfos = new MBeanParameterInfo[]
            {
                new MBeanParameterInfo("id", String.class.getName(), "The unique name for the component instance in the container."),
                new MBeanParameterInfo("domainName", String.class.getName(), "Not currently used."),
                new MBeanParameterInfo("configID", String.class.getName(), "The name of the component's configuration (as known by the Directory Service). Name is equivalent to that returned by IElementIdentity.getName()."),
                new MBeanParameterInfo("start", Boolean.class.getName(), "If true, the component will be automatically started after it has been loaded and initialized."),
                new MBeanParameterInfo("traceMask", Integer.class.getName(), "The initial setting for the components debug mask.")
            };
            OPERATION_INFOS.add(new MBeanOperationInfo("loadComponent", "Load a component to the container.",
                                                       mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));
            // unload component
            mbParamInfos = new MBeanParameterInfo[]
            {
                new MBeanParameterInfo("id", String.class.getName(), "The runtime identity of the component within the container to be unloaded.")
            };
            OPERATION_INFOS.add(new MBeanOperationInfo("unloadComponent", "Unload the given component ID.",
                                                       mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));
            // terminate container
            mbParamInfos = new MBeanParameterInfo[]
            {
                new MBeanParameterInfo("secondsDelay", Integer.class.getName(), "The delay (seconds) before termination.")
            };
            OPERATION_INFOS.add(new MBeanOperationInfo("terminate", "Immediately exit the container (QA mode only).",
                                                       mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));
            // trigger container log file rollover attempt
            OPERATION_INFOS.add(new MBeanOperationInfo("attemptLogFileRollover", "Attempt a container log file rollover (QA mode only).",
                                                       IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));
        }

        //
        // Notifications
        //
        String[] notifTypes = null;

        // shutdown
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            SHUTDOWN_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Agent/Container shutdown initiated."));
        // component load
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            LOAD_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Component load completed."));
        // component unload
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            UNLOAD_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Component unload completed."));
        // log failure
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.LOG_SUBCATEGORY],
            LOGFAILURE_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Failure to write to the container log file."));
        // log threshold
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.LOG_SUBCATEGORY],
            LOGTHRESHOLD_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "The size of the current container log has reached or exceeded (by a 10% increment) the configured threshold."));
        // ContainerState
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            CONTAINER_STATE_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Update of container state "));
        // container FT role change
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            FAILOVER_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Standby container has failed over to become the active container."));
        // permission denied notification
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.SECURITY_SUBCATEGORY],
            MANAGE_PERMISSION_DENIED_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "User has been denied permission to perform a management task."));
        // log message notification
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.LOG_SUBCATEGORY],
            LOG_MESSAGE_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Message has been logged to the container log. Note: The following will not generate this notification: initial container startup messages, final container shutdown messages, trace messages."));
    }

    static void redirectOutput()
    {
        // redirect the stdout/stderr - this is so that anything that is sent directly to stdout/stderr
        // (rather than using logMessage()) can be piped to any of the active log mechanisms
        // NOTE: hardcoded to support the only 2 log mechanisms for the initial MGMT 2.0 release - this
        //       will need modification in the future if we add support for custom loggers
        if (System.getProperty(IContainer.MF_TEEOUPUT_PROPERTY, "true").equals("true"))
        {
            OutputStream os = new OutputStream()
            {
                @Override
                public synchronized void write(int b)
                throws IOException
                {
                    if (Agent.m_consoleLogger != null)
                    {
                        Agent.m_consoleLogger.write(b);
                    }
                    else
                    {
                        try { Agent.m_stdout.write(b); } catch (IOException e) { }
                    }

                    RollingFileLogger fileLogger = Agent.m_fileLogger;

                    if (fileLogger != null)
                    {
                        fileLogger.write(b);
                    }
                }
                @Override
                public synchronized void write(byte[] b, int off, int len)
                throws IOException
                {
                    // is this a special signal we want to send to the console only
                    boolean signal = len > 0 && (b[0] == IContainer.STARTUP_SIGNAL_CHAR || b[0] == IContainer.SHUTDOWN_SIGNAL_CHAR);

                    if (Agent.m_consoleLogger != null)
                    {
                        // just so QA can report that some output does not conform
                        if (IContainer.QA_MODE)
                        {
                            Agent.m_consoleLogger.write('*');
                        }
                        Agent.m_consoleLogger.write(b, off, len);
                    }
                    else
                    {
                        try
                        {
                            if (IContainer.QA_MODE)
                            {
                                Agent.m_stdout.write('*');
                            }
                            Agent.m_stdout.write(b, off, len);
                        }
                        catch (IOException e) { }
                    }

                    if (!signal)
                    {
                        RollingFileLogger fileLogger = Agent.m_fileLogger;

                        if (fileLogger != null)
                        {
                            if (IContainer.QA_MODE)
                            {
                                fileLogger.write('*');
                            }
                            fileLogger.write(b, off, len);
                        }
                    }
                }
            };
            PrintStream ps = new PrintStream(os);

            System.setOut(ps);
            System.setErr(ps);
        }
    }

    public Agent()
    {
        super();

        m_SdfMFTracingIntegration = new SdfMFTracingIntegration();
        m_SdfMFTracingIntegration.register();
    }

    public IHostManager getHostManager()
    {
        return m_hostManager;
    }

    @Override
    public MBeanAttributeInfo[] getAttributeInfos() { return (MBeanAttributeInfo[])ATTRIBUTE_INFOS.toArray(IEmptyArray.EMPTY_ATTRIBUTE_INFO_ARRAY); }

    @Override
    public MBeanOperationInfo[] getOperationInfos() { return (MBeanOperationInfo[])OPERATION_INFOS.toArray(IEmptyArray.EMPTY_OPERATION_INFO_ARRAY); }

    @Override
    public MBeanNotificationInfo[] getNotificationInfos() { return (MBeanNotificationInfo[])NOTIFICATION_INFOS.toArray(IEmptyArray.EMPTY_NOTIFICATION_INFO_ARRAY); }

    public void setContainer(ContainerImpl container)
    {
        m_containerImpl = container;
        m_hostManager = m_containerImpl.getHostManager();
    }

    // IMPORTANT NOTE: if any of the default values for the runtime attributes of Agent
    // changes, make sure you also change the method handleDeletedContainerAttrs.
    // We revert back to default values there if the user deletes an Agent configuration
    // attribute

    @Override
    public void init(IComponentContext context)
    {
        // call the superclass first to do its stuff
        super.init(context);

        loadNonConfiguredINIList();

        //IContainer.DS_START_ACTIVE_PROPERTY is a one-time property - regenerating container.ini to set it to false
        if (Boolean.getBoolean(IContainer.DS_START_ACTIVE_PROPERTY))
        {
            super.m_context.logMessage(IContainer.DS_START_ACTIVE_PROPERTY + " is enabled for this start; resetting it to false in container.ini", Level.INFO);
            generateContainerINI();
        }

        m_agentManager = context.getComponentName().getDomainName() + '.' + "AGENT MANAGER" + IComponentIdentity.DELIMITED_ID_PREFIX + "AGENT MANAGER";

        // get the config and parse its info
        IElement config = context.getConfiguration(true);
        m_configID = config.getIdentity().getName();

        // extract the agent details from the container configuration
        IAttributeSet containerAttrs = config.getAttributes();

        //Console logger
        Boolean logConsole = (Boolean)containerAttrs.getAttribute(IContainerConstants.LOG_TO_CONSOLE_ATTR);
        setLogToConsole(logConsole == null ? Boolean.TRUE : logConsole);
        //File Logger
        String fileName = (String)containerAttrs.getAttribute(IContainerConstants.LOG_FILE_ATTR);
        m_defaultLogFile = getContainerName() + FILELOG_SUFFIX;
        try
        {
            Integer logFileRolloverTimeInterval = (Integer)containerAttrs.getAttribute(IContainerConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_ATTR);
            setLogFileRolloverTimeInterval(logFileRolloverTimeInterval == null ? new Integer(IContainerConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_DEFAULT) : logFileRolloverTimeInterval);
            setLogFile(fileName == null ? m_defaultLogFile : fileName);
            Boolean logToFile = (Boolean)containerAttrs.getAttribute(IContainerConstants.LOG_TO_FILE_ATTR);
            setLogToFile(logToFile == null ? Boolean.TRUE : logToFile);
            Long logFileThresholdSize = (Long)containerAttrs.getAttribute(IContainerConstants.LOG_FILE_SIZE_THRESHOLD_ATTR);
            setLogFileSizeThreshold(logFileThresholdSize == null ? new Long(IContainerConstants.LOG_FILE_SIZE_THRESHOLD_DEFAULT) : logFileThresholdSize);
            Long logFileRolloverThresholdSize = (Long)containerAttrs.getAttribute(IContainerConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_ATTR);
            setLogFileRolloverSizeThreshold(logFileRolloverThresholdSize == null ? new Long(IContainerConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_DEFAULT) : logFileRolloverThresholdSize);
            Boolean actionalLogInterceptor = (Boolean)containerAttrs.getAttribute(IContainerConstants.ACTIONAL_LOG_INTERCEPTOR_ATTR);
            setActionalLogInterceptor(actionalLogInterceptor == null ? m_defaultActionalLogInterceptor : actionalLogInterceptor);
        }
        catch(Exception e)
        {
            super.m_context.logMessage("Failed to setup container log file, trace follows...", e, Level.SEVERE);
        }
        // Centralized logging
        final Boolean centrallyLogMessages = (Boolean)containerAttrs.getAttribute(IContainerConstants.ENABLE_CENTRALIZED_LOGGING_ATTR);
        Runnable centralLoggingConfigurator = new Runnable()
        {
            @Override
            public void run()
            {
                if (Agent.this.waitForContainerToBoot())
                {
                    setCentrallyLogMessages();
                }
            }
        };
        m_context.scheduleTask(centralLoggingConfigurator, new Date());

        m_containerImpl.beginLogging();

        Boolean localCLI = (Boolean)containerAttrs.getAttribute(IContainerConstants.COMMAND_LINE_ATTR);
        try
        {
            setCommandLine(localCLI == null ? Boolean.FALSE : localCLI);
        }
        catch(MFException e)
        {
            super.m_context.logMessage(e.getMessage(), Level.WARNING);
        }

        m_taskScheduler = (TaskScheduler)super.m_container.getTaskScheduler();
        m_notificationPublisher = (NotificationPublisher)super.m_container.getNotificationPublisher();

        // metrics setup
        IMetricInfo[] agentMetricsInfo = Agent.getMetricsInfo();
        IMetricInfo[] taskSchedulerMetricsInfo = TaskScheduler.getMetricsInfo();
        IMetricInfo[] notificationPublisherMetricsInfo = NotificationPublisher.getMetricsInfo();
        IMetricInfo[] infos = new IMetricInfo[agentMetricsInfo.length + taskSchedulerMetricsInfo.length + notificationPublisherMetricsInfo.length] ;
        System.arraycopy(agentMetricsInfo, 0, infos, 0, agentMetricsInfo.length);
        System.arraycopy(taskSchedulerMetricsInfo, 0, infos, agentMetricsInfo.length, taskSchedulerMetricsInfo.length);
        System.arraycopy(notificationPublisherMetricsInfo, 0, infos, agentMetricsInfo.length + taskSchedulerMetricsInfo.length, notificationPublisherMetricsInfo.length);
        m_metricsRegistrar = super.m_context.initMetricsManagement(infos);
        initMetrics();
        m_taskScheduler.initMetrics(m_metricsRegistrar);
        m_notificationPublisher.initMetrics(m_metricsRegistrar);

        // the metrics refresh and collection intervals are maintained
        // by the metrics manager manager

        // needs to be reset, in case it was set prior to initialization
        setTraceMask(super.getTraceMask());

        m_agentName = super.m_context.getComponentName();

        m_initialized = true;
    }

    private void loadNonConfiguredINIList()
    {
        m_nonConfiguredPropSet = new HashSet();
        try
        {
            File iniPropsFile = null;
            if (IContainer.CURRENT_LAUNCHER_VERSION != null)
            {
                File launcherHomeDir = new File(new File(SONICSW_HOME), IContainer.LAUNCHER_DIR);
                iniPropsFile = new File (new File(launcherHomeDir, ContainerSetup.CONTAINER_TEMPLATES_DIR), ContainerSetup.INI_PROPS_FILE);
            }

            if (iniPropsFile != null && iniPropsFile.exists())
            {
                Properties props = ContainerUtil.readProperties(new FileInputStream(iniPropsFile));

                Enumeration iniList = props.keys();

                while (iniList.hasMoreElements())
                {
                    String prop = (String)iniList.nextElement();
                    boolean configured = new Boolean(props.getProperty(prop)).booleanValue();
                    if (!configured)
                    {
                        m_nonConfiguredPropSet.add(prop);
                    }
                }
            }
        }
        catch (IOException e)
        {
            super.m_context.logMessage("Failed to load container.ini information, trace follows...", e, Level.WARNING);
        }
    }

    @Override
    public synchronized void start()
    {
        super.start();

        Agent.this.m_containerStateNotifier = new ContainerStateNotifier(Agent.this, Agent.super.m_context);

        Runnable stateNotifierStarter = new Runnable()
        {
            @Override
            public void run()
            {
                if (Agent.this.waitForContainerToBoot())
                {
                    Agent.this.m_containerStateNotifier.start();
                }
            }
        };
        m_context.scheduleTask(stateNotifierStarter, new Date());
    }

    @Override
    public synchronized void stop()
    {
        // don't send any more container state notifications
        if (m_containerStateNotifier != null)
        {
            m_containerStateNotifier.cleanup();
        }

        // also we we don't want to send a notification when we do call stop
        super.m_state = IComponentState.STATE_OFFLINE;
    }

    @Override
    public void destroy()
    {
        // Sonic00026866
        // It is at this time unclear why on Unix platforms only, setting the context to null for the AGENT
        // component causes a JVM hang. A workaround appears to be to not allow the context to be set to null
        // by overriding the default destroy() method
    }

    @Override
    public synchronized void enableMetrics(IMetricIdentity[] ids)
    {
        // now enable the ones that the agent deals with directly
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(IAgentProxy.SYSTEM_MEMORY_CURRENTUSAGE_METRIC_ID))
            {
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_MEMORY_CURRENTUSAGE_METRIC_ID, m_memoryUsageStatistic);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_MEMORY_MAXUSAGE_METRIC_ID))
            {
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_MEMORY_MAXUSAGE_METRIC_ID, m_maxMemoryUsageStatistic);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_THREADS_CURRENTTOTAL_METRIC_ID))
            {
                m_metricsRegistrar.registerMetric(IAgentProxy.SYSTEM_THREADS_CURRENTTOTAL_METRIC_ID, m_totalThreadsStatistic);
            }
        }

        // deal with the ones that the agent delegates to the task scheduler
        m_taskScheduler.enableMetrics(ids);
        m_notificationPublisher.enableMetrics(ids);
    }

    @Override
    public synchronized void disableMetrics(IMetricIdentity[] ids)
    {
        // now disable the ones that the agent deals with directly
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(IAgentProxy.SYSTEM_MEMORY_CURRENTUSAGE_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_MEMORY_CURRENTUSAGE_METRIC_ID);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_MEMORY_MAXUSAGE_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_MEMORY_MAXUSAGE_METRIC_ID);
            }
            else if (ids[i].equals(IAgentProxy.SYSTEM_THREADS_CURRENTTOTAL_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(IAgentProxy.SYSTEM_THREADS_CURRENTTOTAL_METRIC_ID);
            }
        }

        // deal with the ones that the agent delegates to the task scheduler
        m_taskScheduler.disableMetrics(ids);
        m_notificationPublisher.disableMetrics(ids);
    }

    //
    // The management attributes we will expose (beyond or overriding that handled by
    // AbstractFrameworkComponent).
    //

    public String getHostname() { return MFNotification.HOST; }

    public Boolean getStartupComplete() { return new Boolean(m_frameworkContext.getContainer().isBooted()); }

    /**
     * The container can be configured to present a local command line interface. This
     * functionality can also be enabled or disabled at runtime.
     *
     * @return Returns true if the local command line interface is enabled.
     *
     * @see #setCommandLine(boolean)
     */
    public Boolean getCommandLine() { return new Boolean(m_commandLineEnabled); }

    /**
     * @param enabled If true, the local command line will be enabled. If false the
     *                command line will be disabled.
     *
     * @see #getCommandLine()
     */
    public synchronized void setCommandLine(Boolean enabled) throws MFException
    {
        boolean enableCommandLine = enabled.booleanValue();

        if (!enableCommandLine)
        {
            if (m_forceCLI)
            {
                if (!m_commandLineEnabled)
                {
                    enableCommandLine = true;
                }
                else
                {
                    throw new MFException("CLI cannot be disabled");
                }
            }
        }

        // if already in the desired state get out
        if (m_commandLineEnabled == enableCommandLine)
        {
            return;
        }

        if (enableCommandLine)
        {
            if (!m_cliInitialized)
            {
                try
                {
                    AgentCLI.init(this);
                }
                catch(NoSuchMethodException e)
                {
                    MFException exception = new MFException("Failed to initialize command line");
                    exception.setLinkedException(e);
                    throw exception;
                }
            }

            synchronized(super.m_container)
            {
                if (super.m_container.isBooted())
                {
                    super.m_context.logMessage("Command line started", Level.INFO);
                    AgentCLI.readParseExecute();
                }
                else // wait for boot
                {
                    Thread cliStarter = new Thread(IAgentProxy.ID + " - CLI Reader")
                    {
                        @Override
                        public void run()
                        {
                            synchronized(Agent.super.m_container)
                            {
                                while (!Agent.super.m_container.isBooted())
                                {
                                    if (Agent.super.m_container.isClosing())
                                    {
                                        return;
                                    }
                                    try
                                    {
                                        Agent.super.m_container.wait(1000);
                                    }
                                    catch (InterruptedException e)
                                    {
                                        return;
                                    }
                                }
                                AgentCLI.readParseExecute();
                            }
                        }
                    };
                    cliStarter.setDaemon(true);
                    cliStarter.start();
                }
                m_commandLineEnabled = true;
            }
        }
        else
        {
            try
            {
                AgentCLI.cancelReadParseExecute();
                m_commandLineEnabled = false;
                super.m_context.logMessage("Command line stopped", Level.INFO);
            } catch(Throwable e) { e.printStackTrace(); }
        }

    }

    //File log accessors

    public Boolean getLogToConsole() { return new Boolean(m_logToConsole); }
    public void setLogToConsole(Boolean enabled)
    {
        m_logToConsole = enabled.booleanValue();
        if (m_logToConsole)
        {
            if (m_consoleLogger == null)
            {
            	synchronized (CONSOLE_LOCK_OBJ) 
            	{
                    if (m_consoleLogger == null)
                    {
                        m_consoleLogger = new ConsoleLogger(this, m_stdout);
                        m_containerImpl.addMessageLogger(m_consoleLogger);
                        if (m_initialized)
                        {
                            super.m_context.logMessage("(Re)started logging to console", Level.INFO);
                        }
                    }
				}
            }
        }
        else
        {
           if (m_consoleLogger != null)
           {
               super.m_context.logMessage("Stopped logging to console", Level.INFO);
               m_containerImpl.removeMessageLogger(m_consoleLogger);
               m_consoleLogger = null;
           }
        }
    }

    public Boolean getLogToFile() { return new Boolean(m_logToFile); }
    public void setLogToFile(Boolean enabled)
    throws Exception
    {
        m_logToFile = enabled.booleanValue();

        if (m_logToFile)
        {
            if (m_fileLogger == null)
            {
            	synchronized (FILE_LOCK_OBJ) {
                    if (m_fileLogger == null) {
                        m_fileLogger = new RollingFileLogger(super.m_context, getLogFile(), m_containerImpl.getContainerIdentity().getCanonicalName(), m_logFileRolloverTimeInterval);
                        m_fileLogger.setSizeThreshold(m_logFileThreshold);
                        m_containerImpl.addMessageLogger(m_fileLogger);
                        if (m_initialized)
                        {
                            super.m_context.logMessage("(Re)started logging to disk", Level.INFO);
                        }
                    }
				}
            }
        }
        else
        {
            if (m_fileLogger != null)
            {
                super.m_context.logMessage("Stopped logging to disk", Level.INFO);
                m_containerImpl.removeMessageLogger(m_fileLogger);
                m_fileLogger.close();
                m_fileLogger = null;
            }
        }
    }

    public String getLogFile() { return new File(m_logDir).getAbsolutePath(); }
    public void setLogFile(String path)
    throws Exception
    {
        if (path == null || path.length() == 0)
        {
            path = m_defaultLogFile;
        }

        String oldFile = m_logDir;
        m_logDir = path;

        if (IContainer.CURRENT_LAUNCHER_VERSION != null)
        {
            if (!m_logDir.equals(m_defaultLogFile))
            {
                File logLocationFile = new File(IContainer.LOG_LOCATION_FILE);
                logLocationFile.delete();
                PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(logLocationFile, false)), true);
                writer.print(m_logDir);
                writer.close();
            }
        }

        RollingFileLogger fileLogger = m_fileLogger;

        if (m_logToFile && fileLogger != null)
        {
            try
            {
                fileLogger.resetLogDirectory(path);
            }
            catch(Exception e)
            {
                m_logDir = oldFile;
                throw e;
            }
        }
    }

    public Long getLogFileSize()
    {
        RollingFileLogger fileLogger = m_fileLogger;

        return new Long(fileLogger == null ? 0L : fileLogger.length());
    }

    public Long getLogFileSizeThreshold() { return new Long(m_logFileThreshold); }
    public void setLogFileSizeThreshold(Long thresholdSize)
    {
        RollingFileLogger fileLogger = m_fileLogger;

        if (thresholdSize.longValue() != m_logFileThreshold && fileLogger != null)
        {
            fileLogger.setSizeThreshold(thresholdSize.longValue());
        }

        m_logFileThreshold = thresholdSize.longValue();
    }

    public Long getLogFileRolloverSizeThreshold() { return new Long(m_logFileRolloverThreshold); }
    public void setLogFileRolloverSizeThreshold(Long thresholdSize)
    {
        RollingFileLogger fileLogger = m_fileLogger;

        if (thresholdSize.longValue() != m_logFileRolloverThreshold && fileLogger != null)
        {
            fileLogger.setRolloverThreshold(thresholdSize.longValue());
        }

        m_logFileRolloverThreshold = thresholdSize.longValue();
    }

    public Integer getLogFileRolloverTimeInterval() { return new Integer(m_logFileRolloverTimeInterval); }
    public void setLogFileRolloverTimeInterval(Integer timeInterval)
    {
        RollingFileLogger fileLogger = m_fileLogger;

        if (timeInterval.intValue() != m_logFileRolloverTimeInterval && fileLogger != null)
        {
            fileLogger.set_LogRolloverTimeInterval(timeInterval.intValue());
        }

        m_logFileRolloverTimeInterval = timeInterval.intValue();
    }

    public Boolean getActionalLogInterceptor() { return new Boolean(m_actionalLogInterceptor); }
    public void setActionalLogInterceptor(Boolean enabled)
    throws Exception
    {
        m_actionalLogInterceptor = enabled.booleanValue();

        if (m_actionalLogInterceptor && m_context != null)
        {
            // Check to see if there are any changes in the system properties that the interceptor
            // would be using.
            IElement config = m_context.getConfiguration(true);
            IAttributeSet containerAttrs = config.getAttributes();
            IAttributeSet systemProps = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.SYSTEM_PROPERTIES_ATTR);

            Integer maxMsgSize = null;
            Boolean disableLogOutput = null;
            if (systemProps != null && systemProps.getAttributes().entrySet().size() > 0)
            {
                // System properties are Strings, we need them in the appropriate type for the interceptor properties
                String maxMsgProp = (String)systemProps.getAttribute(SonicLogInterceptor.MAX_MSG_SIZE_ATTR);
                if (maxMsgProp != null)
                {
                    try
                    {
                        maxMsgSize = new Integer(maxMsgProp);
                    }
                    catch (NumberFormatException ex)
                    {
                        maxMsgSize = null;
                    }
                }
                else
                {
                    maxMsgSize = null;
                }

                disableLogOutput = new Boolean((String)systemProps.getAttribute(SonicLogInterceptor.DISABLE_LOG_OUTPUT_ATTR));
            }
            SonicLogInterceptor.updateLoggingProperty(SonicLogInterceptor.MAX_MSG_SIZE_ATTR, maxMsgSize);
            SonicLogInterceptor.updateLoggingProperty(SonicLogInterceptor.DISABLE_LOG_OUTPUT_ATTR, disableLogOutput);

            m_containerImpl.logMessage(null, "Sonic logging interceptor enabled", Level.INFO);
        }
    }

    // settable via container configuration
    public Integer getMaxThreads() { return new Integer(m_taskScheduler.getMaxThreads()); }
    public Integer getMinThreads() { return new Integer(m_taskScheduler.getMinThreads()); }


    //
    // The management operations we will expose (beyond or overriding that handled by
    // AbstractFrameworkComponent).
    //

    /**
     * Checks the container for accessibility.
     */
    public String ping(String pingMessage)
    {
        return pingMessage;
    }

    /**
     * Get the current aggregated state of the container and the individual state of
     * each of the components hosted by the container.
     */
    public IContainerState getContainerState()
    throws Exception
    {
        return super.m_container.getContainerState();
    }

    /**
     * Terminate the Agent/Container immediately .. for QA use only.
     */
    public void terminate(Integer secondsDelay)
    {
        Runnable terminator = new Runnable()
        {
            @Override
            public void run()
            {
                if(m_containerImpl != null)
                {
                    m_containerImpl.logMessage(null, "QA CRASH INITIATED!", Level.INFO);
                }
                Runtime.getRuntime().halt(IContainerExitCodes.UNSPECIFIED_FAILURE_EXIT_CODE);
            }
        };
        super.m_frameworkContext.scheduleTask(terminator, new Date(System.currentTimeMillis() + (secondsDelay.intValue() * 1000)));
    }

    //Executes diagnostics instructions
    public String diagnose(String instructions)
    {
        return m_containerImpl.diagnose(instructions);
    }

    /**
     * Shutdown the Agent/Container.
     */
    public void shutdown() { shutdown(IContainerExitCodes.NORMAL_SHUTDOWN_EXIT_CODE); }

    /**
     * Restart the Agent/Container.
     */
    public void restart()
    throws MFException
    {
        boolean allowRestart = Boolean.getBoolean(IContainer.MF_ALLOW_RESTART_PROPERTY);
        if (!allowRestart)
        {
            throw new MFException("Restart by container's parent shell/process is not supported");
        }

        shutdown(IContainerExitCodes.CONTAINER_RESTART_EXIT_CODE);
    }

    /**
     * Delete all the files from the working directory of the container before restarting (other than the scripts, container.ini and the log)
     */
    public void cleanRestart()
    throws MFException
    {
        boolean allowRestart = Boolean.getBoolean(IContainer.MF_ALLOW_RESTART_PROPERTY);
        if (!allowRestart)
        {
            throw new MFException("Restart by container's parent shell/process is not supported");
        }

        shutdown(IContainerExitCodes.CONTAINER_RESTART_EXIT_CODE, true);
    }

    /**
     * If fine grained security is enabled (manage policies are defined and/or an audit trail is configured),
     * then initialize the appropriate features
     *
     */
    synchronized void initFineGrainedSecurity()
    {
        // permissions must be setup first (as audit will use it)
        m_permissionsManager = new PermissionsManager((IFrameworkComponentContext)super.m_context);
        m_auditManager = new AuditManager((IFrameworkComponentContext)super.m_context, m_containerImpl.m_mbeanServer);
    }

    IPermissionsManager getPermissionsManager()
    {
        return m_permissionsManager;
    }

    IAuditManager getAuditManager()
    {
        return m_auditManager;
    }

    /**
     * Internal use - Shutdown the Agent/Container.
     */
    void shutdown(int exitCode)
    {
        shutdown(exitCode, false);
    }

    /**
     * Internal use - Shutdown the Agent/Container.
     */
    void shutdown(final int exitCode, final boolean cleanRestart)
    {
        // get out if the container is already shutting down
        if (Agent.super.m_container.isClosing())
        {
            return;
        }

        // don't wait for a pooled thread to become available
        new Thread(IAgentProxy.ID + " - Shutdown Handler")
        {
            @Override
            public void run()
            {
                Agent.super.m_container.shutdown(exitCode, cleanRestart);
            }
        }.start();
    }

    // Checks the  accessibility of a remote container
    String remotePing(String containerName, long timeout)
    {
        try
        {
            String retValue = m_containerImpl.remotePing(containerName, "PING", timeout);
            if (retValue != null && retValue.equals("PING"))
            {
                return "Ping succeeded.";
            }
            else
            {
                return "Ping failed.";
            }
        }
        catch (InvokeTimeoutException e)
        {
            return "Destination container unreachable.";
        }
        catch (Exception e)
        {
            return "Ping failed: " + e.toString();
        }
    }


    /**
     * Reload the given component.
     *
     * @param id         The unique name for the component instance in the container.
     *
     * @see #unloadComponent(String)
     * @see com.sonicsw.mf.common.config.IElementIdentity#getName()
     */
    public void reloadComponent(String id)
    throws Exception
    {
        if (id.equals(IAgentProxy.ID) || id.equals(DSComponent.GLOBAL_ID))
        {
            throw new MFException("Cannot reload ID=" + id);
        }
        m_containerImpl.reloadComponent(id);
    }

    /**
     * Reads the requested bytes from the log file and returns them as a String.
     * <p>
     * When evaluating log output to be read, the current log file and previous versions (due to
     * rollover activity) will be effectively viewed as a single file.
     *
    * @param fromPosition The starting position from which to read the log for the given date.
     *                     If null, then the total log length minus the given readLength is
     *                     assumed (i.e. the tail of the logging output). The total log length is
     *                     calculated by summing the size of all log files (current + previous
     *                     versions.
     * @param readLength   The maximum amount of bytes to read from the given (or assumed) starting
     *                     position. The max size for this value is 1Mb. If null, 200Kb is assumed.
     *
     * @return A String representation of the actual bytes read (which may be less than the
     *         requested readLength).
     */
    public String getLogExtract(Long fromPosition, Long readLength)
    {
        if (fromPosition == null)
         {
            fromPosition = new Long(-1); // will cause read to be relative to end of file
        }
        if (readLength == null)
         {
            readLength = new Long(-1); // will cause default read length to be used
        }

        RollingFileLogger fileLogger = m_fileLogger;
        if (m_logToFile && fileLogger != null)
        {
            try
            {
                return new String(fileLogger.read(fromPosition.longValue(), readLength.longValue()));
            }
            catch (IOException e)
            {
                MFRuntimeException mfe = new MFRuntimeException("Failed to read container log file");
                mfe.setLinkedException(e);
                throw mfe;
            }
        }

        return "<log to file disabled>";
    }

    /**
     * Reads the requested bytes from the log file and returns them as a String encapsulated with
     * the current log size. The current log size is the size of the current log file plus the sum
     * of all available prior versions.
     * <p>
     * When evaluating log output to be read, the current log file and previous versions (due to
     * rollover activity) will be effectively viewed as a single file.
     *
     * @param fromPosition The starting position from which to read the log for the given date.
     *                     If null, then the total log length minus the given readLength is
     *                     assumed (i.e. the tail of the logging output). The total log length is
     *                     calculated by summing the size of all log files (current + previous
     *                     versions.
     * @param readLength   The maximum amount of bytes to read from the given (or assumed) starting
     *                     position. The max size for this value is 1Mb. If null, 200Kb is assumed.
     *
     * @return A javax.management.CompositeData object containing key="LogExtract",
     *         value=<log extract String> and key="LogFileSize", value=<total log size Long>.
     *         The log extract is a String representation of the actual bytes read (which may be
     *         less than the requested readLength).
     */
    public CompositeData getLogExtractAndLogFileSize(Long fromPosition, Long readLength)
    {
        RollingFileLogger fileLogger = m_fileLogger;
        if (m_logToFile && fileLogger != null)
        {
            try
            {
                Object[] itemValues = new Object[]
                {
                    getLogExtract(fromPosition, readLength),
                    new Long(m_logToFile ? m_fileLogger.length() : 0)
                };
                return new CompositeDataSupport(LOG_COMPOSITE_TYPE, LOG_COMPOSITE_TYPE_ITEM_NAMES, itemValues);
            }
            catch (OpenDataException e)
            {
                super.m_context.logMessage("Failed to create container log extract, trace follows...", e, Level.WARNING);
                return null;
            }
        }


        try
        {
            Object[] itemValues = new Object[]
            {
                "<log to file disabled>",
                new Long(0)
            };
            return new CompositeDataSupport(LOG_COMPOSITE_TYPE, LOG_COMPOSITE_TYPE_ITEM_NAMES, itemValues);
        }
        catch (OpenDataException e)
        {
            super.m_context.logMessage("Failed to create container log extract, trace follows...", e, Level.WARNING);
            return null;
        }
    }


    /**
     * *** FOR QA/INTERNAL USAGE ONLY ***
     *
     * Load the given component.
     *
     * @param id         The unique name for the component instance in the container.
     * @param domainName Not currently used.
     * @param configID   The name of the component's configuration (as known by the
     *                   Directory Service). Name is equivalent to that returned by
     *                   IElementIdentity.getName().
     * @param start      If true, the component will be automatically started after it
     *                   has been loaded and initialized.
     * @param traceMask  The initial setting for the components debug mask.
     *
     * @see #unloadComponent(String)
     * @see com.sonicsw.mf.common.config.IElementIdentity#getName()
     */
    public void loadComponent(String id, String domainName, String configID, Boolean start, Integer traceMask)
    throws Exception
    {
        super.m_frameworkContext.loadComponent(id, configID, start.booleanValue(), traceMask.intValue());
    }

    /**
     * *** FOR QA/INTERNAL USAGE ONLY ***
     *
     * Unload the named component. If the component is in an online state, it will be
     * stopped prior to destroying and unloading.
     *
     * @param id The unique name for the component instance in the container.
     *
     * @see #loadComponent(String, String, String, boolean)
     */
    public void unloadComponent(String id)
    throws Exception
    {
        if (id.equals(IAgentProxy.ID) || id.equals(DSComponent.GLOBAL_ID))
        {
            throw new MFException("Cannot unload ID=" + id);
        }
        else
        {
            super.m_frameworkContext.unloadComponent(id);

            // now remove any runtime information accumulated in the cache by the component
            m_containerImpl.deleteRuntimeConfiguration(id);
        }
    }


    //
    // Other public methods.
    //

    public void setInitialConfigVersion(long version)
    {
        if (m_initialConfigVersion > -1)
        {
            throw new IllegalStateException("Initial config version already set");
        }
        m_initialConfigVersion = version;
    }

    // handles changes to an existing element asynchronously reported by the DS
    public void receiveChangedElement(IBasicElement element)
    {
        if (waitForContainerToBoot())
        {
            if ((m_traceMask & TRACE_CONFIGURATION_CACHE) > 0 && (m_traceMask & TRACE_DETAIL) > 0)
            {
                m_container.logMessage(null, "Received 1 configuration update", Level.TRACE);
            }
            
            waitForDSUpdateReceiverTurn();
            try
            {
                synchronized (m_asyncDSUpdateReceiverLock)
                {
                    internalReceiveChangedElement(element);
                }
            }
            finally
            {
                notifyNextDSUpdateReceiver();
            }
        }
    }

    // handles changes to existing elements asynchronously reported by the DS
    public void receiveChangedElements(IBasicElement[] elements)
    {
        if (waitForContainerToBoot())
        {
            if ((m_traceMask & TRACE_CONFIGURATION_CACHE) > 0 && (m_traceMask & TRACE_DETAIL) > 0)
            {
                m_container.logMessage(null, "Received " + elements.length + " configuration update(s)", Level.TRACE);
            }
            
            waitForDSUpdateReceiverTurn();
            try
            {
                synchronized (m_asyncDSUpdateReceiverLock)
                {
                    for (int i = 0; i < elements.length; i++)
                    {
                        internalReceiveChangedElement(elements[i]);
                    }
                }
            }
            finally
            {
                notifyNextDSUpdateReceiver();
            }
        }
    }

    // handles new elements asynchronously reported by the DS
    public void receiveChangedElements(IDirElement[] elements)
    {
        if (waitForContainerToBoot())
        {
            if ((m_traceMask & TRACE_CONFIGURATION_CACHE) > 0 && (m_traceMask & TRACE_DETAIL) > 0)
            {
                m_container.logMessage(null, "Received " + elements.length + " new configuration(s)", Level.TRACE);
            }

            waitForDSUpdateReceiverTurn();
            try
            {
                synchronized (m_asyncDSUpdateReceiverLock)
                {
                    m_containerImpl.reportChangedElements(elements);
                }
            }
            finally
            {
                notifyNextDSUpdateReceiver();
            }
        }
    }

    private void internalReceiveChangedElement(IBasicElement element)
    {
        m_containerImpl.reportChangedElement(element);
    }

    private void waitForDSUpdateReceiverTurn()
    {
        // use a crude queuing mechanism to ensure that if the changes have been received in order to
        // here, then they remain in order .. note don't need to limit the queue as the update threads
        // will be limited by the management thread pool
        synchronized (m_asyncDSUpdateReceivers)
        {
            m_asyncDSUpdateReceivers.add(Thread.currentThread());
            // wait until its at the head of the queue
            while (m_asyncDSUpdateReceivers.get(0) != Thread.currentThread())
            {
                try
                {
                    m_asyncDSUpdateReceivers.wait();
                }
                catch (InterruptedException e) { }
            }
        }
    }

    private void notifyNextDSUpdateReceiver()
    {
        // let the next ordered update thread know its time for it to apply its changes
        synchronized (m_asyncDSUpdateReceivers)
        {
            // remove the current thread
            m_asyncDSUpdateReceivers.remove(0);
            // notify other threads to see if its their turn
            m_asyncDSUpdateReceivers.notifyAll();
        }
    }

    private boolean waitForContainerToBoot()
    {
        synchronized (Agent.super.m_container)
        {
            while (!Agent.super.m_container.isBooted())
            {
                if (super.m_container.isClosing())
                {
                    return false;
                }

                try
                {
                    Agent.super.m_container.wait(1000);
                }
                catch (InterruptedException e)
                {
                    return false;
                }
            }

            return !super.m_container.isClosing();
        }
    }

    /**
     * Used to stop a component from the CLI.
     */
    public void stopComponent(String id)
    throws Exception
    {
        if (id.equals(IAgentProxy.ID))
        {
            throw new MFException("Cannot stop ID=" + id);
        }
        else
        {
            super.m_frameworkContext.invoke(getContainerName() + ':' + IComponentIdentity.ID_PREFIX + id, "stop", new Object[] {}, new String[] {}, true, 0);
        }
    }

    /**
     * Used to start a component from the CLI.
     */
    public void startComponent(String id)
    throws Exception
    {
        super.m_frameworkContext.invoke(getContainerName() + ':' + IComponentIdentity.ID_PREFIX + id, "start", new Object[] {}, new String[] {}, true, 0);
    }

    /**
     * Used to clear the errors of a component from the CLI.
     */
    public void clearError(String id)
    throws Exception
    {
        super.m_frameworkContext.invoke(getContainerName() + ':' + IComponentIdentity.ID_PREFIX + id, "clearError", new Object[] {}, new String[] {}, true, 0);
    }

    //File log
    public void clearLogFile()
    {
        RollingFileLogger fileLogger = m_fileLogger;
        if (!m_logToFile || fileLogger == null)
        {
            return;
        }
        try
        {
            m_fileLogger.clearLogFile();
        }
        catch(IOException e)
        {
            MFRuntimeException runtimeException = new MFRuntimeException("Failed to clear container log file");
            runtimeException.setLinkedException(e);
            throw runtimeException;
        }
        super.m_context.logMessage("Container log file truncated", Level.INFO);
    }

    public void saveLogFile(String path)
    {
        RollingFileLogger fileLogger = m_fileLogger;

        if (!m_logToFile || fileLogger == null)
        {
            return;
        }

        try
        {
            fileLogger.saveLogFile(path);
        }
        catch(IOException e)
        {
            MFRuntimeException runtimeException = new MFRuntimeException("Failed to save container log file to " + path);
            runtimeException.setLinkedException(e);
            throw runtimeException;
        }
        super.m_context.logMessage("Container log file copied to " + path, Level.INFO);
    }

    public void attemptLogFileRollover()
    throws IOException
    {
        RollingFileLogger fileLogger = m_fileLogger;

        if (!m_logToFile || fileLogger == null)
        {
            return;
        }

        super.m_context.logMessage("Manual rollover attempt of container log file initiated", Level.TRACE);
        fileLogger.attemptLogFileRollover();
    }

    ///////////// Container FT ///////////////////

    //runtime attributes
    public String getFaultToleranceRole()
    {
        return m_containerImpl.isFTContainer() ? m_containerImpl.getContainerFT().getRole() : "";
    }

    public Short getFaultTolerantState()
    {
        return new Short(m_containerImpl.isFTContainer() ? m_containerImpl.getContainerFT().getState() : IFaultTolerantState.STATE_NOT_FAULT_TOLERANT);
    }

    public String getFaultTolerantStateString()
    {
        return IFaultTolerantState.STATE_TEXT[m_containerImpl.isFTContainer() ? m_containerImpl.getContainerFT().getState() : IFaultTolerantState.STATE_NOT_FAULT_TOLERANT];
    }

    public Boolean getAllowFailover()
    {
        return m_containerImpl.getAllowFailover();
    }

    public void setAllowFailover(Boolean allowFailover)
    {
        m_containerImpl.setAllowFailover(allowFailover);
    }

    // notification
    void sendFailoverNotification()
    {
        INotification notification = m_context.createNotification(INotification.SYSTEM_CATEGORY,
                                                                  INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
                                                                  FAILOVER_NOTIFICATION_TYPE,
                                                                  Level.WARNING);
        notification.setLogType(INotification.WARNING_TYPE);
        notification.setAttribute("FaultToleranceRole", m_containerImpl.getContainerFT().getRole());
        super.m_context.sendNotification(notification);
    }


    //operation
    /**
     * For an active container, this operation attempts to relinquish the active role to
     * the current standby.  The active container will wait up the given number of
     * seconds for a standby to takeover the active role.  If a standby does not
     * takeover the active role in this period, the container will continue its active role.
     * Note: The general use of this operation will be to failback a recovered container,
     * rather than to force failover to a standby.
     *
     * @param waitSeconds The time in seconds the active container will wait up to.
     *
     */

    public void suspendActiveRole(final Integer waitSeconds)
    throws MFException
    {
        short currentState = m_containerImpl.getContainerFT().getState();
        if (currentState != IFaultTolerantState.STATE_ACTIVE)
        {
            throw new MFException("Cannot suspend \"" + IFaultTolerantState.STATE_TEXT[IFaultTolerantState.STATE_ACTIVE] +
                                  "\" state when container is in a \"" + IFaultTolerantState.STATE_TEXT[currentState] +
                                  "\" state.");
        }

        Runnable suspender = new Runnable()
        {
            @Override
            public void run()
            {
                long waitTime ;
                if (waitSeconds == null || waitSeconds.intValue() == 0)
                {
                    waitTime = m_containerImpl.getContainerFT().getFaultDetectionInterval();
                }
                else
                {
                    waitTime = waitSeconds.intValue() * 1000;
                }

                long startTime = System.currentTimeMillis();

                PingThread pingThread = m_containerImpl.getPingThread();

                if (pingThread != null)
                {
                    m_containerImpl.logMessage(null, "Suspending \"" + IFaultTolerantState.STATE_TEXT[IFaultTolerantState.STATE_ACTIVE] + "\" state for " + waitSeconds + " seconds...", Level.WARNING);
                    pingThread.suspendActiveState();

                    while (true)
                    {
                        try { Thread.sleep(2500); } catch(InterruptedException e) { }

                        // is the container closing in the meantime?
                        if (m_containerImpl.isClosing())
                        {
                            break;
                        }

                        if (System.currentTimeMillis() - startTime > waitTime)
                        {
                            if (pingThread != null)
                            {
                                m_containerImpl.logMessage(null, "...suspension complete", Level.INFO);
                                pingThread.resumeAfterSuspend();
                            }
                            break;
                        }
                    } //while true
                }
            } //run()
        };

        super.m_context.scheduleTask(suspender, new Date());

    }

    /////////////////////// end of Container FT ////////////////////////


    /////////////////////// Misc ////////////////////////
    /**
     * Get the system properties of the container's JVM
     */
    public Properties getJVMProperties()
    throws MFException
    {
        return m_hostManager.getJVMProperties();
    }

    /**
     * Makes sure all the latest versions of the archives are cached. NOTE: Will not downalod archives that were replaced (archive path was not modified) if notifications were
     * suspended (DSCompoenet.suspendChangeNotifications was called) prior to replacing the archive(s).
     */
    public void downloadArchives()
    throws Exception
    {
        super.m_context.logMessage("downloadArchives was called: Updating archives... ", Level.INFO);
        m_containerImpl.cacheAllArchivesIfNeeded();
        super.m_context.logMessage("...archives update is done", Level.INFO);
    }

    /**
     * Makes a list of all the archives used by this container. Then,gets the new version of the archives, from the specified dsArchivePath0 DS location.
     * For example, if the container uses MF/8.0/MFcontainer.car and downloadArchives("/Archives", "8.1") is called - fetches from the DS
     * /Archives/MF/8.1/MFcontainer.car and stores in cache. NOTE: Will not downalod archives that were replaced (archive path was not modified) if notifications were
     * suspended (DSCompoenet.suspendChangeNotifications was called) prior to replacing the archive(s).
     */
    public void downloadArchives(String dsArchivePath0, String newVersion)
    throws Exception
    {
        super.m_context.logMessage("downloadArchives was called: Updating archives from Directory Service folder \"" + dsArchivePath0 + "\" newVersion \"" +  newVersion + "\"", Level.INFO);

        //Remove "sonicfs:///" if there
        String dsArchivePath = dsArchivePath0.startsWith(IContainer.DS_CLASSPATH_PROTOCOL) ? dsArchivePath0.substring(IContainer.DS_CLASSPATH_PROTOCOL.length()) : dsArchivePath0;

        if (dsArchivePath.charAt(dsArchivePath.length() - 1) != '/')
        {
            dsArchivePath += "/";
        }

        m_containerImpl.downloadArchives(dsArchivePath, newVersion);

        super.m_context.logMessage("...archives update is done", Level.INFO);
    }



    /**
     * Makes sure all the latest version of the file is caches. NOTE: Will not downalod files that were replaced (file path was not modified) if notifications were
     * suspended (DSCompoenet.suspendChangeNotifications was called) prior to replacing the file.
     */
    public void downloadFile(String dsPath)
    throws Exception
    {
        m_containerImpl.cacheFileIfNeeded(dsPath, false, true);
        super.m_context.logMessage("downloadFile updated Directory Service file \"" + dsPath + "\"", Level.INFO);
    }

    //
    // All container notifications go through the agent.
    //

    // this is now an internal notification only destined for the AM that now sends a container startup notification
    void sendStartupNotification()
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], STARTUP_NOTIFICATION_TYPE, Level.INFO);
        notification.setLogType(INotification.INFORMATION_TYPE);
        super.m_context.sendNotification(notification);
    }

    void sendShutdownNotification(int exitCode)
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], SHUTDOWN_NOTIFICATION_TYPE, Level.INFO);
        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("ExitCode", new Integer(exitCode));
        try
        {
            notification.setAttribute("ExitCodeString", IContainerExitCodes.EXIT_CODE_TEXTS[exitCode]);
        }
        catch(ArrayIndexOutOfBoundsException e)
        {
            notification.setAttribute("ExitCodeString", "n/a");
        }
        super.m_context.sendNotification(notification);
    }

    void sendComponentLoadNotification(String id, String configID, String lastError, Integer lastErrorLevel)
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], LOAD_NOTIFICATION_TYPE, Level.INFO);
        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("ID", id);
        notification.setAttribute("ConfigID", configID);
        notification.setAttribute("LastError", lastError);
        notification.setAttribute("LastErrorLevel", lastErrorLevel);
        super.m_context.sendNotification(notification);
    }

    void sendComponentUnloadNotification(String id)
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], UNLOAD_NOTIFICATION_TYPE, Level.INFO);
        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("ID", id);
        super.m_context.sendNotification(notification);
    }

    void sendLogMessageNotification(String id, String message, int severityLevel)
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.LOG_SUBCATEGORY], LOG_MESSAGE_NOTIFICATION_TYPE, severityLevel);
        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("LogLevel", Level.LEVEL_TEXT[severityLevel]);
        if (id != null)
        {
            notification.setAttribute("ID", id);
        }
        notification.setAttribute("Message", message);
        super.m_context.sendNotification(notification);
    }

    void sendContainerStateNotification()
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], CONTAINER_STATE_NOTIFICATION_TYPE, Level.INFO);
        notification.setLogType(INotification.INFORMATION_TYPE);
        IContainerState containerState = null;
        try
        {
            containerState = getContainerState();
        }
        catch (Exception e) { if (DEBUG)
        {
            System.out.println("Agent.sendContainerStateNotification: error obtaining container state...");
        } }
        notification.setAttribute("ContainerState", containerState);
        super.m_context.sendNotification(notification);
    }

    @Override
    public synchronized void handleElementChange(final IElementChange elementChange)
    {
        // don't handle any changes until startup has completed or we are shutting down
        // - we need to do thius on a separate thread so we don't block startup
        final IContainer container = super.m_container;
        synchronized(container)
        {
            if (!container.isBooted())
            {
                Runnable delayedHandler = new Runnable()
                {
                    @Override
                    public void run()
                    {
                        synchronized(container)
                        {
                            while (!container.isBooted())
                            {
                                if (container.isClosing())
                                {
                                    return;
                                }

                                try { container.wait(1000); } catch(InterruptedException e) { }
                            }
                            Agent.this.handleElementChange(elementChange);
                        }
                    }
                };
                super.m_frameworkContext.scheduleTask(delayedHandler, new Date());
                return;
            }
        }
        if (container.isClosing())
        {
            return;
        }

        IElementIdentity id = elementChange.getElement().getIdentity();
        String configID = id.getName();

        if (configID.equals(m_configID)) // is this the containers own configuration?
        {
            // if the delta is for a version that is older than the initial config version
            // then ignore the change
            long configVersion = id.getVersion();
            if (configVersion < m_initialConfigVersion)
            {
                return;
            }

            if (elementChange.getChangeType() == IElementChange.ELEMENT_UPDATED)
            {

                IDeltaElement changeElement = (IDeltaElement)elementChange.getElement();
                IDeltaAttributeSet attrs = (IDeltaAttributeSet)changeElement.getDeltaAttributes();

                String [] mods1 = attrs.getModifiedAttributesNames();
                String [] mods2 = attrs.getNewAttributesNames();
                String [] mods3 = attrs.getDeletedAttributesNames();


                boolean shouldGenerateINI = false;
                boolean shouldGenerateDSBoot = false;
                for (int i = 0; i < mods1.length; i++)
                {
                    if (com.sonicsw.mf.common.util.Container.iniFileAttribute(mods1[i]))
                    {
                        shouldGenerateINI = true;
                        break;
                    }
                }

                if (!shouldGenerateINI)
                {

                    for (int i = 0; i < mods2.length; i++)
                    {
                        if (com.sonicsw.mf.common.util.Container.iniFileAttribute(mods2[i]))
                        {
                            shouldGenerateINI = true;
                        }
                        if (mods2[i].equals(IContainerConstants.HOSTS_DIRECTORY_SERVICE_ATTR))
                        {
                            shouldGenerateDSBoot = true;
                        }
                    }
                }

                if (!shouldGenerateINI)
                {
                    for (int i = 0; i < mods3.length; i++)
                    {
                        if (com.sonicsw.mf.common.util.Container.iniFileAttribute(mods3[i]))
                        {
                            shouldGenerateINI = true;
                            break;
                        }
                    }
                }

                if (shouldGenerateINI)
                {
                    generateContainerINI();
                }
                if (shouldGenerateDSBoot)
                {
                    generateDSBootFileIfNeeded();
                }

                handleChangeContainerAttrs(mods1, attrs);
                handleChangeContainerAttrs(mods2, attrs);
                handleDeletedContainerAttrs(mods3);
            }
        }
        if (configID.equals(DOMAIN_CONFIG_ID)) // is this the domain configuration?
        {
            // since were just interested in the domain wide setting of the force centralized logging we can follow an existing code
            // path to get there
            setCentrallyLogMessages();
        }

        // also pass changes to the management policy and audit managers
        if (m_permissionsManager != null)
        {
            m_permissionsManager.handleElementChange(elementChange);
        }
        if (m_auditManager != null)
        {
            m_auditManager.handleElementChange(elementChange);
        }
    }

    private void generateContainerINI()
    {

        File iniFile = new File(ContainerSetup.CONTAINER_INI_FILE);

        if (!iniFile.exists() || iniFile.isDirectory())
        {
            return;
        }

        try
        {
            String encryptedPwd = getINIPwd();

            Properties oldProps = getINIProps(iniFile, encryptedPwd);

            if (oldProps == null)
            {
                return;
            }

            File tempIniFile = new File("tmp_"+ ContainerSetup.CONTAINER_INI_FILE);

            if (encryptedPwd != null)
            {
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                PrintWriter writer = new PrintWriter(out);
                generateINIProps(oldProps, writer);
                String clearPwd = new String(PBE.decrypt(encryptedPwd.getBytes()));
                byte[] encryptedData = PBE.encrypt(out.toByteArray(), clearPwd);
                RandomAccessFile encFile = new RandomAccessFile(tempIniFile, "rw");
                encFile.write(encryptedData);
                encFile.close();
            }
            else
            {
                PrintWriter writer =  new PrintWriter(new BufferedWriter(new FileWriter(tempIniFile, false)), true);
                generateINIProps(oldProps, writer);
            }

            if (!iniFile.delete())
            {
                throw new IOException("Could not modify " + iniFile.getAbsolutePath());
            }

            tempIniFile.renameTo(iniFile);
            super.m_context.logMessage("Updated " + iniFile.getAbsolutePath(), Level.INFO);
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Failed to generate an updated container.ini file, trace follows...", e, Level.WARNING);
        }

    }

    private void generateDSBootFileIfNeeded()
    {
        if (IContainer.CURRENT_LAUNCHER_VERSION == null)
        {
            return;
        }

        try
        {

            String dsElementID = ContainerUtil.extractDSConfigIDFromContainerConfig(m_context.getConfiguration(true));

            if (dsElementID == null)
            {
                return;
            }

            m_containerImpl.generateDSBootFile(dsElementID);

        }
        catch (Exception e)
        {
            super.m_context.logMessage("Failed to generate the \"" + IContainer.DEFAULT_DS_BOOTFILE_NAME + "\" file, trace follows...", e, Level.WARNING);
        }
    }


    private void generateINIProps(Properties oldProps, PrintWriter writer) throws Exception
    {
        //Leave only non configures values
        Properties newProps = new Properties();
        Enumeration propItems = oldProps.keys();
        String oldCachePassword = oldProps.getProperty(IContainer.CACHE_PASSWORD_ATTR);
        while (propItems.hasMoreElements())
        {
            String key = (String)propItems.nextElement();

            if (m_nonConfiguredPropSet.contains(key.toUpperCase()) ||
                key.regionMatches(true, 0, IContainer.SYSTEM_PROP_PREFIX, 0, IContainer.SYSTEM_PROP_PREFIX.length()))
            {
                //IContainer.DS_START_ACTIVE_PROPERTY is a one time property, should be false when the INI file is regenerated
                String propVal = (key.indexOf(IContainer.DS_START_ACTIVE_PROPERTY) == -1) ? oldProps.getProperty(key) : "false";
                newProps.setProperty(key,  propVal);
            }
        }


        com.sonicsw.mf.common.util.Container.bootConfigurationToProp(m_context.getConfiguration(true), newProps);
        String newCachePassword = newProps.getProperty(IContainer.CACHE_PASSWORD_ATTR);
        propItems = newProps.keys();

        while (propItems.hasMoreElements())
        {
            String key = (String)propItems.nextElement();
            String value = newProps.getProperty(key);
            if (value == null || value.length() == 0)
            {
                continue;
            }
            writer.println(key + "=" + value);
        }
        writer.close();

        if ((oldCachePassword == null && newCachePassword != null) ||
            (oldCachePassword != null && newCachePassword == null) ||
            (oldCachePassword != null && newCachePassword != null && !newCachePassword.equals(oldCachePassword)))
        {
            requestCacheReset();
        }
    }

    private void requestCacheReset()
    {
        try
        {
            new java.io.FileOutputStream(IContainer.RESET_CACHE_FILE, true).close();
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Failed to create cache reset file, trace follows...", e, Level.WARNING);
        }
    }

    //Returns the encrypted password of the INI file by reading the script that sets it up. Null is returned
    // if the script does not exists (the INI is not encrypted
    private String getINIPwd() throws IOException
    {
        File setPwdScript = new File(SET_ENC_PWD_SH_SCRIPT_NAME);
        if (!setPwdScript.exists())
        {
            return null;
        }

        BufferedReader reader = new BufferedReader (new InputStreamReader(new FileInputStream(setPwdScript)));
        try
        {

            //Skip the #/bin/sh line
            reader.readLine();
            String srcLine = reader.readLine();

            if (null != srcLine)
            {
                String pwdProp = IContainer.MF_PASSWORD_PROPERTY + "=";
                int pwdIndex = srcLine.indexOf(pwdProp);
                String encryptedPwd = srcLine.substring(pwdIndex + pwdProp.length());

                //Remove the closing '"' character
                return encryptedPwd.substring(0, encryptedPwd.length()-1);
            }
        }
        finally
        {
            reader.close();
        }
        return null;
    }

    private Properties getINIProps(File iniFile, String encryptedPwd) throws Exception
    {


        RandomAccessFile raf = new RandomAccessFile(iniFile, "r");
        byte[] bytes = new byte[(int)raf.length()];
        raf.read(bytes);
        raf.close();

        if (encryptedPwd != null)
        {
            bytes = ContainerUtil.decryptBytes(bytes, encryptedPwd);
        }


        ByteArrayInputStream stream = new ByteArrayInputStream(bytes);

        return ContainerUtil.readProperties(stream);

    }

    @Override
    public synchronized void handleFileChange(IFSElementChange fileChange)
    {
        if (m_auditManager != null)
        {
            m_auditManager.handleFileChange(fileChange);
        }
    }

    @Override
    public String getTraceMaskValues() { return (super.getTraceMaskValues() + "," + AGENT_TRACE_MASK_VALUES); }

    @Override
    public void setTraceMask(Integer traceMask)
    {
        setTraceMask(traceMask, false);
    }

    private void setTraceMask(Integer traceMask, boolean fromSDF)
    {
        // If SDF is used then ignore trace setup from other sources
        // If SDF is not used - update it from other sources
        if (!fromSDF)
        {
            if (m_SdfMFTracingIntegration.wasUpdated())
            {
                return;
            }
            else
            {
                m_SdfMFTracingIntegration.setTraceMask(traceMask);
            }
        }

        super.setTraceMask(traceMask);
        if (super.m_container != null)
        {
            m_containerImpl.setTraceMask(super.getTraceMask().intValue());
        }
        if (m_permissionsManager != null)
        {
            m_permissionsManager.setTraceMask(super.getTraceMask().intValue());
        }
    }

    public String getContainerName() { return super.m_container.getContainerIdentity().getCanonicalName(); }

    public String getContainerConfigID() { return super.m_container.getContainerIdentity().getConfigIdentity().getName(); }

    void redoCLIPrompt()
    {
        if (m_commandLineEnabled)
        {
            AgentCLI.redoPrompt();
        }
    }

    // This is here so the container can call it and get the carriage return
    // printed only when a command is not executing. This info is known by the
    // Agent CLI
    void crAfterPrompt() { AgentCLI.crAfterPrompt(); }

    Object[] getAttributeValues(String source, String[] attributeNames)
    throws Exception
    {
        return super.m_frameworkContext.getAttributeValues(source, attributeNames);
    }

    private void handleChangeContainerAttrs(String[] attributeNames, IDeltaAttributeSet modifiedAttrs)
    {
        try
        {
            for (int i = 0; i < attributeNames.length; i++)
            {
                if (attributeNames[i].equals(IContainerConstants.TRACE_MASK_ATTR))
                {
                    setTraceMask((Integer)modifiedAttrs.getNewValue(IContainerConstants.TRACE_MASK_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.COMMAND_LINE_ATTR))
                {
                    setCommandLine((Boolean)modifiedAttrs.getNewValue(IContainerConstants.COMMAND_LINE_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.DIRECTORY_SERVICE_PING_INTERVAL_ATTR))
                {
                    m_containerImpl.setDSPingFrequency((Integer)modifiedAttrs.getNewValue(IContainerConstants.DIRECTORY_SERVICE_PING_INTERVAL_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_TO_CONSOLE_ATTR))
                {
                    setLogToConsole((Boolean)modifiedAttrs.getNewValue(IContainerConstants.LOG_TO_CONSOLE_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_TO_FILE_ATTR))
                {
                    setLogToFile((Boolean)modifiedAttrs.getNewValue(IContainerConstants.LOG_TO_FILE_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_FILE_SIZE_THRESHOLD_ATTR))
                {
                    setLogFileSizeThreshold((Long)modifiedAttrs.getNewValue(IContainerConstants.LOG_FILE_SIZE_THRESHOLD_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_ATTR))
                {
                    setLogFileRolloverSizeThreshold((Long)modifiedAttrs.getNewValue(IContainerConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_ATTR))
                {
                    setLogFileRolloverTimeInterval((Integer)modifiedAttrs.getNewValue(IContainerConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_FILE_ATTR))
                {
                    setLogFile((String)modifiedAttrs.getNewValue(IContainerConstants.LOG_FILE_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.ACTIONAL_LOG_INTERCEPTOR_ATTR))
                {
                    setActionalLogInterceptor((Boolean)modifiedAttrs.getNewValue(IContainerConstants.ACTIONAL_LOG_INTERCEPTOR_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.ENABLE_CENTRALIZED_LOGGING_ATTR))
                {
                    setCentrallyLogMessages();
                }
                else
                if (attributeNames[i].equals(IContainerConstants.MAX_THREADS_ATTR))
                {
                    m_taskScheduler.setMaxThreads(((Integer)modifiedAttrs.getNewValue(IContainerConstants.MAX_THREADS_ATTR)).intValue());
                }
                else
                if (attributeNames[i].equals(IContainerConstants.MIN_THREADS_ATTR))
                {
                    m_taskScheduler.setMinThreads(((Integer)modifiedAttrs.getNewValue(IContainerConstants.MIN_THREADS_ATTR)).intValue());
                }
                else
                if (attributeNames[i].equals(IContainerConstants.NOTIFICATION_DISPATCH_QUEUE_SIZE_ATTR))
                {
                    m_notificationPublisher.setQueueSize(((Integer)modifiedAttrs.getNewValue(IContainerConstants.NOTIFICATION_DISPATCH_QUEUE_SIZE_ATTR)).intValue());
                }
                else
                if (attributeNames[i].equals(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR))
                {
                    m_containerImpl.setSonicArchiveSearchPath((String)modifiedAttrs.getNewValue(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.EXTENSIONS_ATTR))
                {
                    handleChangeExtensionsAttr((IDeltaAttributeSet)modifiedAttrs.getNewValue(IContainerConstants.EXTENSIONS_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.COMPONENTS_ATTR))
                {
                    handleChangeComponentsAttr((IDeltaAttributeSet)modifiedAttrs.getNewValue(IContainerConstants.COMPONENTS_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.CONNECTION_ATTR))
                {
                    handleChangeConnectionAttr((IDeltaAttributeSet)modifiedAttrs.getNewValue(IContainerConstants.CONNECTION_ATTR));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.MONITORING_ATTR))
                {
                    Object newValue = modifiedAttrs.getNewValue(IContainerConstants.MONITORING_ATTR);
                    if (newValue instanceof IDeltaAttributeSet)
                    {
                        handleChangeMonitoringAttr((IDeltaAttributeSet)newValue);
                    }
                    else
                    {
                        handleNewMonitoringAttr((IAttributeSet)newValue);
                    }
                }
                else
                if (attributeNames[i].equals(IContainerConstants.ENABLED_METRICS_ATTR))
                {
                    resetMetrics(IAgentProxy.ID);
                }
                else
                if (attributeNames[i].equals(IContainerConstants.ENABLED_ALERTS_ATTR))
                {
                    resetAlerts(IAgentProxy.ID);
                }
                else
                if (attributeNames[i].equals(IContainerConstants.FAULT_TOLERANCE_PARAMETERS_ATTR))
                {
                    if (m_containerImpl.isFTContainer())
                    {
                        m_containerImpl.getContainerFT().handleChangeFTAttrs((IDeltaAttributeSet)modifiedAttrs.getNewValue(IContainerConstants.FAULT_TOLERANCE_PARAMETERS_ATTR));
                    }
                    else
                    {
                        m_containerImpl.logMessage(null, attributeNames[i] + " cannot be added at runtime. ", Level.WARNING);
                    }
                }
            }
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Error modifying Agent runtime from configuration change, trace follows...", e, Level.WARNING);
        }
    }
    
    private void resetMetrics(String componentID)
    {
        IAttributeSet mas = getEnabledMetricsOrAlertsFromConfiguration(componentID, IContainerConstants.ENABLED_METRICS_ATTR);
        IMetricIdentity[] enabledMetricIds = mas == null ? null : createMetricIds(mas);
        
        String target = m_agentName.getDomainName() + '.' + m_agentName.getContainerName() + ':' + IComponentIdentity.ID_PREFIX + componentID;
        try
        {
            m_containerImpl.invokeLocal(target, REPLACE_ENABLED_METRICS_OPERATION_NAME, new Object[] { enabledMetricIds }, REPLACE_ENABLED_METRICS_OPERATION_SIGNATURE, false);
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Failed to replace enabled metrics for ID=" + componentID + ", trace follows...", e, Level.SEVERE);
        }
    }
    
    private void resetAlerts(String componentID)
    {
        IAttributeSet aas = getEnabledMetricsOrAlertsFromConfiguration(componentID, IContainerConstants.ENABLED_ALERTS_ATTR);
        IAlert[] enabledAlerts = aas == null ? null : createAlerts(aas);
        
        String target = m_agentName.getDomainName() + '.' + m_agentName.getContainerName() + ':' + IComponentIdentity.ID_PREFIX + componentID;
        try
        {
            m_containerImpl.invokeLocal(target, REPLACE_ENABLED_ALERTS_OPERATION_NAME, new Object[] { enabledAlerts }, REPLACE_ENABLED_ALERTS_OPERATION_SIGNATURE, false);
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Failed to replace enabled alerts for ID=" + componentID + ", trace follows...", e, Level.SEVERE);
        }
    }
    
    private IAttributeSet getEnabledMetricsOrAlertsFromConfiguration(String componentID, String attributeName)
    {
        IAttributeSet as = null;
        
        // get the container's Config Element
        IElement containerCE = m_context.getConfiguration(true);

        // get the container attribute set
        IAttributeSet containerAS = (IAttributeSet)containerCE.getAttributes();
        
        // If componentID = "AGENT" get enabled metrics or alerts from container attributes
        if (IAgentProxy.ID.equals(componentID))
        {
            as = (IAttributeSet)containerAS.getAttribute(attributeName);
        }
        else
        {
            // get the "COMPONENTS" attribute set
            IAttributeSet componentAS = (IAttributeSet)containerAS.getAttribute(IContainerConstants.COMPONENTS_ATTR);
            HashMap map = (HashMap)componentAS.getAttributes();
            IAttributeSet spas = (IAttributeSet)map.get(componentID);

            // return if no startup parameters found for the component
            if (spas == null)
            {
                return null ;
            }

            // get the enabled metrics or alerts set
            as = (IAttributeSet)spas.getAttribute(attributeName);
        }
        
        return as;
    }

    // When an attribute is deleted,the runtime equivalent goes back to the
    // default value
    private void handleDeletedContainerAttrs(String[] attributeNames)
    {
        try
        {
            for (int i=0; i< attributeNames.length; i++)
            {
                if (attributeNames[i].equals(IContainerConstants.TRACE_MASK_ATTR))
                {
                    setTraceMask(new Integer(IContainerConstants.TRACE_MASK_DEFAULT));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.COMMAND_LINE_ATTR))
                {
                    setCommandLine(new Boolean(IContainerConstants.COMMAND_LINE_DEFAULT));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.DIRECTORY_SERVICE_PING_INTERVAL_ATTR))
                {
                    m_containerImpl.setDSPingFrequency(new Integer(IContainerConstants.DIRECTORY_SERVICE_PING_INTERVAL_DEFAULT));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_TO_CONSOLE_ATTR))
                {
                    setLogToConsole(new Boolean(IContainerConstants.LOG_TO_CONSOLE_DEFAULT));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_TO_FILE_ATTR))
                {
                    setLogToFile(new Boolean(IContainerConstants.LOG_TO_FILE_DEFAULT));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_FILE_SIZE_THRESHOLD_ATTR))
                {
                    setLogFileSizeThreshold(new Long(IContainerConstants.LOG_FILE_SIZE_THRESHOLD_DEFAULT));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_ATTR))
                {
                    setLogFileRolloverSizeThreshold(new Long(IContainerConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_DEFAULT));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_ATTR))
                {
                    setLogFileRolloverTimeInterval(new Integer(IContainerConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_DEFAULT));
                }
                else
                if (attributeNames[i].equals(IContainerConstants.LOG_FILE_ATTR))
                {
                    setLogFile(m_defaultLogFile);                }
                else
                if (attributeNames[i].equals(IContainerConstants.ACTIONAL_LOG_INTERCEPTOR_ATTR))
                {
                    setActionalLogInterceptor(m_defaultActionalLogInterceptor);
                }
                else
                if (attributeNames[i].equals(IContainerConstants.ENABLE_CENTRALIZED_LOGGING_ATTR))
                {
                    setCentrallyLogMessages();
                }
                else
                if (attributeNames[i].equals(IContainerConstants.MAX_THREADS_ATTR))
                {
                    m_taskScheduler.setMaxThreads(IContainerConstants.MAX_THREADS_DEFAULT);
                }
                else
                if (attributeNames[i].equals(IContainerConstants.MIN_THREADS_ATTR))
                {
                    m_taskScheduler.setMinThreads(IContainerConstants.MIN_THREADS_DEFAULT);
                }
                else
                if (attributeNames[i].equals(IContainerConstants.NOTIFICATION_DISPATCH_QUEUE_SIZE_ATTR))
                {
                    m_notificationPublisher.setQueueSize(IContainerConstants.NOTIFICATION_DISPATCH_QUEUE_SIZE_DEFAULT);
                }
                else
                if (attributeNames[i].equals(IContainerConstants.FAULT_TOLERANCE_PARAMETERS_ATTR))
                {
                    m_containerImpl.logMessage(null, "Dynamic removal of " + attributeNames[i] + " is not supported; a container restart is required", Level.WARNING);
                }
            }
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Error modifying Agent runtime from configuration change", e, Level.SEVERE);
        }
    }

    private void handleChangeExtensionsAttr(IDeltaAttributeSet attrs)
    throws Exception
    {
        handleNewExtensionsAttrs(attrs.getNewAttributesNames(), attrs);
        handleModifiedExtensionsAttrs(attrs.getModifiedAttributesNames(), attrs);
        handleDeletedExtensionsAttrs(attrs.getDeletedAttributesNames());
    }

    private void handleNewExtensionsAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            m_containerImpl.loadConfiguredComponent(names[i], (IAttributeSet)attrs.getNewValue(names[i]), true, false);
        }
    }

    private void handleModifiedExtensionsAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            IDeltaAttributeSet modifiedAttrs = (IDeltaAttributeSet)attrs.getNewValue(names[i]);
            handleNewExtensionAttrs(names[i], modifiedAttrs.getNewAttributesNames(), modifiedAttrs);
            handleModifiedExtensionAttrs(names[i], modifiedAttrs.getModifiedAttributesNames(), modifiedAttrs);
            handleDeletedExtensionAttrs(names[i], modifiedAttrs.getDeletedAttributesNames());
        }
    }

    private void handleDeletedExtensionsAttrs(String[] names)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            synchronized(super.m_container)
            {
                if (!super.m_container.isHostingComponent(names[i]))
                {
                    continue;
                }
            }
            // window still left where we could run into a timing issue, however if that occurs
            // it will not be fatal

            unloadComponent(names[i]);

            // now remove any runtime information accumulated in the cache by the component
            m_containerImpl.deleteRuntimeConfiguration(names[i]);
        }
    }

    private void handleNewExtensionAttrs(String id, String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.EXTENSION_ACTIVE_ATTR))
            {
                unloadComponent(id);
            }
        }
    }

    private void handleModifiedExtensionAttrs(String id, String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.EXTENSION_CONFIG_REF_ATTR))
            {
                logMessage("Dynamic changes to extension config references unsupported: " + names[i], Level.WARNING);
            }
            else
            if (names[i].equals(IContainerConstants.EXTENSION_ACTIVE_ATTR))
            {
                Boolean isActive = (Boolean)attrs.getNewValue(IContainerConstants.EXTENSION_ACTIVE_ATTR);
                if (isActive.booleanValue() && !super.m_container.isHostingComponent(id))
                {
                    IAttributeSet extensionsAttrs = (IAttributeSet)m_context.getConfiguration(true).getAttributes().getAttribute(IContainerConstants.EXTENSIONS_ATTR);
                    m_containerImpl.loadConfiguredComponent(id, (IAttributeSet)extensionsAttrs.getAttribute(id), true, false);
                }
                else
                if (!isActive.booleanValue() && super.m_container.isHostingComponent(id))
                {
                    unloadComponent(id);
                }
            }
        }
    }

    private void handleDeletedExtensionAttrs(String id, String[] names)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.NATIVE_LIBRARIES_ATTR))
            {
                logMessage("Dynamic removal of component native libraries unsupported: " + names[i], Level.WARNING);
            }
            else
            if (names[i].equals(IContainerConstants.EXTENSION_ACTIVE_ATTR))
            {
                if (super.m_container.isHostingComponent(id))
                {
                    unloadComponent(id);
                }
            }
        }
    }

    private void handleChangeComponentsAttr(IDeltaAttributeSet attrs)
    throws Exception
    {
        handleNewComponentsAttrs(attrs.getNewAttributesNames(), attrs);
        handleModifiedComponentsAttrs(attrs.getModifiedAttributesNames(), attrs);
        handleDeletedComponentsAttrs(attrs.getDeletedAttributesNames());
    }

    private void handleNewComponentsAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(DSComponent.GLOBAL_ID))
            {
                try
                {
                    boolean done = com.sonicsw.mf.framework.agent.ci.LauncherJARBuilder.createMFdirectoryJAR();
                    if (done)
                    {
                        super.m_context.logMessage("Added MFdirectory.jar to the launcher lib directory", Level.INFO);
                    }
                }
                catch (java.io.IOException e)
                {
                    super.m_context.logMessage("Failed to add MFdirectory.jar to the launcher lib directory", e, Level.WARNING);
                }

                if (!super.m_container.isHostingComponent(names[i]))
                {
                    super.m_context.logMessage("Container must be restarted to load the " + names[i], Level.WARNING);
                }
                continue;
            }

            m_containerImpl.loadConfiguredComponent(names[i], (IAttributeSet)attrs.getNewValue(names[i]), false, false);
        }
    }

    private void handleModifiedComponentsAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            IDeltaAttributeSet modifiedAttrs = (IDeltaAttributeSet)attrs.getNewValue(names[i]);
            handleNewComponentAttrs(names[i], modifiedAttrs.getNewAttributesNames(), modifiedAttrs);
            handleModifiedComponentAttrs(names[i], modifiedAttrs.getModifiedAttributesNames(), modifiedAttrs);
            handleDeletedComponentAttrs(names[i], modifiedAttrs.getDeletedAttributesNames());
        }
    }

    private void handleDeletedComponentsAttrs(String[] names)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            synchronized(super.m_container)
            {
                if (!super.m_container.isHostingComponent(names[i]))
                {
                    continue;
                }
            }
            // window still left where we could run into a timing issue, however if that occurs
            // it will not be fatal

            if (names[i].equals(DSComponent.GLOBAL_ID))
            {
                super.m_context.logMessage("Container must be restarted to unload the " + names[i], Level.WARNING);
                continue;
            }
            unloadComponent(names[i]);

            // now remove any runtime information accumulated in the cache by the component
            m_containerImpl.deleteRuntimeConfiguration(names[i]);
        }
    }

    private void handleNewComponentAttrs(String id, String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.TRACE_MASK_ATTR))
            {
                setTraceMask(id, (Integer)attrs.getNewValue(IContainerConstants.TRACE_MASK_ATTR));
            }
            else
            if (names[i].equals(IContainerConstants.NATIVE_LIBRARIES_ATTR))
            {
                logMessage("Dynamic additions to component native libraries unsupported: " + names[i], Level.WARNING);
            }
            else
            if (names[i].equals(IContainerConstants.ENABLED_METRICS_ATTR))
            {
                resetMetrics(id);
            }
            else
            if (names[i].equals(IContainerConstants.ENABLED_ALERTS_ATTR))
            {
                resetAlerts(id);
            }
        }
    }

    private void handleModifiedComponentAttrs(String id, String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.TRACE_MASK_ATTR))
            {
                setTraceMask(id, (Integer)attrs.getNewValue(IContainerConstants.TRACE_MASK_ATTR));
            }
            else
            if (names[i].equals(IContainerConstants.CONFIG_REF_ATTR))
            {
                logMessage("Dynamic changes to component config references unsupported: " + names[i], Level.WARNING);
            }
            else
            if (names[i].equals(IContainerConstants.NATIVE_LIBRARIES_ATTR))
            {
                logMessage("Dynamic changes to component native libraries unsupported: " + names[i], Level.WARNING);
            }
            else
            if (names[i].equals(IContainerConstants.ENABLED_METRICS_ATTR))
            {
                resetMetrics(id);
            }
            else
            if (names[i].equals(IContainerConstants.ENABLED_ALERTS_ATTR))
            {
                resetAlerts(id);
            }
        }
    }

    private void handleDeletedComponentAttrs(String id, String[] names)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.TRACE_MASK_ATTR))
            {
                setTraceMask(id, new Integer(0));
            }
            else
            if (names[i].equals(IContainerConstants.NATIVE_LIBRARIES_ATTR))
            {
                logMessage("Dynamic removal of component native libraries unsupported: " + names[i], Level.WARNING);
            }
        }
    }

    private void handleNewMonitoringAttr(IAttributeSet attrs)
    throws Exception
    {
        Object notificationInterval = attrs.getAttribute(IContainerConstants.STATUS_POLL_INTERVAL_ATTR);
        if (notificationInterval == null)
        {
            m_containerStateNotifier.setNotificationInterval( (long) (IContainerConstants.STATUS_POLL_INTERVAL_DEFAULT * 800));  // want 80% of the poll interval, expressed in milliseconds -> 0.8 * 1000 * STATUS_POLL_INTERVAL
        }
        else {
            m_containerStateNotifier.setNotificationInterval(((Integer)notificationInterval).longValue() * 800);  // want 80% of the poll interval, expressed in milliseconds -> 0.8 * 1000 * STATUS_POLL_INTERVAL
        }
    }

    private void handleChangeMonitoringAttr(IDeltaAttributeSet attrs)
    throws Exception
    {
        handleModifiedMonitoringAttrs(attrs.getNewAttributesNames(), attrs);
        handleModifiedMonitoringAttrs(attrs.getModifiedAttributesNames(), attrs);
        handleDeletedMonitoringAttrs(attrs.getDeletedAttributesNames());
    }

    private void handleModifiedMonitoringAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.STATUS_POLL_INTERVAL_ATTR))
             {
                m_containerStateNotifier.setNotificationInterval(((Integer)attrs.getNewValue(names[i])).longValue() * 800);  // want 80% of the poll interval, expressed in milliseconds -> 0.8 * 1000 * STATUS_POLL_INTERVAL
            }
        }
    }

    private void handleDeletedMonitoringAttrs(String[] names)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.STATUS_POLL_INTERVAL_ATTR))
             {
                m_containerStateNotifier.setNotificationInterval((long)(IContainerConstants.STATUS_POLL_INTERVAL_DEFAULT * 800));  // want 80% of the poll interval, expressed in milliseconds -> 0.8 * 1000 * STATUS_POLL_INTERVAL
            }
        }
    }

    private void setTraceMask(String id, Integer traceMask)
    throws Exception
    {
        String target = m_agentName.getDomainName() + '.' + m_agentName.getContainerName() + ':' + IComponentIdentity.ID_PREFIX + id;
        super.m_frameworkContext.setAttributes(target, new String[] { "TraceMask" }, new Object[] { traceMask }, false);
    }

    private void handleChangeConnectionAttr(IDeltaAttributeSet attrs)
    throws Exception
    {
        handleModifiedConnectionAttrs(attrs.getNewAttributesNames(), attrs);
        handleModifiedConnectionAttrs(attrs.getModifiedAttributesNames(), attrs);
        handleDeletedConnectionAttrs(attrs.getDeletedAttributesNames());
    }

    private void handleModifiedConnectionAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.REQUEST_TIMEOUT_ATTR))
            {
                super.m_container.getConnectorServer().setRequestTimeout(((Integer)attrs.getNewValue(names[i])).intValue() * 1000);
            }
        }
    }

    private void handleDeletedConnectionAttrs(String[] names)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.REQUEST_TIMEOUT_ATTR))
            {
                super.m_container.getConnectorServer().setRequestTimeout(ConnectorClient.REQUEST_TIMEOUT_DEFAULT);
            }
        }
    }

    //
    // Utility methods for creating metrics and alerts from config information.
    //

    private IMetricIdentity[] createMetricIds(String[] metricNames)
    {
        IMetricIdentity[] metricIds = new IMetricIdentity[metricNames.length];
        for (int i = 0; i < metricNames.length; i++)
        {
            metricIds[i] = MetricsFactory.createMetricIdentity(metricNames[i]);
        }
        return metricIds;
    }

    private IMetricIdentity[] createMetricIds(IAttributeSet metricsAS)
    {
        HashMap metricsMap = metricsAS.getAttributes();
        Iterator iterator = metricsMap.values().iterator();
        ArrayList metricIDList = new ArrayList();
        
        while (iterator.hasNext())
        {
            String metricName = (String)iterator.next();
            IMetricIdentity metricID = MetricsFactory.createMetricIdentity(metricName);
            metricIDList.add(metricID);
        }
        
        return (IMetricIdentity[])metricIDList.toArray(new IMetricIdentity[metricIDList.size()]);
    }

    private IAlert[] createAlerts(IAttributeSet alertsAS)
    {
        HashMap alertsMap = alertsAS.getAttributes();
        Iterator itor = alertsMap.values().iterator();
        IAlert alerts[];
        
        ArrayList alertsList = new ArrayList();

        while (itor.hasNext())
        {
            Object attr = itor.next();
            if (attr instanceof AttributeSet)
            {
                AttributeSet alertAS = (AttributeSet)attr;
                AttributeName compoundName = alertAS.getCompoundName();
                String metricName = (String)compoundName.getComponent(compoundName.getComponentCount() - 1);
                IMetricIdentity metricID = MetricsFactory.createMetricIdentity(metricName);

                Object thresholds = ((AttributeSet)alertAS).getAttribute(IContainerConstants.HIGH_THRESHOLDS_ATTR);
                if (thresholds instanceof String)
                {
                    alerts = MetricsFactory.createAlerts(metricID, true, (String)thresholds);
                    for (int j = 0; j < alerts.length; j++)
                    {
                        alertsList.add(alerts[j]);
                    }
                }

                thresholds = ((AttributeSet)alertAS).getAttribute(IContainerConstants.LOW_THRESHOLDS_ATTR);
                if (thresholds instanceof String)
                {
                    alerts = MetricsFactory.createAlerts(metricID, false, (String)thresholds);
                    for (int j = 0; j < alerts.length; j++)
                    {
                        alertsList.add(alerts[j]);
                    }
                }
            }
        }

        return (IAlert[])alertsList.toArray(new IAlert[alertsList.size()]);
    }

    private void setCentrallyLogMessages()
    {
        // get the domain wide setting
        boolean centrallyLogMessages = getForceCentralizedLogging();

        if (!centrallyLogMessages)
        {
            // get the container specific setting
            IElement containerConfig = super.m_context.getConfiguration(true);

            IAttributeSet containerAttributes = containerConfig.getAttributes();
            Boolean enableCentralizedLogging = (Boolean)containerAttributes.getAttribute(IContainerConstants.ENABLE_CENTRALIZED_LOGGING_ATTR);

            if (enableCentralizedLogging != null)
            {
                centrallyLogMessages = enableCentralizedLogging.booleanValue();
            }
        }

        boolean currentlyEnabled = m_centrallyLogMessages == null ? false : m_centrallyLogMessages.booleanValue();
        if (centrallyLogMessages)
        {
            m_centrallyLogMessages = Boolean.TRUE;
            if (!currentlyEnabled)
            {
                m_container.logMessage(null, "Centralized logging enabled", Level.INFO);
            }
        }
        else
        {
            if (currentlyEnabled)
            {
                m_container.logMessage(null, "Centralized logging disabled", Level.INFO);
            }
            m_centrallyLogMessages = Boolean.FALSE;
        }
    }

    private boolean getForceCentralizedLogging()
    {
        IElement domainConfig = super.m_context.getConfiguration(DOMAIN_CONFIG_ID, true);
        if (domainConfig == null)
        {
            return false;
        }

        IAttributeSet domainAttributes = domainConfig.getAttributes();
        Boolean forceCentralizedLogging = (Boolean)domainAttributes.getAttribute("FORCE_CENTRALIZED_LOGGING");

        if (forceCentralizedLogging == null)
        {
            return false;
        }

        return forceCentralizedLogging.booleanValue();
    }

    void logMessage(String message, int level) { super.m_context.logMessage(message, level); }

    // returns true if the message can now be disposed of, false if we should try again
    boolean centrallyLogMessage(String timestampedMessage)
    {
        if (m_centrallyLogMessages == null)
        {
            // not yet set so don't dispose of messages yet
            return false;
        }

        if (!m_centrallyLogMessages.booleanValue())
        {
            // don't centrally log so dispose of
            return true;
        }

        // fixup the message to contain the container ID
        if (timestampedMessage.startsWith("["))
        {
            if (timestampedMessage.indexOf("] ID=") == 18)
            {
                timestampedMessage = timestampedMessage.replaceFirst("] ID=", "] " + m_agentName.getContainerName() + ".ID=");
            }
            else
            if (timestampedMessage.indexOf("] (") == 18)
            {
                timestampedMessage = timestampedMessage.replaceFirst("] \\(", "] " + m_agentName.getContainerName() + " (");
            }
        }

        boolean result = false;
        try
        {
            super.m_frameworkContext.invoke(m_agentManager, LOG_MESSAGE_OPERATION_NAME, new Object[] { timestampedMessage }, LOG_MESSAGE_OPERATION_SIGNATURE, false, 0);
            result = true;
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Failed to centrally log message, trace follows...", e, Level.SEVERE);
            result = false;
        }
        try
        {
			Thread.sleep(25); // gives better chance the messages will appear in order in the centralized log since they are sent async
		}
        catch (InterruptedException e) { }
        return result;
    }

    private static IMetricInfo[] getMetricsInfo()
    {
        // create the infos
        IMetricInfo[] infos = new IMetricInfo[3];
        infos[0] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_MEMORY_CURRENTUSAGE_METRIC_ID, IValueType.VALUE,
                                                   "Heap space used by the container and its hosted components.",
                                                   null, false, true, true, false, "bytes" );
        infos[1] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_MEMORY_MAXUSAGE_METRIC_ID, IValueType.VALUE,
                                                   "Maximum heap space used by the container and its hosted components since last metrics reset.",
                                                   null, false, true, false, false, "bytes");
        infos[2] = MetricsFactory.createMetricInfo(IAgentProxy.SYSTEM_THREADS_CURRENTTOTAL_METRIC_ID, IValueType.VALUE,
                                                   "Total number of threads used by the container and its hosted components.",
                                                   null, false, true, true, false, "threads");
        return infos;
    }

    private void initMetrics()
    {
        // memory usage statistic provider and statistics
        IStatisticProvider[] memoryStatisticProviders = new IStatisticProvider[]
        {
            new IStatisticProvider()
            {
                private Runtime runtime = Runtime.getRuntime();
                @Override
                public void updateStatistic(ISampledStatistic statistic)
                {
                    statistic.updateValue(this.runtime.totalMemory() - this.runtime.freeMemory());
                }
                @Override
                public void resetStatistic(ISampledStatistic statistic) { } // no implementation
            }
        };
        m_memoryUsageStatistic = StatisticsFactory.createStatistic(IStatistic.VALUE_MODE, false, memoryStatisticProviders, (short)0);
        m_maxMemoryUsageStatistic = StatisticsFactory.createStatistic(IStatistic.MAXIMUM_MODE, false, memoryStatisticProviders, (short)0);;

        // thread usage statistic provider
        IStatisticProvider[] threadStatisticProviders = new IStatisticProvider[]
        {
            new IStatisticProvider()
            {
                private ThreadGroup rootThreadGroup = Agent.this.getRootThreadGroup();

                @Override
                public void updateStatistic(ISampledStatistic statistic)
                {
                    statistic.updateValue(this.rootThreadGroup.activeCount());
                }
                @Override
                public void resetStatistic(ISampledStatistic statistic) { } // no implementation
            }
        };
        m_totalThreadsStatistic = StatisticsFactory.createStatistic(IStatistic.VALUE_MODE, false, threadStatisticProviders, (short)0);
    }

    private ThreadGroup getRootThreadGroup()
    {
        ThreadGroup threadGroup = Thread.currentThread().getThreadGroup();

        while (threadGroup.getParent() != null)
        {
            threadGroup = threadGroup.getParent();
        }

        return threadGroup;
    }

    long getNotificationInterval()
    {
        return m_containerStateNotifier == null ? JMSConnectorServer.NOTIFICATION_TTL : m_containerStateNotifier.getNotificationInterval();
    }

    private class SdfMFTracingIntegration extends com.sonicsw.sdf.AbstractMFComponentTracing
    {
         private boolean m_updateTraceLevelWasCalled;

         SdfMFTracingIntegration()
         {
             super("sonic.mf.agent", getTraceMaskValues());
             m_updateTraceLevelWasCalled = false;

             setTraceMask();
         }
         
         private void setTraceMask() {
             setTraceMask(new Integer(Agent.this.m_traceMask));
         }

         boolean wasUpdated()
         {
             return m_updateTraceLevelWasCalled;
         }

         @Override
        public void updateTraceLevel(String doiIDNotUsed, java.util.HashMap parameters, StringBuffer buffer)
         {
             super.updateTraceLevel(doiIDNotUsed, parameters, buffer);
             m_updateTraceLevelWasCalled = true;
             Agent.this.setTraceMask(getCurrentMask(), true);
         }
    }

}
