/* Copyright (c) 2009 Progress Software Corporation.  All Rights Reserved. */
package com.sonicsw.mf.framework.agent;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;

import com.sonicsw.sdf.AbstractDiagnosticsProvider;
import com.sonicsw.sdf.DiagnosticsManagerAccess;
import com.sonicsw.sdf.IDiagnosticsConstants;
import com.sonicsw.sdf.IDiagnosticsManager;
import com.sonicsw.sdf.IStateWriter;
import com.sonicsw.sdf.ITracer;

import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;

public class ClassLoaderFactory
{
    private static final boolean DEBUG = false;

    private static HashMap m_delegatingLoaders = new HashMap();
    private static HashMap m_delegateLoaders = new HashMap();
    private static URLClassLoader m_globalLoader;

    // This table provides a way to keep track of which component IDs own a given
    // shared library loader. The table is keyed by the delegate loader name with a value
    // of the owning component ID. It is used during component reloading to determine
    // dependent components.
    private static HashMap m_delegateLoaderOwners = new HashMap();

    // This table provides a way to keep track of which component IDs are using
    // particular delegate loaders and ultimately allows for a given delegate loader
    // to be released when their are no more components using it. The table is keyed
    // by the delegate loader's delegateName and the values are HashSets of
    // component ID's. Since any programmatically created delegating loaders will
    // be created via a component's context, these will still be tracked under the same
    // ID.
    private static HashMap m_delegateLoaderUsages = new HashMap();

    private static volatile IContainer m_container;
    private static int m_traceMask;

    // NOTE: Don't use the constants from IEmptyArray to reduce class loading requirements
    //       for the boot loader
    public static final String[] EMPTY_STRING_ARRAY = new String[0];
    public static final URL[] EMPTY_URL_ARRAY = new URL[0];

    private static String m_devPrivateClasspath = System.getProperty(IContainer.MF_DEV_PRIVATE_CLASSPATH_PROPERTY);
    private static String m_devSharedClasspath = System.getProperty(IContainer.MF_DEV_SHAREDLIBRARY_CLASSPATH_PROPERTY);

    // SDF debugging
    private static final int SDF_TRACE_NOTHING = 0;
    private static final int SDF_TRACE_NAME = 1;
    private static final int SDF_TRACE_LOADER_NAME = 2;
    private static final int SDF_TRACE_LOADED_PATH = 4;
    private static final int SDF_TRACE_LOADING_THREAD = 8;

    private static final String SDF_DOIID_CLASSES = "Classes";
    private static final String SDF_DOIID_RESOURCES = "Resources";

    public static Class m_classLoaderFactoryClass;
    private static final Class[] TRACE_LOADING_SIGNATURE = new Class[] { String.class };
    
    public static int m_sdfClassTraceMask = SDF_TRACE_NOTHING;
    public static int m_sdfResourceTraceMask = SDF_TRACE_NOTHING;

    private static IDiagnosticsManager m_diagnosticsManager;
    private static ClassLoaderDiagnostics m_classLoaderDiagnostics;

    static
    {
        if (ClassLoaderFactory.class.getClassLoader().getClass().getName().equals(DelegatingLoader.class.getName()))
        {
            m_classLoaderFactoryClass = ClassLoaderFactory.class;
            m_diagnosticsManager = DiagnosticsManagerAccess.createManager();
            m_classLoaderDiagnostics = new ClassLoaderDiagnostics();
            m_classLoaderDiagnostics.register();
        }
    }

    // non-sdf tracing
    static void setTraceMask(int traceMask, IContainer container)
    {
        m_traceMask = traceMask;
        m_container = container;
    }

    public static synchronized ClassLoader createDelegatingLoader(String containerName, String id, URL[] classpath, ClassLoader parent, String[] delegateLoaderNames)
    {
        // any development overrides?
        if (m_devPrivateClasspath != null)
        {
            classpath = prependClasspath(m_devPrivateClasspath, classpath);
        }

        escapeURLs(classpath);

        ClassLoader loader = new DelegatingLoader(containerName, id, classpath, parent, delegateLoaderNames);
        m_delegatingLoaders.put(containerName + ':' + id, loader);
        addDelegateLoaderUsages(containerName, id, delegateLoaderNames);

        return loader;
    }

    public static synchronized URLClassLoader getDelegatingLoader(String containerName, String id)
    {
        return (URLClassLoader)m_delegatingLoaders.get(containerName + ':' + id);
    }

    public static synchronized ClassLoader createDelegateLoader(String containerName, String id, String delegateName, URL[] classpath, ClassLoader parent)
    {
        // any development overrides?
        if (m_devSharedClasspath != null)
        {
            classpath = prependClasspath(m_devSharedClasspath, classpath);
        }

        escapeURLs(classpath);

        ClassLoader loader = (ClassLoader)m_delegateLoaders.get(containerName + ':' + delegateName);
        if (loader == null)
        {
            loader = new DelegateLoader(containerName, delegateName, classpath, parent);
            if (DEBUG)
            {
                System.out.println(">> Created delegate loader [" + containerName + ':' + delegateName + "]: " + loader);
            }
        }

        // always need to add the loader in case it was created due to a dependency, but is not
        // currently assigned to the given id
        addDelegateLoader(containerName, id, delegateName, loader);

        return loader;
    }

    public static synchronized URLClassLoader createGlobalLoader(String containerName, URL[] classpath, ClassLoader parent)
    {
        // we will use just one global loader for LSD
        if (m_globalLoader != null && Boolean.getBoolean(IContainer.MF_LSD_COLOCATE_PROPERTY))
        {
            return m_globalLoader;
        }

        // any development overrides?
        String devClasspath = System.getProperty(IContainer.MF_DEV_GLOBAL_CLASSPATH_PROPERTY);
        if (devClasspath != null)
        {
            classpath = prependClasspath(devClasspath, classpath);
        }

        escapeURLs(classpath);

        m_globalLoader = new GlobalLoader(containerName, classpath, parent);

        return m_globalLoader;
    }

    // For use by framework on boot to reregister the global loader created during boot into the agent's class loader namespace
    // (rather than the java default class loader namespace)
    public static synchronized void setGlobalLoader(URLClassLoader globalLoader)
    {
        if (m_globalLoader == null)
        {
            m_globalLoader = globalLoader;
        }
    }

    // For use by framework on boot to reregister agent loader created during boot into the agent's class loader namespace
    // (rather than the java default class loader namespace)
    public static synchronized void addDelegatingLoader(String containerName, String id, String[] delegateLoaderNames, ClassLoader loader)
    {
        m_delegatingLoaders.put(containerName + ':' + id, loader);
        addDelegateLoaderUsages(containerName, id, delegateLoaderNames);
    }

    // For use by framework on boot to reregister delegates created during boot into the agent's class loader namespace
    // (rather than the java default class loader namespace)
    public static synchronized void addDelegateLoader(String containerName, String id, String delegateName, ClassLoader loader)
    {
        m_delegateLoaders.put(containerName + ':' + delegateName, loader);
        if (id != null)
        {
            m_delegateLoaderOwners.put(containerName + ':' + delegateName, containerName + ':' + id);
        }
    }

    // For use by framework on boot to reregister delegate usages created during boot into the agent's class loader namespace
    // (rather than the java default class loader namespace)
    private static synchronized void addDelegateLoaderUsages(String containerName, String id, String[] delegateNames)
    {
        if (delegateNames == null || delegateNames.length == 0)
        {
            return;
        }

        for (int i = 0; i < delegateNames.length; i++)
        {
            HashSet delegateLoaderUsage = (HashSet)m_delegateLoaderUsages.get(containerName + ':' + delegateNames[i]);
            if (delegateLoaderUsage == null)
            {
                delegateLoaderUsage = new HashSet();
                m_delegateLoaderUsages.put(containerName + ':' + delegateNames[i], delegateLoaderUsage);
            }
            delegateLoaderUsage.add(containerName + ':' + id);
        }
    }

    public static synchronized void removeLoaderUsages(String containerName, String id)
    {
        Iterator iterator = m_delegateLoaderUsages.entrySet().iterator();

        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();
            HashSet delegateLoaderUsage = (HashSet)entry.getValue();
            if (delegateLoaderUsage.remove(containerName + ':' + id) && delegateLoaderUsage.isEmpty())
            {
                m_delegateLoaders.remove(entry.getKey());
                iterator.remove();
            }
        }

        iterator = m_delegateLoaderOwners.entrySet().iterator();
        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();
            String ownerID = (String)entry.getValue();
            if (ownerID.equals(containerName + ':' + id))
            {
                iterator.remove();
                break;
            }
        }

        m_delegatingLoaders.remove(containerName + ':' + id);
    }

    // get the list of components that share the given component's shared library
    public static synchronized String[] getDependentComponents(String containerName, String id)
    {
        // find if this component declared a shared library
        Iterator iterator = m_delegateLoaderOwners.entrySet().iterator();
        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();
            String ownerID = (String)entry.getValue();
            if (ownerID.equals(containerName + ':' + id))
            {
                String delegateName = (String)entry.getKey();
                // find components that are using this shared loader
                HashSet ids = (HashSet)m_delegateLoaderUsages.get(delegateName);
                return (String[])ids.toArray(EMPTY_STRING_ARRAY);
            }
        }

        return EMPTY_STRING_ARRAY;
    }

    public static URL[] prependClasspath(String addClasspath, URL[] existingClasspath)
    {
        ArrayList aggregatedClasspath = new ArrayList();

        StringTokenizer st = new StringTokenizer(addClasspath, File.pathSeparator);
        while (st.hasMoreTokens())
        {
            String pathElement = st.nextToken();
            if (pathElement.length() > 0)
            {
                try
                {
                    aggregatedClasspath.add(new File(pathElement).toURL());
                }
                catch(Exception e) { e.printStackTrace(); }
            }
        }

        for (int i = 0; i < existingClasspath.length; i++)
        {
            aggregatedClasspath.add(existingClasspath[i]);
        }

        return (URL[])aggregatedClasspath.toArray(EMPTY_URL_ARRAY);
    }

    public static class DelegatingLoader
    extends URLClassLoader
    {
        private String containerName;
        private String id;
        private String[] delegateLoaderNames;
        private HashSet notFounds = new HashSet();

        private DelegatingLoader(String containerName, String id, URL[] urls, ClassLoader parent, String[] delegateLoaderNames)
        {
            super(urls, parent);

            this.containerName = containerName;
            this.id = id;
            this.delegateLoaderNames = delegateLoaderNames;

            if (DEBUG)
            {
                String loaderNames = "";
                if (delegateLoaderNames != null)
                {
                    for (int i = 0; i < delegateLoaderNames.length; i++)
                    {
                        loaderNames += (i == 0 ? "" : ", ") + delegateLoaderNames[i];
                    }
                }
                System.out.println(">> Created delegating loader [delegates=" + loaderNames + "]: " + this);
            }
        }
        
        public String[] getDelegateLoaderNames() { return (String[])this.delegateLoaderNames.clone(); }

        @Override
        public String toString()
        {
            return  '[' + containerName + ':' + id + "] " + DelegatingLoader.class.getName();
        }

        @Override
        public Class loadClass(String name)
        throws ClassNotFoundException
        {
            return loadClass(name, false);
        }

        @Override
        protected synchronized Class loadClass(String name, boolean resolve)
        throws ClassNotFoundException
        {
            Class loadedClass = null;

            try
            {
                if (name.startsWith("["))
                {
                    loadedClass = Class.forName(name, false, this);
                    return loadedClass;
                }

                // has this class already been loaded
                loadedClass = findLoadedClass(name);
                if (loadedClass != null)
                {
                    return loadedClass;
                }

                String classFilename = name.replace('.', '/') + ".class";

                // should this class be loaded by the global loader
                URL url = ClassLoaderFactory.m_globalLoader.findResource(classFilename);
                if (url != null) // then it was found by the global loader and we should delegate there
                {
                    loadedClass = m_globalLoader.loadClass(name);
                    return loadedClass;
                }

                // should this class be loaded by any of the shared loaders this loader delegates to
                if (this.delegateLoaderNames != null)
                {
                    for (int i = 0; i < this.delegateLoaderNames.length; i++)
                    {
                        URLClassLoader loader = (URLClassLoader)ClassLoaderFactory.m_delegateLoaders.get(this.containerName + ':' + delegateLoaderNames[i]);
                        if (loader != null)
                        {
                            url = loader.findResource(classFilename);
                            if (url != null)
                            {
                                loadedClass = loader.loadClass(name);
                                return loadedClass;
                            }
                        }
                    }
                }

                // so now try self-first loading from this loader
                try
                {
                    loadedClass = findClass(name);
                    ClassLoaderFactory.traceClassLoading(loadedClass, this, findResource(classFilename));
                    return loadedClass;
                }
                catch(ClassNotFoundException e) { } // ignore and move on
                catch(NoClassDefFoundError e) { } // ignore and move on

                // otherwise try regular class loading
                try
                {
                    loadedClass = super.loadClass(name, resolve);
                    return loadedClass;
                }
                catch (java.lang.LinkageError e)
                {
                    logClassLoadingFailure(name, e);
                    throw e;
                }
                catch (ClassNotFoundException e)
                {
                    logClassLoadingFailure(name, e);
                    throw e;
                }
            }
            finally
            {
                if (loadedClass != null && resolve)
                {
                    resolveClass(loadedClass);
                }
            }
        }

        @Override
        public URL getResource(String name)
        {
            synchronized (this.notFounds)
            {
                if (this.notFounds.contains(name))
                {
                    return null;
                }
            }

            URL url = null;

            // should this resource be loaded by the global loader
            url = ClassLoaderFactory.m_globalLoader.findResource(name);
            if (url != null) // then it was found by the global loader and we should delegate there
            {
                if (!name.endsWith(".class"))
                {
                    traceResourceLoading(name, ClassLoaderFactory.m_globalLoader, url);
                }
                return url;
            }

            // should this resource be loaded by any of the shared loaders this loader delegates to
            if (this.delegateLoaderNames != null)
            {
                for (int i = 0; i < this.delegateLoaderNames.length; i++)
                {
                    URLClassLoader loader = (URLClassLoader)ClassLoaderFactory.m_delegateLoaders.get(this.containerName + ':' + delegateLoaderNames[i]);
                    if (loader != null)
                    {
                        url = loader.findResource(name);
                        if (url != null)
                        {
                            if (!name.endsWith(".class"))
                            {
                                traceResourceLoading(name, loader, url);
                            }
                            return url;
                        }
                    }
                }
            }

            // so now try self-first loading from this loader
            url = findResource(name);
            if (url != null)
            {
                if (!name.endsWith(".class"))
                {
                    traceResourceLoading(name, this, url);
                }
                return url;
            }

            // otherwise try regular class loading
            url = super.getResource(name);
            if (url != null)
            {
                if (!name.endsWith(".class"))
                {
                    traceResourceLoading(name, null, url);
                }
            }

            if (url == null)
            {
                synchronized (this.notFounds)
                {
                    this.notFounds.add(name);
                }
            }

            return url;
        }

        @Override
        public InputStream getResourceAsStream(String name)
        {
            URL url = getResource(name);
            try
            {
                return url != null ? url.openStream() : null;
            }
            catch (IOException e)
            {
                return null;
            }
        }

        private void logClassLoadingFailure(String name, Throwable throwable)
        {
            if (ClassLoaderFactory.m_container == null)
            {
                return;
            }
            // don't trace lack of translation bundles
            if (name.indexOf(".prBundle_") > 0)
            {
                return;
            }
            if ((ClassLoaderFactory.m_traceMask & Agent.TRACE_CLASS_LOADING_FAILURES) > 0)
            {
                String logID = this.id.equals(IAgentProxy.ID) ? null : this.id;
                boolean traceDetail = true; //(ClassLoaderFactory.m_traceMask & Agent.TRACE_DETAIL) > 0;
                StringBuffer sb = new StringBuffer("Failed to load class (").append(name).append(")...");
                sb.append(IContainer.NEWLINE);
                sb.append("Class loader graph...");
                sb.append(IContainer.NEWLINE);
                sb.append("\tPrivate loader: " + this);
                if (traceDetail)
                {
                    appendURLs(sb, ((URLClassLoader)this).getURLs());
                }
                if (this.delegateLoaderNames != null)
                {
                    for (int i = 0; i < this.delegateLoaderNames.length; i++)
                    {
                        ClassLoader loader = (ClassLoader)m_delegateLoaders.get(this.containerName + ':' + delegateLoaderNames[i]);
                        if (loader != null)
                        {
                            sb.append(IContainer.NEWLINE);
                            sb.append("\tShared loader : " + loader + " (" + this.containerName + ':' + delegateLoaderNames[i] + ")");
                            if (traceDetail)
                            {
                                appendURLs(sb, ((URLClassLoader)loader).getURLs());
                            }
                        }
                    }
                }
                sb.append(IContainer.NEWLINE);
                sb.append("\tGlobal loader : " + this.getParent());
                if (traceDetail)
                {
                    appendURLs(sb, ((URLClassLoader)this.getParent()).getURLs());
                    if (ClassLoaderFactory.m_container == null)
                    {
                        System.out.println(sb.toString());
                    }
                    else
                    {
                        ClassLoaderFactory.m_container.logMessage(logID, sb.toString(), throwable, Level.TRACE);
                    }
                }
                else
                {
                    sb.append(IContainer.NEWLINE);
                    sb.append(throwable.getClass().getName()).append(": ").append(name);
                    ClassLoaderFactory.m_container.logMessage(logID, sb.toString(), Level.TRACE);
                }
            }
        }

        private void appendURLs(StringBuffer sb, URL[] urls)
        {
            for (int i = 0; i < urls.length; i++)
            {
                sb.append(IContainer.NEWLINE).append("\t\t").append(urls[i].toExternalForm());
            }
        }
    }

    public static class DelegateLoader
    extends URLClassLoader
    {
        private String containerName;
        private String delegateName;
        private HashSet notFounds = new HashSet();

        private DelegateLoader(String containerName, String delegateName, URL[] urls, ClassLoader parent)
        {
            super(urls, parent);

            this.containerName = containerName;
            this.delegateName = delegateName;
        }

        @Override
        public String toString()
        {
            return '[' + containerName + ':' + delegateName + "] " + DelegateLoader.class.getName();
        }

        @Override
        public Class loadClass(String name)
        throws ClassNotFoundException
        {
            return loadClass(name, false);
        }

        @Override
        protected synchronized Class loadClass(String name, boolean resolve)
        throws ClassNotFoundException
        {
            Class loadedClass = null;

            try
            {
                if (name.startsWith("["))
                {
                    loadedClass = Class.forName(name, false, this);
                    return loadedClass;
                }

                // has this class already been loaded
                loadedClass = findLoadedClass(name);
                if (loadedClass != null)
                {
                    return loadedClass;
                }

                String classFilename = name.replace('.', '/') + ".class";

                // should this class be loaded by the global loader
                URL url = ClassLoaderFactory.m_globalLoader.findResource(classFilename);
                if (url != null) // then it was found by the global loader and we should delegate there
                {
                    loadedClass = m_globalLoader.loadClass(name);
                    return loadedClass;
                }

                // so now try self-first loading from this loader
                try
                {
                    loadedClass = findClass(name);
                    ClassLoaderFactory.traceClassLoading(loadedClass, this, this.findResource(classFilename));
                    return loadedClass;
                }
                catch(ClassNotFoundException e) { } // ignore and move on
                catch(NoClassDefFoundError e) { } // ignore and move on

                // otherwise try regular class loading
                loadedClass = super.loadClass(name, resolve);
                return loadedClass;
            }
            finally
            {
                if (loadedClass != null && resolve)
                {
                    resolveClass(loadedClass);
                }
            }
        }

        @Override
        public URL getResource(String name)
        {
            synchronized (this.notFounds)
            {
                if (this.notFounds.contains(name))
                {
                    return null;
                }
            }

            URL url = null;

            // should this resource be loaded by the global loader
            url = ClassLoaderFactory.m_globalLoader.findResource(name);
            if (url != null) // then it was found by the global loader and we should delegate there
            {
                if (!name.endsWith(".class"))
                {
                    traceResourceLoading(name, ClassLoaderFactory.m_globalLoader, url);
                }
                return url;
            }

            // so now try self-first loading from this loader
            url = findResource(name);
            if (url != null)
            {
                if (!name.endsWith(".class"))
                {
                    traceResourceLoading(name, this, url);
                }
                return url;
            }

            // otherwise try regular class loading
            url = super.getResource(name);
            if (url != null)
            {
                if (!name.endsWith(".class"))
                {
                    traceResourceLoading(name, null, url);
                }
            }

            if (url == null)
            {
                synchronized (this.notFounds)
                {
                    this.notFounds.add(name);
                }
            }

            return url;
        }

        @Override
        public InputStream getResourceAsStream(String name)
        {
            URL url = getResource(name);
            try
            {
                return url != null ? url.openStream() : null;
            }
            catch (IOException e)
            {
                return null;
            }
        }
    }

    public static class GlobalLoader
    extends URLClassLoader
    {
        private String containerName;
        private HashSet notFounds = new HashSet();

        private GlobalLoader(String containerName, URL[] urls, ClassLoader parent)
        {
            super(urls, parent);

            this.containerName = containerName;
        }

        @Override
        public String toString()
        {
            return  '[' + containerName + ":GLOBAL] " + GlobalLoader.class.getName();
        }
        @Override
        public Class loadClass(String name)
        throws ClassNotFoundException
        {
            return loadClass(name, false);
        }

        @Override
        protected synchronized Class loadClass(String name, boolean resolve)
        throws ClassNotFoundException
        {
            Class loadedClass = null;
            
            if (name.startsWith("["))
            {
                loadedClass = Class.forName(name, false, this);
                return loadedClass;
            }

            loadedClass = findLoadedClass(name);
            if (loadedClass != null)
            {
                return loadedClass;
            }

            loadedClass = super.loadClass(name, resolve);
            ClassLoaderFactory.traceClassLoading(loadedClass, loadedClass.getClassLoader(), getResource(name.replace('.', '/') + ".class"));

            return loadedClass;
        }

        @Override
        public URL getResource(String name)
        {
            synchronized (this.notFounds)
            {
                if (this.notFounds.contains(name))
                {
                    return null;
                }
            }

            URL url = super.getResource(name);

            if (url != null && !name.endsWith(".class"))
            {
                // find out which loader loaded the resource and trace the information
                ClassLoader loader = this;
                while (loader.getParent() != null)
                {
                    if (loader.getParent().getResource(name) == null)
                     {
                        break; // means it must have been loaded by the current loader
                    }
                    loader = loader.getParent();
                }
                traceResourceLoading(name, loader, url);
            }

            if (url == null)
            {
                synchronized (this.notFounds)
                {
                    this.notFounds.add(name);
                }
            }

            return url;
        }

        @Override
        public InputStream getResourceAsStream(String name)
        {
            URL url = getResource(name);
            try
            {
                return url != null ? url.openStream() : null;
            }
            catch (IOException e)
            {
                return null;
            }
        }
    }

    private static void escapeURLs(URL[] urls)
    {
    	for (int i=0; i< urls.length; i++)
        {
        	if (urls[i].toExternalForm().indexOf(" ") > -1)
            {
                urls[i] = escapeURL(urls[i]);
            }
        }
    }

    public static void traceClassLoading(Class loadedClass, ClassLoader loader, URL url)
    {
        if (m_classLoaderFactoryClass != null)
        {
            try
            {
                Field sdfClassTraceMaskField = m_classLoaderFactoryClass.getField("m_sdfClassTraceMask");
                int sdfClassTraceMask = sdfClassTraceMaskField.getInt(null);
                
                if (sdfClassTraceMask > ClassLoaderFactory.SDF_TRACE_NOTHING)
                {
                    String sdfTraceMessage = ClassLoaderFactory.getSDFTraceMessage(sdfClassTraceMask, "class", loadedClass.getName(), loader, url);
                    Method traceLoadingMethod = m_classLoaderFactoryClass.getMethod("traceLoading", TRACE_LOADING_SIGNATURE);
                    traceLoadingMethod.invoke(null, new Object[] { sdfTraceMessage });
                }
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
        }
    }

    public static void traceResourceLoading(String name, ClassLoader loader, URL url)
    {
        if (m_classLoaderFactoryClass != null)
        {
            try
            {
                Field sdfResourceTraceMaskField = m_classLoaderFactoryClass.getField("m_sdfResourceTraceMask");
                int sdfResourceTraceMask = sdfResourceTraceMaskField.getInt(null);
                
                if (sdfResourceTraceMask > ClassLoaderFactory.SDF_TRACE_NOTHING)
                {
                    String sdfTraceMessage = ClassLoaderFactory.getSDFTraceMessage(sdfResourceTraceMask, "resource", name, loader, url);
                    Method traceLoadingMethod = m_classLoaderFactoryClass.getMethod("traceLoading", TRACE_LOADING_SIGNATURE);
                    traceLoadingMethod.invoke(null, new Object[] { sdfTraceMessage });
                }
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
        }
    }
    
    private static String getSDFTraceMessage(int traceMask, String type, String name, ClassLoader loader, URL path)
    {
        StringBuffer sb = new StringBuffer();
    
        if ((traceMask & ClassLoaderFactory.SDF_TRACE_NAME) > 0)
        {
            sb.append("Loaded ").append(type).append(": ").append(name);
        }
        if ((traceMask & ClassLoaderFactory.SDF_TRACE_LOADER_NAME) > 0)
        {
            if (sb.length() > 0)
            {
                sb.append(IDiagnosticsConstants.NEWLINE);
            }
            sb.append("  - Loader: ").append(loader == null ? "<system loader>" : loader.toString());
        }
        if ((traceMask & ClassLoaderFactory.SDF_TRACE_LOADED_PATH) > 0)
        {
            if (sb.length() > 0)
            {
                sb.append(IDiagnosticsConstants.NEWLINE);
            }
            sb.append("  - Path  : ").append(path.toExternalForm());
        }
        if ((traceMask & ClassLoaderFactory.SDF_TRACE_LOADING_THREAD) > 0)
        {
            if (sb.length() > 0)
            {
                sb.append(IDiagnosticsConstants.NEWLINE);
            }
            sb.append("  - Thread: \"").append(Thread.currentThread()).append("\" of group \"").append(Thread.currentThread().getThreadGroup()).append('"');
        }
        return sb.toString();
    }

    
    public static void traceLoading(String traceMessage)
    {
        ClassLoaderFactory.m_classLoaderDiagnostics.traceLoading(traceMessage);
    }

    private static URL escapeURL( URL url )
    {
        URL result = url;

        try
        {
            StringBuffer newURL = new StringBuffer("");
            StringTokenizer token = new StringTokenizer( url.toExternalForm(), " ", true );
            while( token.hasMoreTokens() )
            {
                String newToken = token.nextToken();

                if( newToken.equals(" ") )
                {
                    newURL.append("%20");
                }
                else
                {
                    newURL.append( newToken );
                }
            }

            result = new URL( newURL.toString() );
        }
        catch( MalformedURLException e )
        {
            // Use the passed in URL
        }

        return result;
    }

    private static class ClassLoaderDiagnostics
    extends AbstractDiagnosticsProvider
    {
        private static String[] OPERATIONS;
        private static HashMap SHOW_TRACE_LEVEL_PARAM_DESCIPTOR = new HashMap();
        private static HashMap UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR = new HashMap();
        private static HashMap PARAM_DESCRIPTOR = new HashMap();
        private static HashMap DUMP_STATE_PARAM_DESCIPTOR = new HashMap();
        private static HashMap DESCRIBE_PARAM_DESCIPTOR = new HashMap();
        private static HashMap LIST_DIAGNOSTICS_INSTANCES_PARAM_DESCIPTOR = new HashMap();

        String description = "Use this subsystem to help diagnose the Sonic MF specific class loading hierarchy."
                           + NEWLINE
                           + NEWLINE
                           + "The diagnosis includes tracing of class and resource loading and dumping of the current class loader hierarchy."
                           + NEWLINE
                           + "Two diagnosed object instances are used to distinguish between classes (\"Classes\") and resources (\"Resources\")."
                           + NEWLINE
                           + "Tracing can be configured so that in addition to loaded class or resource names, the associated loader and/or path can be traced."
                           + NEWLINE
                           + NEWLINE
                           + "Examples:"
                           + NEWLINE
                           + "  sonic.mf.classloading updateTraceLevel doiID=Classes integerTraceLevel=7"
                           + NEWLINE
                           + "  sonic.mf.classloading updateTraceLevel doiID=Resources integerTraceLevel=5"
                           + NEWLINE;

        static
        {
            UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR.put(DIAGNOSTICS_OBJECT_ID_PARAM, "values: \"" + SDF_DOIID_CLASSES + "\", \"" + SDF_DOIID_RESOURCES + "\"");
            UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR.put(INTEGER_TRACE_LEVEL_PARAM, "bits: 1 - trace name, 2 - trace class loader name, 4 - trace loaded path");

            PARAM_DESCRIPTOR.put(DUMP_STATE_OP, DUMP_STATE_PARAM_DESCIPTOR);
            PARAM_DESCRIPTOR.put(DESCRIBE_OP, DESCRIBE_PARAM_DESCIPTOR);
            PARAM_DESCRIPTOR.put(SHOW_TRACE_LEVEL_OP, SHOW_TRACE_LEVEL_PARAM_DESCIPTOR);
            PARAM_DESCRIPTOR.put(UPDATE_TRACE_LEVEL_OP, UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR);
            PARAM_DESCRIPTOR.put(LIST_DIAGNOSTICS_INSTANCES_OP, LIST_DIAGNOSTICS_INSTANCES_PARAM_DESCIPTOR);

            OPERATIONS = toOpnameArray(PARAM_DESCRIPTOR);
        }

        ClassLoaderDiagnostics()
        {
            super("sonic.mf.classloading");
        }

        @Override
        public String describe()
        {
            return this.description;
        }

        @Override
        public String[] getOperations()
        {
            return OPERATIONS;
        }

        @Override
        public HashMap describeParameters(String operationName)
        {
            return (HashMap)PARAM_DESCRIPTOR.get(operationName);
        }

        @Override
        public String[] getDOInstances()
        {
            return new String[] { SDF_DOIID_CLASSES, SDF_DOIID_RESOURCES };
        }

        @Override
        public void updateTraceLevel(String doiID, HashMap parameters, StringBuffer buffer)
        {
            try
            {
                if (doiID.equals(SDF_DOIID_CLASSES))
                {
                    ClassLoaderFactory.m_sdfClassTraceMask = parseTraceLevel(doiID, parameters, buffer, ClassLoaderFactory.m_sdfClassTraceMask, ClassLoaderFactory.m_sdfResourceTraceMask);
                }
                if (doiID.equals(SDF_DOIID_RESOURCES))
                {
                    ClassLoaderFactory.m_sdfResourceTraceMask = parseTraceLevel(doiID, parameters, buffer, ClassLoaderFactory.m_sdfResourceTraceMask, ClassLoaderFactory.m_sdfClassTraceMask);
                }
            }
            catch (Exception e)
            {
                e.printStackTrace();
                buffer.append(e.toString());
            }
        }

        @Override
        public void showTraceLevel(String doiID, HashMap parameters, StringBuffer buffer)
        {
            if (doiID.equals(SDF_DOIID_CLASSES))
            {
                buffer.append("Trace level for \"").append(super.m_subsystemName).append("\" [").append(doiID).append("] is ").append(ClassLoaderFactory.m_sdfClassTraceMask);
            }
            if (doiID.equals(SDF_DOIID_RESOURCES))
            {
                buffer.append("Trace level for \"").append(super.m_subsystemName).append("\" [").append(doiID).append("] is ").append(ClassLoaderFactory.m_sdfResourceTraceMask);
            }
        }

        @Override
        public void appendStateDump(String doiID, HashMap Parameters, StringBuffer buffer)
        {
            if (ClassLoaderFactory.m_globalLoader == null)
            {
                buffer.append("Unable to write state of \"").append(super.m_subsystemName).append(" as core MF class loaders bhave not yet been created");
                return;
            }

            IStateWriter writer = null;
            try
            {
                writer = m_diagnosticsContext.getStateWriter();

                ArrayList parents = new ArrayList();
                parents.add(ClassLoaderFactory.m_globalLoader);
                while (((ClassLoader)parents.get(0)).getParent() != null)
                {
                    parents.add(0, ((ClassLoader)parents.get(0)).getParent());
                }
                for (int i = 0; i < parents.size(); i++)
                {
                    dumpClassLoader(writer, createIndent(i), (ClassLoader)parents.get(i));
                }

                Iterator sharedLoaders = ClassLoaderFactory.m_delegateLoaders.values().iterator();
                while (sharedLoaders.hasNext())
                {
                    dumpClassLoader(writer, createIndent(parents.size()), (ClassLoader)sharedLoaders.next());
                }

                Iterator privateLoaders = ClassLoaderFactory.m_delegatingLoaders.values().iterator();
                while (privateLoaders.hasNext())
                {
                    dumpClassLoader(writer, createIndent(parents.size()), (ClassLoader)privateLoaders.next());
                }
            }
            catch (Exception e)
            {
                buffer.append("Failed to use state file: " + e);
                return;
            }
            finally
            {
                if (writer != null)
                {
                    writer.close();
                }
            }
            
            if (writer != null) {
            	buffer.append("Dump of \"").append(super.m_subsystemName).append("\" written to ").append(writer.getFilePath());
            }
        }

        private String createIndent(int offset)
        {
            String indent = "";
            for (int i = 0; i < offset; i++)
            {
                indent += "  ";
            }

            return indent;
        }

        private void dumpClassLoader(IStateWriter writer, String indent, ClassLoader classLoader)
        throws Exception
        {
            if (classLoader.getClass().getName().equals("com.sonicsw.mf.framework.agent.ClassLoaderFactory$GlobalLoader"))
            {
                writer.write(indent + "+ <sonic global loader> " + classLoader.toString());
            }
            else
            if (classLoader.getClass().getName().startsWith("com.sonicsw.mf.framework.agent.ClassLoaderFactory$DelegateLoader"))
            {
                writer.write(indent + "+ <sonic delegate/shared loader> " + classLoader.toString());
            }
            else
            if (classLoader.getClass().getName().startsWith("com.sonicsw.mf.framework.agent.ClassLoaderFactory$DelegatingLoader"))
            {
                writer.write(indent + "+ <sonic delegating/private loader> " + classLoader.toString());
            }
            else
            {
                writer.write(indent + "+ <system loader> " + classLoader.getClass().getName());
            }
            writer.write(NEWLINE);

            if (classLoader.getClass().getName().startsWith("com.sonicsw.mf.framework.agent.ClassLoaderFactory$DelegatingLoader"))
            {
                writer.write(indent + "  + <depends on>");
                writer.write(NEWLINE);
                Class delegatingLoaderClass = classLoader.getClass();
                Method getDelegateLoaderNamesMethod = delegatingLoaderClass.getMethod("getDelegateLoaderNames", new Class[] { });
                String[] delegates = (String[])getDelegateLoaderNamesMethod.invoke(classLoader, new Object[0]);
                if (delegates == null || delegates.length == 0)
                {
                    writer.write(indent + "    - <none>");
                    writer.write(NEWLINE);
                }
                else
                {
                    for (int i = 0; i < delegates.length; i++)
                    {
                        writer.write(indent + "    - [" + delegates[i] + ']');
                        writer.write(NEWLINE);
                    }
                }
            }

            if (classLoader instanceof URLClassLoader)
            {
                writer.write(indent + "  + <urls>");
                writer.write(NEWLINE);
                URL[] urls = ((URLClassLoader)classLoader).getURLs();
                for (int i = 0; i < urls.length; i++)
                {
                    writer.write(indent + "    - " + urls[i].toExternalForm());
                    writer.write(NEWLINE);
                }
            }
        }

        private int parseTraceLevel(String doiID, HashMap parameters, StringBuffer buffer, int currentTraceMask, int otherTraceMask)
        throws Exception
        {
            String traceLevel = (String)parameters.get(INTEGER_TRACE_LEVEL_PARAM);
            if (traceLevel == null)
            {
                traceLevel = new String("0");
            }

            int traceMask = Integer.parseInt(traceLevel);

            if (traceMask == SDF_TRACE_NOTHING)
            {
                if (currentTraceMask > 0 && otherTraceMask == 0)
                {
                    if (super.m_tracer != null)
                    {
                        super.m_tracer.close();
                        super.m_tracer = null;
                    }
                }
            }
            if (traceMask > SDF_TRACE_NOTHING)
            {
                if (super.m_tracer == null)
                {
                    super.m_tracer = super.m_diagnosticsContext.getTracer();
                }
                buffer.append("Trace file for \"").append(super.m_subsystemName).append("\" [").append(doiID).append("] is \"").append(super.m_tracer.getFilePath()).append("\"");
                super.m_tracer.trace("Start tracing " + super.m_subsystemName + '.' + doiID + " with tracing level " + traceLevel, false);
            }

            return traceMask;
        }

        private void traceLoading(String traceMessage)
        {
            ITracer tracer = m_tracer;
            if (tracer != null)
            {
                try
                {
                    tracer.trace(traceMessage, false);
                }
                catch (IOException e)
                {
                    e.printStackTrace();
                }
            }
        }
    }
}
