package com.sonicsw.mf.framework.agent;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanFeatureInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.ReflectionException;
import javax.naming.Context;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.IComponent;
import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeChangeHandler;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.AttributeSet;
import com.sonicsw.mf.common.config.impl.IChangeRegistration;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.metrics.IAlert;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IMetricsData;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.manager.IMetricsManager;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.metrics.manager.impl.MetricsManager;
import com.sonicsw.mf.common.runtime.ICanonicalName;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.jmx.client.MFNotification;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;

public class ComponentMBean
extends AbstractMBean
{
    private IMetricsManager m_metricsManager;

    private static Method CLEAR_ERROR_METHOD;
    private static Method HANDLE_NOTIFICATION_METHOD;
    private static Method GET_METRICS_INFO_METHOD;
    private static Method ENABLE_METRICS_METHOD;
    private static Method DISABLE_METRICS_METHOD;
    private static Method REPLACE_ENABLED_METRICS_METHOD;
    private static Method GET_ACTIVE_METRICS_METHOD;
    private static Method GET_INSTANCE_METRIC_NAMES_METHOD;
    private static Method GET_ENABLED_METRICS_METHOD;
    private static Method GET_METRICS_DATA_METHOD;
    private static Method RESET_METRICS_METHOD;
    private static Method GET_ENABLED_ALERTS_METHOD;
    private static Method ENABLE_ALERTS_METHOD;
    private static Method DISABLE_ALERTS_METHOD;
    private static Method REPLACE_ENABLED_ALERTS_METHOD;

    private static final String NOTIFICATION_CLASSNAME = INotification.class.getName();
    private static final String METRIC_IDENTITY_ARRAY_CLASSNAME = IMetricIdentity[].class.getName();
    private static final String ALERT_ARRAY_CLASSNAME = IAlert[].class.getName();
    private static final Class[] HANDLE_NOTIFICATION_CLASS_ARRAY = new Class[] { INotification.class };
    private static final Class[] METRIC_IDENTITY_ARRAY_CLASS_ARRAY = new Class[] { IMetricIdentity[].class };
    private static final Class[] ALERT_ARRAY_CLASS_ARRAY = new Class[] { IAlert[].class };

    static
    {
        try
        {
            CLEAR_ERROR_METHOD = AbstractMBean.class.getDeclaredMethod("clearError", IEmptyArray.EMPTY_CLASS_ARRAY);
            HANDLE_NOTIFICATION_METHOD = AbstractMBean.class.getDeclaredMethod("handleNotification", HANDLE_NOTIFICATION_CLASS_ARRAY);
            GET_METRICS_INFO_METHOD = IMetricsManager.class.getDeclaredMethod("getMetricsInfo", IEmptyArray.EMPTY_CLASS_ARRAY);
            ENABLE_METRICS_METHOD = IMetricsManager.class.getDeclaredMethod("enableMetrics", METRIC_IDENTITY_ARRAY_CLASS_ARRAY);
            DISABLE_METRICS_METHOD = IMetricsManager.class.getDeclaredMethod("disableMetrics", METRIC_IDENTITY_ARRAY_CLASS_ARRAY);
            REPLACE_ENABLED_METRICS_METHOD = MetricsManager.class.getDeclaredMethod("replaceEnabledMetrics", METRIC_IDENTITY_ARRAY_CLASS_ARRAY);
            GET_ACTIVE_METRICS_METHOD = IMetricsManager.class.getDeclaredMethod("getActiveMetrics", METRIC_IDENTITY_ARRAY_CLASS_ARRAY);
            GET_INSTANCE_METRIC_NAMES_METHOD = IComponent.class.getDeclaredMethod("getInstanceMetricNames", new Class[] { IMetricIdentity.class });
            GET_ENABLED_METRICS_METHOD = IMetricsManager.class.getDeclaredMethod("getEnabledMetrics", METRIC_IDENTITY_ARRAY_CLASS_ARRAY);
            GET_METRICS_DATA_METHOD = IMetricsManager.class.getDeclaredMethod("getMetricsData", new Class[] { IMetricIdentity[].class, Boolean.class });
            RESET_METRICS_METHOD = IMetricsManager.class.getDeclaredMethod("resetMetrics", IEmptyArray.EMPTY_CLASS_ARRAY);
            GET_ENABLED_ALERTS_METHOD = IMetricsManager.class.getDeclaredMethod("getEnabledAlerts", METRIC_IDENTITY_ARRAY_CLASS_ARRAY);
            ENABLE_ALERTS_METHOD = IMetricsManager.class.getDeclaredMethod("enableAlerts", ALERT_ARRAY_CLASS_ARRAY );
            DISABLE_ALERTS_METHOD = IMetricsManager.class.getDeclaredMethod("disableAlerts", ALERT_ARRAY_CLASS_ARRAY);
            REPLACE_ENABLED_ALERTS_METHOD = MetricsManager.class.getDeclaredMethod("replaceEnabledAlerts", ALERT_ARRAY_CLASS_ARRAY );
        }
        catch(NoSuchMethodException nsme)
        {
            nsme.printStackTrace(); // ok to do as it should never happen
        }

        //
        // Attributes
        //

        //
        // Operations
        //

        // start component
        DEFAULT_OPERATION_INFOS.add(new MBeanOperationInfo("start", "Start providing service.",
                                                           IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(),
                                                           MBeanOperationInfo.ACTION));

        // stop component
        DEFAULT_OPERATION_INFOS.add(new MBeanOperationInfo("stop", "Stop providing service.",
                                                           IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(),
                                                           MBeanOperationInfo.ACTION));

        // clear error condition
        DEFAULT_OPERATION_INFOS.add(new MBeanOperationInfo("clearError", "Clear existing error condition.",
                                                           IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(),
                                                           MBeanOperationInfo.ACTION));

        //
        // Notifications
        //
        String[] notifTypes = null;

        // online notification
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            IComponentState.STATE_TEXT[IComponentState.STATE_ONLINE]
        };
        DEFAULT_NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Component start complete."));

        // offline notification
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            IComponentState.STATE_TEXT[IComponentState.STATE_OFFLINE]
        };
        DEFAULT_NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Component stop complete."));
    }

    public ComponentMBean(ContainerImpl container, IComponent component, String id, String configID)
    throws Exception
    {
        super(container, component, id, configID);

        // did the component set a metrics manager and thus have metrics to expose ?
        // if so add the extra operations [and attributes] we will expose for metrics
        if (m_metricsManager != null)
        {
        	addMetricsOperations();
        	IElement componentConfig = m_componentContext.getConfiguration(true);
        	applyMetricsAttrs(componentConfig);
        	enableMetricsAndAlertsFromConfiguration();
        }
    }

    // Add Metrics and Alerts operations
    private void addMetricsOperations()
    {
        //attributes
        MBeanFeatureInfo mbInfo = null;

        // component metrics refresh interval
        mbInfo = new MBeanAttributeInfo("MetricsRefreshInterval", Integer.class.getName(), "Metrics refresh interval (in seconds)", true, true, false);
        m_attributes.add(mbInfo);

        // component metrics collection interval
        mbInfo = new MBeanAttributeInfo("MetricsCollectionInterval", Integer.class.getName(), "Metrics collection interval (in minutes)", true, true, false);
        m_attributes.add(mbInfo);

        // component repeat metric alert notifications
        mbInfo = new MBeanAttributeInfo("RepeatMetricAlerts", Boolean.class.getName(), "Repeat metric alert notifications.", true, true, false);
        m_attributes.add(mbInfo);

        //operations
        MBeanParameterInfo[] mbParamInfos = null;

        // get metrics info
        m_operations.add(new MBeanOperationInfo("getMetricsInfo", "Gets the meta-data for all metrics supported by this component.",
                                                IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, IMetricInfo[].class.getName(),
                                                MBeanOperationInfo.INFO));

        // get enabled metrics patterns
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("ids", IMetricIdentity[].class.getName(), "A list of metrics identities for which to get the enabled alerts.")
        };
        m_operations.add(new MBeanOperationInfo("getEnabledMetrics", "Get the list of metric patterns that have been enabled for this component.",
                                                mbParamInfos, IMetricIdentity[].class.getName(),
                                                MBeanOperationInfo.INFO));

        // get active metrics
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("ids", IMetricIdentity[].class.getName(), "A list of metrics identity patterns.")
        };
        m_operations.add(new MBeanOperationInfo("getActiveMetrics", "Gets the the list of metrics currently active in this component.",
                                                mbParamInfos, IMetricIdentity[].class.getName(),
                                                MBeanOperationInfo.INFO));

        // get metrics data
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("ids", IMetricIdentity[].class.getName(), "A list of active metrics, either individual metrics or instances."),
            new MBeanParameterInfo("returnTriggeredAlerts", Boolean.class.getName(), "If true, each metric returned should include a list of triggered (exceeded) alerts for these metrics.")
        };
        m_operations.add(new MBeanOperationInfo("getMetricsData", "Gets metric values for a given set of metrics. If a requested metric was not enabled or is otherwise unknown, then no data for that metric will be included in the results.",
                                                mbParamInfos, IMetricsData.class.getName(),
                                                MBeanOperationInfo.INFO));

        // enable metrics
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("ids", IMetricIdentity[].class.getName(), "A list of metrics identity patterns.")
        };
        m_operations.add(new MBeanOperationInfo("enableMetrics", "Enable the given set of dynamic metrics.",
                                                mbParamInfos, Void.class.getName(),
                                                MBeanOperationInfo.ACTION));

        // disable metrics
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("ids", IMetricIdentity[].class.getName(), "A list of metrics identity patterns.")
        };
        m_operations.add(new MBeanOperationInfo("disableMetrics", "Disable the given set of dynamic metrics.",
                                                           mbParamInfos, Void.class.getName(),
                                                           MBeanOperationInfo.ACTION));

        // reset metrics
        m_operations.add(new MBeanOperationInfo("resetMetrics", "Reset all metrics to their initial state.",
                                                IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(),
                                                MBeanOperationInfo.ACTION));

        // does the component have instance metrics or metrics that expose alerts ?
        IMetricInfo[] infos = m_metricsManager.getMetricsInfo();
        // instance metrics
        for (int i = 0; i < infos.length; i++)
        {
            if (infos[i].isInstanceMetric())
            {
                // get instance names
                mbParamInfos = new MBeanParameterInfo[]
                {
                    new MBeanParameterInfo("id", IMetricIdentity.class.getName(), "The instance metric parent identity.")
                };
                m_operations.add(new MBeanOperationInfo("getInstanceMetricNames", "Get the instance names to which the given instance metric may be applied. An array of instance names. If there are no instances the broker will a zero length array. If the metric identity is either unknown or not that of an instance metric, then the broker will return null.",
                                                        mbParamInfos, String[].class.getName(),
                                                        MBeanOperationInfo.INFO));
                break;
            }
        }
        // alerts
        for (int i = 0; i < infos.length; i++)
        {
            if (infos[i].supportsHighThresholdAlerts() || infos[i].supportsLowThresholdAlerts())
            {
                // get enabled alerts
                mbParamInfos = new MBeanParameterInfo[]
                {
                    new MBeanParameterInfo("ids", IMetricIdentity[].class.getName(), "A list of metrics identities for which to get the enabled alerts.")
                };
                m_operations.add(new MBeanOperationInfo("getEnabledAlerts", "Get all metric threshold alerts enabled by this component.",
                                                        mbParamInfos, IAlert[].class.getName(),
                                                        MBeanOperationInfo.INFO));

                // enable alerts
                mbParamInfos = new MBeanParameterInfo[]
                {
                    new MBeanParameterInfo("alerts", IAlert[].class.getName(), "A list of alerts to enable.")
                };
                m_operations.add(new MBeanOperationInfo("enableAlerts", "Enable threshold alerts on a metric.",
                                                        mbParamInfos, Void.class.getName(),
                                                        MBeanOperationInfo.ACTION));

                // disable alerts
                mbParamInfos = new MBeanParameterInfo[]
                {
                    new MBeanParameterInfo("alerts", IAlert[].class.getName(), "A list of alerts to disable.")
                };
                m_operations.add(new MBeanOperationInfo("disableAlerts", "Disable threshold alerts on a metric.",
                                                        mbParamInfos, Void.class.getName(),
                                                        MBeanOperationInfo.ACTION));
                break;
            }
        }
        // alert notifications
        for (int i = 0; i < infos.length; i++)
        {
            if (infos[i].supportsHighThresholdAlerts() || infos[i].supportsLowThresholdAlerts())
            {
                short category = infos[i].getMetricIdentity().getNameComponents()[0].equals(INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY])
                               ? INotification.SYSTEM_CATEGORY : INotification.APPLICATION_CATEGORY;
                String[] notifTypes = new String[]
                {
                    INotification.CATEGORY_TEXT[category],
                    INotification.SUBCATEGORY_TEXT[INotification.ALERT_SUBCATEGORY],
                    infos[i].getMetricIdentity().getName()
                };
                m_notifications.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Alert for metric " + infos[i].getMetricIdentity().getName()));
            }
        }
    }
     
    // Enable metrics and alerts from Container config
    private void enableMetricsAndAlertsFromConfiguration() throws Exception
    {
        IAttributeSet emas = getEnabledMetricsOrAlertsFromConfiguration(IContainerConstants.ENABLED_METRICS_ATTR);
        if (emas != null)
        {
            IMetricIdentity[] metricIDs = createMetricIds(emas);
            if (metricIDs.length > 0)
            {
                internalInvoke("enableMetrics", new Object[] { metricIDs }, new String[] { METRIC_IDENTITY_ARRAY_CLASSNAME });
            }
        }
        
        IAttributeSet eaas = getEnabledMetricsOrAlertsFromConfiguration(IContainerConstants.ENABLED_ALERTS_ATTR);
        if (eaas != null)
        {
            IAlert[] alerts = createAlerts(eaas);
            if (alerts.length > 0)
            {
                m_metricsManager.enableAlerts(alerts);
            }
        }
    }

    private IAttributeSet getEnabledMetricsOrAlertsFromConfiguration(String attributeName)
    {
        IAttributeSet mas = null;
        
        // get the container's Config Element
        IElementIdentity containerIdentity = m_container.getContainerIdentity().getConfigIdentity();
        IElement containerCE = m_componentContext.getConfiguration(containerIdentity.getName(), true);

        // get the container attribute set
        IAttributeSet containerAS = (IAttributeSet) containerCE.getAttributes();
        
        // If componentID = "AGENT" get enabled metrics from container attributes
        String compID = m_componentName.getComponentName();
        if (IAgentProxy.ID.equals(compID))
        {
            mas = (IAttributeSet)containerAS.getAttribute(attributeName);
        }
        else
        {
            // get the "COMPONENTS" attribute set
            IAttributeSet componentAS = (IAttributeSet)containerAS.getAttribute(IContainerConstants.COMPONENTS_ATTR);
            HashMap map = (HashMap)componentAS.getAttributes();
            IAttributeSet spas = (IAttributeSet)map.get(compID);

            // return if no startup parameters found for the component
            if (spas == null)
            {
                return null ;
            }

            // get the enabled metrics set
            mas = (IAttributeSet)spas.getAttribute(attributeName);
        }
        
        return mas ;
    }
    
    //
    // Utility methods for creating metrics and alerts from IAttributeSets.
 
    
    private IMetricIdentity[] createMetricIds(IAttributeSet metricsAS)
    {
        HashMap metricsMap = metricsAS.getAttributes();
        Iterator iterator = metricsMap.values().iterator();
        ArrayList metricIDList = new ArrayList();
        
        while (iterator.hasNext())
        {
            String metricName = (String)iterator.next();
            IMetricIdentity metricID = MetricsFactory.createMetricIdentity(metricName);
            metricIDList.add(metricID);
        }
        
        return (IMetricIdentity[])metricIDList.toArray(new IMetricIdentity[metricIDList.size()]);
    }

    private IAlert[] createAlerts(IAttributeSet alertsAS)
    {
        HashMap alertsMap = alertsAS.getAttributes();
        Iterator itor = alertsMap.values().iterator();
        IAlert alerts[];
        
        ArrayList alertsList = new ArrayList();

        while (itor.hasNext())
        {
            Object attr = itor.next();
            if (attr instanceof AttributeSet)
            {
                AttributeSet alertAS = (AttributeSet)attr;
                AttributeName compoundName = alertAS.getCompoundName();
                String metricName = (String)compoundName.getComponent(compoundName.getComponentCount() - 1);
                IMetricIdentity metricID = MetricsFactory.createMetricIdentity(metricName);

                Object thresholds = ((AttributeSet)alertAS).getAttribute(IContainerConstants.HIGH_THRESHOLDS_ATTR);
                if (thresholds instanceof String)
                {
                    alerts = MetricsFactory.createAlerts(metricID, true, (String)thresholds);
                    for (int j = 0; j < alerts.length; j++)
                    {
                        alertsList.add(alerts[j]);
                    }
                }

                thresholds = ((AttributeSet)alertAS).getAttribute(IContainerConstants.LOW_THRESHOLDS_ATTR);
                if (thresholds instanceof String)
                {
                    alerts = MetricsFactory.createAlerts(metricID, false, (String)thresholds);
                    for (int j = 0; j < alerts.length; j++)
                    {
                        alertsList.add(alerts[j]);
                    }
                }
            }
        }

        return (IAlert[])alertsList.toArray(new IAlert[alertsList.size()]);
    }

    @Override
    IComponentContext getContext() { return new ContextImpl(); }

    @Override
    final void cleanup()
    {
        super.cleanup();
        if (m_metricsManager != null)
        {
            m_metricsManager.cleanup();
            m_metricsManager = null;
        }
    }

    //
    // DynamicMBean interface
    //

    @Override
    public Object getAttribute(String name)
    throws AttributeNotFoundException, MBeanException, ReflectionException
    {
        MBeanAttributeInfo info = super.getAttributeInfo(name);

        if (info == null)
        {
            throw new AttributeNotFoundException(name + " is not a valid attribute.");
        }

        if (!info.isReadable())
        {
            throw new AttributeNotFoundException(name + " is not a readable attribute.");
        }

        // deal with managed attributes before forwarding to the component
        if (name.equals("MetricsRefreshInterval"))
        {
            int refreshInterval = IContainerConstants.REFRESH_INTERVAL_DEFAULT; // default refresh interval in seconds
            if (this.m_metricsManager != null)
            {
                long ri = this.m_metricsManager.getRefreshInterval();  //in milliseconds
                refreshInterval = (int) (ri/1000);  //convert from milliseconds to seconds
            }
            return(new Integer(refreshInterval));  //in seconds
        }
        else if (name.equals("MetricsCollectionInterval"))
        {
            int collectionInterval = IContainerConstants.COLLECTION_INTERVAL_DEFAULT; // default collection interval in minutes
            if (this.m_metricsManager != null)
            {
                long ci = this.m_metricsManager.getCollectionInterval();  //in milliseconds
                collectionInterval = (int) (ci/60000);  //convert from milliseconds to minutes
            }
            return(new Integer(collectionInterval));  //in minutes
        }
        else if (name.equals("RepeatMetricAlerts"))
        {
            boolean repeatAlerts = IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_DEFAULT; // default collection interval in minutes
            if (this.m_metricsManager != null)
            {
                repeatAlerts = this.m_metricsManager.getRepeatMetricAlerts();
            }
            return (new Boolean(repeatAlerts));
        }
        else
        {
            return super.getAttribute(name);
        }
    }

    @Override
    public void setAttribute(Attribute attribute)
    throws AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException
    {
        String name = attribute.getName();
        if (name.equals("MetricsRefreshInterval"))
        {
            MBeanAttributeInfo info = validateAttribute(name);  // will throw exception if MBeanAttributeInfo object not found...
            Object value = getAttributeValue(attribute,info);
            setRefreshInterval(name,(Integer)value);
        }
        else if (name.equals("MetricsCollectionInterval"))
        {
            MBeanAttributeInfo info = validateAttribute(name);  // will throw exception if MBeanAttributeInfo object not found...
            Object value = getAttributeValue(attribute,info);
            setCollectionInterval(name,(Integer)value);
        }
        else if (name.equals("RepeatMetricAlerts"))
        {
            MBeanAttributeInfo info = validateAttribute(name);  // will throw exception if MBeanAttributeInfo object not found...
            Object value = getAttributeValue(attribute,info);
            setRepeatMetricAlerts(name,(Boolean)value);
        }
        else
        {
            super.setAttribute(attribute);
        }
    }

    private void setRefreshInterval(String name, Integer refreshInterval)
    throws InvalidAttributeValueException
    {
        // convert from seconds to milliseconds
        long ri = 1000 * refreshInterval.longValue();

        // delegate to metrics manager, if defined
        try
        {
            if (this.m_metricsManager != null)
            {
                // trace...
                int traceMask = this.m_component.getTraceMask().intValue();
                long originalValue = 0;
                if ((traceMask & (IComponent.TRACE_ATTRIBUTES | IComponent.TRACE_VERBOSE)) > 0)
                {
                    originalValue = this.m_metricsManager.getRefreshInterval();
                }

                // set the new refresh interval
                this.m_metricsManager.setRefreshInterval(ri);

                // trace...
                if ((traceMask & IComponent.TRACE_ATTRIBUTES) > 0)
                {
                    StringBuffer msg = new StringBuffer("Set attribute [" + name + "]");
                    if ((traceMask & IComponent.TRACE_VERBOSE) > 0)
                    {
                        // convert old and new values from milliseconds to seconds for reporting purposes...
                        int oldVal = (int) (originalValue/1000);
                        msg.append(", values...");
                        msg.append(IContainer.NEWLINE).append("\tOriginal value = ").append(oldVal);
                        msg.append(IContainer.NEWLINE).append("\tNew value      = ").append(refreshInterval.intValue());
                    }
                    logMessage(msg.toString(), Level.TRACE);
                }
            }
        }
        catch (MFException mfe)
        {
           throw new InvalidAttributeValueException();
        }
    }

    private void setCollectionInterval(String name, Integer collectionInterval)
    throws InvalidAttributeValueException
    {
        // convert from minutes to milliseconds
        long ci = 60000 * collectionInterval.longValue();

        // delegate to metrics manager, if defined
        try
        {
            if (this.m_metricsManager != null)
            {
                // trace...
                int traceMask = this.m_component.getTraceMask().intValue();
                long originalValue = 0;
                if ((traceMask & (IComponent.TRACE_ATTRIBUTES | IComponent.TRACE_VERBOSE)) > 0)
                {
                    originalValue = this.m_metricsManager.getCollectionInterval();
                }

                // set the new collection interval
                this.m_metricsManager.setCollectionInterval(ci);

                // trace...
                if ((traceMask & IComponent.TRACE_ATTRIBUTES) > 0)
                {
                    StringBuffer msg = new StringBuffer("Set attribute [" + name + "]");
                    if ((traceMask & IComponent.TRACE_VERBOSE) > 0)
                    {
                        int oldVal = (int) (originalValue/60000); // convert old value from milliseconds to seconds for reporting purposes...
                        msg.append(", values...");
                        msg.append(IContainer.NEWLINE).append("\tOriginal value = ").append(oldVal);
                        msg.append(IContainer.NEWLINE).append("\tNew value      = ").append(collectionInterval.intValue());
                    }
                    logMessage(msg.toString(), Level.TRACE);
                }
            }
        }
        catch (MFException mfe)
        {
           throw new InvalidAttributeValueException();
        }
    }

    private void setRepeatMetricAlerts(String name, Boolean repeatMetricAlerts)
    throws InvalidAttributeValueException
    {
        // delegate to metrics manager, if defined
        try
        {
            if (this.m_metricsManager != null)
            {
                // trace...
                int traceMask = this.m_component.getTraceMask().intValue();
                boolean originalValue = false;
                if ((traceMask & (IComponent.TRACE_ATTRIBUTES | IComponent.TRACE_VERBOSE)) > 0)
                {
                    originalValue = this.m_metricsManager.getRepeatMetricAlerts();
                }

                // set the new repeat metric alerts
                this.m_metricsManager.setRepeatMetricAlerts(repeatMetricAlerts.booleanValue());

                // trace...
                if ((traceMask & IComponent.TRACE_ATTRIBUTES) > 0)
                {
                    StringBuffer msg = new StringBuffer("Set attribute [" + name + "]");
                    if ((traceMask & IComponent.TRACE_VERBOSE) > 0)
                    {
                        msg.append(", values...");
                        msg.append(IContainer.NEWLINE).append("\tOriginal value = ").append(originalValue);
                        msg.append(IContainer.NEWLINE).append("\tNew value      = ").append(repeatMetricAlerts.booleanValue());
                    }
                    logMessage(msg.toString(), Level.TRACE);
                }
            }
        }
        catch (MFException mfe)
        {
           throw new InvalidAttributeValueException();
        }
    }

    @Override
    Object internalInvoke(String operationName, Object[] params, String[] signature)
    throws Exception
    {
        Object target = null;
        Method method = null;

        try
        {
            if (operationName.equals("clearError") && signature.length == 0)
            {
                target = this;
                method = CLEAR_ERROR_METHOD;
            }
            else if (operationName.equals("handleNotification") && signature.length == 1 && signature[0].equals(NOTIFICATION_CLASSNAME))
            {
                target = this;
                method = HANDLE_NOTIFICATION_METHOD;
            }
            else if (operationName.equals("getMetricsInfo") && signature.length == 0)
            {
                target = m_metricsManager;
                method = GET_METRICS_INFO_METHOD;
            }
            else if (operationName.equals("enableMetrics") && signature.length == 1 && signature[0].equals(METRIC_IDENTITY_ARRAY_CLASSNAME))
            {
                IMetricIdentity[] ids = (IMetricIdentity[])internalInvoke(m_metricsManager, operationName, ENABLE_METRICS_METHOD, params);
                params[0] = ids;
            }
            else if (operationName.equals("disableMetrics") && signature.length == 1 && signature[0].equals(METRIC_IDENTITY_ARRAY_CLASSNAME))
            {
                IMetricIdentity[] ids = (IMetricIdentity[])internalInvoke(m_metricsManager, operationName, DISABLE_METRICS_METHOD, params);
                params[0] = ids;
            }
            else if (operationName.equals("getActiveMetrics") && signature.length == 1 && signature[0].equals(METRIC_IDENTITY_ARRAY_CLASSNAME))
            {
                target = m_metricsManager;
                method = GET_ACTIVE_METRICS_METHOD;
            }
            else if (operationName.equals("getEnabledMetrics") && signature.length == 1 && signature[0].equals(METRIC_IDENTITY_ARRAY_CLASSNAME))
            {
                target = m_metricsManager;
                method = GET_ENABLED_METRICS_METHOD;
            }
            else if (operationName.equals("getMetricsData") && signature.length == 2
                                             && signature[0].equals(METRIC_IDENTITY_ARRAY_CLASSNAME)
                                             && signature[1].equals(Boolean.class.getName()))
            {
                target = m_metricsManager;
                method = GET_METRICS_DATA_METHOD;
            }
            else if (operationName.equals("resetMetrics") && signature.length == 0)
            {
                target = m_metricsManager;
                method = RESET_METRICS_METHOD;
            }
            else if (operationName.equals("getEnabledAlerts") && signature.length == 1 && signature[0].equals(METRIC_IDENTITY_ARRAY_CLASSNAME))
            {
                target = m_metricsManager;
                method = GET_ENABLED_ALERTS_METHOD;
            }
            else if (operationName.equals("enableAlerts") && signature.length == 1 && signature[0].equals(ALERT_ARRAY_CLASSNAME))
            {
                target = m_metricsManager;
                method = ENABLE_ALERTS_METHOD;
            }
            else if (operationName.equals("disableAlerts") && signature.length == 1 && signature[0].equals(ALERT_ARRAY_CLASSNAME))
            {
                target = m_metricsManager;
                method = DISABLE_ALERTS_METHOD;
            }
            else if (operationName.equals("replaceEnabledMetrics") && signature.length == 1 && signature[0].equals(METRIC_IDENTITY_ARRAY_CLASSNAME))
            {
                IMetricIdentity[][] ids = (IMetricIdentity[][])internalInvoke(m_metricsManager, operationName, REPLACE_ENABLED_METRICS_METHOD, params);
                params[0] = ids[0];
                super.internalInvoke("disableMetrics", params, signature);
                params[0] = ids[1];
                operationName = "enableMetrics";
            }
            else if (operationName.equals("replaceEnabledAlerts") && signature.length == 1 && signature[0].equals(ALERT_ARRAY_CLASSNAME))
            {
                target = m_metricsManager;
                method = REPLACE_ENABLED_ALERTS_METHOD;
            }
            // if target is set .. call directly
            if (target != null)
            {
                return internalInvoke(target, operationName, method, params);
            }
        }
        catch(MFException e) { super.throwWrappedExceptionOnUnload(e); }
        catch(MFRuntimeException e) { super.throwWrappedExceptionOnUnload(e); }
        catch(RuntimeException e)
        {
            if (e.getClass().getName().startsWith("java."))
            {
                super.throwWrappedExceptionOnUnload(e);
            }
            MFRuntimeException mfe = new MFRuntimeException();
            mfe.setLinkedException(e);
            super.throwWrappedExceptionOnUnload(mfe);
        }
        catch(Exception e)
        {
            if (e.getClass().getName().startsWith("java."))
            {
                super.throwWrappedExceptionOnUnload(e);
            }
            MFException mfe = new MFException();
            mfe.setLinkedException(e);
            super.throwWrappedExceptionOnUnload(mfe);
        }
        catch(OutOfMemoryError e)
        {
            logMessage("Failed operation ["  + operationName + "] due to insufficient memory...", e, Level.SEVERE);
            throw e;
        }
        catch(Error e)
        {
            logMessage("Failed operation ["  + operationName + "], trace follows...", e, Level.SEVERE);
            throw e;
        }

        return super.internalInvoke(operationName, params, signature);
    }

    IMetricsRegistrar initMetricsManagement(IMetricInfo[] metricInfos)
    {
        if (m_metricsManager != null)
        {
            throw new IllegalStateException("Metrics management already initialized");
        }

        try
        {
            Class metricsManagerClass = m_componentClassLoader.loadClass(MetricsManager.class.getName());
            Constructor constructor = metricsManagerClass.getConstructor(new Class[] { IMetricInfo[].class });
            m_metricsManager = (IMetricsManager)constructor.newInstance(new Object[] { metricInfos });
        }
        catch(Exception e)
        {
            MFRuntimeException mfe = new MFRuntimeException("Failed to initialize metrics management");
            mfe.setLinkedException(e);
            throw mfe;
        }

        m_metricsManager.init(m_componentContext);
        return (IMetricsRegistrar)m_metricsManager;
    }

    @Override
    void handleElementChange(IElementChange change)
    {
        IBasicElement element = change.getElement();
        
        // we only need to handle changes to this components configuration *or* (special case) the agent
        // manager primary configuration on behalf of the backup
        
        // We will NOT special case an /mq/brokers primary configuration metrics change being applied on behalf of a /mq/backupbrokers.
        // This is because it is not easy to confirm from the change element that the primary broker is in fact associated with the
        // current backup. 
        // For this we would add an || super.getConfigID().startsWith("/mq/backupbrokers/") && configID.startsWith("/mq/brokers/")
        // All elements in the local cache become elements for which change listeners are registered for cache reconciliation purposes.
        // Unlikely, but, a container that holds multiple broker components could get in trouble here. In any event changes to
        // the primary configuration will be applied on backup restart. Also from JMX(runtime properties) the backup metrics behavior can 
        // be changed live.
        String configID = element.getIdentity().getName();
        if (configID.equals(super.getConfigID()) || super.getConfigID().startsWith("/backupmanager/") && configID.startsWith("/manager/"))
        {
            if (element instanceof IDeltaElement)
            {
                IDeltaElement changes = (IDeltaElement)element;
                IDeltaAttributeSet attrs = (IDeltaAttributeSet)changes.getDeltaAttributes();
                String[] mods = attrs.getModifiedAttributesNames();
                
                try
                {
                    for (int i = 0; i < mods.length; i++)
                    {
                        if (mods[i].equals(IContainerConstants.METRICS_ATTR))
                        {
                            // prevents ClassCastException because of incompatible interfaces
                            if (attrs.getNewValue(IContainerConstants.METRICS_ATTR) instanceof IDeltaAttributeSet)
                            {
                                attrs = (IDeltaAttributeSet)attrs.getNewValue(IContainerConstants.METRICS_ATTR);
                                handleModifiedMetricsAttrs(attrs.getNewAttributesNames(), attrs);
                                handleModifiedMetricsAttrs(attrs.getModifiedAttributesNames(), attrs);
                                handleDeletedMetricsAttrs(attrs.getDeletedAttributesNames());
                            }
                        }
                    }
                }
                catch (Exception e)
                {
                    m_componentContext.logMessage("Error modifying " + m_componentName.getComponentName() + " runtime from configuration change, trace follows...", e, Level.WARNING);
                }
            }
            else
            {
                applyMetricsAttrs((IElement)element);
            }
        }
        
        super.handleElementChange(change);
    }
    
    private void applyMetricsAttrs(IElement element)
    {
        IAttributeSet amAttributes = element.getAttributes();
        
        // Note: special case - for the backup agent manager we will get the primary's config to set
        // the metrics configuration
        if (element.getIdentity().getName().startsWith("/backupmanager/"))
        {
            // Note: don't use config constants to avoid a dependency
            IAttributeSet backupRefs = (IAttributeSet)amAttributes.getAttribute("CONFIG_ELEMENT_REFERENCES");
            if (backupRefs == null)
            {
                return; // not an FT AM
            }
            else
            {
                Reference primaryRef = (Reference)backupRefs.getAttribute("PRIMARY_AM_CONFIG_ELEMENT_REF");
                if (primaryRef == null)
                {
                    return; // not an FT AM
                }
                else
                {
                    IElement primaryEl = m_componentContext.getConfiguration(primaryRef.getElementName(), true);
                    amAttributes = primaryEl.getAttributes();
                }
            }
        } else if (element.getIdentity().getName().startsWith("/mq/backupbrokers/"))
        {
        	//MQ-35078 Backup Broker Metrics Collection Interval and Period Cannot Be Changed From Default
            // Note: don't use config constants to avoid a dependency
            IAttributeSet backupRefs = (IAttributeSet)amAttributes.getAttribute("CONFIG_ELEMENT_REFERENCES");
            if (backupRefs == null)
            {
                return; // not an FT Broker
            }
            else
            {
                Reference primaryRef = (Reference)backupRefs.getAttribute("PRIMARY_CONFIG_ELEMENT_REF");
                if (primaryRef == null)
                {
                    return; // not an FT Broker
                }
                else
                {
                    IElement primaryEl = m_componentContext.getConfiguration(primaryRef.getElementName(), true);
                    amAttributes = primaryEl.getAttributes();
                }
            }
        }
        
        // set the metrics manager intervals either from the config or from the defaults if not provided
        try
        {
            IAttributeSet metricsAS = (IAttributeSet)amAttributes.getAttribute(IContainerConstants.METRICS_ATTR);
            if (m_metricsManager != null)
            {
                if (metricsAS != null)
                {

                    Integer refreshInterval    = (Integer)metricsAS.getAttribute(IContainerConstants.REFRESH_INTERVAL_ATTR);
                    Integer collectionInterval = (Integer)metricsAS.getAttribute(IContainerConstants.COLLECTION_INTERVAL_ATTR);
					
                    // need to convert from seconds to milliseconds					
                    long lRefreshInterval = (refreshInterval != null) ? ( refreshInterval.longValue() * 1000 ) : 
                                                                 ((long)(IContainerConstants.REFRESH_INTERVAL_DEFAULT * 1000));

                    // need to convert from minutes to milliseconds
                    long lCollectionInterval = (collectionInterval != null) ? (collectionInterval.longValue() * 60000) : 
                                                                 ((long)(IContainerConstants.COLLECTION_INTERVAL_DEFAULT * 60000));
                    
                    m_metricsManager.setRefreshAndCollectionInterval( lRefreshInterval, lCollectionInterval );
					
					
                    Boolean repeatAlerts = (Boolean) metricsAS.getAttribute(IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_ATTR);
                    if (repeatAlerts != null)
                    {
                        m_metricsManager.setRepeatMetricAlerts(repeatAlerts.booleanValue());
                    }
                    else
                    {
                        m_metricsManager.setRepeatMetricAlerts(IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_DEFAULT);
                    }
                }
                else
                {
                    m_metricsManager.setRefreshAndCollectionInterval((long)(IContainerConstants.REFRESH_INTERVAL_DEFAULT * 1000),
                                                                     (long)(IContainerConstants.COLLECTION_INTERVAL_DEFAULT * 60000) );
					                    
                    m_metricsManager.setRepeatMetricAlerts(IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_DEFAULT);
                }
            }
        }
        catch(Exception e)
        {
            m_componentContext.logMessage("Failed to configure metric settings, trace follows...", e, Level.WARNING);
        }
    }

    private void handleModifiedMetricsAttrs(String[] attributeNames, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < attributeNames.length; i++)
        {
            // use the new attribute value(s)
            if (attributeNames[i].equals(IContainerConstants.REFRESH_INTERVAL_ATTR) || attributeNames[i].equals(IContainerConstants.COLLECTION_INTERVAL_ATTR))
             {
            	 //convert from seconds to milliseconds
            	long refreshInterval = ((Integer)attrs.getNewValue(IContainerConstants.REFRESH_INTERVAL_ATTR)).longValue() * 1000;
            	 //convert from minutes to milliseconds
            	long collectionInterval =  ((Integer)attrs.getNewValue(IContainerConstants.COLLECTION_INTERVAL_ATTR)).longValue() * 60000;
                m_metricsManager.setRefreshAndCollectionInterval(refreshInterval, collectionInterval);                 
            }
            if (attributeNames[i].equals(IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_ATTR))
            {
                m_metricsManager.setRepeatMetricAlerts(((Boolean)attrs.getNewValue(IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_ATTR)).booleanValue());
            }
        }
    }

    private void handleDeletedMetricsAttrs(String[] attributeNames)
    throws Exception
    {
        for (int i = 0; i < attributeNames.length; i++)
        {
            // set attribute(s) back to default value(s)
            if (attributeNames[i].equals(IContainerConstants.REFRESH_INTERVAL_ATTR) || attributeNames[i].equals(IContainerConstants.COLLECTION_INTERVAL_ATTR))
             {
	           	//convert from seconds to milliseconds
	           	long refreshInterval    = (long)IContainerConstants.REFRESH_INTERVAL_DEFAULT    * 1000;
	           	//convert from minutes to milliseconds
	           	long collectionInterval = (long)IContainerConstants.COLLECTION_INTERVAL_DEFAULT * 60000;
	           	
                m_metricsManager.setRefreshAndCollectionInterval(refreshInterval, collectionInterval);
            }
            if (attributeNames[i].equals(IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_ATTR))
            {
                m_metricsManager.setRepeatMetricAlerts((boolean)IContainerConstants.REPEAT_ALERT_NOTIFICATIONS_DEFAULT);
            }
        }
    }

    @Override
    void addSharedPath(URL url)
    throws UnsupportedOperationException
    {
    }

    @Override
    void addSharedClassname(String classname)
    throws UnsupportedOperationException
    {
    }

    public class ContextImpl
    implements IComponentContext, IChangeRegistration
    {
        @Override
        public Context getInitialContext() {return ComponentMBean.super.getInitialContext(); }

        @Override
        public ICanonicalName getComponentName() { return ComponentMBean.super.m_componentName; }

        @Override
        public void setRuntimeConfiguration(IDirElement configuration) { ComponentMBean.super.setRuntimeConfiguration(configuration); }

        @Override
        public IElement getConfiguration(boolean acceptChanges) { return ComponentMBean.super.getConfiguration(null, acceptChanges); }

        @Override
        public IElement getConfiguration(String configID, boolean acceptChanges) { return ComponentMBean.super.getConfiguration(configID, acceptChanges); }

        @Override
        public IElement getConfiguration(String configID, boolean acceptChanges, boolean alwaysFromDS) { return ComponentMBean.super.getConfiguration(configID, acceptChanges, alwaysFromDS); }

        @Override
        public int checkFSConfiguration(String path) { return ComponentMBean.super.checkFSConfiguration(path); }
        
        @Override
        public IElement getFSConfiguration(String path, boolean notify) { return ComponentMBean.super.getFSConfiguration(path, notify); }

        @Override
        public IElement getFSConfiguration(String path, boolean notify, boolean alwaysFromDS) { return ComponentMBean.super.getFSConfiguration(path, notify, alwaysFromDS); }

        @Override
        public String getPrivateSubDir(String baseDir) {return ComponentMBean.super.getPrivateSubDir(baseDir); }
        
        @Override
        public IElement registerFileChangeInterest(String path) {return ComponentMBean.super.registerFileChangeInterest(path); }

        @Override
        public void unregisterFileChangeInterest(String path) {ComponentMBean.super.unregisterFileChangeInterest(path); }

        @Override
        public IElement[] getConfigurations(String[] configIDs, boolean[] acceptChanges)
        {
            // since this could be indirectly called by FGS from an external request thread we must ensure
            // we have full permission to execute
            String currentUserID = TaskScheduler.getCurrentUserID();
            if (TaskScheduler.isExecutionThread())
            {
                ((TaskScheduler.ExecutionThread)Thread.currentThread()).setUserID(IManagementPermission.SUPER_USER_NAME);
            }
            try
            {
                return ComponentMBean.super.getConfigurations(configIDs, acceptChanges);
            }
            finally
            {
                if (TaskScheduler.isExecutionThread())
                {
                    ((TaskScheduler.ExecutionThread)Thread.currentThread()).setUserID(currentUserID);
                }
            }
        }

        @Override
        public IElement[] getFSConfigurations(String[] paths, boolean notify) { return null; } // not implemented

        @Override
        public IElement[] getConfigurations(String dirName, boolean acceptChanges) { return ComponentMBean.super.getConfigurations(dirName, acceptChanges); }

        @Override
        public java.io.File getLocalFile(String path) throws MFException {return ComponentMBean.super.getLocalFile(path);}
        @Override
        public INotification createNotification(short category, String subCategory, String eventName, int severityLevel)
        {
            // application components can only create system notifications if the type is in a predefined set
            if (category == INotification.SYSTEM_CATEGORY)
            {
                if (subCategory.equals(INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY]))
                {
                    if (!(eventName.equals(IComponentState.STATE_TEXT[IComponentState.STATE_ONLINE]) || eventName.equals(IComponentState.STATE_TEXT[IComponentState.STATE_OFFLINE])))
                    {
                        throw new IllegalArgumentException("Illegal to create system notifications");
                    }
                }
                else
                {
                    throw new IllegalArgumentException("Illegal to create system notifications");
                }
            }

            return ComponentMBean.super.createNotification(category, subCategory, eventName, severityLevel);
        }

        @Override
        public void sendNotification(INotification notification) { ComponentMBean.super.sendNotification(notification); }

        @Override
        public void logMessage(String message, int severityLevel) { ComponentMBean.super.logMessage(message, severityLevel); }
        @Override
        public void logMessage(String message, Throwable exception, int severityLevel) { ComponentMBean.super.logMessage(message, exception, severityLevel); }
        @Override
        public void logMessage(Throwable exception, int severityLevel) { ComponentMBean.super.logMessage(exception, severityLevel); }

        @Override
        public boolean registerErrorCondition(String errorMessage, int errorLevel) { return ComponentMBean.super.registerErrorCondition(errorMessage, errorLevel); }

        @Override
        public void clearErrorCondition() { ComponentMBean.super.clearError(); }

        @Override
        public void scheduleTask(Runnable task, Date startTime) { ComponentMBean.super.scheduleTask(task, startTime, false); }

        @Override
        public void cancelTask(Runnable task) { ComponentMBean.super.cancelTask(task); }

        @Override
        public IMetricsRegistrar initMetricsManagement(IMetricInfo[] metricInfos) { return ComponentMBean.this.initMetricsManagement(metricInfos); }

        @Override
        public void fireAttributeChangeHandlers() { ComponentMBean.super.fireAttributeChangeHandlers(); }

        @Override
        public void registerAttributeChangeHandler(AttributeName name, IAttributeChangeHandler handler) { ComponentMBean.super.registerAttributeChangeHandler(name, handler); }

        @Override
        public void unregisterAttributeChangeHandler(AttributeName name) { ComponentMBean.super.unregisterAttributeChangeHandler(name); }

        @Override
        public IElement getContainerConfiguration(boolean acceptsChanges) { return ComponentMBean.super.getContainerConfiguration(acceptsChanges) ;}
        
        @Override
        public IAttributeSet getDeploymentParameters() { return ComponentMBean.super.getDeploymentParameters() ; }

        @Override
        public Object setFaultDetectionConnection(Object jmsConnection)
        throws Exception
        {
            throw new UnsupportedOperationException();
        }

        @Override
        public void restartContainer()
        throws Exception
        {
            final IContainer container = ContainerImpl.getContainer();
            container.shutdown(14, true);
        }

    }
}
