package com.sonicsw.mf.framework.agent;

import java.util.HashMap;
import java.util.StringTokenizer;

import com.sonicsw.mf.comm.IConnectorClient;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.DSNotStartedException;
import com.sonicsw.mf.common.IBlobSource;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IChunkedBlobStreamer;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.impl.Blob;
import com.sonicsw.mf.common.dirconfig.DirectoryDoesNotExistException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceClosedException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.common.view.IDeltaView;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.directory.DSComponent;
import com.sonicsw.mf.framework.util.ContainerCompatibility;


// Presents the DS to the container in a location transparent manner (the same interface is used
// whether or not the DS is local)
public class ContainerDS
implements IChunkedBlobStreamer, IBlobSource
{
    private ContainerImpl m_container;
    private JMSConnectorServer m_connector;
    private DSComponent m_directory;
    private final IContainerIdentity m_containerIdentity;
    private final String m_commType;
    private String m_targetDS;
    private SonicURLStreamHandlerFactoryImpl m_urlStreamHandlerFactory;

    // Creates a proxy for a remote DS
    ContainerDS(ContainerImpl container, JMSConnectorServer connector, IContainerIdentity containerIdentity, String commType)
    {
        m_container = container;
        m_connector = connector;
        m_containerIdentity = containerIdentity;
        m_commType = commType;
        m_targetDS = new CanonicalName(m_containerIdentity.getDomainName(), DSComponent.GLOBAL_ID, DSComponent.GLOBAL_ID).getCanonicalName();
    }

    // Turn the remote proxy into a local one
    public void setLocalDS(DSComponent directory)
    {
        m_directory = directory;
        if (m_urlStreamHandlerFactory != null)
        {
            m_urlStreamHandlerFactory.resetSonicRNHandlerDS();
        }
    }

    IDirectoryAdminService getLocalDS()
    {
        if (m_directory == null)
        {
            return null;
        }
        else
        {
            return m_directory.getDS();
        }
    }
    
    boolean isLocalDS()
    {
        return (m_directory != null);
    }
    
    void setURLStreamHandlerFactory(SonicURLStreamHandlerFactoryImpl urlStreamHandlerFactory)
    {
        m_urlStreamHandlerFactory = urlStreamHandlerFactory;
    }

    String ping(String pingMessage)
    throws Exception
    {
        if (m_directory == null)
        {
            return (String)m_connector.invoke(m_commType, m_targetDS, m_targetDS, "ping", new Object [] { pingMessage }, new String [] { String.class.getName() });
        }
        else
        {
            // the Directory Service is local
            return pingMessage;
        }
    }

    String ping(String containerID, Short[] versionInfo)
    throws Exception
    {
        if (m_directory == null)
        {
            return (String)m_connector.invoke(m_commType, m_targetDS, m_targetDS, "ping", new Object [] { containerID, versionInfo }, new String [] { String.class.getName(), Short[].class.getName() });
        }
        else // the Directory Service is local
        {
            ContainerCompatibility.addContainer(containerID, versionInfo[0].shortValue(), versionInfo[1].shortValue(), versionInfo[2].shortValue(), versionInfo[3].shortValue());
            return "PONG";
        }
    }

    IElement getFSElement(String subscriber, String elementPath)
    {
        try
        {
            if (m_directory == null)
            {
                return (IElement)dsInvoke("getFSElement", new Object [] { elementPath, Boolean.FALSE }, new String [] { String.class.getName(), Boolean.class.getName() });
            }
            else
            {
                // the Directory Service is local
                return m_directory.getFSElement(elementPath, Boolean.FALSE);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get FS configuration: " + elementPath);
        }
    }

    IElement getElement(String subscriber, String configID)
    {
        try
        {
            if (m_directory == null)
            {
                return (IElement)dsInvoke("getElement", new Object [] { subscriber, configID }, new String [] { String.class.getName(), String.class.getName() });
            }
            else
            {
                // the Directory Service is local
                return m_directory.getElement(subscriber, configID);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get configuration: " + configID);
        }
    }

    IElement getElementByLogicalName(String subscriber, String logicalName)
    {
    	return getElementByLogicalName(subscriber, logicalName, true);
    }
    
    IElement getElementByLogicalName(String subscriber, String logicalName, boolean retry)
    {
        try
        {
            if (m_directory == null)
            {
                return (IElement)dsInvoke("getElementByLogicalName", new Object [] { subscriber, logicalName}, new String [] { String.class.getName(), String.class.getName() }, retry);
            }
            else
            {
                // the Directory Service is local
                return m_directory.getElementByLogicalName(subscriber, logicalName);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get configuration: " + logicalName);
        }
    }

    IElement[] getElementsByLogicalNames(String subscriber, String[] logicalNames)
    {
        try
        {
            if (m_directory == null)
            {
                return (IElement[])dsInvoke("getElementsByLogicalNames", new Object [] { subscriber, logicalNames}, new String [] { String.class.getName(), logicalNames.getClass().getName() });
            }
            else
            {
                // the Directory Service is local
                return m_directory.getElementsByLogicalNames(subscriber, logicalNames);
            }
        }
        catch(Exception e)
        {
            String names = "";
            for (int i = 0; i < logicalNames.length; i++)
            {
                names += logicalNames[i] + (i + 1 == logicalNames.length ? "" : ",");
            }
            throw convertException(e, "Failed to get configurations: " + names);
        }
    }

    IDirIdentity[] listDirectories(String parentDir)
    {
        try
        {
            if (m_directory == null)
            {
                return (IDirIdentity[])dsInvoke("listDirectories", new Object[] {parentDir}, new String[] {String.class.getName()});
            }
            else
            {
                // the Directory Service is local
                return m_directory.listDirectories(parentDir);
            }
        }
        catch (Exception e)
        {
            throw convertException(e, "Failed to list directories under parent directory " + parentDir);
        }
    }

    IElementIdentity[] listElements(String dirName)
    {
        try
        {
            if (m_directory == null)
            {
                return (IElementIdentity[])dsInvoke("listElements", new Object[] {dirName}, new String[] {String.class.getName()});
            }
            else
            {
                // the Directory Service is local
                return m_directory.listElements(dirName);
            }
        }
        catch (Exception e)
        {
            throw convertException(e, "Failed to list elements in directory " + dirName);
        }
    }

    IDirElement[] getAllElements(String dirName)
    {
        try
        {
            if (m_directory == null)
            {
                return (IDirElement[])dsInvoke("getAllElements", new Object[] {dirName, Boolean.FALSE}, new String[] {String.class.getName(), Boolean.class.getName()});
            }
            else
            {
                // the Directory Service is local
                return m_directory.getAllElements(dirName, Boolean.FALSE);
            }
        }
        catch (Exception e)
        {
            throw convertException(e, "Failed to get the elements in directory " + dirName);
        }
    }

    com.sonicsw.mf.common.config.IIdentity[] listAll(String dirName)
    {
        try
        {
            if (m_directory == null)
            {
                return (IIdentity[])dsInvoke("listAll", new Object[] {dirName}, new String[] {String.class.getName()});
            }
            else
            {
                // the Directory Service is local
                return m_directory.listAll(dirName);
            }
        }
        catch (Exception e)
        {
            throw convertException(e, "Failed to list elements in directory " + dirName);
        }
    }

    @Override
    public IBlob getBlobByLogicalName(String subscriber, String logicalName)
    {
        try
        {
            if (m_directory == null)
            {
                IBlob blob = (IBlob)dsInvoke("getBlobByLogicalName", new Object[] {subscriber, logicalName}, new String[] {String.class.getName(), String.class.getName()});
               // This blob has a storage name. Do not set the logical flag to true
                // mrd 10/28/2005 Eclipse DS handler blobs gotten through getBlobByLogicalName can only be handled
                // through the FS interface, and their logical name and storage names are the same. If the flag comes
                // back true, set it to true again here. The other alternative would have been to implement getBlob
                // in the EclipseHandler, and have it do the same thing as getFSBlob.
                if (blob == null)
                {
                    return null;
                }
                boolean isLogical = blob.isLogicalName();
                blob = new Blob(blob.getElement(), blob.getBlobBytes(), this);
                if (isLogical)
                {
                    blob.setLogical(true);
                }
                return blob;
            }
            else
            {
                // the Directory Service is local
                return m_directory.getBlobByLogicalName(subscriber, logicalName);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get configuration: " + logicalName);
        }
    }

    /**
     * Intended for use in getting/storing runtime information
     */
    IElement getElementForUpdate(String configID)
    {
        try
        {
            if (m_directory == null)
            {
                return (IElement)dsInvoke("getElement", new Object [] { configID, Boolean.TRUE }, new String [] { String.class.getName(), Boolean.class.getName() });
            }
            else
            {
                // the Directory Service is local
                return m_directory.getElement(configID, Boolean.TRUE);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get configuration: " + configID);
        }
    }

    /**
     * Intended for automatic update of the ds.xml boot file
     */
    String exportDSBootFileString(String dsConfigElement)
    {
        try
        {
            if (m_directory == null)
            {
                return (String)dsInvoke("exportDSBootFileString", new Object [] {dsConfigElement}, new String [] { String.class.getName()});
            }
            else
            {
                // the Directory Service is local
                return m_directory.exportDSBootFileString(dsConfigElement);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to export to XML the DS configuration from: " + dsConfigElement);
        }
    }


    IElement[] getElements(String subscriber, String[] configIDs)
    {
        try
        {
            if (m_directory == null)
            {
                return (IElement[])dsInvoke("getElements", new Object [] { subscriber, configIDs }, new String [] { String.class.getName(), configIDs.getClass().getName() });
            }
            else
            {
                // the Directory Service is local
                return m_directory.getElements(subscriber, configIDs);
            }
        }
        catch(Exception e)
        {
            String configNames = "";
            for (int i = 0; i < configIDs.length; i++)
            {
                configNames += configIDs[i] + (i + 1 == configIDs.length ? "" : ",");
            }
            throw convertException(e, "Failed to get configurations: " + configNames);
        }
    }


    IBlob getFSBlob(String subscriber, String elementPath)
    {
        try
        {
            if (m_directory == null)
            {
                IBlob blob = (IBlob)dsInvoke("getFSBlob", new Object[] {elementPath, Boolean.FALSE}, new String[] {String.class.getName(), Boolean.class.getName()});
                blob = new Blob(blob.getElement(), blob.getBlobBytes(), this);
                blob.setLogical(true);
                return blob;
            }
            else
            {
                // the Directory Service is local
                return m_directory.getFSBlob(elementPath, Boolean.FALSE);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get binary FS attachment: " + elementPath);
        }
    }

    // implement IChunkedBlobStreamer

    private static final String[] GET_PARTIAL_BLOB_SIGNATURE = new String[]
    {
        String.class.getName(),
        Boolean.class.getName(),
        Integer.class.getName()
    };

    // IChunkedBlobStreamer.getBlob. This class is the streamer object for a blob only in the remote case.
    // in the direct case the blobs are returned with a local DS object which implements the streamer methods
    @Override
    public IBlob getBlob(String elementName, boolean forUpdate, int offset)
    throws DirectoryServiceException
    {
        try
        {
            IBlob blob = (IBlob)dsInvoke("getBlob", new Object[] { elementName, new Boolean(forUpdate), new Integer(offset) }, GET_PARTIAL_BLOB_SIGNATURE);
            return new Blob(blob.getElement(), blob.getBlobBytes(), this);
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get binary attachment: " + elementName);
        }
    }

    private static final String[] GET_FS_BLOB_PARTIAL_SIGNATURE = new String[]
    {
        String.class.getName(),
        Boolean.class.getName(),
        Integer.class.getName()
    };

    // IChunkedBlobStreamer.getBlob. This class is the streamer object for a blob only in the remote case.
    // in the direct case the blobs are returned with a local DS object which implements the streamer methods
    @Override
    public IBlob getFSBlob(String elementPath, boolean forUpdate, int offset)
    throws DirectoryServiceException
    {
        try
        {
            IBlob blob = (IBlob)dsInvoke("getFSBlob", new Object[] { elementPath, new Boolean(forUpdate), new Integer(offset) }, GET_FS_BLOB_PARTIAL_SIGNATURE);
            blob = new Blob(blob.getElement(), blob.getBlobBytes(), this);
            blob.setLogical(true);
            return blob;
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get binary attachment: " + elementPath);
        }
    }

    IBlob getBlob(String subscriber, String configID)
    {
        try
        {
            if (m_directory == null)
            {
                IBlob blob =(IBlob)dsInvoke("getBlob", new Object[] {subscriber, configID}, new String[] {String.class.getName(), String.class.getName()});
                blob = new Blob(blob.getElement(), blob.getBlobBytes(), this);
                return blob;
            }
            else
            {
                // the Directory Service is local
                return m_directory.getBlob(subscriber, configID);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get binary attachment: " + configID);
        }
    }

    IElement[] getAllElements(String subscriber, String dirName)
    {
        try
        {
             if (m_directory == null)
            {
                return (IElement[])dsInvoke("getAllElements", new Object [] {subscriber, dirName}, new String [] { String.class.getName(),  String.class.getName() });
            }
            else
            {
                return m_directory.getAllElements(subscriber, dirName);
            }

        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get configuration: " + dirName);
        }
    }

    Object[] reconcileCache(String containerID, Short[] versionInfo, Long myBackupVersion, IElementIdentity[] identities, HashMap deletedConfigurations, String dirNames[], HashMap logicalMap)
    {
        // this method is called during a reconcile .. this could occur for all containers during a failover, thus we
        // use special timeout for the remote call, as a reconcile under load could take longer
        try
        {
             if (m_directory == null)
            {
                return (Object[])m_connector.invoke(m_commType, m_targetDS, m_targetDS, "reconcileCache",
                                                     new Object [] { containerID, versionInfo, myBackupVersion, identities, deletedConfigurations, dirNames, logicalMap},
                                                     new String [] { String.class.getName(), Short[].class.getName(), Long.class.getName(), identities.getClass().getName(), HashMap.class.getName(), String[].class.getName(), HashMap.class.getName() },
                                                     m_connector.getRequestTimeout() << 2);
            }
            else
            {
                return m_directory.reconcileCache(containerID, versionInfo, myBackupVersion, identities, deletedConfigurations, dirNames, logicalMap);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get configuration of updated/new elements.");
        }
    }

    /**
     * Intended for use in getting/storing runtime information
     */
    INextVersionToken setElement(IBasicElement element)
    {
        while (true)
        {
            try
            {
                if (m_directory == null)
                {
                    return (INextVersionToken)dsInvoke("setElement", new Object [] { element, null }, new String [] { IBasicElement.class.getName(), IDeltaView.class.getName() });
                }
                else
                {
                    // the Directory Service is local
                    return m_directory.setElement(element, null);
                }
            }
            catch(DirectoryDoesNotExistException e)
            {
                // try creating the directories as required
                String dirname = "";
                StringTokenizer st = new StringTokenizer(element.getIdentity().getName(), "/");
                for (int i = st.countTokens(); i > 1; i--)
                {
                    dirname = dirname + '/' + st.nextToken();
                    try
                    {
                        createDirectory(dirname);
                    }
                    catch(DirectoryServiceException dse) { } // thrown when directory exists so ignore and move on
                    catch(Exception e1)
                    {
                        throw convertException(e1, "Failed to create DS directory: " + dirname);
                    }
                }
                // now we can loop and retry to set the element
            }
            catch(Exception e)
            {
                throw convertException(e, "Failed to set configuration: " + element.getIdentity().getName());
            }
        }
    }

    /**
     * Intended for use in deleting runtime information
     */
    void deleteElement(String configID)
    {
        while (true)
        {
            try
            {
                if (m_directory == null)
                {
                    dsInvoke("deleteElement", new Object [] { configID, null }, new String [] { String.class.getName(), IDeltaView.class.getName() });
                }
                else
                {
                    // the Directory Service is local
                    m_directory.deleteElement(configID, null);
                }
                break;
            }
            catch (DirectoryDoesNotExistException e)
            {
                break;
            }
            catch (DirectoryServiceClosedException closed)
            {
                break;
            }
            catch (Exception e)
            {
                throw convertException(e, "Failed to delete configuration: " + configID);
            }
        }
    }

    String storageToLogical(String storageName)
    {
        try
        {
            if (m_directory == null)
            {
                return (String)dsInvoke("storageToLogical", new Object[] {storageName}, new String[] {String.class.getName()});
            }
            else
            {
                // the Directory Service is local
                return m_directory.storageToLogical(storageName);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to get logical name for : " + storageName);
        }
    }

    private void createDirectory(String dirName)
    throws Exception
    {
        if (m_directory == null)
        {
            dsInvoke("createDirectory", new Object [] { dirName }, new String [] { String.class.getName() });
        }
        else
        {
            // the Directory Service is local
            m_directory.createDirectory(dirName);
        }
    }

    private MFRuntimeException convertException(Exception e, String msg)
    {
          // DS not started should be handled the same is DS not accessible - so we put InvokeTimeoutException inside
          if (e instanceof DSNotStartedException)
          {
              DSNotStartedException dse = (DSNotStartedException)e;
              dse.setLinkedException(new InvokeTimeoutException(e.toString()));
              throw dse;
          }

          MFRuntimeException mfe = null;
          if (e instanceof MFRuntimeException)
        {
            mfe = (MFRuntimeException)e;
        }
        else
          {
              mfe = new MFRuntimeException(msg);
              mfe.setLinkedException(e);
          }
          return mfe;
    }

    private Object dsInvoke(String operation, Object[] params, String[] signature)
    throws Exception
    {
    	return dsInvoke(operation, params, signature, true);
    }
    
    private Object dsInvoke(String operation, Object[] params, String[] signature, boolean retry)
    throws Exception
    {
        Exception priorException = null;
        while (true)
        {
            try
            {
                Object returnValue = m_connector.invoke(m_commType, m_targetDS, m_targetDS, operation, params, signature);
                if (priorException != null)
                {
                    m_connector.logMessage("...retry successful", Level.INFO);
                }
                return returnValue;
            }
            catch(Exception e)
            {
                if (m_container.isClosing())
                {
                    throw e;
                }
                if (ContainerUtil.isCausedByTimeout(e) && Thread.currentThread().getName().startsWith(IContainer.CONTAINER_BOOT_THREAD_PREFIX))
                {
                    int traceMask = m_connector.getTraceMask();
                    if ((traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                    {
                        if ((traceMask & IConnectorClient.TRACE_DETAIL) > 0)
                        {
                            m_connector.logMessage("Timeout while communicating with the Directory Service, retrying...", e, Level.TRACE);
                        }
                        else
                        {
                            m_connector.logMessage("Timeout while communicating with the Directory Service, retrying...", Level.TRACE);
                        }
                    }
                    else
                    {
                        boolean log = false;
                        if (priorException == null)
                        {
                            log = true;
                        }
                        else if (!e.getClass().isInstance(priorException))
                        {
                            log = true;
                        }
                        else if (priorException.getMessage() != null && !priorException.getMessage().equals(e.getMessage()))
                        {
                            log = true;
                        }
                        else if (e.getMessage() != null && !e.getMessage().equals(priorException.getMessage()))
                        {
                            log = true;
                        }
                        if (log)
                        {
                            m_connector.logMessage("Timeout while communicating with the Directory Service, retrying...", Level.WARNING);
                        }
                    }
                                    
                    if (retry)
                    {
                    	priorException = e;
                    }
                    else
                    {
                    	throw e;
                    }
                }
                else
                {
                    throw e;
                }
            }
        }
    }

    public void registerInterestInChanges(String subscriber, String logicalName)
    {
    	try
        {
            if (m_directory == null)
            {
                dsInvoke("registerInterestInChanges", new Object [] { subscriber, logicalName}, new String [] { String.class.getName(), String.class.getName() });
            }
            else
            {
                // the Directory Service is local
                m_directory.registerInterestInChanges(subscriber, logicalName);
            }
        }
        catch(Exception e)
        {
            throw convertException(e, "Failed to register interest in changes; container: " + subscriber + " element "+ logicalName);
        }
    }
}
