// Copyright (c) 2011 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.mf.framework.agent;

import java.io.IOException;

import javax.jms.JMSException;

import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.common.runtime.IFaultTolerantState;
import com.sonicsw.mf.common.runtime.IStateController;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.NonRecoverableStateChangeException;
import com.sonicsw.mf.common.runtime.RecoverableStateChangeException;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.util.StateManager;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

import progress.message.jclient.Connection;

class ContainerFT
extends StateManager
{
    private ContainerImpl m_container;
    private JMSConnectorServer m_containerConnector; // the container's management connector
    
    private String m_containerFTURLs;
    private String m_containerFTUser;
    private String m_containerFTPassword;
    private int m_containerFTConnectTimeout = CONNECT_TIMEOUT_DEFAULT; // seconds
    private long m_containerFTSocketConnectTimeout = SOCKET_CONNECT_TIMEOUT_DEFAULT; // milliseconds
    
    private ContainerFTConnectorClient m_containerFTConnector; // if configured, the custom connector

    private Connection m_faultDetectionConnection = null;
    private String m_pingTopicForThisContainer;
    private String m_pingTopicForPeerContainer;
    private String m_role; // primary or backup
    private StateManager m_stateManager;
    private PingThread m_pingThread;
    private int m_faultDetectionInterval = IContainerConstants.FAULT_DETECTION_INTERVAL_DEFAULT;
    private int m_faultDetectionTimeout= IContainerConstants.FAULT_DETECTION_TIMEOUT_DEFAULT;
    private boolean m_backupStartActive;

    // default timeout for connection setup
    public static final int CONNECT_TIMEOUT_DEFAULT = 10;
    // minimum timeout for connect setup
    public static final int CONNECT_TIMEOUT_MINIMUM = 10;

    // default timeout for socket connection setup (max per URL in connection URL list)
    public static final long SOCKET_CONNECT_TIMEOUT_DEFAULT = 5000;
    // minimum timeout for socket connect 
    public static final long SOCKET_CONNECT_TIMEOUT_MINIMUM = 0; // implies driven by OS TCP settings

    // TODO: Handle dynamic configuration change of the peer container
    
    ContainerFT(ContainerImpl container, JMSConnectorServer containerConnector)
    throws JMSException, InterruptedException, IOException
    {
        super(IFaultTolerantState.STATE_VALUES, IFaultTolerantState.STATE_WAITING);
        
        m_container = container;
        m_containerConnector = containerConnector;
        
        init();
    }

    String getRole()
    {
        return m_role;
    }

    boolean isActive()
    {
        return m_stateManager.getState(null) == IFaultTolerantState.STATE_ACTIVE;
    }

    short getState()
    {
        return m_stateManager.getState(null);
    }

    PingThread getPingThread()
    {
        return m_pingThread;
    }

    /**
     * General initialization
     * @throws JMSException 
     * @throws InterruptedException 
     * @throws IOException 
     */
    private void init()
    throws JMSException, InterruptedException, IOException
    {
        IContainerIdentity containerIdentity = m_container.getContainerIdentity();
        String containerConfigIndentity = containerIdentity.getConfigIdentity().getName();
        
        IElement containerConfiguration = m_container.getConfiguration(containerConfigIndentity);
        IAttributeSet containerAttrs = containerConfiguration.getAttributes();
        
        IAttributeSet containerFTAttrs = (IAttributeSet)containerAttrs.getAttribute(IContainerConstants.FAULT_TOLERANCE_PARAMETERS_ATTR);
        m_role = (String)containerFTAttrs.getAttribute(IContainerConstants.FAULT_TOLERANCE_ROLE_ATTR);
        
        Reference peerContainerRef = (Reference)containerFTAttrs.getAttribute(IContainerConstants.FAULT_TOLERANCE_PEER_REF_ATTR);
        String peerContainerConfigIndentity = peerContainerRef.getElementName();
        IElement peerContainerConfiguration = m_container.getConfiguration(peerContainerConfigIndentity);
        IAttributeSet peerContainerAttrs = peerContainerConfiguration.getAttributes();

        // Setup the JMS topic names for the pub/sub based ping comms
        m_pingTopicForThisContainer = createPingTopic(containerAttrs);
        m_pingTopicForPeerContainer = createPingTopic(peerContainerAttrs);

        setBackupStartActive((Boolean)containerFTAttrs.getAttribute(IContainerConstants.START_ACTIVE_ATTR));
        
        // The rest of the fault tolerance settings should come from the primary's configuration, so if this is the backup
        // we need to get the primary's config now
        if (m_role.equals(IContainerConstants.FAULT_TOLERANCE_ROLE_BACKUP))
        {
            containerFTAttrs = (IAttributeSet)peerContainerAttrs.getAttribute(IContainerConstants.FAULT_TOLERANCE_PARAMETERS_ATTR);
        }
        
        setFaultDetectionInterval((Integer)containerFTAttrs.getAttribute(IContainerConstants.FAULT_DETECTION_INTERVAL_ATTR));
        setFaultDetectionTimeout((Integer)containerFTAttrs.getAttribute(IContainerConstants.FAULT_DETECTION_TIMEOUT_ATTR));
        
        // log the container's FT role
        m_container.logMessage(null, IContainer.NEWLINE + IContainer.NEWLINE + '\t' + "FT container role \"" + m_role + "\"." + IContainer.NEWLINE, Level.CONFIG);

        initStateManagement();

        // extract the fault detection connection attributes
        initConnection(containerFTAttrs);
    }

    int getFaultDetectionInterval()
    {
        return m_faultDetectionInterval;
    }

    private void setFaultDetectionInterval(Integer faultDetectionInterval)
    {
        if (faultDetectionInterval == null)
        {
            m_faultDetectionInterval = IContainerConstants.FAULT_DETECTION_INTERVAL_DEFAULT;
        }
        else
        {
            m_faultDetectionInterval = faultDetectionInterval.intValue();
        }
        
        // handle dynamic changes
        if (m_pingThread != null)
        {
            m_pingThread.setPingInterval(m_faultDetectionInterval);
        }
    }

    private void setFaultDetectionTimeout(Integer faultDetectionTimeout)
    {
        if (faultDetectionTimeout == null)
        {
            m_faultDetectionTimeout = IContainerConstants.FAULT_DETECTION_TIMEOUT_DEFAULT;
        }
        else
        {
            m_faultDetectionTimeout = faultDetectionTimeout.intValue();
        }
        
        // handle dynamic changes
        if (m_pingThread != null)
        {
            m_pingThread.setPingTimeout(m_faultDetectionTimeout);
        }
    }

    private void setBackupStartActive(Boolean startActive)
    {
        if (startActive == null)
        {
            m_backupStartActive = IContainerConstants.START_ACTIVE_DEFAULT;
        }
        else
        {
            m_backupStartActive = startActive.booleanValue();
        }
    }

    private String createPingTopic(IAttributeSet containerAttrs)
    {
        String containerName = (String) containerAttrs.getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
        String domainName = (String) containerAttrs.getAttribute(IContainerConstants.DOMAIN_NAME_ATTR);

        return ConnectorClient.MF_SUBJECT_ROOT + '.' + domainName + '.' + containerName + ".ping";
    }

    /**
     * Create a state manager and register state change controllers
     */
    private void initStateManagement()
    {
        m_stateManager = new StateManager(IFaultTolerantState.STATE_VALUES, IFaultTolerantState.STATE_WAITING);
        
        m_container.logMessage(null, "FT container initial state: \"" + IFaultTolerantState.STATE_TEXT[m_stateManager.getState(null)] + '"', Level.INFO);

        // for state change reporting
        m_stateManager.registerStateListener(new ContainerFTStateListener(m_container), null);

        IStateController toActiveController = new ToActiveController();
        m_stateManager.registerStateController(toActiveController, IFaultTolerantState.STATE_WAITING, IFaultTolerantState.STATE_ACTIVE, null);
        m_stateManager.registerStateController(toActiveController, IFaultTolerantState.STATE_STANDBY, IFaultTolerantState.STATE_ACTIVE, null);

        IStateController activeToStandbyController = new ActiveToStandbyController();
        m_stateManager.registerStateController(activeToStandbyController, IFaultTolerantState.STATE_ACTIVE, IFaultTolerantState.STATE_STANDBY, null);

        IStateController waitingToStandbyController = new WaitingToStandbyController();
        m_stateManager.registerStateController(waitingToStandbyController, IFaultTolerantState.STATE_WAITING, IFaultTolerantState.STATE_STANDBY, null);
    }

    /**
     * Initializes the fault detection connection
     * @throws JMSException 
     * @throws InterruptedException 
     * @throws IOException 
     */
    private void initConnection(IAttributeSet containerFTAttrs)
    throws JMSException, InterruptedException, IOException
    {
        m_containerFTURLs = (String)containerFTAttrs.getAttribute(IContainerConstants.FAILURE_DETECTION_CONNECTIONURLS_ATTR);
        m_containerFTUser = (String)containerFTAttrs.getAttribute(IContainerConstants.FAILURE_DETECTION_DEFAULT_USER_ATTR);
        m_containerFTPassword = (String)containerFTAttrs.getAttribute(IContainerConstants.FAILURE_DETECTION_DEFAULT_PASSWORD_ATTR);
        
        // if no fault detection connection configuration was found then we default to using the containers
        // management connection for fault detection
        if (m_containerFTURLs != null)
        {
            Integer connectTimeout = (Integer)containerFTAttrs.getAttribute(IContainerConstants.FAILURE_DETECTION_CONNECT_TIMEOUT_ATTR);
            if (connectTimeout != null)
            {
                m_containerFTConnectTimeout = connectTimeout.intValue();
            }
            if (m_containerFTConnectTimeout < CONNECT_TIMEOUT_MINIMUM)
            {
                m_containerFTConnectTimeout = CONNECT_TIMEOUT_MINIMUM;
            }
            Integer socketConnectTimeout = (Integer)containerFTAttrs.getAttribute(IContainerConstants.FAILURE_DETECTION_SOCKET_CONNECT_TIMEOUT_ATTR);
            if (socketConnectTimeout != null)
            {
                m_containerFTSocketConnectTimeout = socketConnectTimeout.intValue() * 1000;
            }
            if (m_containerFTSocketConnectTimeout < SOCKET_CONNECT_TIMEOUT_MINIMUM)
            {
                m_containerFTSocketConnectTimeout = SOCKET_CONNECT_TIMEOUT_MINIMUM;
            }
    
            IContainerIdentity containerID = m_container.getContainerIdentity();
            String domainName = containerID.getDomainName();
            String containerName = containerID.getContainerName();
            m_containerFTConnector = new ContainerFTConnectorClient(this, domainName, containerName, m_role, m_containerFTURLs,
                                                                    m_containerFTUser, m_containerFTPassword,
                                                                    m_containerFTConnectTimeout, m_containerFTSocketConnectTimeout,
                                                                    m_faultDetectionTimeout);
        }
        
        setFaultDetectionConnection();
    }

    private void setFaultDetectionConnection()
    {
        try
        {
            if (m_containerFTConnector == null)
            {
                if (m_containerConnector.isConnected())
                {
                    Connection connection = (Connection)m_containerConnector.getConnection();
                    setFaultDetectionConnection(connection);
                }
            }
            else
            {
                if (m_containerFTConnector.isConnected())
                {
                    Connection connection = m_containerFTConnector.getConnection();
                    setFaultDetectionConnection(connection);
                }
            }
            // if the connector is not connected, don't set the fault detection connection at this point
            // it will get set on reconnect
        }
        catch (JMSException e)
        {
            m_container.logMessage(null, "Failed to set this FT container's fault detection connection, trace follows...", e, Level.SEVERE);
        }
    }

    private synchronized void setFaultDetectionConnection(Connection connection)
    throws JMSException
    {
        // its possible during startup that both the initialization and reconnect handler could
        // be trying to set the fault detection connection, so the combination of synchronization
        // and the check below prevents 2 ping threads from being created
        if (m_pingThread != null && m_pingThread.getConnection() == connection)
        {
            return;
        }
        
        m_faultDetectionConnection = connection;

        cleanupPingThread();

        // Create a new ping thread using the given fault detect connection.
        // Note: If the connection is container's management connection and is not connected,
        //       a JMSException is thrown for now.
        m_pingThread = new PingThread(m_faultDetectionConnection, m_pingTopicForThisContainer, m_pingTopicForPeerContainer,
                                      m_faultDetectionInterval, m_faultDetectionTimeout,
                                      m_role, m_backupStartActive, m_stateManager, this.m_container);
        m_pingThread.start();
    }

    /**
     * The fault detection connection has been reestablished
     */
    void onReconnect()
    {
        setFaultDetectionConnection();
    }

    /**
     * The fault detection connection has been disconnected
     */
    void onDisconnect()
    {
        if (m_containerFTConnector == null || !m_containerFTConnector.isConnected()) {
            cleanupPingThread();
        }
    }

    /**
     * The fault detection connection has failed
     */
    void onFailure(Exception e)
    {
        if (m_containerFTConnector == null || !m_containerFTConnector.isConnected()) {
            cleanupPingThread();
        }
    }

    private synchronized void cleanupPingThread()
    {
        if (m_pingThread != null)
        {
            //clean up old pingThread
            m_pingThread.interrupt();
            m_pingThread = null;
        }
    }

    void handleChangeFTAttrs(IDeltaAttributeSet attrs)
    throws Exception
    {
        handleModifiedFTAttrs(attrs.getNewAttributesNames(), attrs);
        handleModifiedFTAttrs(attrs.getModifiedAttributesNames(), attrs);
        handleDeletedFTAttrs(attrs.getDeletedAttributesNames());
    }

    void handleModifiedFTAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.FAULT_DETECTION_INTERVAL_ATTR))
            {
                setFaultDetectionInterval((Integer)attrs.getNewValue(names[i]));
            }
            else
            if (names[i].equals(IContainerConstants.FAULT_DETECTION_TIMEOUT_ATTR))
            {
                setFaultDetectionTimeout((Integer)attrs.getNewValue(names[i]));
            }
        }
    }

    void handleDeletedFTAttrs(String[] names)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IContainerConstants.FAULT_DETECTION_INTERVAL_ATTR))
            {
                setFaultDetectionInterval(new Integer(IContainerConstants.FAULT_DETECTION_INTERVAL_DEFAULT));
            }
            else
            if (names[i].equals(IContainerConstants.FAULT_DETECTION_TIMEOUT_ATTR))
            {
                setFaultDetectionTimeout(new Integer(IContainerConstants.FAULT_DETECTION_TIMEOUT_DEFAULT));
            }
        }
    }
    
    public void logMessage(String message, int severity)
    {
        m_container.logMessage(null, message, severity);
    }

    public void logMessage(String message, Throwable e, int severity)
    {
        m_container.logMessage(null, message, e, severity);
    }

    public int getTraceMask()
    {
        return m_container.m_agent.getTraceMask().intValue();
    }

    // Waiting to Active, Standby to Active
    private class ToActiveController
    implements IStateController
    {
        @Override
        public boolean changeState()
        throws NonRecoverableStateChangeException, RecoverableStateChangeException
        {
            ContainerFT.this.m_container.startComponents();
            return true;
        }
    }

    // Active to Standby
    private class ActiveToStandbyController
    implements IStateController
    {
        @Override
        public boolean changeState()
        {
            ContainerFT.this.m_container.stopComponents();
            return true;
        }
    }

    // Waiting to Standby
    private class WaitingToStandbyController
    implements IStateController
    {
        @Override
        public boolean changeState()
        {
            return true;
        }
    }
}
