package com.sonicsw.mf.framework.agent;

import javax.jms.JMSException;

import com.sonicsw.mf.comm.IConnectionListener;
import com.sonicsw.mf.comm.IDurableConnectorConsumer;
import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.comm.jms.DurableConnector;

import progress.message.jclient.Connection;
import progress.message.jclient.TopicConnectionFactory;

/**
 * Provides a durable connector for FT container use.
 * <p>
 * Creates an initial connection and when the connection fails, the connector will continue to try to recreate
 * the connection until it is closed.
 * <p>
 * When a connection is successfully created, the connection is reported to ContainerFT (effectively the FT
 * container ping management code).
 */
class ContainerFTConnectorClient
implements IDurableConnectorConsumer, IConnectionListener
{
    private ContainerFT m_containerFT;
    
    private TopicConnectionFactory m_connectionFactory;
    private DurableConnector m_durableConnector;
    
    private static boolean DISABLE_FT_CLIENT;
    private static String DISABLE_FT_CLIENT_PROPERTY = "sonicsw.mf.disableFTClient";

    static
    {
        String disableFTClient = System.getProperty(DISABLE_FT_CLIENT_PROPERTY, "false");
        DISABLE_FT_CLIENT = (disableFTClient != null && disableFTClient.equals("true"));
    }
    
    ContainerFTConnectorClient(ContainerFT containerFT, String domainName, String containerName, String ftRole, String urls, String user, String password, int connectTimeout, long socketConnectTimeout, long pingTimeout)
    throws JMSException, InterruptedException
    {
        m_containerFT = containerFT;
        
        initConnectionFactory(domainName, containerName, ftRole, urls, user, password, connectTimeout, socketConnectTimeout, pingTimeout);
        
        m_durableConnector = new DurableConnector(this, "Fault detection", pingTimeout);
    }

    boolean isConnected()
    {
        if (m_durableConnector == null)
        {
            return false;
        }

        try
        {
            return m_durableConnector.isConnected();
        }
        catch(NullPointerException e)
        {
            return false;
        }
    }

    Connection getConnection()
    {
        if (m_durableConnector == null)
        {
            return null;
        }

        try
        {
            return m_durableConnector.getConnection();
        }
        catch(NullPointerException e)
        {
            return null;
        }
    }

    @Override
    public int getTraceMask()
    {
        return m_containerFT.getTraceMask();
    }

    @Override
    public TopicConnectionFactory getConnectionFactory()
    {
        return m_connectionFactory;
    }

    @Override
    public IConnectionListener getConnectionListener()
    {
        return this;
    }

    @Override
    public void logMessage(String message, int severity)
    {
        m_containerFT.logMessage(message, severity);
    }

    @Override
    public void logMessage(String message, Throwable e, int severity)
    {
        m_containerFT.logMessage(message, e, severity);
    }

    @Override
    public void onReconnect(String localRoutingNode)
    {
        m_containerFT.onReconnect();
    }

    @Override
    public void onDisconnect()
    {
        m_containerFT.onDisconnect();
    }

    @Override
    public void onFailure(Exception e)
    {
        m_containerFT.onFailure(e);
    }

    private void initConnectionFactory(String domainName, String containerName, String ftRole, String urls, String user, String password,
                                       int connectTimeout, long socketConnectTimeout, long pingTimeout)
    throws JMSException
    {
        m_connectionFactory = new TopicConnectionFactory();
        
        // construct the connectID
        String connectID = ConnectorClient.MF_CONNECTID_PREFIX + domainName + '/' + containerName + '/' + ftRole;
        m_connectionFactory.setConnectID(connectID);
        
        m_connectionFactory.setConnectionURLs(urls);
        
        if (user != null)
        {
            m_connectionFactory.setDefaultUser(user);
            m_connectionFactory.setDefaultPassword(password);
        }

        // set the monitor interval (for flow control debugging) if specified as a System property....
        String monitorInterval = System.getProperty("monitorInterval");
        if (monitorInterval != null)
        {
            int val = Integer.parseInt(monitorInterval);
            if (val >= 0)
            {
                m_connectionFactory.setMonitorInterval(new Integer(val));
            }
        }

        m_connectionFactory.setLoadBalancing(true);

        m_connectionFactory.setSequential(true);
        m_connectionFactory.setPersistentDelivery(true);
        
        m_connectionFactory.setInitialConnectTimeout(new Integer(connectTimeout));
        m_connectionFactory.setSocketConnectTimeout(new Integer((int)socketConnectTimeout));
        
        // set the JMS client ping interval at the greater of 90% of the FT ping timeout or 20 seconds
        int pingInterval = (int)((pingTimeout * 9) / 10000);
        if (pingInterval < 20)
        {
            pingInterval = 20;
        }
        m_connectionFactory.setPingInterval(pingInterval);

        // force (SonicMQ, as opposed to MF) FT to be enabled for the "client"
        if (!DISABLE_FT_CLIENT)
        {
            // enable SonicMQ (as opposed to MF) Fault Tolerance for the TopicConnectionFactory
            m_connectionFactory.setFaultTolerant(Boolean.TRUE);

            // set the reconnect timeout
            m_connectionFactory.setFaultTolerantReconnectTimeout(new Integer(connectTimeout));
        }
    }
}
